/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.verifiedpermissions.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration for an identity source that represents a connection to an Amazon Cognito user pool used as an
 * identity provider for Verified Permissions.
 * </p>
 * <p>
 * This data type is used as a field that is part of the <a
 * href="https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_ConfigurationItem.html"
 * >ConfigurationItem</a> structure that is part of the response to <a
 * href="https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_ListIdentitySources.html"
 * >ListIdentitySources</a>.
 * </p>
 * <p>
 * Example:
 * <code>"CognitoUserPoolConfiguration":{"UserPoolArn":"arn:aws:cognito-idp:us-east-1:123456789012:userpool/us-east-1_1a2b3c4d5","ClientIds": ["a1b2c3d4e5f6g7h8i9j0kalbmc"],"groupConfiguration": {"groupEntityType": "MyCorp::Group"}}</code>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CognitoUserPoolConfigurationItem implements SdkPojo, Serializable,
        ToCopyableBuilder<CognitoUserPoolConfigurationItem.Builder, CognitoUserPoolConfigurationItem> {
    private static final SdkField<String> USER_POOL_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("userPoolArn").getter(getter(CognitoUserPoolConfigurationItem::userPoolArn))
            .setter(setter(Builder::userPoolArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("userPoolArn").build()).build();

    private static final SdkField<List<String>> CLIENT_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("clientIds")
            .getter(getter(CognitoUserPoolConfigurationItem::clientIds))
            .setter(setter(Builder::clientIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clientIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ISSUER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("issuer")
            .getter(getter(CognitoUserPoolConfigurationItem::issuer)).setter(setter(Builder::issuer))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("issuer").build()).build();

    private static final SdkField<CognitoGroupConfigurationItem> GROUP_CONFIGURATION_FIELD = SdkField
            .<CognitoGroupConfigurationItem> builder(MarshallingType.SDK_POJO).memberName("groupConfiguration")
            .getter(getter(CognitoUserPoolConfigurationItem::groupConfiguration)).setter(setter(Builder::groupConfiguration))
            .constructor(CognitoGroupConfigurationItem::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("groupConfiguration").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(USER_POOL_ARN_FIELD,
            CLIENT_IDS_FIELD, ISSUER_FIELD, GROUP_CONFIGURATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("userPoolArn", USER_POOL_ARN_FIELD);
                    put("clientIds", CLIENT_IDS_FIELD);
                    put("issuer", ISSUER_FIELD);
                    put("groupConfiguration", GROUP_CONFIGURATION_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String userPoolArn;

    private final List<String> clientIds;

    private final String issuer;

    private final CognitoGroupConfigurationItem groupConfiguration;

    private CognitoUserPoolConfigurationItem(BuilderImpl builder) {
        this.userPoolArn = builder.userPoolArn;
        this.clientIds = builder.clientIds;
        this.issuer = builder.issuer;
        this.groupConfiguration = builder.groupConfiguration;
    }

    /**
     * <p>
     * The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     * (ARN)</a> of the Amazon Cognito user pool that contains the identities to be authorized.
     * </p>
     * <p>
     * Example: <code>"userPoolArn": "arn:aws:cognito-idp:us-east-1:123456789012:userpool/us-east-1_1a2b3c4d5"</code>
     * </p>
     * 
     * @return The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource
     *         Name (ARN)</a> of the Amazon Cognito user pool that contains the identities to be authorized.</p>
     *         <p>
     *         Example:
     *         <code>"userPoolArn": "arn:aws:cognito-idp:us-east-1:123456789012:userpool/us-east-1_1a2b3c4d5"</code>
     */
    public final String userPoolArn() {
        return userPoolArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the ClientIds property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasClientIds() {
        return clientIds != null && !(clientIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The unique application client IDs that are associated with the specified Amazon Cognito user pool.
     * </p>
     * <p>
     * Example: <code>"clientIds": ["&amp;ExampleCogClientId;"]</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasClientIds} method.
     * </p>
     * 
     * @return The unique application client IDs that are associated with the specified Amazon Cognito user pool.</p>
     *         <p>
     *         Example: <code>"clientIds": ["&amp;ExampleCogClientId;"]</code>
     */
    public final List<String> clientIds() {
        return clientIds;
    }

    /**
     * <p>
     * The OpenID Connect (OIDC) <code>issuer</code> ID of the Amazon Cognito user pool that contains the identities to
     * be authorized.
     * </p>
     * <p>
     * Example: <code>"issuer": "https://cognito-idp.us-east-1.amazonaws.com/us-east-1_1a2b3c4d5"</code>
     * </p>
     * 
     * @return The OpenID Connect (OIDC) <code>issuer</code> ID of the Amazon Cognito user pool that contains the
     *         identities to be authorized.</p>
     *         <p>
     *         Example: <code>"issuer": "https://cognito-idp.us-east-1.amazonaws.com/us-east-1_1a2b3c4d5"</code>
     */
    public final String issuer() {
        return issuer;
    }

    /**
     * <p>
     * The type of entity that a policy store maps to groups from an Amazon Cognito user pool identity source.
     * </p>
     * 
     * @return The type of entity that a policy store maps to groups from an Amazon Cognito user pool identity source.
     */
    public final CognitoGroupConfigurationItem groupConfiguration() {
        return groupConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(userPoolArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasClientIds() ? clientIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(issuer());
        hashCode = 31 * hashCode + Objects.hashCode(groupConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CognitoUserPoolConfigurationItem)) {
            return false;
        }
        CognitoUserPoolConfigurationItem other = (CognitoUserPoolConfigurationItem) obj;
        return Objects.equals(userPoolArn(), other.userPoolArn()) && hasClientIds() == other.hasClientIds()
                && Objects.equals(clientIds(), other.clientIds()) && Objects.equals(issuer(), other.issuer())
                && Objects.equals(groupConfiguration(), other.groupConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CognitoUserPoolConfigurationItem").add("UserPoolArn", userPoolArn())
                .add("ClientIds", clientIds() == null ? null : "*** Sensitive Data Redacted ***").add("Issuer", issuer())
                .add("GroupConfiguration", groupConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "userPoolArn":
            return Optional.ofNullable(clazz.cast(userPoolArn()));
        case "clientIds":
            return Optional.ofNullable(clazz.cast(clientIds()));
        case "issuer":
            return Optional.ofNullable(clazz.cast(issuer()));
        case "groupConfiguration":
            return Optional.ofNullable(clazz.cast(groupConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<CognitoUserPoolConfigurationItem, T> g) {
        return obj -> g.apply((CognitoUserPoolConfigurationItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CognitoUserPoolConfigurationItem> {
        /**
         * <p>
         * The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
         * (ARN)</a> of the Amazon Cognito user pool that contains the identities to be authorized.
         * </p>
         * <p>
         * Example:
         * <code>"userPoolArn": "arn:aws:cognito-idp:us-east-1:123456789012:userpool/us-east-1_1a2b3c4d5"</code>
         * </p>
         * 
         * @param userPoolArn
         *        The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
         *        Resource Name (ARN)</a> of the Amazon Cognito user pool that contains the identities to be
         *        authorized.</p>
         *        <p>
         *        Example:
         *        <code>"userPoolArn": "arn:aws:cognito-idp:us-east-1:123456789012:userpool/us-east-1_1a2b3c4d5"</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userPoolArn(String userPoolArn);

        /**
         * <p>
         * The unique application client IDs that are associated with the specified Amazon Cognito user pool.
         * </p>
         * <p>
         * Example: <code>"clientIds": ["&amp;ExampleCogClientId;"]</code>
         * </p>
         * 
         * @param clientIds
         *        The unique application client IDs that are associated with the specified Amazon Cognito user pool.</p>
         *        <p>
         *        Example: <code>"clientIds": ["&amp;ExampleCogClientId;"]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientIds(Collection<String> clientIds);

        /**
         * <p>
         * The unique application client IDs that are associated with the specified Amazon Cognito user pool.
         * </p>
         * <p>
         * Example: <code>"clientIds": ["&amp;ExampleCogClientId;"]</code>
         * </p>
         * 
         * @param clientIds
         *        The unique application client IDs that are associated with the specified Amazon Cognito user pool.</p>
         *        <p>
         *        Example: <code>"clientIds": ["&amp;ExampleCogClientId;"]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientIds(String... clientIds);

        /**
         * <p>
         * The OpenID Connect (OIDC) <code>issuer</code> ID of the Amazon Cognito user pool that contains the identities
         * to be authorized.
         * </p>
         * <p>
         * Example: <code>"issuer": "https://cognito-idp.us-east-1.amazonaws.com/us-east-1_1a2b3c4d5"</code>
         * </p>
         * 
         * @param issuer
         *        The OpenID Connect (OIDC) <code>issuer</code> ID of the Amazon Cognito user pool that contains the
         *        identities to be authorized.</p>
         *        <p>
         *        Example: <code>"issuer": "https://cognito-idp.us-east-1.amazonaws.com/us-east-1_1a2b3c4d5"</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder issuer(String issuer);

        /**
         * <p>
         * The type of entity that a policy store maps to groups from an Amazon Cognito user pool identity source.
         * </p>
         * 
         * @param groupConfiguration
         *        The type of entity that a policy store maps to groups from an Amazon Cognito user pool identity
         *        source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupConfiguration(CognitoGroupConfigurationItem groupConfiguration);

        /**
         * <p>
         * The type of entity that a policy store maps to groups from an Amazon Cognito user pool identity source.
         * </p>
         * This is a convenience method that creates an instance of the {@link CognitoGroupConfigurationItem.Builder}
         * avoiding the need to create one manually via {@link CognitoGroupConfigurationItem#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CognitoGroupConfigurationItem.Builder#build()} is called
         * immediately and its result is passed to {@link #groupConfiguration(CognitoGroupConfigurationItem)}.
         * 
         * @param groupConfiguration
         *        a consumer that will call methods on {@link CognitoGroupConfigurationItem.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #groupConfiguration(CognitoGroupConfigurationItem)
         */
        default Builder groupConfiguration(Consumer<CognitoGroupConfigurationItem.Builder> groupConfiguration) {
            return groupConfiguration(CognitoGroupConfigurationItem.builder().applyMutation(groupConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String userPoolArn;

        private List<String> clientIds = DefaultSdkAutoConstructList.getInstance();

        private String issuer;

        private CognitoGroupConfigurationItem groupConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(CognitoUserPoolConfigurationItem model) {
            userPoolArn(model.userPoolArn);
            clientIds(model.clientIds);
            issuer(model.issuer);
            groupConfiguration(model.groupConfiguration);
        }

        public final String getUserPoolArn() {
            return userPoolArn;
        }

        public final void setUserPoolArn(String userPoolArn) {
            this.userPoolArn = userPoolArn;
        }

        @Override
        public final Builder userPoolArn(String userPoolArn) {
            this.userPoolArn = userPoolArn;
            return this;
        }

        public final Collection<String> getClientIds() {
            if (clientIds instanceof SdkAutoConstructList) {
                return null;
            }
            return clientIds;
        }

        public final void setClientIds(Collection<String> clientIds) {
            this.clientIds = ClientIdsCopier.copy(clientIds);
        }

        @Override
        public final Builder clientIds(Collection<String> clientIds) {
            this.clientIds = ClientIdsCopier.copy(clientIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder clientIds(String... clientIds) {
            clientIds(Arrays.asList(clientIds));
            return this;
        }

        public final String getIssuer() {
            return issuer;
        }

        public final void setIssuer(String issuer) {
            this.issuer = issuer;
        }

        @Override
        public final Builder issuer(String issuer) {
            this.issuer = issuer;
            return this;
        }

        public final CognitoGroupConfigurationItem.Builder getGroupConfiguration() {
            return groupConfiguration != null ? groupConfiguration.toBuilder() : null;
        }

        public final void setGroupConfiguration(CognitoGroupConfigurationItem.BuilderImpl groupConfiguration) {
            this.groupConfiguration = groupConfiguration != null ? groupConfiguration.build() : null;
        }

        @Override
        public final Builder groupConfiguration(CognitoGroupConfigurationItem groupConfiguration) {
            this.groupConfiguration = groupConfiguration;
            return this;
        }

        @Override
        public CognitoUserPoolConfigurationItem build() {
            return new CognitoUserPoolConfigurationItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
