/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.verifiedpermissions.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration of an OpenID Connect (OIDC) identity source for handling access token claims. Contains the claim
 * that you want to identify as the principal in an authorization request, and the values of the <code>aud</code> claim,
 * or audiences, that you want to accept.
 * </p>
 * <p>
 * This data type is part of a <a href=
 * "https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_OpenIdConnectTokenSelectionDetail.html"
 * >OpenIdConnectTokenSelectionDetail</a> structure, which is a parameter of <a
 * href="https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_GetIdentitySource.html"
 * >GetIdentitySource</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OpenIdConnectAccessTokenConfigurationDetail implements SdkPojo, Serializable,
        ToCopyableBuilder<OpenIdConnectAccessTokenConfigurationDetail.Builder, OpenIdConnectAccessTokenConfigurationDetail> {
    private static final SdkField<String> PRINCIPAL_ID_CLAIM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("principalIdClaim").getter(getter(OpenIdConnectAccessTokenConfigurationDetail::principalIdClaim))
            .setter(setter(Builder::principalIdClaim))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("principalIdClaim").build()).build();

    private static final SdkField<List<String>> AUDIENCES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("audiences")
            .getter(getter(OpenIdConnectAccessTokenConfigurationDetail::audiences))
            .setter(setter(Builder::audiences))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("audiences").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PRINCIPAL_ID_CLAIM_FIELD,
            AUDIENCES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String principalIdClaim;

    private final List<String> audiences;

    private OpenIdConnectAccessTokenConfigurationDetail(BuilderImpl builder) {
        this.principalIdClaim = builder.principalIdClaim;
        this.audiences = builder.audiences;
    }

    /**
     * <p>
     * The claim that determines the principal in OIDC access tokens. For example, <code>sub</code>.
     * </p>
     * 
     * @return The claim that determines the principal in OIDC access tokens. For example, <code>sub</code>.
     */
    public final String principalIdClaim() {
        return principalIdClaim;
    }

    /**
     * For responses, this returns true if the service returned a value for the Audiences property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAudiences() {
        return audiences != null && !(audiences instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The access token <code>aud</code> claim values that you want to accept in your policy store. For example,
     * <code>https://myapp.example.com, https://myapp2.example.com</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAudiences} method.
     * </p>
     * 
     * @return The access token <code>aud</code> claim values that you want to accept in your policy store. For example,
     *         <code>https://myapp.example.com, https://myapp2.example.com</code>.
     */
    public final List<String> audiences() {
        return audiences;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(principalIdClaim());
        hashCode = 31 * hashCode + Objects.hashCode(hasAudiences() ? audiences() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OpenIdConnectAccessTokenConfigurationDetail)) {
            return false;
        }
        OpenIdConnectAccessTokenConfigurationDetail other = (OpenIdConnectAccessTokenConfigurationDetail) obj;
        return Objects.equals(principalIdClaim(), other.principalIdClaim()) && hasAudiences() == other.hasAudiences()
                && Objects.equals(audiences(), other.audiences());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("OpenIdConnectAccessTokenConfigurationDetail")
                .add("PrincipalIdClaim", principalIdClaim() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Audiences", hasAudiences() ? audiences() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "principalIdClaim":
            return Optional.ofNullable(clazz.cast(principalIdClaim()));
        case "audiences":
            return Optional.ofNullable(clazz.cast(audiences()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("principalIdClaim", PRINCIPAL_ID_CLAIM_FIELD);
        map.put("audiences", AUDIENCES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<OpenIdConnectAccessTokenConfigurationDetail, T> g) {
        return obj -> g.apply((OpenIdConnectAccessTokenConfigurationDetail) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OpenIdConnectAccessTokenConfigurationDetail> {
        /**
         * <p>
         * The claim that determines the principal in OIDC access tokens. For example, <code>sub</code>.
         * </p>
         * 
         * @param principalIdClaim
         *        The claim that determines the principal in OIDC access tokens. For example, <code>sub</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder principalIdClaim(String principalIdClaim);

        /**
         * <p>
         * The access token <code>aud</code> claim values that you want to accept in your policy store. For example,
         * <code>https://myapp.example.com, https://myapp2.example.com</code>.
         * </p>
         * 
         * @param audiences
         *        The access token <code>aud</code> claim values that you want to accept in your policy store. For
         *        example, <code>https://myapp.example.com, https://myapp2.example.com</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder audiences(Collection<String> audiences);

        /**
         * <p>
         * The access token <code>aud</code> claim values that you want to accept in your policy store. For example,
         * <code>https://myapp.example.com, https://myapp2.example.com</code>.
         * </p>
         * 
         * @param audiences
         *        The access token <code>aud</code> claim values that you want to accept in your policy store. For
         *        example, <code>https://myapp.example.com, https://myapp2.example.com</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder audiences(String... audiences);
    }

    static final class BuilderImpl implements Builder {
        private String principalIdClaim;

        private List<String> audiences = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(OpenIdConnectAccessTokenConfigurationDetail model) {
            principalIdClaim(model.principalIdClaim);
            audiences(model.audiences);
        }

        public final String getPrincipalIdClaim() {
            return principalIdClaim;
        }

        public final void setPrincipalIdClaim(String principalIdClaim) {
            this.principalIdClaim = principalIdClaim;
        }

        @Override
        public final Builder principalIdClaim(String principalIdClaim) {
            this.principalIdClaim = principalIdClaim;
            return this;
        }

        public final Collection<String> getAudiences() {
            if (audiences instanceof SdkAutoConstructList) {
                return null;
            }
            return audiences;
        }

        public final void setAudiences(Collection<String> audiences) {
            this.audiences = AudiencesCopier.copy(audiences);
        }

        @Override
        public final Builder audiences(Collection<String> audiences) {
            this.audiences = AudiencesCopier.copy(audiences);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder audiences(String... audiences) {
            audiences(Arrays.asList(audiences));
            return this;
        }

        @Override
        public OpenIdConnectAccessTokenConfigurationDetail build() {
            return new OpenIdConnectAccessTokenConfigurationDetail(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
