/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.verifiedpermissions.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains configuration details of an OpenID Connect (OIDC) identity provider, or identity source, that Verified
 * Permissions can use to generate entities from authenticated identities. It specifies the issuer URL, token type that
 * you want to use, and policy store entity details.
 * </p>
 * <p>
 * This data type is part of a <a
 * href="https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_Configuration.html">Configuration</a>
 * structure, which is a parameter to <a
 * href="https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_CreateIdentitySource.html"
 * >CreateIdentitySource</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OpenIdConnectConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<OpenIdConnectConfiguration.Builder, OpenIdConnectConfiguration> {
    private static final SdkField<String> ISSUER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("issuer")
            .getter(getter(OpenIdConnectConfiguration::issuer)).setter(setter(Builder::issuer))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("issuer").build()).build();

    private static final SdkField<String> ENTITY_ID_PREFIX_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("entityIdPrefix").getter(getter(OpenIdConnectConfiguration::entityIdPrefix))
            .setter(setter(Builder::entityIdPrefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("entityIdPrefix").build()).build();

    private static final SdkField<OpenIdConnectGroupConfiguration> GROUP_CONFIGURATION_FIELD = SdkField
            .<OpenIdConnectGroupConfiguration> builder(MarshallingType.SDK_POJO).memberName("groupConfiguration")
            .getter(getter(OpenIdConnectConfiguration::groupConfiguration)).setter(setter(Builder::groupConfiguration))
            .constructor(OpenIdConnectGroupConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("groupConfiguration").build())
            .build();

    private static final SdkField<OpenIdConnectTokenSelection> TOKEN_SELECTION_FIELD = SdkField
            .<OpenIdConnectTokenSelection> builder(MarshallingType.SDK_POJO).memberName("tokenSelection")
            .getter(getter(OpenIdConnectConfiguration::tokenSelection)).setter(setter(Builder::tokenSelection))
            .constructor(OpenIdConnectTokenSelection::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tokenSelection").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ISSUER_FIELD,
            ENTITY_ID_PREFIX_FIELD, GROUP_CONFIGURATION_FIELD, TOKEN_SELECTION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String issuer;

    private final String entityIdPrefix;

    private final OpenIdConnectGroupConfiguration groupConfiguration;

    private final OpenIdConnectTokenSelection tokenSelection;

    private OpenIdConnectConfiguration(BuilderImpl builder) {
        this.issuer = builder.issuer;
        this.entityIdPrefix = builder.entityIdPrefix;
        this.groupConfiguration = builder.groupConfiguration;
        this.tokenSelection = builder.tokenSelection;
    }

    /**
     * <p>
     * The issuer URL of an OIDC identity provider. This URL must have an OIDC discovery endpoint at the path
     * <code>.well-known/openid-configuration</code>.
     * </p>
     * 
     * @return The issuer URL of an OIDC identity provider. This URL must have an OIDC discovery endpoint at the path
     *         <code>.well-known/openid-configuration</code>.
     */
    public final String issuer() {
        return issuer;
    }

    /**
     * <p>
     * A descriptive string that you want to prefix to user entities from your OIDC identity provider. For example, if
     * you set an <code>entityIdPrefix</code> of <code>MyOIDCProvider</code>, you can reference principals in your
     * policies in the format <code>MyCorp::User::MyOIDCProvider|Carlos</code>.
     * </p>
     * 
     * @return A descriptive string that you want to prefix to user entities from your OIDC identity provider. For
     *         example, if you set an <code>entityIdPrefix</code> of <code>MyOIDCProvider</code>, you can reference
     *         principals in your policies in the format <code>MyCorp::User::MyOIDCProvider|Carlos</code>.
     */
    public final String entityIdPrefix() {
        return entityIdPrefix;
    }

    /**
     * <p>
     * The claim in OIDC identity provider tokens that indicates a user's group membership, and the entity type that you
     * want to map it to. For example, this object can map the contents of a <code>groups</code> claim to
     * <code>MyCorp::UserGroup</code>.
     * </p>
     * 
     * @return The claim in OIDC identity provider tokens that indicates a user's group membership, and the entity type
     *         that you want to map it to. For example, this object can map the contents of a <code>groups</code> claim
     *         to <code>MyCorp::UserGroup</code>.
     */
    public final OpenIdConnectGroupConfiguration groupConfiguration() {
        return groupConfiguration;
    }

    /**
     * <p>
     * The token type that you want to process from your OIDC identity provider. Your policy store can process either
     * identity (ID) or access tokens from a given OIDC identity source.
     * </p>
     * 
     * @return The token type that you want to process from your OIDC identity provider. Your policy store can process
     *         either identity (ID) or access tokens from a given OIDC identity source.
     */
    public final OpenIdConnectTokenSelection tokenSelection() {
        return tokenSelection;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(issuer());
        hashCode = 31 * hashCode + Objects.hashCode(entityIdPrefix());
        hashCode = 31 * hashCode + Objects.hashCode(groupConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(tokenSelection());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OpenIdConnectConfiguration)) {
            return false;
        }
        OpenIdConnectConfiguration other = (OpenIdConnectConfiguration) obj;
        return Objects.equals(issuer(), other.issuer()) && Objects.equals(entityIdPrefix(), other.entityIdPrefix())
                && Objects.equals(groupConfiguration(), other.groupConfiguration())
                && Objects.equals(tokenSelection(), other.tokenSelection());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("OpenIdConnectConfiguration").add("Issuer", issuer())
                .add("EntityIdPrefix", entityIdPrefix() == null ? null : "*** Sensitive Data Redacted ***")
                .add("GroupConfiguration", groupConfiguration()).add("TokenSelection", tokenSelection()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "issuer":
            return Optional.ofNullable(clazz.cast(issuer()));
        case "entityIdPrefix":
            return Optional.ofNullable(clazz.cast(entityIdPrefix()));
        case "groupConfiguration":
            return Optional.ofNullable(clazz.cast(groupConfiguration()));
        case "tokenSelection":
            return Optional.ofNullable(clazz.cast(tokenSelection()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("issuer", ISSUER_FIELD);
        map.put("entityIdPrefix", ENTITY_ID_PREFIX_FIELD);
        map.put("groupConfiguration", GROUP_CONFIGURATION_FIELD);
        map.put("tokenSelection", TOKEN_SELECTION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<OpenIdConnectConfiguration, T> g) {
        return obj -> g.apply((OpenIdConnectConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OpenIdConnectConfiguration> {
        /**
         * <p>
         * The issuer URL of an OIDC identity provider. This URL must have an OIDC discovery endpoint at the path
         * <code>.well-known/openid-configuration</code>.
         * </p>
         * 
         * @param issuer
         *        The issuer URL of an OIDC identity provider. This URL must have an OIDC discovery endpoint at the path
         *        <code>.well-known/openid-configuration</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder issuer(String issuer);

        /**
         * <p>
         * A descriptive string that you want to prefix to user entities from your OIDC identity provider. For example,
         * if you set an <code>entityIdPrefix</code> of <code>MyOIDCProvider</code>, you can reference principals in
         * your policies in the format <code>MyCorp::User::MyOIDCProvider|Carlos</code>.
         * </p>
         * 
         * @param entityIdPrefix
         *        A descriptive string that you want to prefix to user entities from your OIDC identity provider. For
         *        example, if you set an <code>entityIdPrefix</code> of <code>MyOIDCProvider</code>, you can reference
         *        principals in your policies in the format <code>MyCorp::User::MyOIDCProvider|Carlos</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityIdPrefix(String entityIdPrefix);

        /**
         * <p>
         * The claim in OIDC identity provider tokens that indicates a user's group membership, and the entity type that
         * you want to map it to. For example, this object can map the contents of a <code>groups</code> claim to
         * <code>MyCorp::UserGroup</code>.
         * </p>
         * 
         * @param groupConfiguration
         *        The claim in OIDC identity provider tokens that indicates a user's group membership, and the entity
         *        type that you want to map it to. For example, this object can map the contents of a
         *        <code>groups</code> claim to <code>MyCorp::UserGroup</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupConfiguration(OpenIdConnectGroupConfiguration groupConfiguration);

        /**
         * <p>
         * The claim in OIDC identity provider tokens that indicates a user's group membership, and the entity type that
         * you want to map it to. For example, this object can map the contents of a <code>groups</code> claim to
         * <code>MyCorp::UserGroup</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link OpenIdConnectGroupConfiguration.Builder}
         * avoiding the need to create one manually via {@link OpenIdConnectGroupConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OpenIdConnectGroupConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #groupConfiguration(OpenIdConnectGroupConfiguration)}.
         * 
         * @param groupConfiguration
         *        a consumer that will call methods on {@link OpenIdConnectGroupConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #groupConfiguration(OpenIdConnectGroupConfiguration)
         */
        default Builder groupConfiguration(Consumer<OpenIdConnectGroupConfiguration.Builder> groupConfiguration) {
            return groupConfiguration(OpenIdConnectGroupConfiguration.builder().applyMutation(groupConfiguration).build());
        }

        /**
         * <p>
         * The token type that you want to process from your OIDC identity provider. Your policy store can process
         * either identity (ID) or access tokens from a given OIDC identity source.
         * </p>
         * 
         * @param tokenSelection
         *        The token type that you want to process from your OIDC identity provider. Your policy store can
         *        process either identity (ID) or access tokens from a given OIDC identity source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tokenSelection(OpenIdConnectTokenSelection tokenSelection);

        /**
         * <p>
         * The token type that you want to process from your OIDC identity provider. Your policy store can process
         * either identity (ID) or access tokens from a given OIDC identity source.
         * </p>
         * This is a convenience method that creates an instance of the {@link OpenIdConnectTokenSelection.Builder}
         * avoiding the need to create one manually via {@link OpenIdConnectTokenSelection#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OpenIdConnectTokenSelection.Builder#build()} is called
         * immediately and its result is passed to {@link #tokenSelection(OpenIdConnectTokenSelection)}.
         * 
         * @param tokenSelection
         *        a consumer that will call methods on {@link OpenIdConnectTokenSelection.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tokenSelection(OpenIdConnectTokenSelection)
         */
        default Builder tokenSelection(Consumer<OpenIdConnectTokenSelection.Builder> tokenSelection) {
            return tokenSelection(OpenIdConnectTokenSelection.builder().applyMutation(tokenSelection).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String issuer;

        private String entityIdPrefix;

        private OpenIdConnectGroupConfiguration groupConfiguration;

        private OpenIdConnectTokenSelection tokenSelection;

        private BuilderImpl() {
        }

        private BuilderImpl(OpenIdConnectConfiguration model) {
            issuer(model.issuer);
            entityIdPrefix(model.entityIdPrefix);
            groupConfiguration(model.groupConfiguration);
            tokenSelection(model.tokenSelection);
        }

        public final String getIssuer() {
            return issuer;
        }

        public final void setIssuer(String issuer) {
            this.issuer = issuer;
        }

        @Override
        public final Builder issuer(String issuer) {
            this.issuer = issuer;
            return this;
        }

        public final String getEntityIdPrefix() {
            return entityIdPrefix;
        }

        public final void setEntityIdPrefix(String entityIdPrefix) {
            this.entityIdPrefix = entityIdPrefix;
        }

        @Override
        public final Builder entityIdPrefix(String entityIdPrefix) {
            this.entityIdPrefix = entityIdPrefix;
            return this;
        }

        public final OpenIdConnectGroupConfiguration.Builder getGroupConfiguration() {
            return groupConfiguration != null ? groupConfiguration.toBuilder() : null;
        }

        public final void setGroupConfiguration(OpenIdConnectGroupConfiguration.BuilderImpl groupConfiguration) {
            this.groupConfiguration = groupConfiguration != null ? groupConfiguration.build() : null;
        }

        @Override
        public final Builder groupConfiguration(OpenIdConnectGroupConfiguration groupConfiguration) {
            this.groupConfiguration = groupConfiguration;
            return this;
        }

        public final OpenIdConnectTokenSelection.Builder getTokenSelection() {
            return tokenSelection != null ? tokenSelection.toBuilder() : null;
        }

        public final void setTokenSelection(OpenIdConnectTokenSelection.BuilderImpl tokenSelection) {
            this.tokenSelection = tokenSelection != null ? tokenSelection.build() : null;
        }

        @Override
        public final Builder tokenSelection(OpenIdConnectTokenSelection tokenSelection) {
            this.tokenSelection = tokenSelection;
            return this;
        }

        @Override
        public OpenIdConnectConfiguration build() {
            return new OpenIdConnectConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
