/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.voiceid;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.voiceid.model.AccessDeniedException;
import software.amazon.awssdk.services.voiceid.model.ConflictException;
import software.amazon.awssdk.services.voiceid.model.CreateDomainRequest;
import software.amazon.awssdk.services.voiceid.model.CreateDomainResponse;
import software.amazon.awssdk.services.voiceid.model.DeleteDomainRequest;
import software.amazon.awssdk.services.voiceid.model.DeleteDomainResponse;
import software.amazon.awssdk.services.voiceid.model.DeleteFraudsterRequest;
import software.amazon.awssdk.services.voiceid.model.DeleteFraudsterResponse;
import software.amazon.awssdk.services.voiceid.model.DeleteSpeakerRequest;
import software.amazon.awssdk.services.voiceid.model.DeleteSpeakerResponse;
import software.amazon.awssdk.services.voiceid.model.DescribeDomainRequest;
import software.amazon.awssdk.services.voiceid.model.DescribeDomainResponse;
import software.amazon.awssdk.services.voiceid.model.DescribeFraudsterRegistrationJobRequest;
import software.amazon.awssdk.services.voiceid.model.DescribeFraudsterRegistrationJobResponse;
import software.amazon.awssdk.services.voiceid.model.DescribeFraudsterRequest;
import software.amazon.awssdk.services.voiceid.model.DescribeFraudsterResponse;
import software.amazon.awssdk.services.voiceid.model.DescribeSpeakerEnrollmentJobRequest;
import software.amazon.awssdk.services.voiceid.model.DescribeSpeakerEnrollmentJobResponse;
import software.amazon.awssdk.services.voiceid.model.DescribeSpeakerRequest;
import software.amazon.awssdk.services.voiceid.model.DescribeSpeakerResponse;
import software.amazon.awssdk.services.voiceid.model.EvaluateSessionRequest;
import software.amazon.awssdk.services.voiceid.model.EvaluateSessionResponse;
import software.amazon.awssdk.services.voiceid.model.InternalServerException;
import software.amazon.awssdk.services.voiceid.model.ListDomainsRequest;
import software.amazon.awssdk.services.voiceid.model.ListDomainsResponse;
import software.amazon.awssdk.services.voiceid.model.ListFraudsterRegistrationJobsRequest;
import software.amazon.awssdk.services.voiceid.model.ListFraudsterRegistrationJobsResponse;
import software.amazon.awssdk.services.voiceid.model.ListSpeakerEnrollmentJobsRequest;
import software.amazon.awssdk.services.voiceid.model.ListSpeakerEnrollmentJobsResponse;
import software.amazon.awssdk.services.voiceid.model.ListSpeakersRequest;
import software.amazon.awssdk.services.voiceid.model.ListSpeakersResponse;
import software.amazon.awssdk.services.voiceid.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.voiceid.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.voiceid.model.OptOutSpeakerRequest;
import software.amazon.awssdk.services.voiceid.model.OptOutSpeakerResponse;
import software.amazon.awssdk.services.voiceid.model.ResourceNotFoundException;
import software.amazon.awssdk.services.voiceid.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.voiceid.model.StartFraudsterRegistrationJobRequest;
import software.amazon.awssdk.services.voiceid.model.StartFraudsterRegistrationJobResponse;
import software.amazon.awssdk.services.voiceid.model.StartSpeakerEnrollmentJobRequest;
import software.amazon.awssdk.services.voiceid.model.StartSpeakerEnrollmentJobResponse;
import software.amazon.awssdk.services.voiceid.model.TagResourceRequest;
import software.amazon.awssdk.services.voiceid.model.TagResourceResponse;
import software.amazon.awssdk.services.voiceid.model.ThrottlingException;
import software.amazon.awssdk.services.voiceid.model.UntagResourceRequest;
import software.amazon.awssdk.services.voiceid.model.UntagResourceResponse;
import software.amazon.awssdk.services.voiceid.model.UpdateDomainRequest;
import software.amazon.awssdk.services.voiceid.model.UpdateDomainResponse;
import software.amazon.awssdk.services.voiceid.model.ValidationException;
import software.amazon.awssdk.services.voiceid.model.VoiceIdException;
import software.amazon.awssdk.services.voiceid.model.VoiceIdRequest;
import software.amazon.awssdk.services.voiceid.paginators.ListDomainsPublisher;
import software.amazon.awssdk.services.voiceid.paginators.ListFraudsterRegistrationJobsPublisher;
import software.amazon.awssdk.services.voiceid.paginators.ListSpeakerEnrollmentJobsPublisher;
import software.amazon.awssdk.services.voiceid.paginators.ListSpeakersPublisher;
import software.amazon.awssdk.services.voiceid.transform.CreateDomainRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DeleteDomainRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DeleteFraudsterRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DeleteSpeakerRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DescribeDomainRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DescribeFraudsterRegistrationJobRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DescribeFraudsterRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DescribeSpeakerEnrollmentJobRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DescribeSpeakerRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.EvaluateSessionRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.ListDomainsRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.ListFraudsterRegistrationJobsRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.ListSpeakerEnrollmentJobsRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.ListSpeakersRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.OptOutSpeakerRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.StartFraudsterRegistrationJobRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.StartSpeakerEnrollmentJobRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.UpdateDomainRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link VoiceIdAsyncClient}.
 *
 * @see VoiceIdAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultVoiceIdAsyncClient implements VoiceIdAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultVoiceIdAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultVoiceIdAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a domain that contains all Amazon Connect Voice ID data, such as speakers, fraudsters, customer audio,
     * and voiceprints.
     * </p>
     *
     * @param createDomainRequest
     * @return A Java Future containing the result of the CreateDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded the service quota. Refer to <a
     *         href="https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html#voiceid-quotas"
     *         >Voice ID Service Quotas</a> and try your request again.</li>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>ConflictException The request failed due to a conflict. Check the <code>ConflictType</code> and error
     *         message for more details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.CreateDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/CreateDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDomainResponse> createDomain(CreateDomainRequest createDomainRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDomainRequest, CreateDomainResponse>()
                            .withOperationName("CreateDomain").withMarshaller(new CreateDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createDomainRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createDomainRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified domain from the Amazon Connect Voice ID system.
     * </p>
     *
     * @param deleteDomainRequest
     * @return A Java Future containing the result of the DeleteDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>ConflictException The request failed due to a conflict. Check the <code>ConflictType</code> and error
     *         message for more details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.DeleteDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DeleteDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDomainResponse> deleteDomain(DeleteDomainRequest deleteDomainRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDomainRequest, DeleteDomainResponse>()
                            .withOperationName("DeleteDomain").withMarshaller(new DeleteDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDomainRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteDomainRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified fraudster from the Amazon Connect Voice ID system.
     * </p>
     *
     * @param deleteFraudsterRequest
     * @return A Java Future containing the result of the DeleteFraudster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.DeleteFraudster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DeleteFraudster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFraudsterResponse> deleteFraudster(DeleteFraudsterRequest deleteFraudsterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFraudsterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFraudster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteFraudsterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteFraudsterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteFraudsterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteFraudsterRequest, DeleteFraudsterResponse>()
                            .withOperationName("DeleteFraudster")
                            .withMarshaller(new DeleteFraudsterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteFraudsterRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteFraudsterRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteFraudsterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified speaker from the Amazon Connect Voice ID system.
     * </p>
     *
     * @param deleteSpeakerRequest
     * @return A Java Future containing the result of the DeleteSpeaker operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>ConflictException The request failed due to a conflict. Check the <code>ConflictType</code> and error
     *         message for more details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.DeleteSpeaker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DeleteSpeaker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSpeakerResponse> deleteSpeaker(DeleteSpeakerRequest deleteSpeakerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSpeakerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSpeaker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSpeakerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteSpeakerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSpeakerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSpeakerRequest, DeleteSpeakerResponse>()
                            .withOperationName("DeleteSpeaker")
                            .withMarshaller(new DeleteSpeakerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteSpeakerRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteSpeakerRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteSpeakerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the specified domain.
     * </p>
     *
     * @param describeDomainRequest
     * @return A Java Future containing the result of the DescribeDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.DescribeDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DescribeDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDomainResponse> describeDomain(DescribeDomainRequest describeDomainRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDomainRequest, DescribeDomainResponse>()
                            .withOperationName("DescribeDomain")
                            .withMarshaller(new DescribeDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeDomainRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeDomainRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the specified fraudster.
     * </p>
     *
     * @param describeFraudsterRequest
     * @return A Java Future containing the result of the DescribeFraudster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.DescribeFraudster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DescribeFraudster" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFraudsterResponse> describeFraudster(DescribeFraudsterRequest describeFraudsterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFraudsterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFraudster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFraudsterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeFraudsterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeFraudsterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFraudsterRequest, DescribeFraudsterResponse>()
                            .withOperationName("DescribeFraudster")
                            .withMarshaller(new DescribeFraudsterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeFraudsterRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeFraudsterRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeFraudsterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the specified fraudster registration job.
     * </p>
     *
     * @param describeFraudsterRegistrationJobRequest
     * @return A Java Future containing the result of the DescribeFraudsterRegistrationJob operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.DescribeFraudsterRegistrationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DescribeFraudsterRegistrationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFraudsterRegistrationJobResponse> describeFraudsterRegistrationJob(
            DescribeFraudsterRegistrationJobRequest describeFraudsterRegistrationJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeFraudsterRegistrationJobRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFraudsterRegistrationJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFraudsterRegistrationJobResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeFraudsterRegistrationJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeFraudsterRegistrationJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFraudsterRegistrationJobRequest, DescribeFraudsterRegistrationJobResponse>()
                            .withOperationName("DescribeFraudsterRegistrationJob")
                            .withMarshaller(new DescribeFraudsterRegistrationJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeFraudsterRegistrationJobRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeFraudsterRegistrationJobRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<DescribeFraudsterRegistrationJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the specified speaker.
     * </p>
     *
     * @param describeSpeakerRequest
     * @return A Java Future containing the result of the DescribeSpeaker operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.DescribeSpeaker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DescribeSpeaker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSpeakerResponse> describeSpeaker(DescribeSpeakerRequest describeSpeakerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSpeakerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSpeaker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSpeakerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSpeakerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSpeakerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSpeakerRequest, DescribeSpeakerResponse>()
                            .withOperationName("DescribeSpeaker")
                            .withMarshaller(new DescribeSpeakerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeSpeakerRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeSpeakerRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeSpeakerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the specified speaker enrollment job.
     * </p>
     *
     * @param describeSpeakerEnrollmentJobRequest
     * @return A Java Future containing the result of the DescribeSpeakerEnrollmentJob operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.DescribeSpeakerEnrollmentJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DescribeSpeakerEnrollmentJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSpeakerEnrollmentJobResponse> describeSpeakerEnrollmentJob(
            DescribeSpeakerEnrollmentJobRequest describeSpeakerEnrollmentJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSpeakerEnrollmentJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSpeakerEnrollmentJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSpeakerEnrollmentJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSpeakerEnrollmentJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSpeakerEnrollmentJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSpeakerEnrollmentJobRequest, DescribeSpeakerEnrollmentJobResponse>()
                            .withOperationName("DescribeSpeakerEnrollmentJob")
                            .withMarshaller(new DescribeSpeakerEnrollmentJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeSpeakerEnrollmentJobRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeSpeakerEnrollmentJobRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeSpeakerEnrollmentJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Evaluates a specified session based on audio data accumulated during a streaming Amazon Connect Voice ID call.
     * </p>
     *
     * @param evaluateSessionRequest
     * @return A Java Future containing the result of the EvaluateSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.EvaluateSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/EvaluateSession" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<EvaluateSessionResponse> evaluateSession(EvaluateSessionRequest evaluateSessionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, evaluateSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EvaluateSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EvaluateSessionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, EvaluateSessionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EvaluateSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EvaluateSessionRequest, EvaluateSessionResponse>()
                            .withOperationName("EvaluateSession")
                            .withMarshaller(new EvaluateSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(evaluateSessionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = evaluateSessionRequest.overrideConfiguration().orElse(null);
            CompletableFuture<EvaluateSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the domains in the Amazon Web Services account.
     * </p>
     *
     * @param listDomainsRequest
     * @return A Java Future containing the result of the ListDomains operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.ListDomains
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/ListDomains" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDomainsResponse> listDomains(ListDomainsRequest listDomainsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDomainsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDomains");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDomainsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDomainsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDomainsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDomainsRequest, ListDomainsResponse>()
                            .withOperationName("ListDomains").withMarshaller(new ListDomainsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDomainsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listDomainsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListDomainsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the domains in the Amazon Web Services account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listDomains(software.amazon.awssdk.services.voiceid.model.ListDomainsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.voiceid.paginators.ListDomainsPublisher publisher = client.listDomainsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.voiceid.paginators.ListDomainsPublisher publisher = client.listDomainsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.voiceid.model.ListDomainsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.voiceid.model.ListDomainsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDomains(software.amazon.awssdk.services.voiceid.model.ListDomainsRequest)} operation.</b>
     * </p>
     *
     * @param listDomainsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.ListDomains
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/ListDomains" target="_top">AWS API
     *      Documentation</a>
     */
    public ListDomainsPublisher listDomainsPaginator(ListDomainsRequest listDomainsRequest) {
        return new ListDomainsPublisher(this, applyPaginatorUserAgent(listDomainsRequest));
    }

    /**
     * <p>
     * Lists all the fraudster registration jobs in the domain with the given <code>JobStatus</code>. If
     * <code>JobStatus</code> is not provided, this lists all fraudster registration jobs in the given domain.
     * </p>
     *
     * @param listFraudsterRegistrationJobsRequest
     * @return A Java Future containing the result of the ListFraudsterRegistrationJobs operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.ListFraudsterRegistrationJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/ListFraudsterRegistrationJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListFraudsterRegistrationJobsResponse> listFraudsterRegistrationJobs(
            ListFraudsterRegistrationJobsRequest listFraudsterRegistrationJobsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listFraudsterRegistrationJobsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFraudsterRegistrationJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFraudsterRegistrationJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListFraudsterRegistrationJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFraudsterRegistrationJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFraudsterRegistrationJobsRequest, ListFraudsterRegistrationJobsResponse>()
                            .withOperationName("ListFraudsterRegistrationJobs")
                            .withMarshaller(new ListFraudsterRegistrationJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listFraudsterRegistrationJobsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listFraudsterRegistrationJobsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListFraudsterRegistrationJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the fraudster registration jobs in the domain with the given <code>JobStatus</code>. If
     * <code>JobStatus</code> is not provided, this lists all fraudster registration jobs in the given domain.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listFraudsterRegistrationJobs(software.amazon.awssdk.services.voiceid.model.ListFraudsterRegistrationJobsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.voiceid.paginators.ListFraudsterRegistrationJobsPublisher publisher = client.listFraudsterRegistrationJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.voiceid.paginators.ListFraudsterRegistrationJobsPublisher publisher = client.listFraudsterRegistrationJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.voiceid.model.ListFraudsterRegistrationJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.voiceid.model.ListFraudsterRegistrationJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listFraudsterRegistrationJobs(software.amazon.awssdk.services.voiceid.model.ListFraudsterRegistrationJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listFraudsterRegistrationJobsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.ListFraudsterRegistrationJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/ListFraudsterRegistrationJobs"
     *      target="_top">AWS API Documentation</a>
     */
    public ListFraudsterRegistrationJobsPublisher listFraudsterRegistrationJobsPaginator(
            ListFraudsterRegistrationJobsRequest listFraudsterRegistrationJobsRequest) {
        return new ListFraudsterRegistrationJobsPublisher(this, applyPaginatorUserAgent(listFraudsterRegistrationJobsRequest));
    }

    /**
     * <p>
     * Lists all the speaker enrollment jobs in the domain with the specified <code>JobStatus</code>. If
     * <code>JobStatus</code> is not provided, this lists all jobs with all possible speaker enrollment job statuses.
     * </p>
     *
     * @param listSpeakerEnrollmentJobsRequest
     * @return A Java Future containing the result of the ListSpeakerEnrollmentJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.ListSpeakerEnrollmentJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/ListSpeakerEnrollmentJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSpeakerEnrollmentJobsResponse> listSpeakerEnrollmentJobs(
            ListSpeakerEnrollmentJobsRequest listSpeakerEnrollmentJobsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSpeakerEnrollmentJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSpeakerEnrollmentJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSpeakerEnrollmentJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSpeakerEnrollmentJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSpeakerEnrollmentJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSpeakerEnrollmentJobsRequest, ListSpeakerEnrollmentJobsResponse>()
                            .withOperationName("ListSpeakerEnrollmentJobs")
                            .withMarshaller(new ListSpeakerEnrollmentJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listSpeakerEnrollmentJobsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listSpeakerEnrollmentJobsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListSpeakerEnrollmentJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the speaker enrollment jobs in the domain with the specified <code>JobStatus</code>. If
     * <code>JobStatus</code> is not provided, this lists all jobs with all possible speaker enrollment job statuses.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSpeakerEnrollmentJobs(software.amazon.awssdk.services.voiceid.model.ListSpeakerEnrollmentJobsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.voiceid.paginators.ListSpeakerEnrollmentJobsPublisher publisher = client.listSpeakerEnrollmentJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.voiceid.paginators.ListSpeakerEnrollmentJobsPublisher publisher = client.listSpeakerEnrollmentJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.voiceid.model.ListSpeakerEnrollmentJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.voiceid.model.ListSpeakerEnrollmentJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSpeakerEnrollmentJobs(software.amazon.awssdk.services.voiceid.model.ListSpeakerEnrollmentJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSpeakerEnrollmentJobsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.ListSpeakerEnrollmentJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/ListSpeakerEnrollmentJobs"
     *      target="_top">AWS API Documentation</a>
     */
    public ListSpeakerEnrollmentJobsPublisher listSpeakerEnrollmentJobsPaginator(
            ListSpeakerEnrollmentJobsRequest listSpeakerEnrollmentJobsRequest) {
        return new ListSpeakerEnrollmentJobsPublisher(this, applyPaginatorUserAgent(listSpeakerEnrollmentJobsRequest));
    }

    /**
     * <p>
     * Lists all speakers in a specified domain.
     * </p>
     *
     * @param listSpeakersRequest
     * @return A Java Future containing the result of the ListSpeakers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.ListSpeakers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/ListSpeakers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListSpeakersResponse> listSpeakers(ListSpeakersRequest listSpeakersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSpeakersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSpeakers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSpeakersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListSpeakersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSpeakersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSpeakersRequest, ListSpeakersResponse>()
                            .withOperationName("ListSpeakers").withMarshaller(new ListSpeakersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listSpeakersRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listSpeakersRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListSpeakersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all speakers in a specified domain.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listSpeakers(software.amazon.awssdk.services.voiceid.model.ListSpeakersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.voiceid.paginators.ListSpeakersPublisher publisher = client.listSpeakersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.voiceid.paginators.ListSpeakersPublisher publisher = client.listSpeakersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.voiceid.model.ListSpeakersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.voiceid.model.ListSpeakersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSpeakers(software.amazon.awssdk.services.voiceid.model.ListSpeakersRequest)} operation.</b>
     * </p>
     *
     * @param listSpeakersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.ListSpeakers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/ListSpeakers" target="_top">AWS API
     *      Documentation</a>
     */
    public ListSpeakersPublisher listSpeakersPaginator(ListSpeakersRequest listSpeakersRequest) {
        return new ListSpeakersPublisher(this, applyPaginatorUserAgent(listSpeakersRequest));
    }

    /**
     * <p>
     * Lists all tags associated with a specified Voice ID resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsForResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Opts out a speaker from Voice ID system. A speaker can be opted out regardless of whether or not they already
     * exist in the system. If they don't yet exist, a new speaker is created in an opted out state. If they already
     * exist, their existing status is overridden and they are opted out. Enrollment and evaluation authentication
     * requests are rejected for opted out speakers, and opted out speakers have no voice embeddings stored in the
     * system.
     * </p>
     *
     * @param optOutSpeakerRequest
     * @return A Java Future containing the result of the OptOutSpeaker operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>ConflictException The request failed due to a conflict. Check the <code>ConflictType</code> and error
     *         message for more details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.OptOutSpeaker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/OptOutSpeaker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<OptOutSpeakerResponse> optOutSpeaker(OptOutSpeakerRequest optOutSpeakerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, optOutSpeakerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "OptOutSpeaker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<OptOutSpeakerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    OptOutSpeakerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<OptOutSpeakerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<OptOutSpeakerRequest, OptOutSpeakerResponse>()
                            .withOperationName("OptOutSpeaker")
                            .withMarshaller(new OptOutSpeakerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(optOutSpeakerRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = optOutSpeakerRequest.overrideConfiguration().orElse(null);
            CompletableFuture<OptOutSpeakerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a new batch fraudster registration job using provided details.
     * </p>
     *
     * @param startFraudsterRegistrationJobRequest
     * @return A Java Future containing the result of the StartFraudsterRegistrationJob operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded the service quota. Refer to <a
     *         href="https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html#voiceid-quotas"
     *         >Voice ID Service Quotas</a> and try your request again.</li>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>ConflictException The request failed due to a conflict. Check the <code>ConflictType</code> and error
     *         message for more details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.StartFraudsterRegistrationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/StartFraudsterRegistrationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartFraudsterRegistrationJobResponse> startFraudsterRegistrationJob(
            StartFraudsterRegistrationJobRequest startFraudsterRegistrationJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                startFraudsterRegistrationJobRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartFraudsterRegistrationJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartFraudsterRegistrationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartFraudsterRegistrationJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartFraudsterRegistrationJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartFraudsterRegistrationJobRequest, StartFraudsterRegistrationJobResponse>()
                            .withOperationName("StartFraudsterRegistrationJob")
                            .withMarshaller(new StartFraudsterRegistrationJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startFraudsterRegistrationJobRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = startFraudsterRegistrationJobRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<StartFraudsterRegistrationJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a new batch speaker enrollment job using specified details.
     * </p>
     *
     * @param startSpeakerEnrollmentJobRequest
     * @return A Java Future containing the result of the StartSpeakerEnrollmentJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded the service quota. Refer to <a
     *         href="https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html#voiceid-quotas"
     *         >Voice ID Service Quotas</a> and try your request again.</li>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>ConflictException The request failed due to a conflict. Check the <code>ConflictType</code> and error
     *         message for more details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.StartSpeakerEnrollmentJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/StartSpeakerEnrollmentJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartSpeakerEnrollmentJobResponse> startSpeakerEnrollmentJob(
            StartSpeakerEnrollmentJobRequest startSpeakerEnrollmentJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startSpeakerEnrollmentJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartSpeakerEnrollmentJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartSpeakerEnrollmentJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartSpeakerEnrollmentJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartSpeakerEnrollmentJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartSpeakerEnrollmentJobRequest, StartSpeakerEnrollmentJobResponse>()
                            .withOperationName("StartSpeakerEnrollmentJob")
                            .withMarshaller(new StartSpeakerEnrollmentJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startSpeakerEnrollmentJobRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = startSpeakerEnrollmentJobRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<StartSpeakerEnrollmentJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Tags an Amazon Connect Voice ID resource with the provided list of tags.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>ConflictException The request failed due to a conflict. Check the <code>ConflictType</code> and error
     *         message for more details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = tagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes specified tags from a specified Amazon Connect Voice ID resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>ConflictException The request failed due to a conflict. Check the <code>ConflictType</code> and error
     *         message for more details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = untagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified domain. This API has clobber behavior, and clears and replaces all attributes. If an
     * optional field, such as 'Description' is not provided, it is removed from the domain.
     * </p>
     *
     * @param updateDomainRequest
     * @return A Java Future containing the result of the UpdateDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>ConflictException The request failed due to a conflict. Check the <code>ConflictType</code> and error
     *         message for more details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.UpdateDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/UpdateDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDomainResponse> updateDomain(UpdateDomainRequest updateDomainRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDomainRequest, UpdateDomainResponse>()
                            .withOperationName("UpdateDomain").withMarshaller(new UpdateDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateDomainRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateDomainRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(VoiceIdException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends VoiceIdRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
