/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.voiceid.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The fraud detection result produced by Voice ID, processed against the current session state and streamed audio of
 * the speaker.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FraudDetectionResult implements SdkPojo, Serializable,
        ToCopyableBuilder<FraudDetectionResult.Builder, FraudDetectionResult> {
    private static final SdkField<Instant> AUDIO_AGGREGATION_ENDED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("AudioAggregationEndedAt").getter(getter(FraudDetectionResult::audioAggregationEndedAt))
            .setter(setter(Builder::audioAggregationEndedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AudioAggregationEndedAt").build())
            .build();

    private static final SdkField<Instant> AUDIO_AGGREGATION_STARTED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT).memberName("AudioAggregationStartedAt")
            .getter(getter(FraudDetectionResult::audioAggregationStartedAt)).setter(setter(Builder::audioAggregationStartedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AudioAggregationStartedAt").build())
            .build();

    private static final SdkField<FraudDetectionConfiguration> CONFIGURATION_FIELD = SdkField
            .<FraudDetectionConfiguration> builder(MarshallingType.SDK_POJO).memberName("Configuration")
            .getter(getter(FraudDetectionResult::configuration)).setter(setter(Builder::configuration))
            .constructor(FraudDetectionConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Configuration").build()).build();

    private static final SdkField<String> DECISION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Decision").getter(getter(FraudDetectionResult::decisionAsString)).setter(setter(Builder::decision))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Decision").build()).build();

    private static final SdkField<String> FRAUD_DETECTION_RESULT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FraudDetectionResultId").getter(getter(FraudDetectionResult::fraudDetectionResultId))
            .setter(setter(Builder::fraudDetectionResultId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FraudDetectionResultId").build())
            .build();

    private static final SdkField<List<String>> REASONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Reasons")
            .getter(getter(FraudDetectionResult::reasonsAsStrings))
            .setter(setter(Builder::reasonsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Reasons").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<FraudRiskDetails> RISK_DETAILS_FIELD = SdkField
            .<FraudRiskDetails> builder(MarshallingType.SDK_POJO).memberName("RiskDetails")
            .getter(getter(FraudDetectionResult::riskDetails)).setter(setter(Builder::riskDetails))
            .constructor(FraudRiskDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RiskDetails").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            AUDIO_AGGREGATION_ENDED_AT_FIELD, AUDIO_AGGREGATION_STARTED_AT_FIELD, CONFIGURATION_FIELD, DECISION_FIELD,
            FRAUD_DETECTION_RESULT_ID_FIELD, REASONS_FIELD, RISK_DETAILS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Instant audioAggregationEndedAt;

    private final Instant audioAggregationStartedAt;

    private final FraudDetectionConfiguration configuration;

    private final String decision;

    private final String fraudDetectionResultId;

    private final List<String> reasons;

    private final FraudRiskDetails riskDetails;

    private FraudDetectionResult(BuilderImpl builder) {
        this.audioAggregationEndedAt = builder.audioAggregationEndedAt;
        this.audioAggregationStartedAt = builder.audioAggregationStartedAt;
        this.configuration = builder.configuration;
        this.decision = builder.decision;
        this.fraudDetectionResultId = builder.fraudDetectionResultId;
        this.reasons = builder.reasons;
        this.riskDetails = builder.riskDetails;
    }

    /**
     * <p>
     * A timestamp indicating when audio aggregation ended for this fraud detection result.
     * </p>
     * 
     * @return A timestamp indicating when audio aggregation ended for this fraud detection result.
     */
    public final Instant audioAggregationEndedAt() {
        return audioAggregationEndedAt;
    }

    /**
     * <p>
     * A timestamp indicating when audio aggregation started for this fraud detection result.
     * </p>
     * 
     * @return A timestamp indicating when audio aggregation started for this fraud detection result.
     */
    public final Instant audioAggregationStartedAt() {
        return audioAggregationStartedAt;
    }

    /**
     * <p>
     * The <code>FraudDetectionConfiguration</code> used to generate this fraud detection result.
     * </p>
     * 
     * @return The <code>FraudDetectionConfiguration</code> used to generate this fraud detection result.
     */
    public final FraudDetectionConfiguration configuration() {
        return configuration;
    }

    /**
     * <p>
     * The fraud detection decision produced by Voice ID, processed against the current session state and streamed audio
     * of the speaker.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #decision} will
     * return {@link FraudDetectionDecision#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #decisionAsString}.
     * </p>
     * 
     * @return The fraud detection decision produced by Voice ID, processed against the current session state and
     *         streamed audio of the speaker.
     * @see FraudDetectionDecision
     */
    public final FraudDetectionDecision decision() {
        return FraudDetectionDecision.fromValue(decision);
    }

    /**
     * <p>
     * The fraud detection decision produced by Voice ID, processed against the current session state and streamed audio
     * of the speaker.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #decision} will
     * return {@link FraudDetectionDecision#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #decisionAsString}.
     * </p>
     * 
     * @return The fraud detection decision produced by Voice ID, processed against the current session state and
     *         streamed audio of the speaker.
     * @see FraudDetectionDecision
     */
    public final String decisionAsString() {
        return decision;
    }

    /**
     * <p>
     * The unique identifier for this fraud detection result. Given there can be multiple fraud detections for a given
     * session, this field helps in identifying if the returned result is from previous streaming activity or a new
     * result. Note that in the absence of any new streaming activity or risk threshold changes, Voice ID always returns
     * cached Fraud Detection result for this API.
     * </p>
     * 
     * @return The unique identifier for this fraud detection result. Given there can be multiple fraud detections for a
     *         given session, this field helps in identifying if the returned result is from previous streaming activity
     *         or a new result. Note that in the absence of any new streaming activity or risk threshold changes, Voice
     *         ID always returns cached Fraud Detection result for this API.
     */
    public final String fraudDetectionResultId() {
        return fraudDetectionResultId;
    }

    /**
     * <p>
     * The reason speaker was flagged by the fraud detection system. This is only be populated if fraud detection
     * Decision is <code>HIGH_RISK</code>, and only has one possible value: <code>KNOWN_FRAUDSTER</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasReasons} method.
     * </p>
     * 
     * @return The reason speaker was flagged by the fraud detection system. This is only be populated if fraud
     *         detection Decision is <code>HIGH_RISK</code>, and only has one possible value:
     *         <code>KNOWN_FRAUDSTER</code>.
     */
    public final List<FraudDetectionReason> reasons() {
        return FraudDetectionReasonsCopier.copyStringToEnum(reasons);
    }

    /**
     * For responses, this returns true if the service returned a value for the Reasons property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasReasons() {
        return reasons != null && !(reasons instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The reason speaker was flagged by the fraud detection system. This is only be populated if fraud detection
     * Decision is <code>HIGH_RISK</code>, and only has one possible value: <code>KNOWN_FRAUDSTER</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasReasons} method.
     * </p>
     * 
     * @return The reason speaker was flagged by the fraud detection system. This is only be populated if fraud
     *         detection Decision is <code>HIGH_RISK</code>, and only has one possible value:
     *         <code>KNOWN_FRAUDSTER</code>.
     */
    public final List<String> reasonsAsStrings() {
        return reasons;
    }

    /**
     * <p>
     * Details about each risk analyzed for this speaker.
     * </p>
     * 
     * @return Details about each risk analyzed for this speaker.
     */
    public final FraudRiskDetails riskDetails() {
        return riskDetails;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(audioAggregationEndedAt());
        hashCode = 31 * hashCode + Objects.hashCode(audioAggregationStartedAt());
        hashCode = 31 * hashCode + Objects.hashCode(configuration());
        hashCode = 31 * hashCode + Objects.hashCode(decisionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(fraudDetectionResultId());
        hashCode = 31 * hashCode + Objects.hashCode(hasReasons() ? reasonsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(riskDetails());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FraudDetectionResult)) {
            return false;
        }
        FraudDetectionResult other = (FraudDetectionResult) obj;
        return Objects.equals(audioAggregationEndedAt(), other.audioAggregationEndedAt())
                && Objects.equals(audioAggregationStartedAt(), other.audioAggregationStartedAt())
                && Objects.equals(configuration(), other.configuration())
                && Objects.equals(decisionAsString(), other.decisionAsString())
                && Objects.equals(fraudDetectionResultId(), other.fraudDetectionResultId()) && hasReasons() == other.hasReasons()
                && Objects.equals(reasonsAsStrings(), other.reasonsAsStrings())
                && Objects.equals(riskDetails(), other.riskDetails());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FraudDetectionResult").add("AudioAggregationEndedAt", audioAggregationEndedAt())
                .add("AudioAggregationStartedAt", audioAggregationStartedAt()).add("Configuration", configuration())
                .add("Decision", decisionAsString()).add("FraudDetectionResultId", fraudDetectionResultId())
                .add("Reasons", hasReasons() ? reasonsAsStrings() : null).add("RiskDetails", riskDetails()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AudioAggregationEndedAt":
            return Optional.ofNullable(clazz.cast(audioAggregationEndedAt()));
        case "AudioAggregationStartedAt":
            return Optional.ofNullable(clazz.cast(audioAggregationStartedAt()));
        case "Configuration":
            return Optional.ofNullable(clazz.cast(configuration()));
        case "Decision":
            return Optional.ofNullable(clazz.cast(decisionAsString()));
        case "FraudDetectionResultId":
            return Optional.ofNullable(clazz.cast(fraudDetectionResultId()));
        case "Reasons":
            return Optional.ofNullable(clazz.cast(reasonsAsStrings()));
        case "RiskDetails":
            return Optional.ofNullable(clazz.cast(riskDetails()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<FraudDetectionResult, T> g) {
        return obj -> g.apply((FraudDetectionResult) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FraudDetectionResult> {
        /**
         * <p>
         * A timestamp indicating when audio aggregation ended for this fraud detection result.
         * </p>
         * 
         * @param audioAggregationEndedAt
         *        A timestamp indicating when audio aggregation ended for this fraud detection result.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder audioAggregationEndedAt(Instant audioAggregationEndedAt);

        /**
         * <p>
         * A timestamp indicating when audio aggregation started for this fraud detection result.
         * </p>
         * 
         * @param audioAggregationStartedAt
         *        A timestamp indicating when audio aggregation started for this fraud detection result.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder audioAggregationStartedAt(Instant audioAggregationStartedAt);

        /**
         * <p>
         * The <code>FraudDetectionConfiguration</code> used to generate this fraud detection result.
         * </p>
         * 
         * @param configuration
         *        The <code>FraudDetectionConfiguration</code> used to generate this fraud detection result.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configuration(FraudDetectionConfiguration configuration);

        /**
         * <p>
         * The <code>FraudDetectionConfiguration</code> used to generate this fraud detection result.
         * </p>
         * This is a convenience that creates an instance of the {@link FraudDetectionConfiguration.Builder} avoiding
         * the need to create one manually via {@link FraudDetectionConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link FraudDetectionConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #configuration(FraudDetectionConfiguration)}.
         * 
         * @param configuration
         *        a consumer that will call methods on {@link FraudDetectionConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #configuration(FraudDetectionConfiguration)
         */
        default Builder configuration(Consumer<FraudDetectionConfiguration.Builder> configuration) {
            return configuration(FraudDetectionConfiguration.builder().applyMutation(configuration).build());
        }

        /**
         * <p>
         * The fraud detection decision produced by Voice ID, processed against the current session state and streamed
         * audio of the speaker.
         * </p>
         * 
         * @param decision
         *        The fraud detection decision produced by Voice ID, processed against the current session state and
         *        streamed audio of the speaker.
         * @see FraudDetectionDecision
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FraudDetectionDecision
         */
        Builder decision(String decision);

        /**
         * <p>
         * The fraud detection decision produced by Voice ID, processed against the current session state and streamed
         * audio of the speaker.
         * </p>
         * 
         * @param decision
         *        The fraud detection decision produced by Voice ID, processed against the current session state and
         *        streamed audio of the speaker.
         * @see FraudDetectionDecision
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FraudDetectionDecision
         */
        Builder decision(FraudDetectionDecision decision);

        /**
         * <p>
         * The unique identifier for this fraud detection result. Given there can be multiple fraud detections for a
         * given session, this field helps in identifying if the returned result is from previous streaming activity or
         * a new result. Note that in the absence of any new streaming activity or risk threshold changes, Voice ID
         * always returns cached Fraud Detection result for this API.
         * </p>
         * 
         * @param fraudDetectionResultId
         *        The unique identifier for this fraud detection result. Given there can be multiple fraud detections
         *        for a given session, this field helps in identifying if the returned result is from previous streaming
         *        activity or a new result. Note that in the absence of any new streaming activity or risk threshold
         *        changes, Voice ID always returns cached Fraud Detection result for this API.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fraudDetectionResultId(String fraudDetectionResultId);

        /**
         * <p>
         * The reason speaker was flagged by the fraud detection system. This is only be populated if fraud detection
         * Decision is <code>HIGH_RISK</code>, and only has one possible value: <code>KNOWN_FRAUDSTER</code>.
         * </p>
         * 
         * @param reasons
         *        The reason speaker was flagged by the fraud detection system. This is only be populated if fraud
         *        detection Decision is <code>HIGH_RISK</code>, and only has one possible value:
         *        <code>KNOWN_FRAUDSTER</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reasonsWithStrings(Collection<String> reasons);

        /**
         * <p>
         * The reason speaker was flagged by the fraud detection system. This is only be populated if fraud detection
         * Decision is <code>HIGH_RISK</code>, and only has one possible value: <code>KNOWN_FRAUDSTER</code>.
         * </p>
         * 
         * @param reasons
         *        The reason speaker was flagged by the fraud detection system. This is only be populated if fraud
         *        detection Decision is <code>HIGH_RISK</code>, and only has one possible value:
         *        <code>KNOWN_FRAUDSTER</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reasonsWithStrings(String... reasons);

        /**
         * <p>
         * The reason speaker was flagged by the fraud detection system. This is only be populated if fraud detection
         * Decision is <code>HIGH_RISK</code>, and only has one possible value: <code>KNOWN_FRAUDSTER</code>.
         * </p>
         * 
         * @param reasons
         *        The reason speaker was flagged by the fraud detection system. This is only be populated if fraud
         *        detection Decision is <code>HIGH_RISK</code>, and only has one possible value:
         *        <code>KNOWN_FRAUDSTER</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reasons(Collection<FraudDetectionReason> reasons);

        /**
         * <p>
         * The reason speaker was flagged by the fraud detection system. This is only be populated if fraud detection
         * Decision is <code>HIGH_RISK</code>, and only has one possible value: <code>KNOWN_FRAUDSTER</code>.
         * </p>
         * 
         * @param reasons
         *        The reason speaker was flagged by the fraud detection system. This is only be populated if fraud
         *        detection Decision is <code>HIGH_RISK</code>, and only has one possible value:
         *        <code>KNOWN_FRAUDSTER</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reasons(FraudDetectionReason... reasons);

        /**
         * <p>
         * Details about each risk analyzed for this speaker.
         * </p>
         * 
         * @param riskDetails
         *        Details about each risk analyzed for this speaker.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder riskDetails(FraudRiskDetails riskDetails);

        /**
         * <p>
         * Details about each risk analyzed for this speaker.
         * </p>
         * This is a convenience that creates an instance of the {@link FraudRiskDetails.Builder} avoiding the need to
         * create one manually via {@link FraudRiskDetails#builder()}.
         *
         * When the {@link Consumer} completes, {@link FraudRiskDetails.Builder#build()} is called immediately and its
         * result is passed to {@link #riskDetails(FraudRiskDetails)}.
         * 
         * @param riskDetails
         *        a consumer that will call methods on {@link FraudRiskDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #riskDetails(FraudRiskDetails)
         */
        default Builder riskDetails(Consumer<FraudRiskDetails.Builder> riskDetails) {
            return riskDetails(FraudRiskDetails.builder().applyMutation(riskDetails).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Instant audioAggregationEndedAt;

        private Instant audioAggregationStartedAt;

        private FraudDetectionConfiguration configuration;

        private String decision;

        private String fraudDetectionResultId;

        private List<String> reasons = DefaultSdkAutoConstructList.getInstance();

        private FraudRiskDetails riskDetails;

        private BuilderImpl() {
        }

        private BuilderImpl(FraudDetectionResult model) {
            audioAggregationEndedAt(model.audioAggregationEndedAt);
            audioAggregationStartedAt(model.audioAggregationStartedAt);
            configuration(model.configuration);
            decision(model.decision);
            fraudDetectionResultId(model.fraudDetectionResultId);
            reasonsWithStrings(model.reasons);
            riskDetails(model.riskDetails);
        }

        public final Instant getAudioAggregationEndedAt() {
            return audioAggregationEndedAt;
        }

        public final void setAudioAggregationEndedAt(Instant audioAggregationEndedAt) {
            this.audioAggregationEndedAt = audioAggregationEndedAt;
        }

        @Override
        @Transient
        public final Builder audioAggregationEndedAt(Instant audioAggregationEndedAt) {
            this.audioAggregationEndedAt = audioAggregationEndedAt;
            return this;
        }

        public final Instant getAudioAggregationStartedAt() {
            return audioAggregationStartedAt;
        }

        public final void setAudioAggregationStartedAt(Instant audioAggregationStartedAt) {
            this.audioAggregationStartedAt = audioAggregationStartedAt;
        }

        @Override
        @Transient
        public final Builder audioAggregationStartedAt(Instant audioAggregationStartedAt) {
            this.audioAggregationStartedAt = audioAggregationStartedAt;
            return this;
        }

        public final FraudDetectionConfiguration.Builder getConfiguration() {
            return configuration != null ? configuration.toBuilder() : null;
        }

        public final void setConfiguration(FraudDetectionConfiguration.BuilderImpl configuration) {
            this.configuration = configuration != null ? configuration.build() : null;
        }

        @Override
        @Transient
        public final Builder configuration(FraudDetectionConfiguration configuration) {
            this.configuration = configuration;
            return this;
        }

        public final String getDecision() {
            return decision;
        }

        public final void setDecision(String decision) {
            this.decision = decision;
        }

        @Override
        @Transient
        public final Builder decision(String decision) {
            this.decision = decision;
            return this;
        }

        @Override
        @Transient
        public final Builder decision(FraudDetectionDecision decision) {
            this.decision(decision == null ? null : decision.toString());
            return this;
        }

        public final String getFraudDetectionResultId() {
            return fraudDetectionResultId;
        }

        public final void setFraudDetectionResultId(String fraudDetectionResultId) {
            this.fraudDetectionResultId = fraudDetectionResultId;
        }

        @Override
        @Transient
        public final Builder fraudDetectionResultId(String fraudDetectionResultId) {
            this.fraudDetectionResultId = fraudDetectionResultId;
            return this;
        }

        public final Collection<String> getReasons() {
            if (reasons instanceof SdkAutoConstructList) {
                return null;
            }
            return reasons;
        }

        public final void setReasons(Collection<String> reasons) {
            this.reasons = FraudDetectionReasonsCopier.copy(reasons);
        }

        @Override
        @Transient
        public final Builder reasonsWithStrings(Collection<String> reasons) {
            this.reasons = FraudDetectionReasonsCopier.copy(reasons);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder reasonsWithStrings(String... reasons) {
            reasonsWithStrings(Arrays.asList(reasons));
            return this;
        }

        @Override
        @Transient
        public final Builder reasons(Collection<FraudDetectionReason> reasons) {
            this.reasons = FraudDetectionReasonsCopier.copyEnumToString(reasons);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder reasons(FraudDetectionReason... reasons) {
            reasons(Arrays.asList(reasons));
            return this;
        }

        public final FraudRiskDetails.Builder getRiskDetails() {
            return riskDetails != null ? riskDetails.toBuilder() : null;
        }

        public final void setRiskDetails(FraudRiskDetails.BuilderImpl riskDetails) {
            this.riskDetails = riskDetails != null ? riskDetails.build() : null;
        }

        @Override
        @Transient
        public final Builder riskDetails(FraudRiskDetails riskDetails) {
            this.riskDetails = riskDetails;
            return this;
        }

        @Override
        public FraudDetectionResult build() {
            return new FraudDetectionResult(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
