/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.voiceid;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.voiceid.model.AccessDeniedException;
import software.amazon.awssdk.services.voiceid.model.AssociateFraudsterRequest;
import software.amazon.awssdk.services.voiceid.model.AssociateFraudsterResponse;
import software.amazon.awssdk.services.voiceid.model.ConflictException;
import software.amazon.awssdk.services.voiceid.model.CreateDomainRequest;
import software.amazon.awssdk.services.voiceid.model.CreateDomainResponse;
import software.amazon.awssdk.services.voiceid.model.CreateWatchlistRequest;
import software.amazon.awssdk.services.voiceid.model.CreateWatchlistResponse;
import software.amazon.awssdk.services.voiceid.model.DeleteDomainRequest;
import software.amazon.awssdk.services.voiceid.model.DeleteDomainResponse;
import software.amazon.awssdk.services.voiceid.model.DeleteFraudsterRequest;
import software.amazon.awssdk.services.voiceid.model.DeleteFraudsterResponse;
import software.amazon.awssdk.services.voiceid.model.DeleteSpeakerRequest;
import software.amazon.awssdk.services.voiceid.model.DeleteSpeakerResponse;
import software.amazon.awssdk.services.voiceid.model.DeleteWatchlistRequest;
import software.amazon.awssdk.services.voiceid.model.DeleteWatchlistResponse;
import software.amazon.awssdk.services.voiceid.model.DescribeDomainRequest;
import software.amazon.awssdk.services.voiceid.model.DescribeDomainResponse;
import software.amazon.awssdk.services.voiceid.model.DescribeFraudsterRegistrationJobRequest;
import software.amazon.awssdk.services.voiceid.model.DescribeFraudsterRegistrationJobResponse;
import software.amazon.awssdk.services.voiceid.model.DescribeFraudsterRequest;
import software.amazon.awssdk.services.voiceid.model.DescribeFraudsterResponse;
import software.amazon.awssdk.services.voiceid.model.DescribeSpeakerEnrollmentJobRequest;
import software.amazon.awssdk.services.voiceid.model.DescribeSpeakerEnrollmentJobResponse;
import software.amazon.awssdk.services.voiceid.model.DescribeSpeakerRequest;
import software.amazon.awssdk.services.voiceid.model.DescribeSpeakerResponse;
import software.amazon.awssdk.services.voiceid.model.DescribeWatchlistRequest;
import software.amazon.awssdk.services.voiceid.model.DescribeWatchlistResponse;
import software.amazon.awssdk.services.voiceid.model.DisassociateFraudsterRequest;
import software.amazon.awssdk.services.voiceid.model.DisassociateFraudsterResponse;
import software.amazon.awssdk.services.voiceid.model.EvaluateSessionRequest;
import software.amazon.awssdk.services.voiceid.model.EvaluateSessionResponse;
import software.amazon.awssdk.services.voiceid.model.InternalServerException;
import software.amazon.awssdk.services.voiceid.model.ListDomainsRequest;
import software.amazon.awssdk.services.voiceid.model.ListDomainsResponse;
import software.amazon.awssdk.services.voiceid.model.ListFraudsterRegistrationJobsRequest;
import software.amazon.awssdk.services.voiceid.model.ListFraudsterRegistrationJobsResponse;
import software.amazon.awssdk.services.voiceid.model.ListFraudstersRequest;
import software.amazon.awssdk.services.voiceid.model.ListFraudstersResponse;
import software.amazon.awssdk.services.voiceid.model.ListSpeakerEnrollmentJobsRequest;
import software.amazon.awssdk.services.voiceid.model.ListSpeakerEnrollmentJobsResponse;
import software.amazon.awssdk.services.voiceid.model.ListSpeakersRequest;
import software.amazon.awssdk.services.voiceid.model.ListSpeakersResponse;
import software.amazon.awssdk.services.voiceid.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.voiceid.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.voiceid.model.ListWatchlistsRequest;
import software.amazon.awssdk.services.voiceid.model.ListWatchlistsResponse;
import software.amazon.awssdk.services.voiceid.model.OptOutSpeakerRequest;
import software.amazon.awssdk.services.voiceid.model.OptOutSpeakerResponse;
import software.amazon.awssdk.services.voiceid.model.ResourceNotFoundException;
import software.amazon.awssdk.services.voiceid.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.voiceid.model.StartFraudsterRegistrationJobRequest;
import software.amazon.awssdk.services.voiceid.model.StartFraudsterRegistrationJobResponse;
import software.amazon.awssdk.services.voiceid.model.StartSpeakerEnrollmentJobRequest;
import software.amazon.awssdk.services.voiceid.model.StartSpeakerEnrollmentJobResponse;
import software.amazon.awssdk.services.voiceid.model.TagResourceRequest;
import software.amazon.awssdk.services.voiceid.model.TagResourceResponse;
import software.amazon.awssdk.services.voiceid.model.ThrottlingException;
import software.amazon.awssdk.services.voiceid.model.UntagResourceRequest;
import software.amazon.awssdk.services.voiceid.model.UntagResourceResponse;
import software.amazon.awssdk.services.voiceid.model.UpdateDomainRequest;
import software.amazon.awssdk.services.voiceid.model.UpdateDomainResponse;
import software.amazon.awssdk.services.voiceid.model.UpdateWatchlistRequest;
import software.amazon.awssdk.services.voiceid.model.UpdateWatchlistResponse;
import software.amazon.awssdk.services.voiceid.model.ValidationException;
import software.amazon.awssdk.services.voiceid.model.VoiceIdException;
import software.amazon.awssdk.services.voiceid.transform.AssociateFraudsterRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.CreateDomainRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.CreateWatchlistRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DeleteDomainRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DeleteFraudsterRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DeleteSpeakerRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DeleteWatchlistRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DescribeDomainRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DescribeFraudsterRegistrationJobRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DescribeFraudsterRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DescribeSpeakerEnrollmentJobRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DescribeSpeakerRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DescribeWatchlistRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DisassociateFraudsterRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.EvaluateSessionRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.ListDomainsRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.ListFraudsterRegistrationJobsRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.ListFraudstersRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.ListSpeakerEnrollmentJobsRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.ListSpeakersRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.ListWatchlistsRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.OptOutSpeakerRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.StartFraudsterRegistrationJobRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.StartSpeakerEnrollmentJobRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.UpdateDomainRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.UpdateWatchlistRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link VoiceIdClient}.
 *
 * @see VoiceIdClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultVoiceIdClient implements VoiceIdClient {
    private static final Logger log = Logger.loggerFor(DefaultVoiceIdClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final VoiceIdServiceClientConfiguration serviceClientConfiguration;

    protected DefaultVoiceIdClient(VoiceIdServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Associates the fraudsters with the watchlist specified in the same domain.
     * </p>
     *
     * @param associateFraudsterRequest
     * @return Result of the AssociateFraudster operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeded the service quota. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html#voiceid-quotas"
     *         >Voice ID Service Quotas</a> and try your request again.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws ConflictException
     *         The request failed due to a conflict. Check the <code>ConflictType</code> and error message for more
     *         details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.AssociateFraudster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/AssociateFraudster" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public AssociateFraudsterResponse associateFraudster(AssociateFraudsterRequest associateFraudsterRequest)
            throws ServiceQuotaExceededException, ResourceNotFoundException, ValidationException, ConflictException,
            InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateFraudsterResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateFraudsterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateFraudsterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateFraudster");

            return clientHandler.execute(new ClientExecutionParams<AssociateFraudsterRequest, AssociateFraudsterResponse>()
                    .withOperationName("AssociateFraudster").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(associateFraudsterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AssociateFraudsterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a domain that contains all Amazon Connect Voice ID data, such as speakers, fraudsters, customer audio,
     * and voiceprints. Every domain is created with a default watchlist that fraudsters can be a part of.
     * </p>
     *
     * @param createDomainRequest
     * @return Result of the CreateDomain operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeded the service quota. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html#voiceid-quotas"
     *         >Voice ID Service Quotas</a> and try your request again.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws ConflictException
     *         The request failed due to a conflict. Check the <code>ConflictType</code> and error message for more
     *         details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.CreateDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/CreateDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateDomainResponse createDomain(CreateDomainRequest createDomainRequest) throws ServiceQuotaExceededException,
            ResourceNotFoundException, ValidationException, ConflictException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDomain");

            return clientHandler.execute(new ClientExecutionParams<CreateDomainRequest, CreateDomainResponse>()
                    .withOperationName("CreateDomain").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createDomainRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateDomainRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a watchlist that fraudsters can be a part of.
     * </p>
     *
     * @param createWatchlistRequest
     * @return Result of the CreateWatchlist operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeded the service quota. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html#voiceid-quotas"
     *         >Voice ID Service Quotas</a> and try your request again.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws ConflictException
     *         The request failed due to a conflict. Check the <code>ConflictType</code> and error message for more
     *         details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.CreateWatchlist
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/CreateWatchlist" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateWatchlistResponse createWatchlist(CreateWatchlistRequest createWatchlistRequest)
            throws ServiceQuotaExceededException, ResourceNotFoundException, ValidationException, ConflictException,
            InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateWatchlistResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateWatchlistResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWatchlistRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWatchlist");

            return clientHandler.execute(new ClientExecutionParams<CreateWatchlistRequest, CreateWatchlistResponse>()
                    .withOperationName("CreateWatchlist").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createWatchlistRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateWatchlistRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified domain from Voice ID.
     * </p>
     *
     * @param deleteDomainRequest
     * @return Result of the DeleteDomain operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws ConflictException
     *         The request failed due to a conflict. Check the <code>ConflictType</code> and error message for more
     *         details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.DeleteDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DeleteDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteDomainResponse deleteDomain(DeleteDomainRequest deleteDomainRequest) throws ResourceNotFoundException,
            ValidationException, ConflictException, InternalServerException, ThrottlingException, AccessDeniedException,
            AwsServiceException, SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDomain");

            return clientHandler.execute(new ClientExecutionParams<DeleteDomainRequest, DeleteDomainResponse>()
                    .withOperationName("DeleteDomain").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteDomainRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteDomainRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified fraudster from Voice ID. This action disassociates the fraudster from any watchlists it is
     * a part of.
     * </p>
     *
     * @param deleteFraudsterRequest
     * @return Result of the DeleteFraudster operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws ConflictException
     *         The request failed due to a conflict. Check the <code>ConflictType</code> and error message for more
     *         details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.DeleteFraudster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DeleteFraudster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteFraudsterResponse deleteFraudster(DeleteFraudsterRequest deleteFraudsterRequest)
            throws ResourceNotFoundException, ValidationException, ConflictException, InternalServerException,
            ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteFraudsterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteFraudsterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFraudsterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFraudster");

            return clientHandler.execute(new ClientExecutionParams<DeleteFraudsterRequest, DeleteFraudsterResponse>()
                    .withOperationName("DeleteFraudster").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteFraudsterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteFraudsterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified speaker from Voice ID.
     * </p>
     *
     * @param deleteSpeakerRequest
     * @return Result of the DeleteSpeaker operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws ConflictException
     *         The request failed due to a conflict. Check the <code>ConflictType</code> and error message for more
     *         details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.DeleteSpeaker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DeleteSpeaker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteSpeakerResponse deleteSpeaker(DeleteSpeakerRequest deleteSpeakerRequest) throws ResourceNotFoundException,
            ValidationException, ConflictException, InternalServerException, ThrottlingException, AccessDeniedException,
            AwsServiceException, SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSpeakerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteSpeakerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSpeakerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSpeaker");

            return clientHandler.execute(new ClientExecutionParams<DeleteSpeakerRequest, DeleteSpeakerResponse>()
                    .withOperationName("DeleteSpeaker").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteSpeakerRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteSpeakerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified watchlist from Voice ID. This API throws an exception when there are fraudsters in the
     * watchlist that you are trying to delete. You must delete the fraudsters, and then delete the watchlist. Every
     * domain has a default watchlist which cannot be deleted.
     * </p>
     *
     * @param deleteWatchlistRequest
     * @return Result of the DeleteWatchlist operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws ConflictException
     *         The request failed due to a conflict. Check the <code>ConflictType</code> and error message for more
     *         details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.DeleteWatchlist
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DeleteWatchlist" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteWatchlistResponse deleteWatchlist(DeleteWatchlistRequest deleteWatchlistRequest)
            throws ResourceNotFoundException, ValidationException, ConflictException, InternalServerException,
            ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteWatchlistResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteWatchlistResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWatchlistRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWatchlist");

            return clientHandler.execute(new ClientExecutionParams<DeleteWatchlistRequest, DeleteWatchlistResponse>()
                    .withOperationName("DeleteWatchlist").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteWatchlistRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteWatchlistRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the specified domain.
     * </p>
     *
     * @param describeDomainRequest
     * @return Result of the DescribeDomain operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.DescribeDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DescribeDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeDomainResponse describeDomain(DescribeDomainRequest describeDomainRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDomain");

            return clientHandler.execute(new ClientExecutionParams<DescribeDomainRequest, DescribeDomainResponse>()
                    .withOperationName("DescribeDomain").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeDomainRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeDomainRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the specified fraudster.
     * </p>
     *
     * @param describeFraudsterRequest
     * @return Result of the DescribeFraudster operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.DescribeFraudster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DescribeFraudster" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeFraudsterResponse describeFraudster(DescribeFraudsterRequest describeFraudsterRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeFraudsterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeFraudsterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFraudsterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFraudster");

            return clientHandler.execute(new ClientExecutionParams<DescribeFraudsterRequest, DescribeFraudsterResponse>()
                    .withOperationName("DescribeFraudster").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeFraudsterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeFraudsterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the specified fraudster registration job.
     * </p>
     *
     * @param describeFraudsterRegistrationJobRequest
     * @return Result of the DescribeFraudsterRegistrationJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.DescribeFraudsterRegistrationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DescribeFraudsterRegistrationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeFraudsterRegistrationJobResponse describeFraudsterRegistrationJob(
            DescribeFraudsterRegistrationJobRequest describeFraudsterRegistrationJobRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeFraudsterRegistrationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeFraudsterRegistrationJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeFraudsterRegistrationJobRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFraudsterRegistrationJob");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeFraudsterRegistrationJobRequest, DescribeFraudsterRegistrationJobResponse>()
                            .withOperationName("DescribeFraudsterRegistrationJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeFraudsterRegistrationJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeFraudsterRegistrationJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the specified speaker.
     * </p>
     *
     * @param describeSpeakerRequest
     * @return Result of the DescribeSpeaker operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.DescribeSpeaker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DescribeSpeaker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeSpeakerResponse describeSpeaker(DescribeSpeakerRequest describeSpeakerRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeSpeakerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeSpeakerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSpeakerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSpeaker");

            return clientHandler.execute(new ClientExecutionParams<DescribeSpeakerRequest, DescribeSpeakerResponse>()
                    .withOperationName("DescribeSpeaker").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeSpeakerRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeSpeakerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the specified speaker enrollment job.
     * </p>
     *
     * @param describeSpeakerEnrollmentJobRequest
     * @return Result of the DescribeSpeakerEnrollmentJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.DescribeSpeakerEnrollmentJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DescribeSpeakerEnrollmentJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeSpeakerEnrollmentJobResponse describeSpeakerEnrollmentJob(
            DescribeSpeakerEnrollmentJobRequest describeSpeakerEnrollmentJobRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeSpeakerEnrollmentJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeSpeakerEnrollmentJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSpeakerEnrollmentJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSpeakerEnrollmentJob");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeSpeakerEnrollmentJobRequest, DescribeSpeakerEnrollmentJobResponse>()
                            .withOperationName("DescribeSpeakerEnrollmentJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeSpeakerEnrollmentJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeSpeakerEnrollmentJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the specified watchlist.
     * </p>
     *
     * @param describeWatchlistRequest
     * @return Result of the DescribeWatchlist operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.DescribeWatchlist
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DescribeWatchlist" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeWatchlistResponse describeWatchlist(DescribeWatchlistRequest describeWatchlistRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeWatchlistResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeWatchlistResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeWatchlistRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeWatchlist");

            return clientHandler.execute(new ClientExecutionParams<DescribeWatchlistRequest, DescribeWatchlistResponse>()
                    .withOperationName("DescribeWatchlist").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeWatchlistRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeWatchlistRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disassociates the fraudsters from the watchlist specified. Voice ID always expects a fraudster to be a part of at
     * least one watchlist. If you try to disassociate a fraudster from its only watchlist, a
     * <code>ValidationException</code> is thrown.
     * </p>
     *
     * @param disassociateFraudsterRequest
     * @return Result of the DisassociateFraudster operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws ConflictException
     *         The request failed due to a conflict. Check the <code>ConflictType</code> and error message for more
     *         details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.DisassociateFraudster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DisassociateFraudster"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateFraudsterResponse disassociateFraudster(DisassociateFraudsterRequest disassociateFraudsterRequest)
            throws ResourceNotFoundException, ValidationException, ConflictException, InternalServerException,
            ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateFraudsterResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateFraudsterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateFraudsterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateFraudster");

            return clientHandler.execute(new ClientExecutionParams<DisassociateFraudsterRequest, DisassociateFraudsterResponse>()
                    .withOperationName("DisassociateFraudster").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(disassociateFraudsterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DisassociateFraudsterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Evaluates a specified session based on audio data accumulated during a streaming Amazon Connect Voice ID call.
     * </p>
     *
     * @param evaluateSessionRequest
     * @return Result of the EvaluateSession operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws ConflictException
     *         The request failed due to a conflict. Check the <code>ConflictType</code> and error message for more
     *         details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.EvaluateSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/EvaluateSession" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public EvaluateSessionResponse evaluateSession(EvaluateSessionRequest evaluateSessionRequest)
            throws ResourceNotFoundException, ValidationException, ConflictException, InternalServerException,
            ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EvaluateSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                EvaluateSessionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, evaluateSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EvaluateSession");

            return clientHandler.execute(new ClientExecutionParams<EvaluateSessionRequest, EvaluateSessionResponse>()
                    .withOperationName("EvaluateSession").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(evaluateSessionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new EvaluateSessionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the domains in the Amazon Web Services account.
     * </p>
     *
     * @param listDomainsRequest
     * @return Result of the ListDomains operation returned by the service.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.ListDomains
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/ListDomains" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDomainsResponse listDomains(ListDomainsRequest listDomainsRequest) throws ValidationException,
            InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDomainsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDomainsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDomainsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDomains");

            return clientHandler.execute(new ClientExecutionParams<ListDomainsRequest, ListDomainsResponse>()
                    .withOperationName("ListDomains").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listDomainsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDomainsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the fraudster registration jobs in the domain with the given <code>JobStatus</code>. If
     * <code>JobStatus</code> is not provided, this lists all fraudster registration jobs in the given domain.
     * </p>
     *
     * @param listFraudsterRegistrationJobsRequest
     * @return Result of the ListFraudsterRegistrationJobs operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.ListFraudsterRegistrationJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/ListFraudsterRegistrationJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListFraudsterRegistrationJobsResponse listFraudsterRegistrationJobs(
            ListFraudsterRegistrationJobsRequest listFraudsterRegistrationJobsRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListFraudsterRegistrationJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListFraudsterRegistrationJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listFraudsterRegistrationJobsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFraudsterRegistrationJobs");

            return clientHandler
                    .execute(new ClientExecutionParams<ListFraudsterRegistrationJobsRequest, ListFraudsterRegistrationJobsResponse>()
                            .withOperationName("ListFraudsterRegistrationJobs").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listFraudsterRegistrationJobsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListFraudsterRegistrationJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all fraudsters in a specified watchlist or domain.
     * </p>
     *
     * @param listFraudstersRequest
     * @return Result of the ListFraudsters operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.ListFraudsters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/ListFraudsters" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListFraudstersResponse listFraudsters(ListFraudstersRequest listFraudstersRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListFraudstersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListFraudstersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFraudstersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFraudsters");

            return clientHandler.execute(new ClientExecutionParams<ListFraudstersRequest, ListFraudstersResponse>()
                    .withOperationName("ListFraudsters").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listFraudstersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListFraudstersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the speaker enrollment jobs in the domain with the specified <code>JobStatus</code>. If
     * <code>JobStatus</code> is not provided, this lists all jobs with all possible speaker enrollment job statuses.
     * </p>
     *
     * @param listSpeakerEnrollmentJobsRequest
     * @return Result of the ListSpeakerEnrollmentJobs operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.ListSpeakerEnrollmentJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/ListSpeakerEnrollmentJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSpeakerEnrollmentJobsResponse listSpeakerEnrollmentJobs(
            ListSpeakerEnrollmentJobsRequest listSpeakerEnrollmentJobsRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSpeakerEnrollmentJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSpeakerEnrollmentJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSpeakerEnrollmentJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSpeakerEnrollmentJobs");

            return clientHandler
                    .execute(new ClientExecutionParams<ListSpeakerEnrollmentJobsRequest, ListSpeakerEnrollmentJobsResponse>()
                            .withOperationName("ListSpeakerEnrollmentJobs").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listSpeakerEnrollmentJobsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListSpeakerEnrollmentJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all speakers in a specified domain.
     * </p>
     *
     * @param listSpeakersRequest
     * @return Result of the ListSpeakers operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.ListSpeakers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/ListSpeakers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListSpeakersResponse listSpeakers(ListSpeakersRequest listSpeakersRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSpeakersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListSpeakersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSpeakersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSpeakers");

            return clientHandler.execute(new ClientExecutionParams<ListSpeakersRequest, ListSpeakersResponse>()
                    .withOperationName("ListSpeakers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listSpeakersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListSpeakersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all tags associated with a specified Voice ID resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all watchlists in a specified domain.
     * </p>
     *
     * @param listWatchlistsRequest
     * @return Result of the ListWatchlists operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.ListWatchlists
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/ListWatchlists" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListWatchlistsResponse listWatchlists(ListWatchlistsRequest listWatchlistsRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListWatchlistsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListWatchlistsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWatchlistsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWatchlists");

            return clientHandler.execute(new ClientExecutionParams<ListWatchlistsRequest, ListWatchlistsResponse>()
                    .withOperationName("ListWatchlists").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listWatchlistsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListWatchlistsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Opts out a speaker from Voice ID. A speaker can be opted out regardless of whether or not they already exist in
     * Voice ID. If they don't yet exist, a new speaker is created in an opted out state. If they already exist, their
     * existing status is overridden and they are opted out. Enrollment and evaluation authentication requests are
     * rejected for opted out speakers, and opted out speakers have no voice embeddings stored in Voice ID.
     * </p>
     *
     * @param optOutSpeakerRequest
     * @return Result of the OptOutSpeaker operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeded the service quota. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html#voiceid-quotas"
     *         >Voice ID Service Quotas</a> and try your request again.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws ConflictException
     *         The request failed due to a conflict. Check the <code>ConflictType</code> and error message for more
     *         details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.OptOutSpeaker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/OptOutSpeaker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public OptOutSpeakerResponse optOutSpeaker(OptOutSpeakerRequest optOutSpeakerRequest) throws ServiceQuotaExceededException,
            ResourceNotFoundException, ValidationException, ConflictException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<OptOutSpeakerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                OptOutSpeakerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, optOutSpeakerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "OptOutSpeaker");

            return clientHandler.execute(new ClientExecutionParams<OptOutSpeakerRequest, OptOutSpeakerResponse>()
                    .withOperationName("OptOutSpeaker").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(optOutSpeakerRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new OptOutSpeakerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a new batch fraudster registration job using provided details.
     * </p>
     *
     * @param startFraudsterRegistrationJobRequest
     * @return Result of the StartFraudsterRegistrationJob operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeded the service quota. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html#voiceid-quotas"
     *         >Voice ID Service Quotas</a> and try your request again.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws ConflictException
     *         The request failed due to a conflict. Check the <code>ConflictType</code> and error message for more
     *         details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.StartFraudsterRegistrationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/StartFraudsterRegistrationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartFraudsterRegistrationJobResponse startFraudsterRegistrationJob(
            StartFraudsterRegistrationJobRequest startFraudsterRegistrationJobRequest) throws ServiceQuotaExceededException,
            ResourceNotFoundException, ValidationException, ConflictException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartFraudsterRegistrationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartFraudsterRegistrationJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                startFraudsterRegistrationJobRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartFraudsterRegistrationJob");

            return clientHandler
                    .execute(new ClientExecutionParams<StartFraudsterRegistrationJobRequest, StartFraudsterRegistrationJobResponse>()
                            .withOperationName("StartFraudsterRegistrationJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(startFraudsterRegistrationJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartFraudsterRegistrationJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a new batch speaker enrollment job using specified details.
     * </p>
     *
     * @param startSpeakerEnrollmentJobRequest
     * @return Result of the StartSpeakerEnrollmentJob operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The request exceeded the service quota. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html#voiceid-quotas"
     *         >Voice ID Service Quotas</a> and try your request again.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws ConflictException
     *         The request failed due to a conflict. Check the <code>ConflictType</code> and error message for more
     *         details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.StartSpeakerEnrollmentJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/StartSpeakerEnrollmentJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartSpeakerEnrollmentJobResponse startSpeakerEnrollmentJob(
            StartSpeakerEnrollmentJobRequest startSpeakerEnrollmentJobRequest) throws ServiceQuotaExceededException,
            ResourceNotFoundException, ValidationException, ConflictException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartSpeakerEnrollmentJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartSpeakerEnrollmentJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startSpeakerEnrollmentJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartSpeakerEnrollmentJob");

            return clientHandler
                    .execute(new ClientExecutionParams<StartSpeakerEnrollmentJobRequest, StartSpeakerEnrollmentJobResponse>()
                            .withOperationName("StartSpeakerEnrollmentJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(startSpeakerEnrollmentJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartSpeakerEnrollmentJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Tags a Voice ID resource with the provided list of tags.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws ConflictException
     *         The request failed due to a conflict. Check the <code>ConflictType</code> and error message for more
     *         details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            ValidationException, ConflictException, InternalServerException, ThrottlingException, AccessDeniedException,
            AwsServiceException, SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes specified tags from a specified Amazon Connect Voice ID resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws ConflictException
     *         The request failed due to a conflict. Check the <code>ConflictType</code> and error message for more
     *         details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            ValidationException, ConflictException, InternalServerException, ThrottlingException, AccessDeniedException,
            AwsServiceException, SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the specified domain. This API has clobber behavior, and clears and replaces all attributes. If an
     * optional field, such as 'Description' is not provided, it is removed from the domain.
     * </p>
     *
     * @param updateDomainRequest
     * @return Result of the UpdateDomain operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws ConflictException
     *         The request failed due to a conflict. Check the <code>ConflictType</code> and error message for more
     *         details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.UpdateDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/UpdateDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateDomainResponse updateDomain(UpdateDomainRequest updateDomainRequest) throws ResourceNotFoundException,
            ValidationException, ConflictException, InternalServerException, ThrottlingException, AccessDeniedException,
            AwsServiceException, SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDomain");

            return clientHandler.execute(new ClientExecutionParams<UpdateDomainRequest, UpdateDomainResponse>()
                    .withOperationName("UpdateDomain").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateDomainRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateDomainRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the specified watchlist. Every domain has a default watchlist which cannot be updated.
     * </p>
     *
     * @param updateWatchlistRequest
     * @return Result of the UpdateWatchlist operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found. Check the <code>ResourceType</code> and error message for more
     *         details.
     * @throws ValidationException
     *         The request failed one or more validations; check the error message for more details.
     * @throws ConflictException
     *         The request failed due to a conflict. Check the <code>ConflictType</code> and error message for more
     *         details.
     * @throws InternalServerException
     *         The request failed due to an unknown error on the server side.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Please slow down your request rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Check the error message and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws VoiceIdException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample VoiceIdClient.UpdateWatchlist
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/UpdateWatchlist" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateWatchlistResponse updateWatchlist(UpdateWatchlistRequest updateWatchlistRequest)
            throws ResourceNotFoundException, ValidationException, ConflictException, InternalServerException,
            ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException, VoiceIdException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateWatchlistResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateWatchlistResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateWatchlistRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateWatchlist");

            return clientHandler.execute(new ClientExecutionParams<UpdateWatchlistRequest, UpdateWatchlistResponse>()
                    .withOperationName("UpdateWatchlist").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateWatchlistRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateWatchlistRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(VoiceIdException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    @Override
    public final VoiceIdServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
