/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.voiceid.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The fraud detection configuration to be used during the batch speaker enrollment job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EnrollmentJobFraudDetectionConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<EnrollmentJobFraudDetectionConfig.Builder, EnrollmentJobFraudDetectionConfig> {
    private static final SdkField<String> FRAUD_DETECTION_ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FraudDetectionAction").getter(getter(EnrollmentJobFraudDetectionConfig::fraudDetectionActionAsString))
            .setter(setter(Builder::fraudDetectionAction))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FraudDetectionAction").build())
            .build();

    private static final SdkField<Integer> RISK_THRESHOLD_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("RiskThreshold").getter(getter(EnrollmentJobFraudDetectionConfig::riskThreshold))
            .setter(setter(Builder::riskThreshold))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RiskThreshold").build()).build();

    private static final SdkField<List<String>> WATCHLIST_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("WatchlistIds")
            .getter(getter(EnrollmentJobFraudDetectionConfig::watchlistIds))
            .setter(setter(Builder::watchlistIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WatchlistIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FRAUD_DETECTION_ACTION_FIELD,
            RISK_THRESHOLD_FIELD, WATCHLIST_IDS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String fraudDetectionAction;

    private final Integer riskThreshold;

    private final List<String> watchlistIds;

    private EnrollmentJobFraudDetectionConfig(BuilderImpl builder) {
        this.fraudDetectionAction = builder.fraudDetectionAction;
        this.riskThreshold = builder.riskThreshold;
        this.watchlistIds = builder.watchlistIds;
    }

    /**
     * <p>
     * The action to take when the given speaker is flagged by the fraud detection system. The default value is
     * <code>FAIL</code>, which fails the speaker enrollment. Changing this value to <code>IGNORE</code> results in the
     * speaker being enrolled even if they are flagged by the fraud detection system.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #fraudDetectionAction} will return {@link FraudDetectionAction#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #fraudDetectionActionAsString}.
     * </p>
     * 
     * @return The action to take when the given speaker is flagged by the fraud detection system. The default value is
     *         <code>FAIL</code>, which fails the speaker enrollment. Changing this value to <code>IGNORE</code> results
     *         in the speaker being enrolled even if they are flagged by the fraud detection system.
     * @see FraudDetectionAction
     */
    public final FraudDetectionAction fraudDetectionAction() {
        return FraudDetectionAction.fromValue(fraudDetectionAction);
    }

    /**
     * <p>
     * The action to take when the given speaker is flagged by the fraud detection system. The default value is
     * <code>FAIL</code>, which fails the speaker enrollment. Changing this value to <code>IGNORE</code> results in the
     * speaker being enrolled even if they are flagged by the fraud detection system.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #fraudDetectionAction} will return {@link FraudDetectionAction#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #fraudDetectionActionAsString}.
     * </p>
     * 
     * @return The action to take when the given speaker is flagged by the fraud detection system. The default value is
     *         <code>FAIL</code>, which fails the speaker enrollment. Changing this value to <code>IGNORE</code> results
     *         in the speaker being enrolled even if they are flagged by the fraud detection system.
     * @see FraudDetectionAction
     */
    public final String fraudDetectionActionAsString() {
        return fraudDetectionAction;
    }

    /**
     * <p>
     * Threshold value for determining whether the speaker is a high risk to be fraudulent. If the detected risk score
     * calculated by Voice ID is greater than or equal to the threshold, the speaker is considered a fraudster.
     * </p>
     * 
     * @return Threshold value for determining whether the speaker is a high risk to be fraudulent. If the detected risk
     *         score calculated by Voice ID is greater than or equal to the threshold, the speaker is considered a
     *         fraudster.
     */
    public final Integer riskThreshold() {
        return riskThreshold;
    }

    /**
     * For responses, this returns true if the service returned a value for the WatchlistIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasWatchlistIds() {
        return watchlistIds != null && !(watchlistIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The identifier of watchlists against which fraud detection is performed.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasWatchlistIds} method.
     * </p>
     * 
     * @return The identifier of watchlists against which fraud detection is performed.
     */
    public final List<String> watchlistIds() {
        return watchlistIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(fraudDetectionActionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(riskThreshold());
        hashCode = 31 * hashCode + Objects.hashCode(hasWatchlistIds() ? watchlistIds() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EnrollmentJobFraudDetectionConfig)) {
            return false;
        }
        EnrollmentJobFraudDetectionConfig other = (EnrollmentJobFraudDetectionConfig) obj;
        return Objects.equals(fraudDetectionActionAsString(), other.fraudDetectionActionAsString())
                && Objects.equals(riskThreshold(), other.riskThreshold()) && hasWatchlistIds() == other.hasWatchlistIds()
                && Objects.equals(watchlistIds(), other.watchlistIds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EnrollmentJobFraudDetectionConfig").add("FraudDetectionAction", fraudDetectionActionAsString())
                .add("RiskThreshold", riskThreshold()).add("WatchlistIds", hasWatchlistIds() ? watchlistIds() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FraudDetectionAction":
            return Optional.ofNullable(clazz.cast(fraudDetectionActionAsString()));
        case "RiskThreshold":
            return Optional.ofNullable(clazz.cast(riskThreshold()));
        case "WatchlistIds":
            return Optional.ofNullable(clazz.cast(watchlistIds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("FraudDetectionAction", FRAUD_DETECTION_ACTION_FIELD);
        map.put("RiskThreshold", RISK_THRESHOLD_FIELD);
        map.put("WatchlistIds", WATCHLIST_IDS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<EnrollmentJobFraudDetectionConfig, T> g) {
        return obj -> g.apply((EnrollmentJobFraudDetectionConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EnrollmentJobFraudDetectionConfig> {
        /**
         * <p>
         * The action to take when the given speaker is flagged by the fraud detection system. The default value is
         * <code>FAIL</code>, which fails the speaker enrollment. Changing this value to <code>IGNORE</code> results in
         * the speaker being enrolled even if they are flagged by the fraud detection system.
         * </p>
         * 
         * @param fraudDetectionAction
         *        The action to take when the given speaker is flagged by the fraud detection system. The default value
         *        is <code>FAIL</code>, which fails the speaker enrollment. Changing this value to <code>IGNORE</code>
         *        results in the speaker being enrolled even if they are flagged by the fraud detection system.
         * @see FraudDetectionAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FraudDetectionAction
         */
        Builder fraudDetectionAction(String fraudDetectionAction);

        /**
         * <p>
         * The action to take when the given speaker is flagged by the fraud detection system. The default value is
         * <code>FAIL</code>, which fails the speaker enrollment. Changing this value to <code>IGNORE</code> results in
         * the speaker being enrolled even if they are flagged by the fraud detection system.
         * </p>
         * 
         * @param fraudDetectionAction
         *        The action to take when the given speaker is flagged by the fraud detection system. The default value
         *        is <code>FAIL</code>, which fails the speaker enrollment. Changing this value to <code>IGNORE</code>
         *        results in the speaker being enrolled even if they are flagged by the fraud detection system.
         * @see FraudDetectionAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FraudDetectionAction
         */
        Builder fraudDetectionAction(FraudDetectionAction fraudDetectionAction);

        /**
         * <p>
         * Threshold value for determining whether the speaker is a high risk to be fraudulent. If the detected risk
         * score calculated by Voice ID is greater than or equal to the threshold, the speaker is considered a
         * fraudster.
         * </p>
         * 
         * @param riskThreshold
         *        Threshold value for determining whether the speaker is a high risk to be fraudulent. If the detected
         *        risk score calculated by Voice ID is greater than or equal to the threshold, the speaker is considered
         *        a fraudster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder riskThreshold(Integer riskThreshold);

        /**
         * <p>
         * The identifier of watchlists against which fraud detection is performed.
         * </p>
         * 
         * @param watchlistIds
         *        The identifier of watchlists against which fraud detection is performed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder watchlistIds(Collection<String> watchlistIds);

        /**
         * <p>
         * The identifier of watchlists against which fraud detection is performed.
         * </p>
         * 
         * @param watchlistIds
         *        The identifier of watchlists against which fraud detection is performed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder watchlistIds(String... watchlistIds);
    }

    static final class BuilderImpl implements Builder {
        private String fraudDetectionAction;

        private Integer riskThreshold;

        private List<String> watchlistIds = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(EnrollmentJobFraudDetectionConfig model) {
            fraudDetectionAction(model.fraudDetectionAction);
            riskThreshold(model.riskThreshold);
            watchlistIds(model.watchlistIds);
        }

        public final String getFraudDetectionAction() {
            return fraudDetectionAction;
        }

        public final void setFraudDetectionAction(String fraudDetectionAction) {
            this.fraudDetectionAction = fraudDetectionAction;
        }

        @Override
        public final Builder fraudDetectionAction(String fraudDetectionAction) {
            this.fraudDetectionAction = fraudDetectionAction;
            return this;
        }

        @Override
        public final Builder fraudDetectionAction(FraudDetectionAction fraudDetectionAction) {
            this.fraudDetectionAction(fraudDetectionAction == null ? null : fraudDetectionAction.toString());
            return this;
        }

        public final Integer getRiskThreshold() {
            return riskThreshold;
        }

        public final void setRiskThreshold(Integer riskThreshold) {
            this.riskThreshold = riskThreshold;
        }

        @Override
        public final Builder riskThreshold(Integer riskThreshold) {
            this.riskThreshold = riskThreshold;
            return this;
        }

        public final Collection<String> getWatchlistIds() {
            if (watchlistIds instanceof SdkAutoConstructList) {
                return null;
            }
            return watchlistIds;
        }

        public final void setWatchlistIds(Collection<String> watchlistIds) {
            this.watchlistIds = EnrollmentJobFraudDetectionConfigWatchlistIdsCopier.copy(watchlistIds);
        }

        @Override
        public final Builder watchlistIds(Collection<String> watchlistIds) {
            this.watchlistIds = EnrollmentJobFraudDetectionConfigWatchlistIdsCopier.copy(watchlistIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder watchlistIds(String... watchlistIds) {
            watchlistIds(Arrays.asList(watchlistIds));
            return this;
        }

        @Override
        public EnrollmentJobFraudDetectionConfig build() {
            return new EnrollmentJobFraudDetectionConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
