/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.waf.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <note>
 * <p>
 * This is <b>AWS WAF Classic</b> documentation. For more information, see <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/classic-waf-chapter.html">AWS WAF Classic</a> in the
 * developer guide.
 * </p>
 * <p>
 * <b>For the latest version of AWS WAF</b>, use the AWS WAFV2 API and see the <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">AWS WAF Developer Guide</a>. With the
 * latest version, AWS WAF has a single set of endpoints for regional and global use.
 * </p>
 * </note>
 * <p>
 * The Amazon Kinesis Data Firehose, <code>RedactedFields</code> information, and the web ACL Amazon Resource Name
 * (ARN).
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LoggingConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<LoggingConfiguration.Builder, LoggingConfiguration> {
    private static final SdkField<String> RESOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceArn").getter(getter(LoggingConfiguration::resourceArn)).setter(setter(Builder::resourceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceArn").build()).build();

    private static final SdkField<List<String>> LOG_DESTINATION_CONFIGS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("LogDestinationConfigs")
            .getter(getter(LoggingConfiguration::logDestinationConfigs))
            .setter(setter(Builder::logDestinationConfigs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogDestinationConfigs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<FieldToMatch>> REDACTED_FIELDS_FIELD = SdkField
            .<List<FieldToMatch>> builder(MarshallingType.LIST)
            .memberName("RedactedFields")
            .getter(getter(LoggingConfiguration::redactedFields))
            .setter(setter(Builder::redactedFields))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RedactedFields").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<FieldToMatch> builder(MarshallingType.SDK_POJO)
                                            .constructor(FieldToMatch::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESOURCE_ARN_FIELD,
            LOG_DESTINATION_CONFIGS_FIELD, REDACTED_FIELDS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String resourceArn;

    private final List<String> logDestinationConfigs;

    private final List<FieldToMatch> redactedFields;

    private LoggingConfiguration(BuilderImpl builder) {
        this.resourceArn = builder.resourceArn;
        this.logDestinationConfigs = builder.logDestinationConfigs;
        this.redactedFields = builder.redactedFields;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the web ACL that you want to associate with <code>LogDestinationConfigs</code>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the web ACL that you want to associate with
     *         <code>LogDestinationConfigs</code>.
     */
    public final String resourceArn() {
        return resourceArn;
    }

    /**
     * Returns true if the LogDestinationConfigs property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasLogDestinationConfigs() {
        return logDestinationConfigs != null && !(logDestinationConfigs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of Amazon Kinesis Data Firehose ARNs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasLogDestinationConfigs()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of Amazon Kinesis Data Firehose ARNs.
     */
    public final List<String> logDestinationConfigs() {
        return logDestinationConfigs;
    }

    /**
     * Returns true if the RedactedFields property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasRedactedFields() {
        return redactedFields != null && !(redactedFields instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The parts of the request that you want redacted from the logs. For example, if you redact the cookie field, the
     * cookie field in the firehose will be <code>xxx</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRedactedFields()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The parts of the request that you want redacted from the logs. For example, if you redact the cookie
     *         field, the cookie field in the firehose will be <code>xxx</code>.
     */
    public final List<FieldToMatch> redactedFields() {
        return redactedFields;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resourceArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasLogDestinationConfigs() ? logDestinationConfigs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasRedactedFields() ? redactedFields() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LoggingConfiguration)) {
            return false;
        }
        LoggingConfiguration other = (LoggingConfiguration) obj;
        return Objects.equals(resourceArn(), other.resourceArn())
                && hasLogDestinationConfigs() == other.hasLogDestinationConfigs()
                && Objects.equals(logDestinationConfigs(), other.logDestinationConfigs())
                && hasRedactedFields() == other.hasRedactedFields() && Objects.equals(redactedFields(), other.redactedFields());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LoggingConfiguration").add("ResourceArn", resourceArn())
                .add("LogDestinationConfigs", hasLogDestinationConfigs() ? logDestinationConfigs() : null)
                .add("RedactedFields", hasRedactedFields() ? redactedFields() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ResourceArn":
            return Optional.ofNullable(clazz.cast(resourceArn()));
        case "LogDestinationConfigs":
            return Optional.ofNullable(clazz.cast(logDestinationConfigs()));
        case "RedactedFields":
            return Optional.ofNullable(clazz.cast(redactedFields()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LoggingConfiguration, T> g) {
        return obj -> g.apply((LoggingConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LoggingConfiguration> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the web ACL that you want to associate with
         * <code>LogDestinationConfigs</code>.
         * </p>
         * 
         * @param resourceArn
         *        The Amazon Resource Name (ARN) of the web ACL that you want to associate with
         *        <code>LogDestinationConfigs</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceArn(String resourceArn);

        /**
         * <p>
         * An array of Amazon Kinesis Data Firehose ARNs.
         * </p>
         * 
         * @param logDestinationConfigs
         *        An array of Amazon Kinesis Data Firehose ARNs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logDestinationConfigs(Collection<String> logDestinationConfigs);

        /**
         * <p>
         * An array of Amazon Kinesis Data Firehose ARNs.
         * </p>
         * 
         * @param logDestinationConfigs
         *        An array of Amazon Kinesis Data Firehose ARNs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logDestinationConfigs(String... logDestinationConfigs);

        /**
         * <p>
         * The parts of the request that you want redacted from the logs. For example, if you redact the cookie field,
         * the cookie field in the firehose will be <code>xxx</code>.
         * </p>
         * 
         * @param redactedFields
         *        The parts of the request that you want redacted from the logs. For example, if you redact the cookie
         *        field, the cookie field in the firehose will be <code>xxx</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder redactedFields(Collection<FieldToMatch> redactedFields);

        /**
         * <p>
         * The parts of the request that you want redacted from the logs. For example, if you redact the cookie field,
         * the cookie field in the firehose will be <code>xxx</code>.
         * </p>
         * 
         * @param redactedFields
         *        The parts of the request that you want redacted from the logs. For example, if you redact the cookie
         *        field, the cookie field in the firehose will be <code>xxx</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder redactedFields(FieldToMatch... redactedFields);

        /**
         * <p>
         * The parts of the request that you want redacted from the logs. For example, if you redact the cookie field,
         * the cookie field in the firehose will be <code>xxx</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<FieldToMatch>.Builder} avoiding the need to
         * create one manually via {@link List<FieldToMatch>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<FieldToMatch>.Builder#build()} is called immediately and its
         * result is passed to {@link #redactedFields(List<FieldToMatch>)}.
         * 
         * @param redactedFields
         *        a consumer that will call methods on {@link List<FieldToMatch>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #redactedFields(List<FieldToMatch>)
         */
        Builder redactedFields(Consumer<FieldToMatch.Builder>... redactedFields);
    }

    static final class BuilderImpl implements Builder {
        private String resourceArn;

        private List<String> logDestinationConfigs = DefaultSdkAutoConstructList.getInstance();

        private List<FieldToMatch> redactedFields = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(LoggingConfiguration model) {
            resourceArn(model.resourceArn);
            logDestinationConfigs(model.logDestinationConfigs);
            redactedFields(model.redactedFields);
        }

        public final String getResourceArn() {
            return resourceArn;
        }

        @Override
        public final Builder resourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
            return this;
        }

        public final void setResourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
        }

        public final Collection<String> getLogDestinationConfigs() {
            if (logDestinationConfigs instanceof SdkAutoConstructList) {
                return null;
            }
            return logDestinationConfigs;
        }

        @Override
        public final Builder logDestinationConfigs(Collection<String> logDestinationConfigs) {
            this.logDestinationConfigs = LogDestinationConfigsCopier.copy(logDestinationConfigs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder logDestinationConfigs(String... logDestinationConfigs) {
            logDestinationConfigs(Arrays.asList(logDestinationConfigs));
            return this;
        }

        public final void setLogDestinationConfigs(Collection<String> logDestinationConfigs) {
            this.logDestinationConfigs = LogDestinationConfigsCopier.copy(logDestinationConfigs);
        }

        public final List<FieldToMatch.Builder> getRedactedFields() {
            List<FieldToMatch.Builder> result = RedactedFieldsCopier.copyToBuilder(this.redactedFields);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder redactedFields(Collection<FieldToMatch> redactedFields) {
            this.redactedFields = RedactedFieldsCopier.copy(redactedFields);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder redactedFields(FieldToMatch... redactedFields) {
            redactedFields(Arrays.asList(redactedFields));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder redactedFields(Consumer<FieldToMatch.Builder>... redactedFields) {
            redactedFields(Stream.of(redactedFields).map(c -> FieldToMatch.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setRedactedFields(Collection<FieldToMatch.BuilderImpl> redactedFields) {
            this.redactedFields = RedactedFieldsCopier.copyFromBuilder(redactedFields);
        }

        @Override
        public LoggingConfiguration build() {
            return new LoggingConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
