/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.waf.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <note>
 * <p>
 * This is <b>AWS WAF Classic</b> documentation. For more information, see <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/classic-waf-chapter.html">AWS WAF Classic</a> in the
 * developer guide.
 * </p>
 * <p>
 * <b>For the latest version of AWS WAF</b>, use the AWS WAFV2 API and see the <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">AWS WAF Developer Guide</a>. With the
 * latest version, AWS WAF has a single set of endpoints for regional and global use.
 * </p>
 * </note>
 * <p>
 * Contains one or more IP addresses or blocks of IP addresses specified in Classless Inter-Domain Routing (CIDR)
 * notation. AWS WAF supports IPv4 address ranges: /8 and any range between /16 through /32. AWS WAF supports IPv6
 * address ranges: /24, /32, /48, /56, /64, and /128.
 * </p>
 * <p>
 * To specify an individual IP address, you specify the four-part IP address followed by a <code>/32</code>, for
 * example, 192.0.2.0/32. To block a range of IP addresses, you can specify /8 or any range between /16 through /32 (for
 * IPv4) or /24, /32, /48, /56, /64, or /128 (for IPv6). For more information about CIDR notation, see the Wikipedia
 * entry <a href="https://en.wikipedia.org/wiki/Classless_Inter-Domain_Routing">Classless Inter-Domain Routing</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IPSet implements SdkPojo, Serializable, ToCopyableBuilder<IPSet.Builder, IPSet> {
    private static final SdkField<String> IP_SET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IPSetId").getter(getter(IPSet::ipSetId)).setter(setter(Builder::ipSetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IPSetId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(IPSet::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<List<IPSetDescriptor>> IP_SET_DESCRIPTORS_FIELD = SdkField
            .<List<IPSetDescriptor>> builder(MarshallingType.LIST)
            .memberName("IPSetDescriptors")
            .getter(getter(IPSet::ipSetDescriptors))
            .setter(setter(Builder::ipSetDescriptors))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IPSetDescriptors").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<IPSetDescriptor> builder(MarshallingType.SDK_POJO)
                                            .constructor(IPSetDescriptor::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IP_SET_ID_FIELD, NAME_FIELD,
            IP_SET_DESCRIPTORS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String ipSetId;

    private final String name;

    private final List<IPSetDescriptor> ipSetDescriptors;

    private IPSet(BuilderImpl builder) {
        this.ipSetId = builder.ipSetId;
        this.name = builder.name;
        this.ipSetDescriptors = builder.ipSetDescriptors;
    }

    /**
     * <p>
     * The <code>IPSetId</code> for an <code>IPSet</code>. You use <code>IPSetId</code> to get information about an
     * <code>IPSet</code> (see <a>GetIPSet</a>), update an <code>IPSet</code> (see <a>UpdateIPSet</a>), insert an
     * <code>IPSet</code> into a <code>Rule</code> or delete one from a <code>Rule</code> (see <a>UpdateRule</a>), and
     * delete an <code>IPSet</code> from AWS WAF (see <a>DeleteIPSet</a>).
     * </p>
     * <p>
     * <code>IPSetId</code> is returned by <a>CreateIPSet</a> and by <a>ListIPSets</a>.
     * </p>
     * 
     * @return The <code>IPSetId</code> for an <code>IPSet</code>. You use <code>IPSetId</code> to get information about
     *         an <code>IPSet</code> (see <a>GetIPSet</a>), update an <code>IPSet</code> (see <a>UpdateIPSet</a>),
     *         insert an <code>IPSet</code> into a <code>Rule</code> or delete one from a <code>Rule</code> (see
     *         <a>UpdateRule</a>), and delete an <code>IPSet</code> from AWS WAF (see <a>DeleteIPSet</a>).</p>
     *         <p>
     *         <code>IPSetId</code> is returned by <a>CreateIPSet</a> and by <a>ListIPSets</a>.
     */
    public final String ipSetId() {
        return ipSetId;
    }

    /**
     * <p>
     * A friendly name or description of the <a>IPSet</a>. You can't change the name of an <code>IPSet</code> after you
     * create it.
     * </p>
     * 
     * @return A friendly name or description of the <a>IPSet</a>. You can't change the name of an <code>IPSet</code>
     *         after you create it.
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the IPSetDescriptors property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIpSetDescriptors() {
        return ipSetDescriptors != null && !(ipSetDescriptors instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IP address type (<code>IPV4</code> or <code>IPV6</code>) and the IP address range (in CIDR notation) that web
     * requests originate from. If the <code>WebACL</code> is associated with a CloudFront distribution and the viewer
     * did not use an HTTP proxy or a load balancer to send the request, this is the value of the c-ip field in the
     * CloudFront access logs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIpSetDescriptors} method.
     * </p>
     * 
     * @return The IP address type (<code>IPV4</code> or <code>IPV6</code>) and the IP address range (in CIDR notation)
     *         that web requests originate from. If the <code>WebACL</code> is associated with a CloudFront distribution
     *         and the viewer did not use an HTTP proxy or a load balancer to send the request, this is the value of the
     *         c-ip field in the CloudFront access logs.
     */
    public final List<IPSetDescriptor> ipSetDescriptors() {
        return ipSetDescriptors;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ipSetId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasIpSetDescriptors() ? ipSetDescriptors() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IPSet)) {
            return false;
        }
        IPSet other = (IPSet) obj;
        return Objects.equals(ipSetId(), other.ipSetId()) && Objects.equals(name(), other.name())
                && hasIpSetDescriptors() == other.hasIpSetDescriptors()
                && Objects.equals(ipSetDescriptors(), other.ipSetDescriptors());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("IPSet").add("IPSetId", ipSetId()).add("Name", name())
                .add("IPSetDescriptors", hasIpSetDescriptors() ? ipSetDescriptors() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "IPSetId":
            return Optional.ofNullable(clazz.cast(ipSetId()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "IPSetDescriptors":
            return Optional.ofNullable(clazz.cast(ipSetDescriptors()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("IPSetId", IP_SET_ID_FIELD);
        map.put("Name", NAME_FIELD);
        map.put("IPSetDescriptors", IP_SET_DESCRIPTORS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<IPSet, T> g) {
        return obj -> g.apply((IPSet) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IPSet> {
        /**
         * <p>
         * The <code>IPSetId</code> for an <code>IPSet</code>. You use <code>IPSetId</code> to get information about an
         * <code>IPSet</code> (see <a>GetIPSet</a>), update an <code>IPSet</code> (see <a>UpdateIPSet</a>), insert an
         * <code>IPSet</code> into a <code>Rule</code> or delete one from a <code>Rule</code> (see <a>UpdateRule</a>),
         * and delete an <code>IPSet</code> from AWS WAF (see <a>DeleteIPSet</a>).
         * </p>
         * <p>
         * <code>IPSetId</code> is returned by <a>CreateIPSet</a> and by <a>ListIPSets</a>.
         * </p>
         * 
         * @param ipSetId
         *        The <code>IPSetId</code> for an <code>IPSet</code>. You use <code>IPSetId</code> to get information
         *        about an <code>IPSet</code> (see <a>GetIPSet</a>), update an <code>IPSet</code> (see
         *        <a>UpdateIPSet</a>), insert an <code>IPSet</code> into a <code>Rule</code> or delete one from a
         *        <code>Rule</code> (see <a>UpdateRule</a>), and delete an <code>IPSet</code> from AWS WAF (see
         *        <a>DeleteIPSet</a>).</p>
         *        <p>
         *        <code>IPSetId</code> is returned by <a>CreateIPSet</a> and by <a>ListIPSets</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipSetId(String ipSetId);

        /**
         * <p>
         * A friendly name or description of the <a>IPSet</a>. You can't change the name of an <code>IPSet</code> after
         * you create it.
         * </p>
         * 
         * @param name
         *        A friendly name or description of the <a>IPSet</a>. You can't change the name of an <code>IPSet</code>
         *        after you create it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The IP address type (<code>IPV4</code> or <code>IPV6</code>) and the IP address range (in CIDR notation) that
         * web requests originate from. If the <code>WebACL</code> is associated with a CloudFront distribution and the
         * viewer did not use an HTTP proxy or a load balancer to send the request, this is the value of the c-ip field
         * in the CloudFront access logs.
         * </p>
         * 
         * @param ipSetDescriptors
         *        The IP address type (<code>IPV4</code> or <code>IPV6</code>) and the IP address range (in CIDR
         *        notation) that web requests originate from. If the <code>WebACL</code> is associated with a CloudFront
         *        distribution and the viewer did not use an HTTP proxy or a load balancer to send the request, this is
         *        the value of the c-ip field in the CloudFront access logs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipSetDescriptors(Collection<IPSetDescriptor> ipSetDescriptors);

        /**
         * <p>
         * The IP address type (<code>IPV4</code> or <code>IPV6</code>) and the IP address range (in CIDR notation) that
         * web requests originate from. If the <code>WebACL</code> is associated with a CloudFront distribution and the
         * viewer did not use an HTTP proxy or a load balancer to send the request, this is the value of the c-ip field
         * in the CloudFront access logs.
         * </p>
         * 
         * @param ipSetDescriptors
         *        The IP address type (<code>IPV4</code> or <code>IPV6</code>) and the IP address range (in CIDR
         *        notation) that web requests originate from. If the <code>WebACL</code> is associated with a CloudFront
         *        distribution and the viewer did not use an HTTP proxy or a load balancer to send the request, this is
         *        the value of the c-ip field in the CloudFront access logs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipSetDescriptors(IPSetDescriptor... ipSetDescriptors);

        /**
         * <p>
         * The IP address type (<code>IPV4</code> or <code>IPV6</code>) and the IP address range (in CIDR notation) that
         * web requests originate from. If the <code>WebACL</code> is associated with a CloudFront distribution and the
         * viewer did not use an HTTP proxy or a load balancer to send the request, this is the value of the c-ip field
         * in the CloudFront access logs.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.waf.model.IPSetDescriptor.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.waf.model.IPSetDescriptor#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.waf.model.IPSetDescriptor.Builder#build()} is called immediately and
         * its result is passed to {@link #ipSetDescriptors(List<IPSetDescriptor>)}.
         * 
         * @param ipSetDescriptors
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.waf.model.IPSetDescriptor.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ipSetDescriptors(java.util.Collection<IPSetDescriptor>)
         */
        Builder ipSetDescriptors(Consumer<IPSetDescriptor.Builder>... ipSetDescriptors);
    }

    static final class BuilderImpl implements Builder {
        private String ipSetId;

        private String name;

        private List<IPSetDescriptor> ipSetDescriptors = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(IPSet model) {
            ipSetId(model.ipSetId);
            name(model.name);
            ipSetDescriptors(model.ipSetDescriptors);
        }

        public final String getIpSetId() {
            return ipSetId;
        }

        public final void setIpSetId(String ipSetId) {
            this.ipSetId = ipSetId;
        }

        @Override
        public final Builder ipSetId(String ipSetId) {
            this.ipSetId = ipSetId;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final List<IPSetDescriptor.Builder> getIpSetDescriptors() {
            List<IPSetDescriptor.Builder> result = IPSetDescriptorsCopier.copyToBuilder(this.ipSetDescriptors);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setIpSetDescriptors(Collection<IPSetDescriptor.BuilderImpl> ipSetDescriptors) {
            this.ipSetDescriptors = IPSetDescriptorsCopier.copyFromBuilder(ipSetDescriptors);
        }

        @Override
        public final Builder ipSetDescriptors(Collection<IPSetDescriptor> ipSetDescriptors) {
            this.ipSetDescriptors = IPSetDescriptorsCopier.copy(ipSetDescriptors);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ipSetDescriptors(IPSetDescriptor... ipSetDescriptors) {
            ipSetDescriptors(Arrays.asList(ipSetDescriptors));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ipSetDescriptors(Consumer<IPSetDescriptor.Builder>... ipSetDescriptors) {
            ipSetDescriptors(Stream.of(ipSetDescriptors).map(c -> IPSetDescriptor.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public IPSet build() {
            return new IPSet(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
