/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.waf.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <note>
 * <p>
 * This is <b>AWS WAF Classic</b> documentation. For more information, see <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/classic-waf-chapter.html">AWS WAF Classic</a> in the
 * developer guide.
 * </p>
 * <p>
 * <b>For the latest version of AWS WAF</b>, use the AWS WAFV2 API and see the <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">AWS WAF Developer Guide</a>. With the
 * latest version, AWS WAF has a single set of endpoints for regional and global use.
 * </p>
 * </note>
 * <p>
 * A combination of <a>ByteMatchSet</a>, <a>IPSet</a>, and/or <a>SqlInjectionMatchSet</a> objects that identify the web
 * requests that you want to allow, block, or count. For example, you might create a <code>Rule</code> that includes the
 * following predicates:
 * </p>
 * <ul>
 * <li>
 * <p>
 * An <code>IPSet</code> that causes AWS WAF to search for web requests that originate from the IP address
 * <code>192.0.2.44</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * A <code>ByteMatchSet</code> that causes AWS WAF to search for web requests for which the value of the
 * <code>User-Agent</code> header is <code>BadBot</code>.
 * </p>
 * </li>
 * </ul>
 * <p>
 * To match the settings in this <code>Rule</code>, a request must originate from <code>192.0.2.44</code> AND include a
 * <code>User-Agent</code> header for which the value is <code>BadBot</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Rule implements SdkPojo, Serializable, ToCopyableBuilder<Rule.Builder, Rule> {
    private static final SdkField<String> RULE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("RuleId")
            .getter(getter(Rule::ruleId)).setter(setter(Builder::ruleId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Rule::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> METRIC_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MetricName").getter(getter(Rule::metricName)).setter(setter(Builder::metricName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricName").build()).build();

    private static final SdkField<List<Predicate>> PREDICATES_FIELD = SdkField
            .<List<Predicate>> builder(MarshallingType.LIST)
            .memberName("Predicates")
            .getter(getter(Rule::predicates))
            .setter(setter(Builder::predicates))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Predicates").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Predicate> builder(MarshallingType.SDK_POJO)
                                            .constructor(Predicate::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RULE_ID_FIELD, NAME_FIELD,
            METRIC_NAME_FIELD, PREDICATES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String ruleId;

    private final String name;

    private final String metricName;

    private final List<Predicate> predicates;

    private Rule(BuilderImpl builder) {
        this.ruleId = builder.ruleId;
        this.name = builder.name;
        this.metricName = builder.metricName;
        this.predicates = builder.predicates;
    }

    /**
     * <p>
     * A unique identifier for a <code>Rule</code>. You use <code>RuleId</code> to get more information about a
     * <code>Rule</code> (see <a>GetRule</a>), update a <code>Rule</code> (see <a>UpdateRule</a>), insert a
     * <code>Rule</code> into a <code>WebACL</code> or delete a one from a <code>WebACL</code> (see
     * <a>UpdateWebACL</a>), or delete a <code>Rule</code> from AWS WAF (see <a>DeleteRule</a>).
     * </p>
     * <p>
     * <code>RuleId</code> is returned by <a>CreateRule</a> and by <a>ListRules</a>.
     * </p>
     * 
     * @return A unique identifier for a <code>Rule</code>. You use <code>RuleId</code> to get more information about a
     *         <code>Rule</code> (see <a>GetRule</a>), update a <code>Rule</code> (see <a>UpdateRule</a>), insert a
     *         <code>Rule</code> into a <code>WebACL</code> or delete a one from a <code>WebACL</code> (see
     *         <a>UpdateWebACL</a>), or delete a <code>Rule</code> from AWS WAF (see <a>DeleteRule</a>).</p>
     *         <p>
     *         <code>RuleId</code> is returned by <a>CreateRule</a> and by <a>ListRules</a>.
     */
    public final String ruleId() {
        return ruleId;
    }

    /**
     * <p>
     * The friendly name or description for the <code>Rule</code>. You can't change the name of a <code>Rule</code>
     * after you create it.
     * </p>
     * 
     * @return The friendly name or description for the <code>Rule</code>. You can't change the name of a
     *         <code>Rule</code> after you create it.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * A friendly name or description for the metrics for this <code>Rule</code>. The name can contain only alphanumeric
     * characters (A-Z, a-z, 0-9), with maximum length 128 and minimum length one. It can't contain whitespace or metric
     * names reserved for AWS WAF, including "All" and "Default_Action." You can't change <code>MetricName</code> after
     * you create the <code>Rule</code>.
     * </p>
     * 
     * @return A friendly name or description for the metrics for this <code>Rule</code>. The name can contain only
     *         alphanumeric characters (A-Z, a-z, 0-9), with maximum length 128 and minimum length one. It can't contain
     *         whitespace or metric names reserved for AWS WAF, including "All" and "Default_Action." You can't change
     *         <code>MetricName</code> after you create the <code>Rule</code>.
     */
    public final String metricName() {
        return metricName;
    }

    /**
     * For responses, this returns true if the service returned a value for the Predicates property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasPredicates() {
        return predicates != null && !(predicates instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The <code>Predicates</code> object contains one <code>Predicate</code> element for each <a>ByteMatchSet</a>,
     * <a>IPSet</a>, or <a>SqlInjectionMatchSet</a> object that you want to include in a <code>Rule</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPredicates} method.
     * </p>
     * 
     * @return The <code>Predicates</code> object contains one <code>Predicate</code> element for each
     *         <a>ByteMatchSet</a>, <a>IPSet</a>, or <a>SqlInjectionMatchSet</a> object that you want to include in a
     *         <code>Rule</code>.
     */
    public final List<Predicate> predicates() {
        return predicates;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ruleId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(metricName());
        hashCode = 31 * hashCode + Objects.hashCode(hasPredicates() ? predicates() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Rule)) {
            return false;
        }
        Rule other = (Rule) obj;
        return Objects.equals(ruleId(), other.ruleId()) && Objects.equals(name(), other.name())
                && Objects.equals(metricName(), other.metricName()) && hasPredicates() == other.hasPredicates()
                && Objects.equals(predicates(), other.predicates());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Rule").add("RuleId", ruleId()).add("Name", name()).add("MetricName", metricName())
                .add("Predicates", hasPredicates() ? predicates() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RuleId":
            return Optional.ofNullable(clazz.cast(ruleId()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "MetricName":
            return Optional.ofNullable(clazz.cast(metricName()));
        case "Predicates":
            return Optional.ofNullable(clazz.cast(predicates()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("RuleId", RULE_ID_FIELD);
        map.put("Name", NAME_FIELD);
        map.put("MetricName", METRIC_NAME_FIELD);
        map.put("Predicates", PREDICATES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Rule, T> g) {
        return obj -> g.apply((Rule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Rule> {
        /**
         * <p>
         * A unique identifier for a <code>Rule</code>. You use <code>RuleId</code> to get more information about a
         * <code>Rule</code> (see <a>GetRule</a>), update a <code>Rule</code> (see <a>UpdateRule</a>), insert a
         * <code>Rule</code> into a <code>WebACL</code> or delete a one from a <code>WebACL</code> (see
         * <a>UpdateWebACL</a>), or delete a <code>Rule</code> from AWS WAF (see <a>DeleteRule</a>).
         * </p>
         * <p>
         * <code>RuleId</code> is returned by <a>CreateRule</a> and by <a>ListRules</a>.
         * </p>
         * 
         * @param ruleId
         *        A unique identifier for a <code>Rule</code>. You use <code>RuleId</code> to get more information about
         *        a <code>Rule</code> (see <a>GetRule</a>), update a <code>Rule</code> (see <a>UpdateRule</a>), insert a
         *        <code>Rule</code> into a <code>WebACL</code> or delete a one from a <code>WebACL</code> (see
         *        <a>UpdateWebACL</a>), or delete a <code>Rule</code> from AWS WAF (see <a>DeleteRule</a>).</p>
         *        <p>
         *        <code>RuleId</code> is returned by <a>CreateRule</a> and by <a>ListRules</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleId(String ruleId);

        /**
         * <p>
         * The friendly name or description for the <code>Rule</code>. You can't change the name of a <code>Rule</code>
         * after you create it.
         * </p>
         * 
         * @param name
         *        The friendly name or description for the <code>Rule</code>. You can't change the name of a
         *        <code>Rule</code> after you create it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A friendly name or description for the metrics for this <code>Rule</code>. The name can contain only
         * alphanumeric characters (A-Z, a-z, 0-9), with maximum length 128 and minimum length one. It can't contain
         * whitespace or metric names reserved for AWS WAF, including "All" and "Default_Action." You can't change
         * <code>MetricName</code> after you create the <code>Rule</code>.
         * </p>
         * 
         * @param metricName
         *        A friendly name or description for the metrics for this <code>Rule</code>. The name can contain only
         *        alphanumeric characters (A-Z, a-z, 0-9), with maximum length 128 and minimum length one. It can't
         *        contain whitespace or metric names reserved for AWS WAF, including "All" and "Default_Action." You
         *        can't change <code>MetricName</code> after you create the <code>Rule</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricName(String metricName);

        /**
         * <p>
         * The <code>Predicates</code> object contains one <code>Predicate</code> element for each <a>ByteMatchSet</a>,
         * <a>IPSet</a>, or <a>SqlInjectionMatchSet</a> object that you want to include in a <code>Rule</code>.
         * </p>
         * 
         * @param predicates
         *        The <code>Predicates</code> object contains one <code>Predicate</code> element for each
         *        <a>ByteMatchSet</a>, <a>IPSet</a>, or <a>SqlInjectionMatchSet</a> object that you want to include in a
         *        <code>Rule</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder predicates(Collection<Predicate> predicates);

        /**
         * <p>
         * The <code>Predicates</code> object contains one <code>Predicate</code> element for each <a>ByteMatchSet</a>,
         * <a>IPSet</a>, or <a>SqlInjectionMatchSet</a> object that you want to include in a <code>Rule</code>.
         * </p>
         * 
         * @param predicates
         *        The <code>Predicates</code> object contains one <code>Predicate</code> element for each
         *        <a>ByteMatchSet</a>, <a>IPSet</a>, or <a>SqlInjectionMatchSet</a> object that you want to include in a
         *        <code>Rule</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder predicates(Predicate... predicates);

        /**
         * <p>
         * The <code>Predicates</code> object contains one <code>Predicate</code> element for each <a>ByteMatchSet</a>,
         * <a>IPSet</a>, or <a>SqlInjectionMatchSet</a> object that you want to include in a <code>Rule</code>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.waf.model.Predicate.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.waf.model.Predicate#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.waf.model.Predicate.Builder#build()} is called immediately and its
         * result is passed to {@link #predicates(List<Predicate>)}.
         * 
         * @param predicates
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.waf.model.Predicate.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #predicates(java.util.Collection<Predicate>)
         */
        Builder predicates(Consumer<Predicate.Builder>... predicates);
    }

    static final class BuilderImpl implements Builder {
        private String ruleId;

        private String name;

        private String metricName;

        private List<Predicate> predicates = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Rule model) {
            ruleId(model.ruleId);
            name(model.name);
            metricName(model.metricName);
            predicates(model.predicates);
        }

        public final String getRuleId() {
            return ruleId;
        }

        public final void setRuleId(String ruleId) {
            this.ruleId = ruleId;
        }

        @Override
        public final Builder ruleId(String ruleId) {
            this.ruleId = ruleId;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getMetricName() {
            return metricName;
        }

        public final void setMetricName(String metricName) {
            this.metricName = metricName;
        }

        @Override
        public final Builder metricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        public final List<Predicate.Builder> getPredicates() {
            List<Predicate.Builder> result = PredicatesCopier.copyToBuilder(this.predicates);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPredicates(Collection<Predicate.BuilderImpl> predicates) {
            this.predicates = PredicatesCopier.copyFromBuilder(predicates);
        }

        @Override
        public final Builder predicates(Collection<Predicate> predicates) {
            this.predicates = PredicatesCopier.copy(predicates);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder predicates(Predicate... predicates) {
            predicates(Arrays.asList(predicates));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder predicates(Consumer<Predicate.Builder>... predicates) {
            predicates(Stream.of(predicates).map(c -> Predicate.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Rule build() {
            return new Rule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
