/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.wafv2.model.AssociateWebAclRequest;
import software.amazon.awssdk.services.wafv2.model.AssociateWebAclResponse;
import software.amazon.awssdk.services.wafv2.model.CheckCapacityRequest;
import software.amazon.awssdk.services.wafv2.model.CheckCapacityResponse;
import software.amazon.awssdk.services.wafv2.model.CreateIpSetRequest;
import software.amazon.awssdk.services.wafv2.model.CreateIpSetResponse;
import software.amazon.awssdk.services.wafv2.model.CreateRegexPatternSetRequest;
import software.amazon.awssdk.services.wafv2.model.CreateRegexPatternSetResponse;
import software.amazon.awssdk.services.wafv2.model.CreateRuleGroupRequest;
import software.amazon.awssdk.services.wafv2.model.CreateRuleGroupResponse;
import software.amazon.awssdk.services.wafv2.model.CreateWebAclRequest;
import software.amazon.awssdk.services.wafv2.model.CreateWebAclResponse;
import software.amazon.awssdk.services.wafv2.model.DeleteFirewallManagerRuleGroupsRequest;
import software.amazon.awssdk.services.wafv2.model.DeleteFirewallManagerRuleGroupsResponse;
import software.amazon.awssdk.services.wafv2.model.DeleteIpSetRequest;
import software.amazon.awssdk.services.wafv2.model.DeleteIpSetResponse;
import software.amazon.awssdk.services.wafv2.model.DeleteLoggingConfigurationRequest;
import software.amazon.awssdk.services.wafv2.model.DeleteLoggingConfigurationResponse;
import software.amazon.awssdk.services.wafv2.model.DeletePermissionPolicyRequest;
import software.amazon.awssdk.services.wafv2.model.DeletePermissionPolicyResponse;
import software.amazon.awssdk.services.wafv2.model.DeleteRegexPatternSetRequest;
import software.amazon.awssdk.services.wafv2.model.DeleteRegexPatternSetResponse;
import software.amazon.awssdk.services.wafv2.model.DeleteRuleGroupRequest;
import software.amazon.awssdk.services.wafv2.model.DeleteRuleGroupResponse;
import software.amazon.awssdk.services.wafv2.model.DeleteWebAclRequest;
import software.amazon.awssdk.services.wafv2.model.DeleteWebAclResponse;
import software.amazon.awssdk.services.wafv2.model.DescribeManagedRuleGroupRequest;
import software.amazon.awssdk.services.wafv2.model.DescribeManagedRuleGroupResponse;
import software.amazon.awssdk.services.wafv2.model.DisassociateWebAclRequest;
import software.amazon.awssdk.services.wafv2.model.DisassociateWebAclResponse;
import software.amazon.awssdk.services.wafv2.model.GetIpSetRequest;
import software.amazon.awssdk.services.wafv2.model.GetIpSetResponse;
import software.amazon.awssdk.services.wafv2.model.GetLoggingConfigurationRequest;
import software.amazon.awssdk.services.wafv2.model.GetLoggingConfigurationResponse;
import software.amazon.awssdk.services.wafv2.model.GetPermissionPolicyRequest;
import software.amazon.awssdk.services.wafv2.model.GetPermissionPolicyResponse;
import software.amazon.awssdk.services.wafv2.model.GetRateBasedStatementManagedKeysRequest;
import software.amazon.awssdk.services.wafv2.model.GetRateBasedStatementManagedKeysResponse;
import software.amazon.awssdk.services.wafv2.model.GetRegexPatternSetRequest;
import software.amazon.awssdk.services.wafv2.model.GetRegexPatternSetResponse;
import software.amazon.awssdk.services.wafv2.model.GetRuleGroupRequest;
import software.amazon.awssdk.services.wafv2.model.GetRuleGroupResponse;
import software.amazon.awssdk.services.wafv2.model.GetSampledRequestsRequest;
import software.amazon.awssdk.services.wafv2.model.GetSampledRequestsResponse;
import software.amazon.awssdk.services.wafv2.model.GetWebAclForResourceRequest;
import software.amazon.awssdk.services.wafv2.model.GetWebAclForResourceResponse;
import software.amazon.awssdk.services.wafv2.model.GetWebAclRequest;
import software.amazon.awssdk.services.wafv2.model.GetWebAclResponse;
import software.amazon.awssdk.services.wafv2.model.ListAvailableManagedRuleGroupsRequest;
import software.amazon.awssdk.services.wafv2.model.ListAvailableManagedRuleGroupsResponse;
import software.amazon.awssdk.services.wafv2.model.ListIpSetsRequest;
import software.amazon.awssdk.services.wafv2.model.ListIpSetsResponse;
import software.amazon.awssdk.services.wafv2.model.ListLoggingConfigurationsRequest;
import software.amazon.awssdk.services.wafv2.model.ListLoggingConfigurationsResponse;
import software.amazon.awssdk.services.wafv2.model.ListRegexPatternSetsRequest;
import software.amazon.awssdk.services.wafv2.model.ListRegexPatternSetsResponse;
import software.amazon.awssdk.services.wafv2.model.ListResourcesForWebAclRequest;
import software.amazon.awssdk.services.wafv2.model.ListResourcesForWebAclResponse;
import software.amazon.awssdk.services.wafv2.model.ListRuleGroupsRequest;
import software.amazon.awssdk.services.wafv2.model.ListRuleGroupsResponse;
import software.amazon.awssdk.services.wafv2.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.wafv2.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.wafv2.model.ListWebAcLsRequest;
import software.amazon.awssdk.services.wafv2.model.ListWebAcLsResponse;
import software.amazon.awssdk.services.wafv2.model.PutLoggingConfigurationRequest;
import software.amazon.awssdk.services.wafv2.model.PutLoggingConfigurationResponse;
import software.amazon.awssdk.services.wafv2.model.PutPermissionPolicyRequest;
import software.amazon.awssdk.services.wafv2.model.PutPermissionPolicyResponse;
import software.amazon.awssdk.services.wafv2.model.TagResourceRequest;
import software.amazon.awssdk.services.wafv2.model.TagResourceResponse;
import software.amazon.awssdk.services.wafv2.model.UntagResourceRequest;
import software.amazon.awssdk.services.wafv2.model.UntagResourceResponse;
import software.amazon.awssdk.services.wafv2.model.UpdateIpSetRequest;
import software.amazon.awssdk.services.wafv2.model.UpdateIpSetResponse;
import software.amazon.awssdk.services.wafv2.model.UpdateRegexPatternSetRequest;
import software.amazon.awssdk.services.wafv2.model.UpdateRegexPatternSetResponse;
import software.amazon.awssdk.services.wafv2.model.UpdateRuleGroupRequest;
import software.amazon.awssdk.services.wafv2.model.UpdateRuleGroupResponse;
import software.amazon.awssdk.services.wafv2.model.UpdateWebAclRequest;
import software.amazon.awssdk.services.wafv2.model.UpdateWebAclResponse;
import software.amazon.awssdk.services.wafv2.model.WafAssociatedItemException;
import software.amazon.awssdk.services.wafv2.model.WafDuplicateItemException;
import software.amazon.awssdk.services.wafv2.model.WafInternalErrorException;
import software.amazon.awssdk.services.wafv2.model.WafInvalidOperationException;
import software.amazon.awssdk.services.wafv2.model.WafInvalidParameterException;
import software.amazon.awssdk.services.wafv2.model.WafInvalidPermissionPolicyException;
import software.amazon.awssdk.services.wafv2.model.WafInvalidResourceException;
import software.amazon.awssdk.services.wafv2.model.WafLimitsExceededException;
import software.amazon.awssdk.services.wafv2.model.WafNonexistentItemException;
import software.amazon.awssdk.services.wafv2.model.WafOptimisticLockException;
import software.amazon.awssdk.services.wafv2.model.WafServiceLinkedRoleErrorException;
import software.amazon.awssdk.services.wafv2.model.WafSubscriptionNotFoundException;
import software.amazon.awssdk.services.wafv2.model.WafTagOperationException;
import software.amazon.awssdk.services.wafv2.model.WafTagOperationInternalErrorException;
import software.amazon.awssdk.services.wafv2.model.WafUnavailableEntityException;
import software.amazon.awssdk.services.wafv2.model.Wafv2Exception;
import software.amazon.awssdk.services.wafv2.transform.AssociateWebAclRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.CheckCapacityRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.CreateIpSetRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.CreateRegexPatternSetRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.CreateRuleGroupRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.CreateWebAclRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.DeleteFirewallManagerRuleGroupsRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.DeleteIpSetRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.DeleteLoggingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.DeletePermissionPolicyRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.DeleteRegexPatternSetRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.DeleteRuleGroupRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.DeleteWebAclRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.DescribeManagedRuleGroupRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.DisassociateWebAclRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.GetIpSetRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.GetLoggingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.GetPermissionPolicyRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.GetRateBasedStatementManagedKeysRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.GetRegexPatternSetRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.GetRuleGroupRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.GetSampledRequestsRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.GetWebAclForResourceRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.GetWebAclRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.ListAvailableManagedRuleGroupsRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.ListIpSetsRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.ListLoggingConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.ListRegexPatternSetsRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.ListResourcesForWebAclRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.ListRuleGroupsRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.ListWebAcLsRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.PutLoggingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.PutPermissionPolicyRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.UpdateIpSetRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.UpdateRegexPatternSetRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.UpdateRuleGroupRequestMarshaller;
import software.amazon.awssdk.services.wafv2.transform.UpdateWebAclRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link Wafv2AsyncClient}.
 *
 * @see Wafv2AsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultWafv2AsyncClient implements Wafv2AsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultWafv2AsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultWafv2AsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Associates a Web ACL with a regional application resource, to protect the resource. A regional application can be
     * an Application Load Balancer (ALB), an API Gateway REST API, or an AppSync GraphQL API.
     * </p>
     * <p>
     * For AWS CloudFront, don't use this call. Instead, use your CloudFront distribution configuration. To associate a
     * Web ACL, in the CloudFront call <code>UpdateDistribution</code>, set the web ACL ID to the Amazon Resource Name
     * (ARN) of the Web ACL. For information, see <a
     * href="https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateDistribution.html"
     * >UpdateDistribution</a>.
     * </p>
     *
     * @param associateWebAclRequest
     * @return A Java Future containing the result of the AssociateWebACL operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafUnavailableEntityException AWS WAF couldn’t retrieve the resource that you requested. Retry your
     *         request.</li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.AssociateWebACL
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/AssociateWebACL" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateWebAclResponse> associateWebACL(AssociateWebAclRequest associateWebAclRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateWebAclRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateWebACL");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateWebAclResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateWebAclResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateWebAclResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateWebAclRequest, AssociateWebAclResponse>()
                            .withOperationName("AssociateWebACL")
                            .withMarshaller(new AssociateWebAclRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(associateWebAclRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = associateWebAclRequest.overrideConfiguration().orElse(null);
            CompletableFuture<AssociateWebAclResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the web ACL capacity unit (WCU) requirements for a specified scope and set of rules. You can use this to
     * check the capacity requirements for the rules you want to use in a <a>RuleGroup</a> or <a>WebACL</a>.
     * </p>
     * <p>
     * AWS WAF uses WCUs to calculate and control the operating resources that are used to run your rules, rule groups,
     * and web ACLs. AWS WAF calculates capacity differently for each rule type, to reflect the relative cost of each
     * rule. Simple rules that cost little to run use fewer WCUs than more complex rules that use more processing power.
     * Rule group capacity is fixed at creation, which helps users plan their web ACL WCU usage when they use a rule
     * group. The WCU limit for web ACLs is 1,500.
     * </p>
     *
     * @param checkCapacityRequest
     * @return A Java Future containing the result of the CheckCapacity operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafLimitsExceededException AWS WAF couldn’t perform the operation because you exceeded your resource
     *         limit. For example, the maximum number of <code>WebACL</code> objects that you can create for an AWS
     *         account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">Limits</a> in the <i>AWS WAF
     *         Developer Guide</i>.</li>
     *         <li>WafInvalidResourceException AWS WAF couldn’t perform the operation because the resource that you
     *         requested isn’t valid. Check the resource, and try again.</li>
     *         <li>WafUnavailableEntityException AWS WAF couldn’t retrieve the resource that you requested. Retry your
     *         request.</li>
     *         <li>WafSubscriptionNotFoundException You tried to use a managed rule group that's available by
     *         subscription, but you aren't subscribed to it yet.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.CheckCapacity
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/CheckCapacity" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CheckCapacityResponse> checkCapacity(CheckCapacityRequest checkCapacityRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, checkCapacityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CheckCapacity");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CheckCapacityResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CheckCapacityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CheckCapacityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CheckCapacityRequest, CheckCapacityResponse>()
                            .withOperationName("CheckCapacity")
                            .withMarshaller(new CheckCapacityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(checkCapacityRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = checkCapacityRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CheckCapacityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an <a>IPSet</a>, which you use to identify web requests that originate from specific IP addresses or
     * ranges of IP addresses. For example, if you're receiving a lot of requests from a ranges of IP addresses, you can
     * configure AWS WAF to block them using an IPSet that lists those IP addresses.
     * </p>
     *
     * @param createIpSetRequest
     * @return A Java Future containing the result of the CreateIPSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafDuplicateItemException AWS WAF couldn’t perform the operation because the resource that you tried
     *         to save is a duplicate of an existing one.</li>
     *         <li>WafOptimisticLockException AWS WAF couldn’t save your changes because you tried to update or delete a
     *         resource that has changed since you last retrieved it. Get the resource again, make any changes you need
     *         to make to the new copy, and retry your operation.</li>
     *         <li>WafLimitsExceededException AWS WAF couldn’t perform the operation because you exceeded your resource
     *         limit. For example, the maximum number of <code>WebACL</code> objects that you can create for an AWS
     *         account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">Limits</a> in the <i>AWS WAF
     *         Developer Guide</i>.</li>
     *         <li>WafTagOperationException An error occurred during the tagging operation. Retry your request.</li>
     *         <li>WafTagOperationInternalErrorException AWS WAF couldn’t perform your tagging operation because of an
     *         internal error. Retry your request.</li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.CreateIPSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/CreateIPSet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateIpSetResponse> createIPSet(CreateIpSetRequest createIpSetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createIpSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateIPSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateIpSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateIpSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateIpSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateIpSetRequest, CreateIpSetResponse>()
                            .withOperationName("CreateIPSet").withMarshaller(new CreateIpSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createIpSetRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createIpSetRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateIpSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a <a>RegexPatternSet</a>, which you reference in a <a>RegexPatternSetReferenceStatement</a>, to have AWS
     * WAF inspect a web request component for the specified patterns.
     * </p>
     *
     * @param createRegexPatternSetRequest
     * @return A Java Future containing the result of the CreateRegexPatternSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafDuplicateItemException AWS WAF couldn’t perform the operation because the resource that you tried
     *         to save is a duplicate of an existing one.</li>
     *         <li>WafOptimisticLockException AWS WAF couldn’t save your changes because you tried to update or delete a
     *         resource that has changed since you last retrieved it. Get the resource again, make any changes you need
     *         to make to the new copy, and retry your operation.</li>
     *         <li>WafLimitsExceededException AWS WAF couldn’t perform the operation because you exceeded your resource
     *         limit. For example, the maximum number of <code>WebACL</code> objects that you can create for an AWS
     *         account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">Limits</a> in the <i>AWS WAF
     *         Developer Guide</i>.</li>
     *         <li>WafTagOperationException An error occurred during the tagging operation. Retry your request.</li>
     *         <li>WafTagOperationInternalErrorException AWS WAF couldn’t perform your tagging operation because of an
     *         internal error. Retry your request.</li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.CreateRegexPatternSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/CreateRegexPatternSet" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRegexPatternSetResponse> createRegexPatternSet(
            CreateRegexPatternSetRequest createRegexPatternSetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRegexPatternSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRegexPatternSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRegexPatternSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateRegexPatternSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateRegexPatternSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRegexPatternSetRequest, CreateRegexPatternSetResponse>()
                            .withOperationName("CreateRegexPatternSet")
                            .withMarshaller(new CreateRegexPatternSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createRegexPatternSetRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createRegexPatternSetRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<CreateRegexPatternSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a <a>RuleGroup</a> per the specifications provided.
     * </p>
     * <p>
     * A rule group defines a collection of rules to inspect and control web requests that you can use in a
     * <a>WebACL</a>. When you create a rule group, you define an immutable capacity limit. If you update a rule group,
     * you must stay within the capacity. This allows others to reuse the rule group with confidence in its capacity
     * requirements.
     * </p>
     *
     * @param createRuleGroupRequest
     * @return A Java Future containing the result of the CreateRuleGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafDuplicateItemException AWS WAF couldn’t perform the operation because the resource that you tried
     *         to save is a duplicate of an existing one.</li>
     *         <li>WafOptimisticLockException AWS WAF couldn’t save your changes because you tried to update or delete a
     *         resource that has changed since you last retrieved it. Get the resource again, make any changes you need
     *         to make to the new copy, and retry your operation.</li>
     *         <li>WafLimitsExceededException AWS WAF couldn’t perform the operation because you exceeded your resource
     *         limit. For example, the maximum number of <code>WebACL</code> objects that you can create for an AWS
     *         account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">Limits</a> in the <i>AWS WAF
     *         Developer Guide</i>.</li>
     *         <li>WafUnavailableEntityException AWS WAF couldn’t retrieve the resource that you requested. Retry your
     *         request.</li>
     *         <li>WafTagOperationException An error occurred during the tagging operation. Retry your request.</li>
     *         <li>WafTagOperationInternalErrorException AWS WAF couldn’t perform your tagging operation because of an
     *         internal error. Retry your request.</li>
     *         <li>WafSubscriptionNotFoundException You tried to use a managed rule group that's available by
     *         subscription, but you aren't subscribed to it yet.</li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.CreateRuleGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/CreateRuleGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRuleGroupResponse> createRuleGroup(CreateRuleGroupRequest createRuleGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRuleGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRuleGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRuleGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateRuleGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateRuleGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRuleGroupRequest, CreateRuleGroupResponse>()
                            .withOperationName("CreateRuleGroup")
                            .withMarshaller(new CreateRuleGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createRuleGroupRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createRuleGroupRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateRuleGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a <a>WebACL</a> per the specifications provided.
     * </p>
     * <p>
     * A Web ACL defines a collection of rules to use to inspect and control web requests. Each rule has an action
     * defined (allow, block, or count) for requests that match the statement of the rule. In the Web ACL, you assign a
     * default action to take (allow, block) for any request that does not match any of the rules. The rules in a Web
     * ACL can be a combination of the types <a>Rule</a>, <a>RuleGroup</a>, and managed rule group. You can associate a
     * Web ACL with one or more AWS resources to protect. The resources can be Amazon CloudFront, an Amazon API Gateway
     * REST API, an Application Load Balancer, or an AWS AppSync GraphQL API.
     * </p>
     *
     * @param createWebAclRequest
     * @return A Java Future containing the result of the CreateWebACL operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafDuplicateItemException AWS WAF couldn’t perform the operation because the resource that you tried
     *         to save is a duplicate of an existing one.</li>
     *         <li>WafOptimisticLockException AWS WAF couldn’t save your changes because you tried to update or delete a
     *         resource that has changed since you last retrieved it. Get the resource again, make any changes you need
     *         to make to the new copy, and retry your operation.</li>
     *         <li>WafLimitsExceededException AWS WAF couldn’t perform the operation because you exceeded your resource
     *         limit. For example, the maximum number of <code>WebACL</code> objects that you can create for an AWS
     *         account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">Limits</a> in the <i>AWS WAF
     *         Developer Guide</i>.</li>
     *         <li>WafInvalidResourceException AWS WAF couldn’t perform the operation because the resource that you
     *         requested isn’t valid. Check the resource, and try again.</li>
     *         <li>WafUnavailableEntityException AWS WAF couldn’t retrieve the resource that you requested. Retry your
     *         request.</li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafTagOperationException An error occurred during the tagging operation. Retry your request.</li>
     *         <li>WafTagOperationInternalErrorException AWS WAF couldn’t perform your tagging operation because of an
     *         internal error. Retry your request.</li>
     *         <li>WafSubscriptionNotFoundException You tried to use a managed rule group that's available by
     *         subscription, but you aren't subscribed to it yet.</li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.CreateWebACL
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/CreateWebACL" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateWebAclResponse> createWebACL(CreateWebAclRequest createWebAclRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWebAclRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWebACL");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateWebAclResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateWebAclResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateWebAclResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateWebAclRequest, CreateWebAclResponse>()
                            .withOperationName("CreateWebACL").withMarshaller(new CreateWebAclRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createWebAclRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createWebAclRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateWebAclResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes all rule groups that are managed by AWS Firewall Manager for the specified web ACL.
     * </p>
     * <p>
     * You can only use this if <code>ManagedByFirewallManager</code> is false in the specified <a>WebACL</a>.
     * </p>
     *
     * @param deleteFirewallManagerRuleGroupsRequest
     * @return A Java Future containing the result of the DeleteFirewallManagerRuleGroups operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafOptimisticLockException AWS WAF couldn’t save your changes because you tried to update or delete a
     *         resource that has changed since you last retrieved it. Get the resource again, make any changes you need
     *         to make to the new copy, and retry your operation.</li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.DeleteFirewallManagerRuleGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/DeleteFirewallManagerRuleGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFirewallManagerRuleGroupsResponse> deleteFirewallManagerRuleGroups(
            DeleteFirewallManagerRuleGroupsRequest deleteFirewallManagerRuleGroupsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteFirewallManagerRuleGroupsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFirewallManagerRuleGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteFirewallManagerRuleGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteFirewallManagerRuleGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteFirewallManagerRuleGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteFirewallManagerRuleGroupsRequest, DeleteFirewallManagerRuleGroupsResponse>()
                            .withOperationName("DeleteFirewallManagerRuleGroups")
                            .withMarshaller(new DeleteFirewallManagerRuleGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteFirewallManagerRuleGroupsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteFirewallManagerRuleGroupsRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<DeleteFirewallManagerRuleGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified <a>IPSet</a>.
     * </p>
     *
     * @param deleteIpSetRequest
     * @return A Java Future containing the result of the DeleteIPSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafOptimisticLockException AWS WAF couldn’t save your changes because you tried to update or delete a
     *         resource that has changed since you last retrieved it. Get the resource again, make any changes you need
     *         to make to the new copy, and retry your operation.</li>
     *         <li>WafAssociatedItemException AWS WAF couldn’t perform the operation because your resource is being used
     *         by another resource or it’s associated with another resource.</li>
     *         <li>WafTagOperationException An error occurred during the tagging operation. Retry your request.</li>
     *         <li>WafTagOperationInternalErrorException AWS WAF couldn’t perform your tagging operation because of an
     *         internal error. Retry your request.</li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.DeleteIPSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/DeleteIPSet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteIpSetResponse> deleteIPSet(DeleteIpSetRequest deleteIpSetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteIpSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteIPSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteIpSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteIpSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteIpSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteIpSetRequest, DeleteIpSetResponse>()
                            .withOperationName("DeleteIPSet").withMarshaller(new DeleteIpSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteIpSetRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteIpSetRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteIpSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the <a>LoggingConfiguration</a> from the specified web ACL.
     * </p>
     *
     * @param deleteLoggingConfigurationRequest
     * @return A Java Future containing the result of the DeleteLoggingConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafOptimisticLockException AWS WAF couldn’t save your changes because you tried to update or delete a
     *         resource that has changed since you last retrieved it. Get the resource again, make any changes you need
     *         to make to the new copy, and retry your operation.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.DeleteLoggingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/DeleteLoggingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLoggingConfigurationResponse> deleteLoggingConfiguration(
            DeleteLoggingConfigurationRequest deleteLoggingConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLoggingConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLoggingConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLoggingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteLoggingConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLoggingConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLoggingConfigurationRequest, DeleteLoggingConfigurationResponse>()
                            .withOperationName("DeleteLoggingConfiguration")
                            .withMarshaller(new DeleteLoggingConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteLoggingConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteLoggingConfigurationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteLoggingConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Permanently deletes an IAM policy from the specified rule group.
     * </p>
     * <p>
     * You must be the owner of the rule group to perform this operation.
     * </p>
     *
     * @param deletePermissionPolicyRequest
     * @return A Java Future containing the result of the DeletePermissionPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.DeletePermissionPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/DeletePermissionPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePermissionPolicyResponse> deletePermissionPolicy(
            DeletePermissionPolicyRequest deletePermissionPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePermissionPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePermissionPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePermissionPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePermissionPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePermissionPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePermissionPolicyRequest, DeletePermissionPolicyResponse>()
                            .withOperationName("DeletePermissionPolicy")
                            .withMarshaller(new DeletePermissionPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deletePermissionPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deletePermissionPolicyRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DeletePermissionPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified <a>RegexPatternSet</a>.
     * </p>
     *
     * @param deleteRegexPatternSetRequest
     * @return A Java Future containing the result of the DeleteRegexPatternSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafOptimisticLockException AWS WAF couldn’t save your changes because you tried to update or delete a
     *         resource that has changed since you last retrieved it. Get the resource again, make any changes you need
     *         to make to the new copy, and retry your operation.</li>
     *         <li>WafAssociatedItemException AWS WAF couldn’t perform the operation because your resource is being used
     *         by another resource or it’s associated with another resource.</li>
     *         <li>WafTagOperationException An error occurred during the tagging operation. Retry your request.</li>
     *         <li>WafTagOperationInternalErrorException AWS WAF couldn’t perform your tagging operation because of an
     *         internal error. Retry your request.</li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.DeleteRegexPatternSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/DeleteRegexPatternSet" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRegexPatternSetResponse> deleteRegexPatternSet(
            DeleteRegexPatternSetRequest deleteRegexPatternSetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRegexPatternSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRegexPatternSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRegexPatternSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRegexPatternSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRegexPatternSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRegexPatternSetRequest, DeleteRegexPatternSetResponse>()
                            .withOperationName("DeleteRegexPatternSet")
                            .withMarshaller(new DeleteRegexPatternSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteRegexPatternSetRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteRegexPatternSetRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DeleteRegexPatternSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified <a>RuleGroup</a>.
     * </p>
     *
     * @param deleteRuleGroupRequest
     * @return A Java Future containing the result of the DeleteRuleGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafOptimisticLockException AWS WAF couldn’t save your changes because you tried to update or delete a
     *         resource that has changed since you last retrieved it. Get the resource again, make any changes you need
     *         to make to the new copy, and retry your operation.</li>
     *         <li>WafAssociatedItemException AWS WAF couldn’t perform the operation because your resource is being used
     *         by another resource or it’s associated with another resource.</li>
     *         <li>WafTagOperationException An error occurred during the tagging operation. Retry your request.</li>
     *         <li>WafTagOperationInternalErrorException AWS WAF couldn’t perform your tagging operation because of an
     *         internal error. Retry your request.</li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.DeleteRuleGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/DeleteRuleGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRuleGroupResponse> deleteRuleGroup(DeleteRuleGroupRequest deleteRuleGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRuleGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRuleGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRuleGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRuleGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRuleGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRuleGroupRequest, DeleteRuleGroupResponse>()
                            .withOperationName("DeleteRuleGroup")
                            .withMarshaller(new DeleteRuleGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteRuleGroupRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteRuleGroupRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteRuleGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified <a>WebACL</a>.
     * </p>
     * <p>
     * You can only use this if <code>ManagedByFirewallManager</code> is false in the specified <a>WebACL</a>.
     * </p>
     *
     * @param deleteWebAclRequest
     * @return A Java Future containing the result of the DeleteWebACL operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafOptimisticLockException AWS WAF couldn’t save your changes because you tried to update or delete a
     *         resource that has changed since you last retrieved it. Get the resource again, make any changes you need
     *         to make to the new copy, and retry your operation.</li>
     *         <li>WafAssociatedItemException AWS WAF couldn’t perform the operation because your resource is being used
     *         by another resource or it’s associated with another resource.</li>
     *         <li>WafTagOperationException An error occurred during the tagging operation. Retry your request.</li>
     *         <li>WafTagOperationInternalErrorException AWS WAF couldn’t perform your tagging operation because of an
     *         internal error. Retry your request.</li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.DeleteWebACL
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/DeleteWebACL" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWebAclResponse> deleteWebACL(DeleteWebAclRequest deleteWebAclRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWebAclRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWebACL");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteWebAclResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteWebAclResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteWebAclResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWebAclRequest, DeleteWebAclResponse>()
                            .withOperationName("DeleteWebACL").withMarshaller(new DeleteWebAclRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteWebAclRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteWebAclRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteWebAclResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides high-level information for a managed rule group, including descriptions of the rules.
     * </p>
     *
     * @param describeManagedRuleGroupRequest
     * @return A Java Future containing the result of the DescribeManagedRuleGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafInvalidResourceException AWS WAF couldn’t perform the operation because the resource that you
     *         requested isn’t valid. Check the resource, and try again.</li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.DescribeManagedRuleGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/DescribeManagedRuleGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeManagedRuleGroupResponse> describeManagedRuleGroup(
            DescribeManagedRuleGroupRequest describeManagedRuleGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeManagedRuleGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeManagedRuleGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeManagedRuleGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeManagedRuleGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeManagedRuleGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeManagedRuleGroupRequest, DescribeManagedRuleGroupResponse>()
                            .withOperationName("DescribeManagedRuleGroup")
                            .withMarshaller(new DescribeManagedRuleGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeManagedRuleGroupRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeManagedRuleGroupRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeManagedRuleGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates a Web ACL from a regional application resource. A regional application can be an Application Load
     * Balancer (ALB), an API Gateway REST API, or an AppSync GraphQL API.
     * </p>
     * <p>
     * For AWS CloudFront, don't use this call. Instead, use your CloudFront distribution configuration. To disassociate
     * a Web ACL, provide an empty web ACL ID in the CloudFront call <code>UpdateDistribution</code>. For information,
     * see <a href="https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateDistribution.html">
     * UpdateDistribution</a>.
     * </p>
     *
     * @param disassociateWebAclRequest
     * @return A Java Future containing the result of the DisassociateWebACL operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.DisassociateWebACL
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/DisassociateWebACL" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateWebAclResponse> disassociateWebACL(DisassociateWebAclRequest disassociateWebAclRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateWebAclRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateWebACL");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateWebAclResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateWebAclResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateWebAclResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateWebAclRequest, DisassociateWebAclResponse>()
                            .withOperationName("DisassociateWebACL")
                            .withMarshaller(new DisassociateWebAclRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disassociateWebAclRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = disassociateWebAclRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DisassociateWebAclResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the specified <a>IPSet</a>.
     * </p>
     *
     * @param getIpSetRequest
     * @return A Java Future containing the result of the GetIPSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.GetIPSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/GetIPSet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetIpSetResponse> getIPSet(GetIpSetRequest getIpSetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getIpSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetIPSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetIpSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetIpSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetIpSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetIpSetRequest, GetIpSetResponse>().withOperationName("GetIPSet")
                            .withMarshaller(new GetIpSetRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getIpSetRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getIpSetRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetIpSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the <a>LoggingConfiguration</a> for the specified web ACL.
     * </p>
     *
     * @param getLoggingConfigurationRequest
     * @return A Java Future containing the result of the GetLoggingConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.GetLoggingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/GetLoggingConfiguration" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetLoggingConfigurationResponse> getLoggingConfiguration(
            GetLoggingConfigurationRequest getLoggingConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLoggingConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLoggingConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLoggingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetLoggingConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLoggingConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLoggingConfigurationRequest, GetLoggingConfigurationResponse>()
                            .withOperationName("GetLoggingConfiguration")
                            .withMarshaller(new GetLoggingConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getLoggingConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getLoggingConfigurationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetLoggingConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the IAM policy that is attached to the specified rule group.
     * </p>
     * <p>
     * You must be the owner of the rule group to perform this operation.
     * </p>
     *
     * @param getPermissionPolicyRequest
     * @return A Java Future containing the result of the GetPermissionPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.GetPermissionPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/GetPermissionPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetPermissionPolicyResponse> getPermissionPolicy(
            GetPermissionPolicyRequest getPermissionPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPermissionPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPermissionPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPermissionPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPermissionPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPermissionPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPermissionPolicyRequest, GetPermissionPolicyResponse>()
                            .withOperationName("GetPermissionPolicy")
                            .withMarshaller(new GetPermissionPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getPermissionPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getPermissionPolicyRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetPermissionPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the keys that are currently blocked by a rate-based rule. The maximum number of managed keys that can
     * be blocked for a single rate-based rule is 10,000. If more than 10,000 addresses exceed the rate limit, those
     * with the highest rates are blocked.
     * </p>
     *
     * @param getRateBasedStatementManagedKeysRequest
     * @return A Java Future containing the result of the GetRateBasedStatementManagedKeys operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.GetRateBasedStatementManagedKeys
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/GetRateBasedStatementManagedKeys"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRateBasedStatementManagedKeysResponse> getRateBasedStatementManagedKeys(
            GetRateBasedStatementManagedKeysRequest getRateBasedStatementManagedKeysRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getRateBasedStatementManagedKeysRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRateBasedStatementManagedKeys");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRateBasedStatementManagedKeysResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetRateBasedStatementManagedKeysResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRateBasedStatementManagedKeysResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRateBasedStatementManagedKeysRequest, GetRateBasedStatementManagedKeysResponse>()
                            .withOperationName("GetRateBasedStatementManagedKeys")
                            .withMarshaller(new GetRateBasedStatementManagedKeysRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getRateBasedStatementManagedKeysRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getRateBasedStatementManagedKeysRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<GetRateBasedStatementManagedKeysResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the specified <a>RegexPatternSet</a>.
     * </p>
     *
     * @param getRegexPatternSetRequest
     * @return A Java Future containing the result of the GetRegexPatternSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.GetRegexPatternSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/GetRegexPatternSet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetRegexPatternSetResponse> getRegexPatternSet(GetRegexPatternSetRequest getRegexPatternSetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRegexPatternSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRegexPatternSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRegexPatternSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRegexPatternSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRegexPatternSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRegexPatternSetRequest, GetRegexPatternSetResponse>()
                            .withOperationName("GetRegexPatternSet")
                            .withMarshaller(new GetRegexPatternSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getRegexPatternSetRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getRegexPatternSetRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetRegexPatternSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the specified <a>RuleGroup</a>.
     * </p>
     *
     * @param getRuleGroupRequest
     * @return A Java Future containing the result of the GetRuleGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.GetRuleGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/GetRuleGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetRuleGroupResponse> getRuleGroup(GetRuleGroupRequest getRuleGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRuleGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRuleGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRuleGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetRuleGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRuleGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRuleGroupRequest, GetRuleGroupResponse>()
                            .withOperationName("GetRuleGroup").withMarshaller(new GetRuleGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getRuleGroupRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getRuleGroupRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetRuleGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets detailed information about a specified number of requests--a sample--that AWS WAF randomly selects from
     * among the first 5,000 requests that your AWS resource received during a time range that you choose. You can
     * specify a sample size of up to 500 requests, and you can specify any time range in the previous three hours.
     * </p>
     * <p>
     * <code>GetSampledRequests</code> returns a time range, which is usually the time range that you specified.
     * However, if your resource (such as a CloudFront distribution) received 5,000 requests before the specified time
     * range elapsed, <code>GetSampledRequests</code> returns an updated time range. This new time range indicates the
     * actual period during which AWS WAF selected the requests in the sample.
     * </p>
     *
     * @param getSampledRequestsRequest
     * @return A Java Future containing the result of the GetSampledRequests operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.GetSampledRequests
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/GetSampledRequests" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSampledRequestsResponse> getSampledRequests(GetSampledRequestsRequest getSampledRequestsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSampledRequestsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSampledRequests");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSampledRequestsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSampledRequestsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSampledRequestsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSampledRequestsRequest, GetSampledRequestsResponse>()
                            .withOperationName("GetSampledRequests")
                            .withMarshaller(new GetSampledRequestsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getSampledRequestsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getSampledRequestsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetSampledRequestsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the specified <a>WebACL</a>.
     * </p>
     *
     * @param getWebAclRequest
     * @return A Java Future containing the result of the GetWebACL operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.GetWebACL
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/GetWebACL" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetWebAclResponse> getWebACL(GetWebAclRequest getWebAclRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getWebAclRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetWebACL");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetWebAclResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetWebAclResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetWebAclResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetWebAclRequest, GetWebAclResponse>().withOperationName("GetWebACL")
                            .withMarshaller(new GetWebAclRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getWebAclRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getWebAclRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetWebAclResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the <a>WebACL</a> for the specified resource.
     * </p>
     *
     * @param getWebAclForResourceRequest
     * @return A Java Future containing the result of the GetWebACLForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafUnavailableEntityException AWS WAF couldn’t retrieve the resource that you requested. Retry your
     *         request.</li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.GetWebACLForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/GetWebACLForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetWebAclForResourceResponse> getWebACLForResource(
            GetWebAclForResourceRequest getWebAclForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getWebAclForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetWebACLForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetWebAclForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetWebAclForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetWebAclForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetWebAclForResourceRequest, GetWebAclForResourceResponse>()
                            .withOperationName("GetWebACLForResource")
                            .withMarshaller(new GetWebAclForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getWebAclForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getWebAclForResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetWebAclForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves an array of managed rule groups that are available for you to use. This list includes all AWS Managed
     * Rules rule groups and the AWS Marketplace managed rule groups that you're subscribed to.
     * </p>
     *
     * @param listAvailableManagedRuleGroupsRequest
     * @return A Java Future containing the result of the ListAvailableManagedRuleGroups operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.ListAvailableManagedRuleGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/ListAvailableManagedRuleGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAvailableManagedRuleGroupsResponse> listAvailableManagedRuleGroups(
            ListAvailableManagedRuleGroupsRequest listAvailableManagedRuleGroupsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listAvailableManagedRuleGroupsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAvailableManagedRuleGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAvailableManagedRuleGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAvailableManagedRuleGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAvailableManagedRuleGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAvailableManagedRuleGroupsRequest, ListAvailableManagedRuleGroupsResponse>()
                            .withOperationName("ListAvailableManagedRuleGroups")
                            .withMarshaller(new ListAvailableManagedRuleGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAvailableManagedRuleGroupsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listAvailableManagedRuleGroupsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListAvailableManagedRuleGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves an array of <a>IPSetSummary</a> objects for the IP sets that you manage.
     * </p>
     *
     * @param listIpSetsRequest
     * @return A Java Future containing the result of the ListIPSets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.ListIPSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/ListIPSets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListIpSetsResponse> listIPSets(ListIpSetsRequest listIpSetsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listIpSetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListIPSets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListIpSetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListIpSetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListIpSetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListIpSetsRequest, ListIpSetsResponse>().withOperationName("ListIPSets")
                            .withMarshaller(new ListIpSetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listIpSetsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listIpSetsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListIpSetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves an array of your <a>LoggingConfiguration</a> objects.
     * </p>
     *
     * @param listLoggingConfigurationsRequest
     * @return A Java Future containing the result of the ListLoggingConfigurations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.ListLoggingConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/ListLoggingConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLoggingConfigurationsResponse> listLoggingConfigurations(
            ListLoggingConfigurationsRequest listLoggingConfigurationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLoggingConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLoggingConfigurations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLoggingConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListLoggingConfigurationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLoggingConfigurationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLoggingConfigurationsRequest, ListLoggingConfigurationsResponse>()
                            .withOperationName("ListLoggingConfigurations")
                            .withMarshaller(new ListLoggingConfigurationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLoggingConfigurationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listLoggingConfigurationsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListLoggingConfigurationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves an array of <a>RegexPatternSetSummary</a> objects for the regex pattern sets that you manage.
     * </p>
     *
     * @param listRegexPatternSetsRequest
     * @return A Java Future containing the result of the ListRegexPatternSets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.ListRegexPatternSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/ListRegexPatternSets" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRegexPatternSetsResponse> listRegexPatternSets(
            ListRegexPatternSetsRequest listRegexPatternSetsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRegexPatternSetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRegexPatternSets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRegexPatternSetsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRegexPatternSetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRegexPatternSetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRegexPatternSetsRequest, ListRegexPatternSetsResponse>()
                            .withOperationName("ListRegexPatternSets")
                            .withMarshaller(new ListRegexPatternSetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listRegexPatternSetsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listRegexPatternSetsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListRegexPatternSetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves an array of the Amazon Resource Names (ARNs) for the regional resources that are associated with the
     * specified web ACL. If you want the list of AWS CloudFront resources, use the AWS CloudFront call
     * <code>ListDistributionsByWebACLId</code>.
     * </p>
     *
     * @param listResourcesForWebAclRequest
     * @return A Java Future containing the result of the ListResourcesForWebACL operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.ListResourcesForWebACL
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/ListResourcesForWebACL" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListResourcesForWebAclResponse> listResourcesForWebACL(
            ListResourcesForWebAclRequest listResourcesForWebAclRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourcesForWebAclRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResourcesForWebACL");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResourcesForWebAclResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListResourcesForWebAclResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResourcesForWebAclResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResourcesForWebAclRequest, ListResourcesForWebAclResponse>()
                            .withOperationName("ListResourcesForWebACL")
                            .withMarshaller(new ListResourcesForWebAclRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listResourcesForWebAclRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listResourcesForWebAclRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListResourcesForWebAclResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves an array of <a>RuleGroupSummary</a> objects for the rule groups that you manage.
     * </p>
     *
     * @param listRuleGroupsRequest
     * @return A Java Future containing the result of the ListRuleGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.ListRuleGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/ListRuleGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListRuleGroupsResponse> listRuleGroups(ListRuleGroupsRequest listRuleGroupsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRuleGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRuleGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRuleGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRuleGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRuleGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRuleGroupsRequest, ListRuleGroupsResponse>()
                            .withOperationName("ListRuleGroups")
                            .withMarshaller(new ListRuleGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listRuleGroupsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listRuleGroupsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListRuleGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the <a>TagInfoForResource</a> for the specified resource. Tags are key:value pairs that you can use to
     * categorize and manage your resources, for purposes like billing. For example, you might set the tag key to
     * "customer" and the value to the customer name or ID. You can specify one or more tags to add to each AWS
     * resource, up to 50 tags for a resource.
     * </p>
     * <p>
     * You can tag the AWS resources that you manage through AWS WAF: web ACLs, rule groups, IP sets, and regex pattern
     * sets. You can't manage or view tags through the AWS WAF console.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafTagOperationException An error occurred during the tagging operation. Retry your request.</li>
     *         <li>WafTagOperationInternalErrorException AWS WAF couldn’t perform your tagging operation because of an
     *         internal error. Retry your request.</li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsForResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves an array of <a>WebACLSummary</a> objects for the web ACLs that you manage.
     * </p>
     *
     * @param listWebAcLsRequest
     * @return A Java Future containing the result of the ListWebACLs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.ListWebACLs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/ListWebACLs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListWebAcLsResponse> listWebACLs(ListWebAcLsRequest listWebAcLsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWebAcLsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWebACLs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWebAcLsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListWebAcLsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListWebAcLsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWebAcLsRequest, ListWebAcLsResponse>()
                            .withOperationName("ListWebACLs").withMarshaller(new ListWebAcLsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listWebAcLsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listWebAcLsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListWebAcLsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables the specified <a>LoggingConfiguration</a>, to start logging from a web ACL, according to the
     * configuration provided.
     * </p>
     * <p>
     * You can access information about all traffic that AWS WAF inspects using the following steps:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Create an Amazon Kinesis Data Firehose.
     * </p>
     * <p>
     * Create the data firehose with a PUT source and in the Region that you are operating. If you are capturing logs
     * for Amazon CloudFront, always create the firehose in US East (N. Virginia).
     * </p>
     * <p>
     * Give the data firehose a name that starts with the prefix <code>aws-waf-logs-</code>. For example,
     * <code>aws-waf-logs-us-east-2-analytics</code>.
     * </p>
     * <note>
     * <p>
     * Do not create the data firehose using a <code>Kinesis stream</code> as your source.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * Associate that firehose to your web ACL using a <code>PutLoggingConfiguration</code> request.
     * </p>
     * </li>
     * </ol>
     * <p>
     * When you successfully enable logging using a <code>PutLoggingConfiguration</code> request, AWS WAF will create a
     * service linked role with the necessary permissions to write logs to the Amazon Kinesis Data Firehose. For more
     * information, see <a href="https://docs.aws.amazon.com/waf/latest/developerguide/logging.html">Logging Web ACL
     * Traffic Information</a> in the <i>AWS WAF Developer Guide</i>.
     * </p>
     *
     * @param putLoggingConfigurationRequest
     * @return A Java Future containing the result of the PutLoggingConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafOptimisticLockException AWS WAF couldn’t save your changes because you tried to update or delete a
     *         resource that has changed since you last retrieved it. Get the resource again, make any changes you need
     *         to make to the new copy, and retry your operation.</li>
     *         <li>WafServiceLinkedRoleErrorException AWS WAF is not able to access the service linked role. This can be
     *         caused by a previous <code>PutLoggingConfiguration</code> request, which can lock the service linked role
     *         for about 20 seconds. Please try your request again. The service linked role can also be locked by a
     *         previous <code>DeleteServiceLinkedRole</code> request, which can lock the role for 15 minutes or more. If
     *         you recently made a call to <code>DeleteServiceLinkedRole</code>, wait at least 15 minutes and try the
     *         request again. If you receive this same exception again, you will have to wait additional time until the
     *         role is unlocked.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>WafLimitsExceededException AWS WAF couldn’t perform the operation because you exceeded your resource
     *         limit. For example, the maximum number of <code>WebACL</code> objects that you can create for an AWS
     *         account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">Limits</a> in the <i>AWS WAF
     *         Developer Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.PutLoggingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/PutLoggingConfiguration" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutLoggingConfigurationResponse> putLoggingConfiguration(
            PutLoggingConfigurationRequest putLoggingConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putLoggingConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutLoggingConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutLoggingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutLoggingConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutLoggingConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutLoggingConfigurationRequest, PutLoggingConfigurationResponse>()
                            .withOperationName("PutLoggingConfiguration")
                            .withMarshaller(new PutLoggingConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putLoggingConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putLoggingConfigurationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<PutLoggingConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Attaches an IAM policy to the specified resource. Use this to share a rule group across accounts.
     * </p>
     * <p>
     * You must be the owner of the rule group to perform this operation.
     * </p>
     * <p>
     * This action is subject to the following restrictions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * You can attach only one policy with each <code>PutPermissionPolicy</code> request.
     * </p>
     * </li>
     * <li>
     * <p>
     * The ARN in the request must be a valid WAF <a>RuleGroup</a> ARN and the rule group must exist in the same region.
     * </p>
     * </li>
     * <li>
     * <p>
     * The user making the request must be the owner of the rule group.
     * </p>
     * </li>
     * </ul>
     *
     * @param putPermissionPolicyRequest
     * @return A Java Future containing the result of the PutPermissionPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafInvalidPermissionPolicyException The operation failed because the specified policy isn't in the
     *         proper format. </p>
     *         <p>
     *         The policy specifications must conform to the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The policy must be composed using IAM Policy version 2012-10-17 or version 2015-01-01.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The policy must include specifications for <code>Effect</code>, <code>Action</code>, and
     *         <code>Principal</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Effect</code> must specify <code>Allow</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Action</code> must specify <code>wafv2:CreateWebACL</code>, <code>wafv2:UpdateWebACL</code>, and
     *         <code>wafv2:PutFirewallManagerRuleGroups</code>. AWS WAF rejects any extra actions or wildcard actions in
     *         the policy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The policy must not include a <code>Resource</code> parameter.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies.html">IAM Policies</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.PutPermissionPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/PutPermissionPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutPermissionPolicyResponse> putPermissionPolicy(
            PutPermissionPolicyRequest putPermissionPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putPermissionPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutPermissionPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutPermissionPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutPermissionPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutPermissionPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutPermissionPolicyRequest, PutPermissionPolicyResponse>()
                            .withOperationName("PutPermissionPolicy")
                            .withMarshaller(new PutPermissionPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putPermissionPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putPermissionPolicyRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<PutPermissionPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates tags with the specified AWS resource. Tags are key:value pairs that you can use to categorize and
     * manage your resources, for purposes like billing. For example, you might set the tag key to "customer" and the
     * value to the customer name or ID. You can specify one or more tags to add to each AWS resource, up to 50 tags for
     * a resource.
     * </p>
     * <p>
     * You can tag the AWS resources that you manage through AWS WAF: web ACLs, rule groups, IP sets, and regex pattern
     * sets. You can't manage or view tags through the AWS WAF console.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafLimitsExceededException AWS WAF couldn’t perform the operation because you exceeded your resource
     *         limit. For example, the maximum number of <code>WebACL</code> objects that you can create for an AWS
     *         account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">Limits</a> in the <i>AWS WAF
     *         Developer Guide</i>.</li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafTagOperationException An error occurred during the tagging operation. Retry your request.</li>
     *         <li>WafTagOperationInternalErrorException AWS WAF couldn’t perform your tagging operation because of an
     *         internal error. Retry your request.</li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = tagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates tags from an AWS resource. Tags are key:value pairs that you can associate with AWS resources. For
     * example, the tag key might be "customer" and the tag value might be "companyA." You can specify one or more tags
     * to add to each container. You can add up to 50 tags to each AWS resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafTagOperationException An error occurred during the tagging operation. Retry your request.</li>
     *         <li>WafTagOperationInternalErrorException AWS WAF couldn’t perform your tagging operation because of an
     *         internal error. Retry your request.</li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = untagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified <a>IPSet</a>.
     * </p>
     * <note>
     * <p>
     * This operation completely replaces any IP address specifications that you already have in the IP set with the
     * ones that you provide to this call. If you want to add to or modify the addresses that are already in the IP set,
     * retrieve those by calling <a>GetIPSet</a>, update them, and provide the complete updated array of IP addresses to
     * this call.
     * </p>
     * </note>
     *
     * @param updateIpSetRequest
     * @return A Java Future containing the result of the UpdateIPSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafDuplicateItemException AWS WAF couldn’t perform the operation because the resource that you tried
     *         to save is a duplicate of an existing one.</li>
     *         <li>WafOptimisticLockException AWS WAF couldn’t save your changes because you tried to update or delete a
     *         resource that has changed since you last retrieved it. Get the resource again, make any changes you need
     *         to make to the new copy, and retry your operation.</li>
     *         <li>WafLimitsExceededException AWS WAF couldn’t perform the operation because you exceeded your resource
     *         limit. For example, the maximum number of <code>WebACL</code> objects that you can create for an AWS
     *         account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">Limits</a> in the <i>AWS WAF
     *         Developer Guide</i>.</li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.UpdateIPSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/UpdateIPSet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateIpSetResponse> updateIPSet(UpdateIpSetRequest updateIpSetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateIpSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateIPSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateIpSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateIpSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateIpSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateIpSetRequest, UpdateIpSetResponse>()
                            .withOperationName("UpdateIPSet").withMarshaller(new UpdateIpSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateIpSetRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateIpSetRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateIpSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified <a>RegexPatternSet</a>.
     * </p>
     *
     * @param updateRegexPatternSetRequest
     * @return A Java Future containing the result of the UpdateRegexPatternSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafDuplicateItemException AWS WAF couldn’t perform the operation because the resource that you tried
     *         to save is a duplicate of an existing one.</li>
     *         <li>WafOptimisticLockException AWS WAF couldn’t save your changes because you tried to update or delete a
     *         resource that has changed since you last retrieved it. Get the resource again, make any changes you need
     *         to make to the new copy, and retry your operation.</li>
     *         <li>WafLimitsExceededException AWS WAF couldn’t perform the operation because you exceeded your resource
     *         limit. For example, the maximum number of <code>WebACL</code> objects that you can create for an AWS
     *         account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">Limits</a> in the <i>AWS WAF
     *         Developer Guide</i>.</li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.UpdateRegexPatternSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/UpdateRegexPatternSet" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRegexPatternSetResponse> updateRegexPatternSet(
            UpdateRegexPatternSetRequest updateRegexPatternSetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRegexPatternSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRegexPatternSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRegexPatternSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateRegexPatternSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateRegexPatternSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRegexPatternSetRequest, UpdateRegexPatternSetResponse>()
                            .withOperationName("UpdateRegexPatternSet")
                            .withMarshaller(new UpdateRegexPatternSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateRegexPatternSetRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateRegexPatternSetRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<UpdateRegexPatternSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified <a>RuleGroup</a>.
     * </p>
     * <p>
     * A rule group defines a collection of rules to inspect and control web requests that you can use in a
     * <a>WebACL</a>. When you create a rule group, you define an immutable capacity limit. If you update a rule group,
     * you must stay within the capacity. This allows others to reuse the rule group with confidence in its capacity
     * requirements.
     * </p>
     *
     * @param updateRuleGroupRequest
     * @return A Java Future containing the result of the UpdateRuleGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafDuplicateItemException AWS WAF couldn’t perform the operation because the resource that you tried
     *         to save is a duplicate of an existing one.</li>
     *         <li>WafOptimisticLockException AWS WAF couldn’t save your changes because you tried to update or delete a
     *         resource that has changed since you last retrieved it. Get the resource again, make any changes you need
     *         to make to the new copy, and retry your operation.</li>
     *         <li>WafLimitsExceededException AWS WAF couldn’t perform the operation because you exceeded your resource
     *         limit. For example, the maximum number of <code>WebACL</code> objects that you can create for an AWS
     *         account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">Limits</a> in the <i>AWS WAF
     *         Developer Guide</i>.</li>
     *         <li>WafUnavailableEntityException AWS WAF couldn’t retrieve the resource that you requested. Retry your
     *         request.</li>
     *         <li>WafSubscriptionNotFoundException You tried to use a managed rule group that's available by
     *         subscription, but you aren't subscribed to it yet.</li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.UpdateRuleGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/UpdateRuleGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRuleGroupResponse> updateRuleGroup(UpdateRuleGroupRequest updateRuleGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRuleGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRuleGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRuleGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateRuleGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateRuleGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRuleGroupRequest, UpdateRuleGroupResponse>()
                            .withOperationName("UpdateRuleGroup")
                            .withMarshaller(new UpdateRuleGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateRuleGroupRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateRuleGroupRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateRuleGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified <a>WebACL</a>.
     * </p>
     * <p>
     * A Web ACL defines a collection of rules to use to inspect and control web requests. Each rule has an action
     * defined (allow, block, or count) for requests that match the statement of the rule. In the Web ACL, you assign a
     * default action to take (allow, block) for any request that does not match any of the rules. The rules in a Web
     * ACL can be a combination of the types <a>Rule</a>, <a>RuleGroup</a>, and managed rule group. You can associate a
     * Web ACL with one or more AWS resources to protect. The resources can be Amazon CloudFront, an Amazon API Gateway
     * REST API, an Application Load Balancer, or an AWS AppSync GraphQL API.
     * </p>
     *
     * @param updateWebAclRequest
     * @return A Java Future containing the result of the UpdateWebACL operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>WafInternalErrorException Your request is valid, but AWS WAF couldn’t perform the operation because
     *         of a system problem. Retry your request.</li>
     *         <li>WafInvalidParameterException The operation failed because AWS WAF didn't recognize a parameter in the
     *         request. For example: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You specified an invalid parameter name or value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your nested statement isn't valid. You might have tried to nest a statement that can’t be nested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a <code>WebACL</code> with a <code>DefaultAction</code> that isn't among the types
     *         available at <a>DefaultAction</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your request references an ARN that is malformed, or corresponds to a resource with which a Web ACL
     *         cannot be associated.
     *         </p>
     *         </li></li>
     *         <li>WafNonexistentItemException AWS WAF couldn’t perform the operation because your resource doesn’t
     *         exist.</li>
     *         <li>WafDuplicateItemException AWS WAF couldn’t perform the operation because the resource that you tried
     *         to save is a duplicate of an existing one.</li>
     *         <li>WafOptimisticLockException AWS WAF couldn’t save your changes because you tried to update or delete a
     *         resource that has changed since you last retrieved it. Get the resource again, make any changes you need
     *         to make to the new copy, and retry your operation.</li>
     *         <li>WafLimitsExceededException AWS WAF couldn’t perform the operation because you exceeded your resource
     *         limit. For example, the maximum number of <code>WebACL</code> objects that you can create for an AWS
     *         account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">Limits</a> in the <i>AWS WAF
     *         Developer Guide</i>.</li>
     *         <li>WafInvalidResourceException AWS WAF couldn’t perform the operation because the resource that you
     *         requested isn’t valid. Check the resource, and try again.</li>
     *         <li>WafUnavailableEntityException AWS WAF couldn’t retrieve the resource that you requested. Retry your
     *         request.</li>
     *         <li>WafSubscriptionNotFoundException You tried to use a managed rule group that's available by
     *         subscription, but you aren't subscribed to it yet.</li>
     *         <li>WafInvalidOperationException The operation isn't valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Wafv2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Wafv2AsyncClient.UpdateWebACL
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/UpdateWebACL" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateWebAclResponse> updateWebACL(UpdateWebAclRequest updateWebAclRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateWebAclRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WAFV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateWebACL");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateWebAclResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateWebAclResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateWebAclResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateWebAclRequest, UpdateWebAclResponse>()
                            .withOperationName("UpdateWebACL").withMarshaller(new UpdateWebAclRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateWebAclRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateWebAclRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateWebAclResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(Wafv2Exception::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("WAFUnavailableEntityException")
                                .exceptionBuilderSupplier(WafUnavailableEntityException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("WAFServiceLinkedRoleErrorException")
                                .exceptionBuilderSupplier(WafServiceLinkedRoleErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("WAFOptimisticLockException")
                                .exceptionBuilderSupplier(WafOptimisticLockException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("WAFInvalidPermissionPolicyException")
                                .exceptionBuilderSupplier(WafInvalidPermissionPolicyException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("WAFSubscriptionNotFoundException")
                                .exceptionBuilderSupplier(WafSubscriptionNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("WAFAssociatedItemException")
                                .exceptionBuilderSupplier(WafAssociatedItemException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("WAFLimitsExceededException")
                                .exceptionBuilderSupplier(WafLimitsExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("WAFTagOperationException")
                                .exceptionBuilderSupplier(WafTagOperationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("WAFInvalidParameterException")
                                .exceptionBuilderSupplier(WafInvalidParameterException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("WAFTagOperationInternalErrorException")
                                .exceptionBuilderSupplier(WafTagOperationInternalErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("WAFDuplicateItemException")
                                .exceptionBuilderSupplier(WafDuplicateItemException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("WAFInvalidOperationException")
                                .exceptionBuilderSupplier(WafInvalidOperationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("WAFNonexistentItemException")
                                .exceptionBuilderSupplier(WafNonexistentItemException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("WAFInternalErrorException")
                                .exceptionBuilderSupplier(WafInternalErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("WAFInvalidResourceException")
                                .exceptionBuilderSupplier(WafInvalidResourceException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
