/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A custom response to send to the client. You can define a custom response for rule actions and default web ACL
 * actions that are set to <a>BlockAction</a>.
 * </p>
 * <p>
 * For information about customizing web requests and responses, see <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-custom-request-response.html">Customizing web
 * requests and responses in WAF</a> in the <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">WAF Developer Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CustomResponse implements SdkPojo, Serializable, ToCopyableBuilder<CustomResponse.Builder, CustomResponse> {
    private static final SdkField<Integer> RESPONSE_CODE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ResponseCode").getter(getter(CustomResponse::responseCode)).setter(setter(Builder::responseCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResponseCode").build()).build();

    private static final SdkField<String> CUSTOM_RESPONSE_BODY_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CustomResponseBodyKey").getter(getter(CustomResponse::customResponseBodyKey))
            .setter(setter(Builder::customResponseBodyKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomResponseBodyKey").build())
            .build();

    private static final SdkField<List<CustomHTTPHeader>> RESPONSE_HEADERS_FIELD = SdkField
            .<List<CustomHTTPHeader>> builder(MarshallingType.LIST)
            .memberName("ResponseHeaders")
            .getter(getter(CustomResponse::responseHeaders))
            .setter(setter(Builder::responseHeaders))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResponseHeaders").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<CustomHTTPHeader> builder(MarshallingType.SDK_POJO)
                                            .constructor(CustomHTTPHeader::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESPONSE_CODE_FIELD,
            CUSTOM_RESPONSE_BODY_KEY_FIELD, RESPONSE_HEADERS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer responseCode;

    private final String customResponseBodyKey;

    private final List<CustomHTTPHeader> responseHeaders;

    private CustomResponse(BuilderImpl builder) {
        this.responseCode = builder.responseCode;
        this.customResponseBodyKey = builder.customResponseBodyKey;
        this.responseHeaders = builder.responseHeaders;
    }

    /**
     * <p>
     * The HTTP status code to return to the client.
     * </p>
     * <p>
     * For a list of status codes that you can use in your custom reqponses, see <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/customizing-the-response-status-codes.html">Supported
     * status codes for custom response</a> in the <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">WAF Developer Guide</a>.
     * </p>
     * 
     * @return The HTTP status code to return to the client. </p>
     *         <p>
     *         For a list of status codes that you can use in your custom reqponses, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/customizing-the-response-status-codes.html"
     *         >Supported status codes for custom response</a> in the <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">WAF Developer Guide</a>.
     */
    public final Integer responseCode() {
        return responseCode;
    }

    /**
     * <p>
     * References the response body that you want WAF to return to the web request client. You can define a custom
     * response for a rule action or a default web ACL action that is set to block. To do this, you first define the
     * response body key and value in the <code>CustomResponseBodies</code> setting for the <a>WebACL</a> or
     * <a>RuleGroup</a> where you want to use it. Then, in the rule action or web ACL default action
     * <code>BlockAction</code> setting, you reference the response body using this key.
     * </p>
     * 
     * @return References the response body that you want WAF to return to the web request client. You can define a
     *         custom response for a rule action or a default web ACL action that is set to block. To do this, you first
     *         define the response body key and value in the <code>CustomResponseBodies</code> setting for the
     *         <a>WebACL</a> or <a>RuleGroup</a> where you want to use it. Then, in the rule action or web ACL default
     *         action <code>BlockAction</code> setting, you reference the response body using this key.
     */
    public final String customResponseBodyKey() {
        return customResponseBodyKey;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResponseHeaders property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResponseHeaders() {
        return responseHeaders != null && !(responseHeaders instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The HTTP headers to use in the response. Duplicate header names are not allowed.
     * </p>
     * <p>
     * For information about the limits on count and size for custom request and response settings, see <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">WAF quotas</a> in the <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">WAF Developer Guide</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResponseHeaders} method.
     * </p>
     * 
     * @return The HTTP headers to use in the response. Duplicate header names are not allowed. </p>
     *         <p>
     *         For information about the limits on count and size for custom request and response settings, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">WAF quotas</a> in the <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">WAF Developer Guide</a>.
     */
    public final List<CustomHTTPHeader> responseHeaders() {
        return responseHeaders;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(responseCode());
        hashCode = 31 * hashCode + Objects.hashCode(customResponseBodyKey());
        hashCode = 31 * hashCode + Objects.hashCode(hasResponseHeaders() ? responseHeaders() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CustomResponse)) {
            return false;
        }
        CustomResponse other = (CustomResponse) obj;
        return Objects.equals(responseCode(), other.responseCode())
                && Objects.equals(customResponseBodyKey(), other.customResponseBodyKey())
                && hasResponseHeaders() == other.hasResponseHeaders()
                && Objects.equals(responseHeaders(), other.responseHeaders());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CustomResponse").add("ResponseCode", responseCode())
                .add("CustomResponseBodyKey", customResponseBodyKey())
                .add("ResponseHeaders", hasResponseHeaders() ? responseHeaders() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ResponseCode":
            return Optional.ofNullable(clazz.cast(responseCode()));
        case "CustomResponseBodyKey":
            return Optional.ofNullable(clazz.cast(customResponseBodyKey()));
        case "ResponseHeaders":
            return Optional.ofNullable(clazz.cast(responseHeaders()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CustomResponse, T> g) {
        return obj -> g.apply((CustomResponse) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CustomResponse> {
        /**
         * <p>
         * The HTTP status code to return to the client.
         * </p>
         * <p>
         * For a list of status codes that you can use in your custom reqponses, see <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/customizing-the-response-status-codes.html"
         * >Supported status codes for custom response</a> in the <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">WAF Developer Guide</a>.
         * </p>
         * 
         * @param responseCode
         *        The HTTP status code to return to the client. </p>
         *        <p>
         *        For a list of status codes that you can use in your custom reqponses, see <a href=
         *        "https://docs.aws.amazon.com/waf/latest/developerguide/customizing-the-response-status-codes.html"
         *        >Supported status codes for custom response</a> in the <a
         *        href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">WAF Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseCode(Integer responseCode);

        /**
         * <p>
         * References the response body that you want WAF to return to the web request client. You can define a custom
         * response for a rule action or a default web ACL action that is set to block. To do this, you first define the
         * response body key and value in the <code>CustomResponseBodies</code> setting for the <a>WebACL</a> or
         * <a>RuleGroup</a> where you want to use it. Then, in the rule action or web ACL default action
         * <code>BlockAction</code> setting, you reference the response body using this key.
         * </p>
         * 
         * @param customResponseBodyKey
         *        References the response body that you want WAF to return to the web request client. You can define a
         *        custom response for a rule action or a default web ACL action that is set to block. To do this, you
         *        first define the response body key and value in the <code>CustomResponseBodies</code> setting for the
         *        <a>WebACL</a> or <a>RuleGroup</a> where you want to use it. Then, in the rule action or web ACL
         *        default action <code>BlockAction</code> setting, you reference the response body using this key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customResponseBodyKey(String customResponseBodyKey);

        /**
         * <p>
         * The HTTP headers to use in the response. Duplicate header names are not allowed.
         * </p>
         * <p>
         * For information about the limits on count and size for custom request and response settings, see <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">WAF quotas</a> in the <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">WAF Developer Guide</a>.
         * </p>
         * 
         * @param responseHeaders
         *        The HTTP headers to use in the response. Duplicate header names are not allowed. </p>
         *        <p>
         *        For information about the limits on count and size for custom request and response settings, see <a
         *        href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">WAF quotas</a> in the <a
         *        href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">WAF Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseHeaders(Collection<CustomHTTPHeader> responseHeaders);

        /**
         * <p>
         * The HTTP headers to use in the response. Duplicate header names are not allowed.
         * </p>
         * <p>
         * For information about the limits on count and size for custom request and response settings, see <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">WAF quotas</a> in the <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">WAF Developer Guide</a>.
         * </p>
         * 
         * @param responseHeaders
         *        The HTTP headers to use in the response. Duplicate header names are not allowed. </p>
         *        <p>
         *        For information about the limits on count and size for custom request and response settings, see <a
         *        href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">WAF quotas</a> in the <a
         *        href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">WAF Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseHeaders(CustomHTTPHeader... responseHeaders);

        /**
         * <p>
         * The HTTP headers to use in the response. Duplicate header names are not allowed.
         * </p>
         * <p>
         * For information about the limits on count and size for custom request and response settings, see <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">WAF quotas</a> in the <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">WAF Developer Guide</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<CustomHTTPHeader>.Builder} avoiding the
         * need to create one manually via {@link List<CustomHTTPHeader>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<CustomHTTPHeader>.Builder#build()} is called immediately and
         * its result is passed to {@link #responseHeaders(List<CustomHTTPHeader>)}.
         * 
         * @param responseHeaders
         *        a consumer that will call methods on {@link List<CustomHTTPHeader>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #responseHeaders(List<CustomHTTPHeader>)
         */
        Builder responseHeaders(Consumer<CustomHTTPHeader.Builder>... responseHeaders);
    }

    static final class BuilderImpl implements Builder {
        private Integer responseCode;

        private String customResponseBodyKey;

        private List<CustomHTTPHeader> responseHeaders = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CustomResponse model) {
            responseCode(model.responseCode);
            customResponseBodyKey(model.customResponseBodyKey);
            responseHeaders(model.responseHeaders);
        }

        public final Integer getResponseCode() {
            return responseCode;
        }

        public final void setResponseCode(Integer responseCode) {
            this.responseCode = responseCode;
        }

        @Override
        @Transient
        public final Builder responseCode(Integer responseCode) {
            this.responseCode = responseCode;
            return this;
        }

        public final String getCustomResponseBodyKey() {
            return customResponseBodyKey;
        }

        public final void setCustomResponseBodyKey(String customResponseBodyKey) {
            this.customResponseBodyKey = customResponseBodyKey;
        }

        @Override
        @Transient
        public final Builder customResponseBodyKey(String customResponseBodyKey) {
            this.customResponseBodyKey = customResponseBodyKey;
            return this;
        }

        public final List<CustomHTTPHeader.Builder> getResponseHeaders() {
            List<CustomHTTPHeader.Builder> result = CustomHTTPHeadersCopier.copyToBuilder(this.responseHeaders);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setResponseHeaders(Collection<CustomHTTPHeader.BuilderImpl> responseHeaders) {
            this.responseHeaders = CustomHTTPHeadersCopier.copyFromBuilder(responseHeaders);
        }

        @Override
        @Transient
        public final Builder responseHeaders(Collection<CustomHTTPHeader> responseHeaders) {
            this.responseHeaders = CustomHTTPHeadersCopier.copy(responseHeaders);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder responseHeaders(CustomHTTPHeader... responseHeaders) {
            responseHeaders(Arrays.asList(responseHeaders));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder responseHeaders(Consumer<CustomHTTPHeader.Builder>... responseHeaders) {
            responseHeaders(Stream.of(responseHeaders).map(c -> CustomHTTPHeader.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public CustomResponse build() {
            return new CustomResponse(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
