/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A single rule, which you can use in a <a>WebACL</a> or <a>RuleGroup</a> to identify web requests that you want to
 * allow, block, or count. Each rule includes one top-level <a>Statement</a> that WAF uses to identify matching web
 * requests, and parameters that govern how WAF handles them.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Rule implements SdkPojo, Serializable, ToCopyableBuilder<Rule.Builder, Rule> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Rule::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<Integer> PRIORITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Priority").getter(getter(Rule::priority)).setter(setter(Builder::priority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Priority").build()).build();

    private static final SdkField<Statement> STATEMENT_FIELD = SdkField.<Statement> builder(MarshallingType.SDK_POJO)
            .memberName("Statement").getter(getter(Rule::statement)).setter(setter(Builder::statement))
            .constructor(Statement::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Statement").build()).build();

    private static final SdkField<RuleAction> ACTION_FIELD = SdkField.<RuleAction> builder(MarshallingType.SDK_POJO)
            .memberName("Action").getter(getter(Rule::action)).setter(setter(Builder::action)).constructor(RuleAction::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Action").build()).build();

    private static final SdkField<OverrideAction> OVERRIDE_ACTION_FIELD = SdkField
            .<OverrideAction> builder(MarshallingType.SDK_POJO).memberName("OverrideAction").getter(getter(Rule::overrideAction))
            .setter(setter(Builder::overrideAction)).constructor(OverrideAction::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OverrideAction").build()).build();

    private static final SdkField<List<Label>> RULE_LABELS_FIELD = SdkField
            .<List<Label>> builder(MarshallingType.LIST)
            .memberName("RuleLabels")
            .getter(getter(Rule::ruleLabels))
            .setter(setter(Builder::ruleLabels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleLabels").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Label> builder(MarshallingType.SDK_POJO)
                                            .constructor(Label::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<VisibilityConfig> VISIBILITY_CONFIG_FIELD = SdkField
            .<VisibilityConfig> builder(MarshallingType.SDK_POJO).memberName("VisibilityConfig")
            .getter(getter(Rule::visibilityConfig)).setter(setter(Builder::visibilityConfig))
            .constructor(VisibilityConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VisibilityConfig").build()).build();

    private static final SdkField<CaptchaConfig> CAPTCHA_CONFIG_FIELD = SdkField
            .<CaptchaConfig> builder(MarshallingType.SDK_POJO).memberName("CaptchaConfig").getter(getter(Rule::captchaConfig))
            .setter(setter(Builder::captchaConfig)).constructor(CaptchaConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CaptchaConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, PRIORITY_FIELD,
            STATEMENT_FIELD, ACTION_FIELD, OVERRIDE_ACTION_FIELD, RULE_LABELS_FIELD, VISIBILITY_CONFIG_FIELD,
            CAPTCHA_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final Integer priority;

    private final Statement statement;

    private final RuleAction action;

    private final OverrideAction overrideAction;

    private final List<Label> ruleLabels;

    private final VisibilityConfig visibilityConfig;

    private final CaptchaConfig captchaConfig;

    private Rule(BuilderImpl builder) {
        this.name = builder.name;
        this.priority = builder.priority;
        this.statement = builder.statement;
        this.action = builder.action;
        this.overrideAction = builder.overrideAction;
        this.ruleLabels = builder.ruleLabels;
        this.visibilityConfig = builder.visibilityConfig;
        this.captchaConfig = builder.captchaConfig;
    }

    /**
     * <p>
     * The name of the rule. You can't change the name of a <code>Rule</code> after you create it.
     * </p>
     * 
     * @return The name of the rule. You can't change the name of a <code>Rule</code> after you create it.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * If you define more than one <code>Rule</code> in a <code>WebACL</code>, WAF evaluates each request against the
     * <code>Rules</code> in order based on the value of <code>Priority</code>. WAF processes rules with lower priority
     * first. The priorities don't need to be consecutive, but they must all be different.
     * </p>
     * 
     * @return If you define more than one <code>Rule</code> in a <code>WebACL</code>, WAF evaluates each request
     *         against the <code>Rules</code> in order based on the value of <code>Priority</code>. WAF processes rules
     *         with lower priority first. The priorities don't need to be consecutive, but they must all be different.
     */
    public final Integer priority() {
        return priority;
    }

    /**
     * <p>
     * The WAF processing statement for the rule, for example <a>ByteMatchStatement</a> or
     * <a>SizeConstraintStatement</a>.
     * </p>
     * 
     * @return The WAF processing statement for the rule, for example <a>ByteMatchStatement</a> or
     *         <a>SizeConstraintStatement</a>.
     */
    public final Statement statement() {
        return statement;
    }

    /**
     * <p>
     * The action that WAF should take on a web request when it matches the rule statement. Settings at the web ACL
     * level can override the rule action setting.
     * </p>
     * <p>
     * This is used only for rules whose statements do not reference a rule group. Rule statements that reference a rule
     * group include <code>RuleGroupReferenceStatement</code> and <code>ManagedRuleGroupStatement</code>.
     * </p>
     * <p>
     * You must specify either this <code>Action</code> setting or the rule <code>OverrideAction</code> setting, but not
     * both:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the rule statement does not reference a rule group, use this rule action setting and not the rule override
     * action setting.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the rule statement references a rule group, use the override action setting and not this action setting.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The action that WAF should take on a web request when it matches the rule statement. Settings at the web
     *         ACL level can override the rule action setting. </p>
     *         <p>
     *         This is used only for rules whose statements do not reference a rule group. Rule statements that
     *         reference a rule group include <code>RuleGroupReferenceStatement</code> and
     *         <code>ManagedRuleGroupStatement</code>.
     *         </p>
     *         <p>
     *         You must specify either this <code>Action</code> setting or the rule <code>OverrideAction</code> setting,
     *         but not both:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If the rule statement does not reference a rule group, use this rule action setting and not the rule
     *         override action setting.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the rule statement references a rule group, use the override action setting and not this action
     *         setting.
     *         </p>
     *         </li>
     */
    public final RuleAction action() {
        return action;
    }

    /**
     * <p>
     * The action to use in the place of the action that results from the rule group evaluation. Set the override action
     * to none to leave the result of the rule group alone. Set it to count to override the result to count only.
     * </p>
     * <p>
     * You can only use this for rule statements that reference a rule group, like
     * <code>RuleGroupReferenceStatement</code> and <code>ManagedRuleGroupStatement</code>.
     * </p>
     * <note>
     * <p>
     * This option is usually set to none. It does not affect how the rules in the rule group are evaluated. If you want
     * the rules in the rule group to only count matches, do not use this and instead exclude those rules in your rule
     * group reference statement settings.
     * </p>
     * </note>
     * 
     * @return The action to use in the place of the action that results from the rule group evaluation. Set the
     *         override action to none to leave the result of the rule group alone. Set it to count to override the
     *         result to count only. </p>
     *         <p>
     *         You can only use this for rule statements that reference a rule group, like
     *         <code>RuleGroupReferenceStatement</code> and <code>ManagedRuleGroupStatement</code>.
     *         </p>
     *         <note>
     *         <p>
     *         This option is usually set to none. It does not affect how the rules in the rule group are evaluated. If
     *         you want the rules in the rule group to only count matches, do not use this and instead exclude those
     *         rules in your rule group reference statement settings.
     *         </p>
     */
    public final OverrideAction overrideAction() {
        return overrideAction;
    }

    /**
     * For responses, this returns true if the service returned a value for the RuleLabels property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasRuleLabels() {
        return ruleLabels != null && !(ruleLabels instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Labels to apply to web requests that match the rule match statement. WAF applies fully qualified labels to
     * matching web requests. A fully qualified label is the concatenation of a label namespace and a rule label. The
     * rule's rule group or web ACL defines the label namespace.
     * </p>
     * <p>
     * Rules that run after this rule in the web ACL can match against these labels using a
     * <code>LabelMatchStatement</code>.
     * </p>
     * <p>
     * For each label, provide a case-sensitive string containing optional namespaces and a label name, according to the
     * following guidelines:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Separate each component of the label with a colon.
     * </p>
     * </li>
     * <li>
     * <p>
     * Each namespace or name can have up to 128 characters.
     * </p>
     * </li>
     * <li>
     * <p>
     * You can specify up to 5 namespaces in a label.
     * </p>
     * </li>
     * <li>
     * <p>
     * Don't use the following reserved words in your label specification: <code>aws</code>, <code>waf</code>,
     * <code>managed</code>, <code>rulegroup</code>, <code>webacl</code>, <code>regexpatternset</code>, or
     * <code>ipset</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For example, <code>myLabelName</code> or <code>nameSpace1:nameSpace2:myLabelName</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRuleLabels} method.
     * </p>
     * 
     * @return Labels to apply to web requests that match the rule match statement. WAF applies fully qualified labels
     *         to matching web requests. A fully qualified label is the concatenation of a label namespace and a rule
     *         label. The rule's rule group or web ACL defines the label namespace. </p>
     *         <p>
     *         Rules that run after this rule in the web ACL can match against these labels using a
     *         <code>LabelMatchStatement</code>.
     *         </p>
     *         <p>
     *         For each label, provide a case-sensitive string containing optional namespaces and a label name,
     *         according to the following guidelines:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Separate each component of the label with a colon.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Each namespace or name can have up to 128 characters.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You can specify up to 5 namespaces in a label.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Don't use the following reserved words in your label specification: <code>aws</code>, <code>waf</code>,
     *         <code>managed</code>, <code>rulegroup</code>, <code>webacl</code>, <code>regexpatternset</code>, or
     *         <code>ipset</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For example, <code>myLabelName</code> or <code>nameSpace1:nameSpace2:myLabelName</code>.
     */
    public final List<Label> ruleLabels() {
        return ruleLabels;
    }

    /**
     * <p>
     * Defines and enables Amazon CloudWatch metrics and web request sample collection.
     * </p>
     * 
     * @return Defines and enables Amazon CloudWatch metrics and web request sample collection.
     */
    public final VisibilityConfig visibilityConfig() {
        return visibilityConfig;
    }

    /**
     * <p>
     * Specifies how WAF should handle <code>CAPTCHA</code> evaluations. If you don't specify this, WAF uses the
     * <code>CAPTCHA</code> configuration that's defined for the web ACL.
     * </p>
     * 
     * @return Specifies how WAF should handle <code>CAPTCHA</code> evaluations. If you don't specify this, WAF uses the
     *         <code>CAPTCHA</code> configuration that's defined for the web ACL.
     */
    public final CaptchaConfig captchaConfig() {
        return captchaConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(priority());
        hashCode = 31 * hashCode + Objects.hashCode(statement());
        hashCode = 31 * hashCode + Objects.hashCode(action());
        hashCode = 31 * hashCode + Objects.hashCode(overrideAction());
        hashCode = 31 * hashCode + Objects.hashCode(hasRuleLabels() ? ruleLabels() : null);
        hashCode = 31 * hashCode + Objects.hashCode(visibilityConfig());
        hashCode = 31 * hashCode + Objects.hashCode(captchaConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Rule)) {
            return false;
        }
        Rule other = (Rule) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(priority(), other.priority())
                && Objects.equals(statement(), other.statement()) && Objects.equals(action(), other.action())
                && Objects.equals(overrideAction(), other.overrideAction()) && hasRuleLabels() == other.hasRuleLabels()
                && Objects.equals(ruleLabels(), other.ruleLabels())
                && Objects.equals(visibilityConfig(), other.visibilityConfig())
                && Objects.equals(captchaConfig(), other.captchaConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Rule").add("Name", name()).add("Priority", priority()).add("Statement", statement())
                .add("Action", action()).add("OverrideAction", overrideAction())
                .add("RuleLabels", hasRuleLabels() ? ruleLabels() : null).add("VisibilityConfig", visibilityConfig())
                .add("CaptchaConfig", captchaConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Priority":
            return Optional.ofNullable(clazz.cast(priority()));
        case "Statement":
            return Optional.ofNullable(clazz.cast(statement()));
        case "Action":
            return Optional.ofNullable(clazz.cast(action()));
        case "OverrideAction":
            return Optional.ofNullable(clazz.cast(overrideAction()));
        case "RuleLabels":
            return Optional.ofNullable(clazz.cast(ruleLabels()));
        case "VisibilityConfig":
            return Optional.ofNullable(clazz.cast(visibilityConfig()));
        case "CaptchaConfig":
            return Optional.ofNullable(clazz.cast(captchaConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Rule, T> g) {
        return obj -> g.apply((Rule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Rule> {
        /**
         * <p>
         * The name of the rule. You can't change the name of a <code>Rule</code> after you create it.
         * </p>
         * 
         * @param name
         *        The name of the rule. You can't change the name of a <code>Rule</code> after you create it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * If you define more than one <code>Rule</code> in a <code>WebACL</code>, WAF evaluates each request against
         * the <code>Rules</code> in order based on the value of <code>Priority</code>. WAF processes rules with lower
         * priority first. The priorities don't need to be consecutive, but they must all be different.
         * </p>
         * 
         * @param priority
         *        If you define more than one <code>Rule</code> in a <code>WebACL</code>, WAF evaluates each request
         *        against the <code>Rules</code> in order based on the value of <code>Priority</code>. WAF processes
         *        rules with lower priority first. The priorities don't need to be consecutive, but they must all be
         *        different.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priority(Integer priority);

        /**
         * <p>
         * The WAF processing statement for the rule, for example <a>ByteMatchStatement</a> or
         * <a>SizeConstraintStatement</a>.
         * </p>
         * 
         * @param statement
         *        The WAF processing statement for the rule, for example <a>ByteMatchStatement</a> or
         *        <a>SizeConstraintStatement</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statement(Statement statement);

        /**
         * <p>
         * The WAF processing statement for the rule, for example <a>ByteMatchStatement</a> or
         * <a>SizeConstraintStatement</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link Statement.Builder} avoiding the need to create
         * one manually via {@link Statement#builder()}.
         *
         * When the {@link Consumer} completes, {@link Statement.Builder#build()} is called immediately and its result
         * is passed to {@link #statement(Statement)}.
         * 
         * @param statement
         *        a consumer that will call methods on {@link Statement.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #statement(Statement)
         */
        default Builder statement(Consumer<Statement.Builder> statement) {
            return statement(Statement.builder().applyMutation(statement).build());
        }

        /**
         * <p>
         * The action that WAF should take on a web request when it matches the rule statement. Settings at the web ACL
         * level can override the rule action setting.
         * </p>
         * <p>
         * This is used only for rules whose statements do not reference a rule group. Rule statements that reference a
         * rule group include <code>RuleGroupReferenceStatement</code> and <code>ManagedRuleGroupStatement</code>.
         * </p>
         * <p>
         * You must specify either this <code>Action</code> setting or the rule <code>OverrideAction</code> setting, but
         * not both:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the rule statement does not reference a rule group, use this rule action setting and not the rule override
         * action setting.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the rule statement references a rule group, use the override action setting and not this action setting.
         * </p>
         * </li>
         * </ul>
         * 
         * @param action
         *        The action that WAF should take on a web request when it matches the rule statement. Settings at the
         *        web ACL level can override the rule action setting. </p>
         *        <p>
         *        This is used only for rules whose statements do not reference a rule group. Rule statements that
         *        reference a rule group include <code>RuleGroupReferenceStatement</code> and
         *        <code>ManagedRuleGroupStatement</code>.
         *        </p>
         *        <p>
         *        You must specify either this <code>Action</code> setting or the rule <code>OverrideAction</code>
         *        setting, but not both:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the rule statement does not reference a rule group, use this rule action setting and not the rule
         *        override action setting.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the rule statement references a rule group, use the override action setting and not this action
         *        setting.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder action(RuleAction action);

        /**
         * <p>
         * The action that WAF should take on a web request when it matches the rule statement. Settings at the web ACL
         * level can override the rule action setting.
         * </p>
         * <p>
         * This is used only for rules whose statements do not reference a rule group. Rule statements that reference a
         * rule group include <code>RuleGroupReferenceStatement</code> and <code>ManagedRuleGroupStatement</code>.
         * </p>
         * <p>
         * You must specify either this <code>Action</code> setting or the rule <code>OverrideAction</code> setting, but
         * not both:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the rule statement does not reference a rule group, use this rule action setting and not the rule override
         * action setting.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the rule statement references a rule group, use the override action setting and not this action setting.
         * </p>
         * </li>
         * </ul>
         * This is a convenience that creates an instance of the {@link RuleAction.Builder} avoiding the need to create
         * one manually via {@link RuleAction#builder()}.
         *
         * When the {@link Consumer} completes, {@link RuleAction.Builder#build()} is called immediately and its result
         * is passed to {@link #action(RuleAction)}.
         * 
         * @param action
         *        a consumer that will call methods on {@link RuleAction.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #action(RuleAction)
         */
        default Builder action(Consumer<RuleAction.Builder> action) {
            return action(RuleAction.builder().applyMutation(action).build());
        }

        /**
         * <p>
         * The action to use in the place of the action that results from the rule group evaluation. Set the override
         * action to none to leave the result of the rule group alone. Set it to count to override the result to count
         * only.
         * </p>
         * <p>
         * You can only use this for rule statements that reference a rule group, like
         * <code>RuleGroupReferenceStatement</code> and <code>ManagedRuleGroupStatement</code>.
         * </p>
         * <note>
         * <p>
         * This option is usually set to none. It does not affect how the rules in the rule group are evaluated. If you
         * want the rules in the rule group to only count matches, do not use this and instead exclude those rules in
         * your rule group reference statement settings.
         * </p>
         * </note>
         * 
         * @param overrideAction
         *        The action to use in the place of the action that results from the rule group evaluation. Set the
         *        override action to none to leave the result of the rule group alone. Set it to count to override the
         *        result to count only. </p>
         *        <p>
         *        You can only use this for rule statements that reference a rule group, like
         *        <code>RuleGroupReferenceStatement</code> and <code>ManagedRuleGroupStatement</code>.
         *        </p>
         *        <note>
         *        <p>
         *        This option is usually set to none. It does not affect how the rules in the rule group are evaluated.
         *        If you want the rules in the rule group to only count matches, do not use this and instead exclude
         *        those rules in your rule group reference statement settings.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder overrideAction(OverrideAction overrideAction);

        /**
         * <p>
         * The action to use in the place of the action that results from the rule group evaluation. Set the override
         * action to none to leave the result of the rule group alone. Set it to count to override the result to count
         * only.
         * </p>
         * <p>
         * You can only use this for rule statements that reference a rule group, like
         * <code>RuleGroupReferenceStatement</code> and <code>ManagedRuleGroupStatement</code>.
         * </p>
         * <note>
         * <p>
         * This option is usually set to none. It does not affect how the rules in the rule group are evaluated. If you
         * want the rules in the rule group to only count matches, do not use this and instead exclude those rules in
         * your rule group reference statement settings.
         * </p>
         * </note> This is a convenience that creates an instance of the {@link OverrideAction.Builder} avoiding the
         * need to create one manually via {@link OverrideAction#builder()}.
         *
         * When the {@link Consumer} completes, {@link OverrideAction.Builder#build()} is called immediately and its
         * result is passed to {@link #overrideAction(OverrideAction)}.
         * 
         * @param overrideAction
         *        a consumer that will call methods on {@link OverrideAction.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #overrideAction(OverrideAction)
         */
        default Builder overrideAction(Consumer<OverrideAction.Builder> overrideAction) {
            return overrideAction(OverrideAction.builder().applyMutation(overrideAction).build());
        }

        /**
         * <p>
         * Labels to apply to web requests that match the rule match statement. WAF applies fully qualified labels to
         * matching web requests. A fully qualified label is the concatenation of a label namespace and a rule label.
         * The rule's rule group or web ACL defines the label namespace.
         * </p>
         * <p>
         * Rules that run after this rule in the web ACL can match against these labels using a
         * <code>LabelMatchStatement</code>.
         * </p>
         * <p>
         * For each label, provide a case-sensitive string containing optional namespaces and a label name, according to
         * the following guidelines:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Separate each component of the label with a colon.
         * </p>
         * </li>
         * <li>
         * <p>
         * Each namespace or name can have up to 128 characters.
         * </p>
         * </li>
         * <li>
         * <p>
         * You can specify up to 5 namespaces in a label.
         * </p>
         * </li>
         * <li>
         * <p>
         * Don't use the following reserved words in your label specification: <code>aws</code>, <code>waf</code>,
         * <code>managed</code>, <code>rulegroup</code>, <code>webacl</code>, <code>regexpatternset</code>, or
         * <code>ipset</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For example, <code>myLabelName</code> or <code>nameSpace1:nameSpace2:myLabelName</code>.
         * </p>
         * 
         * @param ruleLabels
         *        Labels to apply to web requests that match the rule match statement. WAF applies fully qualified
         *        labels to matching web requests. A fully qualified label is the concatenation of a label namespace and
         *        a rule label. The rule's rule group or web ACL defines the label namespace. </p>
         *        <p>
         *        Rules that run after this rule in the web ACL can match against these labels using a
         *        <code>LabelMatchStatement</code>.
         *        </p>
         *        <p>
         *        For each label, provide a case-sensitive string containing optional namespaces and a label name,
         *        according to the following guidelines:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Separate each component of the label with a colon.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Each namespace or name can have up to 128 characters.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You can specify up to 5 namespaces in a label.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Don't use the following reserved words in your label specification: <code>aws</code>, <code>waf</code>, <code>managed</code>, <code>rulegroup</code>, <code>webacl</code>, <code>regexpatternset</code>, or
         *        <code>ipset</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For example, <code>myLabelName</code> or <code>nameSpace1:nameSpace2:myLabelName</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleLabels(Collection<Label> ruleLabels);

        /**
         * <p>
         * Labels to apply to web requests that match the rule match statement. WAF applies fully qualified labels to
         * matching web requests. A fully qualified label is the concatenation of a label namespace and a rule label.
         * The rule's rule group or web ACL defines the label namespace.
         * </p>
         * <p>
         * Rules that run after this rule in the web ACL can match against these labels using a
         * <code>LabelMatchStatement</code>.
         * </p>
         * <p>
         * For each label, provide a case-sensitive string containing optional namespaces and a label name, according to
         * the following guidelines:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Separate each component of the label with a colon.
         * </p>
         * </li>
         * <li>
         * <p>
         * Each namespace or name can have up to 128 characters.
         * </p>
         * </li>
         * <li>
         * <p>
         * You can specify up to 5 namespaces in a label.
         * </p>
         * </li>
         * <li>
         * <p>
         * Don't use the following reserved words in your label specification: <code>aws</code>, <code>waf</code>,
         * <code>managed</code>, <code>rulegroup</code>, <code>webacl</code>, <code>regexpatternset</code>, or
         * <code>ipset</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For example, <code>myLabelName</code> or <code>nameSpace1:nameSpace2:myLabelName</code>.
         * </p>
         * 
         * @param ruleLabels
         *        Labels to apply to web requests that match the rule match statement. WAF applies fully qualified
         *        labels to matching web requests. A fully qualified label is the concatenation of a label namespace and
         *        a rule label. The rule's rule group or web ACL defines the label namespace. </p>
         *        <p>
         *        Rules that run after this rule in the web ACL can match against these labels using a
         *        <code>LabelMatchStatement</code>.
         *        </p>
         *        <p>
         *        For each label, provide a case-sensitive string containing optional namespaces and a label name,
         *        according to the following guidelines:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Separate each component of the label with a colon.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Each namespace or name can have up to 128 characters.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You can specify up to 5 namespaces in a label.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Don't use the following reserved words in your label specification: <code>aws</code>, <code>waf</code>, <code>managed</code>, <code>rulegroup</code>, <code>webacl</code>, <code>regexpatternset</code>, or
         *        <code>ipset</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For example, <code>myLabelName</code> or <code>nameSpace1:nameSpace2:myLabelName</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleLabels(Label... ruleLabels);

        /**
         * <p>
         * Labels to apply to web requests that match the rule match statement. WAF applies fully qualified labels to
         * matching web requests. A fully qualified label is the concatenation of a label namespace and a rule label.
         * The rule's rule group or web ACL defines the label namespace.
         * </p>
         * <p>
         * Rules that run after this rule in the web ACL can match against these labels using a
         * <code>LabelMatchStatement</code>.
         * </p>
         * <p>
         * For each label, provide a case-sensitive string containing optional namespaces and a label name, according to
         * the following guidelines:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Separate each component of the label with a colon.
         * </p>
         * </li>
         * <li>
         * <p>
         * Each namespace or name can have up to 128 characters.
         * </p>
         * </li>
         * <li>
         * <p>
         * You can specify up to 5 namespaces in a label.
         * </p>
         * </li>
         * <li>
         * <p>
         * Don't use the following reserved words in your label specification: <code>aws</code>, <code>waf</code>,
         * <code>managed</code>, <code>rulegroup</code>, <code>webacl</code>, <code>regexpatternset</code>, or
         * <code>ipset</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For example, <code>myLabelName</code> or <code>nameSpace1:nameSpace2:myLabelName</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Label>.Builder} avoiding the need to create
         * one manually via {@link List<Label>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Label>.Builder#build()} is called immediately and its result
         * is passed to {@link #ruleLabels(List<Label>)}.
         * 
         * @param ruleLabels
         *        a consumer that will call methods on {@link List<Label>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ruleLabels(List<Label>)
         */
        Builder ruleLabels(Consumer<Label.Builder>... ruleLabels);

        /**
         * <p>
         * Defines and enables Amazon CloudWatch metrics and web request sample collection.
         * </p>
         * 
         * @param visibilityConfig
         *        Defines and enables Amazon CloudWatch metrics and web request sample collection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder visibilityConfig(VisibilityConfig visibilityConfig);

        /**
         * <p>
         * Defines and enables Amazon CloudWatch metrics and web request sample collection.
         * </p>
         * This is a convenience that creates an instance of the {@link VisibilityConfig.Builder} avoiding the need to
         * create one manually via {@link VisibilityConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link VisibilityConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #visibilityConfig(VisibilityConfig)}.
         * 
         * @param visibilityConfig
         *        a consumer that will call methods on {@link VisibilityConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #visibilityConfig(VisibilityConfig)
         */
        default Builder visibilityConfig(Consumer<VisibilityConfig.Builder> visibilityConfig) {
            return visibilityConfig(VisibilityConfig.builder().applyMutation(visibilityConfig).build());
        }

        /**
         * <p>
         * Specifies how WAF should handle <code>CAPTCHA</code> evaluations. If you don't specify this, WAF uses the
         * <code>CAPTCHA</code> configuration that's defined for the web ACL.
         * </p>
         * 
         * @param captchaConfig
         *        Specifies how WAF should handle <code>CAPTCHA</code> evaluations. If you don't specify this, WAF uses
         *        the <code>CAPTCHA</code> configuration that's defined for the web ACL.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder captchaConfig(CaptchaConfig captchaConfig);

        /**
         * <p>
         * Specifies how WAF should handle <code>CAPTCHA</code> evaluations. If you don't specify this, WAF uses the
         * <code>CAPTCHA</code> configuration that's defined for the web ACL.
         * </p>
         * This is a convenience that creates an instance of the {@link CaptchaConfig.Builder} avoiding the need to
         * create one manually via {@link CaptchaConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link CaptchaConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #captchaConfig(CaptchaConfig)}.
         * 
         * @param captchaConfig
         *        a consumer that will call methods on {@link CaptchaConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #captchaConfig(CaptchaConfig)
         */
        default Builder captchaConfig(Consumer<CaptchaConfig.Builder> captchaConfig) {
            return captchaConfig(CaptchaConfig.builder().applyMutation(captchaConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private Integer priority;

        private Statement statement;

        private RuleAction action;

        private OverrideAction overrideAction;

        private List<Label> ruleLabels = DefaultSdkAutoConstructList.getInstance();

        private VisibilityConfig visibilityConfig;

        private CaptchaConfig captchaConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(Rule model) {
            name(model.name);
            priority(model.priority);
            statement(model.statement);
            action(model.action);
            overrideAction(model.overrideAction);
            ruleLabels(model.ruleLabels);
            visibilityConfig(model.visibilityConfig);
            captchaConfig(model.captchaConfig);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Integer getPriority() {
            return priority;
        }

        public final void setPriority(Integer priority) {
            this.priority = priority;
        }

        @Override
        @Transient
        public final Builder priority(Integer priority) {
            this.priority = priority;
            return this;
        }

        public final Statement.Builder getStatement() {
            return statement != null ? statement.toBuilder() : null;
        }

        public final void setStatement(Statement.BuilderImpl statement) {
            this.statement = statement != null ? statement.build() : null;
        }

        @Override
        @Transient
        public final Builder statement(Statement statement) {
            this.statement = statement;
            return this;
        }

        public final RuleAction.Builder getAction() {
            return action != null ? action.toBuilder() : null;
        }

        public final void setAction(RuleAction.BuilderImpl action) {
            this.action = action != null ? action.build() : null;
        }

        @Override
        @Transient
        public final Builder action(RuleAction action) {
            this.action = action;
            return this;
        }

        public final OverrideAction.Builder getOverrideAction() {
            return overrideAction != null ? overrideAction.toBuilder() : null;
        }

        public final void setOverrideAction(OverrideAction.BuilderImpl overrideAction) {
            this.overrideAction = overrideAction != null ? overrideAction.build() : null;
        }

        @Override
        @Transient
        public final Builder overrideAction(OverrideAction overrideAction) {
            this.overrideAction = overrideAction;
            return this;
        }

        public final List<Label.Builder> getRuleLabels() {
            List<Label.Builder> result = LabelsCopier.copyToBuilder(this.ruleLabels);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRuleLabels(Collection<Label.BuilderImpl> ruleLabels) {
            this.ruleLabels = LabelsCopier.copyFromBuilder(ruleLabels);
        }

        @Override
        @Transient
        public final Builder ruleLabels(Collection<Label> ruleLabels) {
            this.ruleLabels = LabelsCopier.copy(ruleLabels);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder ruleLabels(Label... ruleLabels) {
            ruleLabels(Arrays.asList(ruleLabels));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder ruleLabels(Consumer<Label.Builder>... ruleLabels) {
            ruleLabels(Stream.of(ruleLabels).map(c -> Label.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final VisibilityConfig.Builder getVisibilityConfig() {
            return visibilityConfig != null ? visibilityConfig.toBuilder() : null;
        }

        public final void setVisibilityConfig(VisibilityConfig.BuilderImpl visibilityConfig) {
            this.visibilityConfig = visibilityConfig != null ? visibilityConfig.build() : null;
        }

        @Override
        @Transient
        public final Builder visibilityConfig(VisibilityConfig visibilityConfig) {
            this.visibilityConfig = visibilityConfig;
            return this;
        }

        public final CaptchaConfig.Builder getCaptchaConfig() {
            return captchaConfig != null ? captchaConfig.toBuilder() : null;
        }

        public final void setCaptchaConfig(CaptchaConfig.BuilderImpl captchaConfig) {
            this.captchaConfig = captchaConfig != null ? captchaConfig.build() : null;
        }

        @Override
        @Transient
        public final Builder captchaConfig(CaptchaConfig captchaConfig) {
            this.captchaConfig = captchaConfig;
            return this;
        }

        @Override
        public Rule build() {
            return new Rule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
