/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A rule group defines a collection of rules to inspect and control web requests that you can use in a <a>WebACL</a>.
 * When you create a rule group, you define an immutable capacity limit. If you update a rule group, you must stay
 * within the capacity. This allows others to reuse the rule group with confidence in its capacity requirements.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RuleGroup implements SdkPojo, Serializable, ToCopyableBuilder<RuleGroup.Builder, RuleGroup> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(RuleGroup::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(RuleGroup::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<Long> CAPACITY_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("Capacity")
            .getter(getter(RuleGroup::capacity)).setter(setter(Builder::capacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Capacity").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("ARN")
            .getter(getter(RuleGroup::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ARN").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(RuleGroup::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<List<Rule>> RULES_FIELD = SdkField
            .<List<Rule>> builder(MarshallingType.LIST)
            .memberName("Rules")
            .getter(getter(RuleGroup::rules))
            .setter(setter(Builder::rules))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Rules").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Rule> builder(MarshallingType.SDK_POJO)
                                            .constructor(Rule::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<VisibilityConfig> VISIBILITY_CONFIG_FIELD = SdkField
            .<VisibilityConfig> builder(MarshallingType.SDK_POJO).memberName("VisibilityConfig")
            .getter(getter(RuleGroup::visibilityConfig)).setter(setter(Builder::visibilityConfig))
            .constructor(VisibilityConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VisibilityConfig").build()).build();

    private static final SdkField<String> LABEL_NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LabelNamespace").getter(getter(RuleGroup::labelNamespace)).setter(setter(Builder::labelNamespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LabelNamespace").build()).build();

    private static final SdkField<Map<String, CustomResponseBody>> CUSTOM_RESPONSE_BODIES_FIELD = SdkField
            .<Map<String, CustomResponseBody>> builder(MarshallingType.MAP)
            .memberName("CustomResponseBodies")
            .getter(getter(RuleGroup::customResponseBodies))
            .setter(setter(Builder::customResponseBodies))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomResponseBodies").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<CustomResponseBody> builder(MarshallingType.SDK_POJO)
                                            .constructor(CustomResponseBody::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<LabelSummary>> AVAILABLE_LABELS_FIELD = SdkField
            .<List<LabelSummary>> builder(MarshallingType.LIST)
            .memberName("AvailableLabels")
            .getter(getter(RuleGroup::availableLabels))
            .setter(setter(Builder::availableLabels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailableLabels").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LabelSummary> builder(MarshallingType.SDK_POJO)
                                            .constructor(LabelSummary::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<LabelSummary>> CONSUMED_LABELS_FIELD = SdkField
            .<List<LabelSummary>> builder(MarshallingType.LIST)
            .memberName("ConsumedLabels")
            .getter(getter(RuleGroup::consumedLabels))
            .setter(setter(Builder::consumedLabels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConsumedLabels").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LabelSummary> builder(MarshallingType.SDK_POJO)
                                            .constructor(LabelSummary::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ID_FIELD,
            CAPACITY_FIELD, ARN_FIELD, DESCRIPTION_FIELD, RULES_FIELD, VISIBILITY_CONFIG_FIELD, LABEL_NAMESPACE_FIELD,
            CUSTOM_RESPONSE_BODIES_FIELD, AVAILABLE_LABELS_FIELD, CONSUMED_LABELS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String id;

    private final Long capacity;

    private final String arn;

    private final String description;

    private final List<Rule> rules;

    private final VisibilityConfig visibilityConfig;

    private final String labelNamespace;

    private final Map<String, CustomResponseBody> customResponseBodies;

    private final List<LabelSummary> availableLabels;

    private final List<LabelSummary> consumedLabels;

    private RuleGroup(BuilderImpl builder) {
        this.name = builder.name;
        this.id = builder.id;
        this.capacity = builder.capacity;
        this.arn = builder.arn;
        this.description = builder.description;
        this.rules = builder.rules;
        this.visibilityConfig = builder.visibilityConfig;
        this.labelNamespace = builder.labelNamespace;
        this.customResponseBodies = builder.customResponseBodies;
        this.availableLabels = builder.availableLabels;
        this.consumedLabels = builder.consumedLabels;
    }

    /**
     * <p>
     * The name of the rule group. You cannot change the name of a rule group after you create it.
     * </p>
     * 
     * @return The name of the rule group. You cannot change the name of a rule group after you create it.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * A unique identifier for the rule group. This ID is returned in the responses to create and list commands. You
     * provide it to operations like update and delete.
     * </p>
     * 
     * @return A unique identifier for the rule group. This ID is returned in the responses to create and list commands.
     *         You provide it to operations like update and delete.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The web ACL capacity units (WCUs) required for this rule group.
     * </p>
     * <p>
     * When you create your own rule group, you define this, and you cannot change it after creation. When you add or
     * modify the rules in a rule group, WAF enforces this limit. You can check the capacity for a set of rules using
     * <a>CheckCapacity</a>.
     * </p>
     * <p>
     * WAF uses WCUs to calculate and control the operating resources that are used to run your rules, rule groups, and
     * web ACLs. WAF calculates capacity differently for each rule type, to reflect the relative cost of each rule.
     * Simple rules that cost little to run use fewer WCUs than more complex rules that use more processing power. Rule
     * group capacity is fixed at creation, which helps users plan their web ACL WCU usage when they use a rule group.
     * The WCU limit for web ACLs is 1,500.
     * </p>
     * 
     * @return The web ACL capacity units (WCUs) required for this rule group.</p>
     *         <p>
     *         When you create your own rule group, you define this, and you cannot change it after creation. When you
     *         add or modify the rules in a rule group, WAF enforces this limit. You can check the capacity for a set of
     *         rules using <a>CheckCapacity</a>.
     *         </p>
     *         <p>
     *         WAF uses WCUs to calculate and control the operating resources that are used to run your rules, rule
     *         groups, and web ACLs. WAF calculates capacity differently for each rule type, to reflect the relative
     *         cost of each rule. Simple rules that cost little to run use fewer WCUs than more complex rules that use
     *         more processing power. Rule group capacity is fixed at creation, which helps users plan their web ACL WCU
     *         usage when they use a rule group. The WCU limit for web ACLs is 1,500.
     */
    public final Long capacity() {
        return capacity;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the entity.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the entity.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * A description of the rule group that helps with identification.
     * </p>
     * 
     * @return A description of the rule group that helps with identification.
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the Rules property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasRules() {
        return rules != null && !(rules instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The <a>Rule</a> statements used to identify the web requests that you want to allow, block, or count. Each rule
     * includes one top-level statement that WAF uses to identify matching web requests, and parameters that govern how
     * WAF handles them.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRules} method.
     * </p>
     * 
     * @return The <a>Rule</a> statements used to identify the web requests that you want to allow, block, or count.
     *         Each rule includes one top-level statement that WAF uses to identify matching web requests, and
     *         parameters that govern how WAF handles them.
     */
    public final List<Rule> rules() {
        return rules;
    }

    /**
     * <p>
     * Defines and enables Amazon CloudWatch metrics and web request sample collection.
     * </p>
     * 
     * @return Defines and enables Amazon CloudWatch metrics and web request sample collection.
     */
    public final VisibilityConfig visibilityConfig() {
        return visibilityConfig;
    }

    /**
     * <p>
     * The label namespace prefix for this rule group. All labels added by rules in this rule group have this prefix.
     * </p>
     * <ul>
     * <li>
     * <p>
     * The syntax for the label namespace prefix for your rule groups is the following:
     * </p>
     * <p>
     * <code>awswaf:&lt;account ID&gt;:rulegroup:&lt;rule group name&gt;:</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * When a rule with a label matches a web request, WAF adds the fully qualified label to the request. A fully
     * qualified label is made up of the label namespace from the rule group or web ACL where the rule is defined and
     * the label from the rule, separated by a colon:
     * </p>
     * <p>
     * <code>&lt;label namespace&gt;:&lt;label from rule&gt;</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The label namespace prefix for this rule group. All labels added by rules in this rule group have this
     *         prefix. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The syntax for the label namespace prefix for your rule groups is the following:
     *         </p>
     *         <p>
     *         <code>awswaf:&lt;account ID&gt;:rulegroup:&lt;rule group name&gt;:</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         When a rule with a label matches a web request, WAF adds the fully qualified label to the request. A
     *         fully qualified label is made up of the label namespace from the rule group or web ACL where the rule is
     *         defined and the label from the rule, separated by a colon:
     *         </p>
     *         <p>
     *         <code>&lt;label namespace&gt;:&lt;label from rule&gt;</code>
     *         </p>
     *         </li>
     */
    public final String labelNamespace() {
        return labelNamespace;
    }

    /**
     * For responses, this returns true if the service returned a value for the CustomResponseBodies property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCustomResponseBodies() {
        return customResponseBodies != null && !(customResponseBodies instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of custom response keys and content bodies. When you create a rule with a block action, you can send a
     * custom response to the web request. You define these for the rule group, and then use them in the rules that you
     * define in the rule group.
     * </p>
     * <p>
     * For information about customizing web requests and responses, see <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-custom-request-response.html">Customizing web
     * requests and responses in WAF</a> in the <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">WAF Developer Guide</a>.
     * </p>
     * <p>
     * For information about the limits on count and size for custom request and response settings, see <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">WAF quotas</a> in the <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">WAF Developer Guide</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCustomResponseBodies} method.
     * </p>
     * 
     * @return A map of custom response keys and content bodies. When you create a rule with a block action, you can
     *         send a custom response to the web request. You define these for the rule group, and then use them in the
     *         rules that you define in the rule group. </p>
     *         <p>
     *         For information about customizing web requests and responses, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-custom-request-response.html">Customizing
     *         web requests and responses in WAF</a> in the <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">WAF Developer Guide</a>.
     *         </p>
     *         <p>
     *         For information about the limits on count and size for custom request and response settings, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">WAF quotas</a> in the <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">WAF Developer Guide</a>.
     */
    public final Map<String, CustomResponseBody> customResponseBodies() {
        return customResponseBodies;
    }

    /**
     * For responses, this returns true if the service returned a value for the AvailableLabels property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAvailableLabels() {
        return availableLabels != null && !(availableLabels instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The labels that one or more rules in this rule group add to matching web requests. These labels are defined in
     * the <code>RuleLabels</code> for a <a>Rule</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAvailableLabels} method.
     * </p>
     * 
     * @return The labels that one or more rules in this rule group add to matching web requests. These labels are
     *         defined in the <code>RuleLabels</code> for a <a>Rule</a>.
     */
    public final List<LabelSummary> availableLabels() {
        return availableLabels;
    }

    /**
     * For responses, this returns true if the service returned a value for the ConsumedLabels property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasConsumedLabels() {
        return consumedLabels != null && !(consumedLabels instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The labels that one or more rules in this rule group match against in label match statements. These labels are
     * defined in a <code>LabelMatchStatement</code> specification, in the <a>Statement</a> definition of a rule.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasConsumedLabels} method.
     * </p>
     * 
     * @return The labels that one or more rules in this rule group match against in label match statements. These
     *         labels are defined in a <code>LabelMatchStatement</code> specification, in the <a>Statement</a>
     *         definition of a rule.
     */
    public final List<LabelSummary> consumedLabels() {
        return consumedLabels;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(capacity());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasRules() ? rules() : null);
        hashCode = 31 * hashCode + Objects.hashCode(visibilityConfig());
        hashCode = 31 * hashCode + Objects.hashCode(labelNamespace());
        hashCode = 31 * hashCode + Objects.hashCode(hasCustomResponseBodies() ? customResponseBodies() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAvailableLabels() ? availableLabels() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasConsumedLabels() ? consumedLabels() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RuleGroup)) {
            return false;
        }
        RuleGroup other = (RuleGroup) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(id(), other.id())
                && Objects.equals(capacity(), other.capacity()) && Objects.equals(arn(), other.arn())
                && Objects.equals(description(), other.description()) && hasRules() == other.hasRules()
                && Objects.equals(rules(), other.rules()) && Objects.equals(visibilityConfig(), other.visibilityConfig())
                && Objects.equals(labelNamespace(), other.labelNamespace())
                && hasCustomResponseBodies() == other.hasCustomResponseBodies()
                && Objects.equals(customResponseBodies(), other.customResponseBodies())
                && hasAvailableLabels() == other.hasAvailableLabels()
                && Objects.equals(availableLabels(), other.availableLabels()) && hasConsumedLabels() == other.hasConsumedLabels()
                && Objects.equals(consumedLabels(), other.consumedLabels());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RuleGroup").add("Name", name()).add("Id", id()).add("Capacity", capacity()).add("ARN", arn())
                .add("Description", description()).add("Rules", hasRules() ? rules() : null)
                .add("VisibilityConfig", visibilityConfig()).add("LabelNamespace", labelNamespace())
                .add("CustomResponseBodies", hasCustomResponseBodies() ? customResponseBodies() : null)
                .add("AvailableLabels", hasAvailableLabels() ? availableLabels() : null)
                .add("ConsumedLabels", hasConsumedLabels() ? consumedLabels() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Capacity":
            return Optional.ofNullable(clazz.cast(capacity()));
        case "ARN":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Rules":
            return Optional.ofNullable(clazz.cast(rules()));
        case "VisibilityConfig":
            return Optional.ofNullable(clazz.cast(visibilityConfig()));
        case "LabelNamespace":
            return Optional.ofNullable(clazz.cast(labelNamespace()));
        case "CustomResponseBodies":
            return Optional.ofNullable(clazz.cast(customResponseBodies()));
        case "AvailableLabels":
            return Optional.ofNullable(clazz.cast(availableLabels()));
        case "ConsumedLabels":
            return Optional.ofNullable(clazz.cast(consumedLabels()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RuleGroup, T> g) {
        return obj -> g.apply((RuleGroup) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RuleGroup> {
        /**
         * <p>
         * The name of the rule group. You cannot change the name of a rule group after you create it.
         * </p>
         * 
         * @param name
         *        The name of the rule group. You cannot change the name of a rule group after you create it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A unique identifier for the rule group. This ID is returned in the responses to create and list commands. You
         * provide it to operations like update and delete.
         * </p>
         * 
         * @param id
         *        A unique identifier for the rule group. This ID is returned in the responses to create and list
         *        commands. You provide it to operations like update and delete.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The web ACL capacity units (WCUs) required for this rule group.
         * </p>
         * <p>
         * When you create your own rule group, you define this, and you cannot change it after creation. When you add
         * or modify the rules in a rule group, WAF enforces this limit. You can check the capacity for a set of rules
         * using <a>CheckCapacity</a>.
         * </p>
         * <p>
         * WAF uses WCUs to calculate and control the operating resources that are used to run your rules, rule groups,
         * and web ACLs. WAF calculates capacity differently for each rule type, to reflect the relative cost of each
         * rule. Simple rules that cost little to run use fewer WCUs than more complex rules that use more processing
         * power. Rule group capacity is fixed at creation, which helps users plan their web ACL WCU usage when they use
         * a rule group. The WCU limit for web ACLs is 1,500.
         * </p>
         * 
         * @param capacity
         *        The web ACL capacity units (WCUs) required for this rule group.</p>
         *        <p>
         *        When you create your own rule group, you define this, and you cannot change it after creation. When
         *        you add or modify the rules in a rule group, WAF enforces this limit. You can check the capacity for a
         *        set of rules using <a>CheckCapacity</a>.
         *        </p>
         *        <p>
         *        WAF uses WCUs to calculate and control the operating resources that are used to run your rules, rule
         *        groups, and web ACLs. WAF calculates capacity differently for each rule type, to reflect the relative
         *        cost of each rule. Simple rules that cost little to run use fewer WCUs than more complex rules that
         *        use more processing power. Rule group capacity is fixed at creation, which helps users plan their web
         *        ACL WCU usage when they use a rule group. The WCU limit for web ACLs is 1,500.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacity(Long capacity);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the entity.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the entity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * A description of the rule group that helps with identification.
         * </p>
         * 
         * @param description
         *        A description of the rule group that helps with identification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The <a>Rule</a> statements used to identify the web requests that you want to allow, block, or count. Each
         * rule includes one top-level statement that WAF uses to identify matching web requests, and parameters that
         * govern how WAF handles them.
         * </p>
         * 
         * @param rules
         *        The <a>Rule</a> statements used to identify the web requests that you want to allow, block, or count.
         *        Each rule includes one top-level statement that WAF uses to identify matching web requests, and
         *        parameters that govern how WAF handles them.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rules(Collection<Rule> rules);

        /**
         * <p>
         * The <a>Rule</a> statements used to identify the web requests that you want to allow, block, or count. Each
         * rule includes one top-level statement that WAF uses to identify matching web requests, and parameters that
         * govern how WAF handles them.
         * </p>
         * 
         * @param rules
         *        The <a>Rule</a> statements used to identify the web requests that you want to allow, block, or count.
         *        Each rule includes one top-level statement that WAF uses to identify matching web requests, and
         *        parameters that govern how WAF handles them.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rules(Rule... rules);

        /**
         * <p>
         * The <a>Rule</a> statements used to identify the web requests that you want to allow, block, or count. Each
         * rule includes one top-level statement that WAF uses to identify matching web requests, and parameters that
         * govern how WAF handles them.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Rule>.Builder} avoiding the need to create
         * one manually via {@link List<Rule>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Rule>.Builder#build()} is called immediately and its result
         * is passed to {@link #rules(List<Rule>)}.
         * 
         * @param rules
         *        a consumer that will call methods on {@link List<Rule>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rules(List<Rule>)
         */
        Builder rules(Consumer<Rule.Builder>... rules);

        /**
         * <p>
         * Defines and enables Amazon CloudWatch metrics and web request sample collection.
         * </p>
         * 
         * @param visibilityConfig
         *        Defines and enables Amazon CloudWatch metrics and web request sample collection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder visibilityConfig(VisibilityConfig visibilityConfig);

        /**
         * <p>
         * Defines and enables Amazon CloudWatch metrics and web request sample collection.
         * </p>
         * This is a convenience that creates an instance of the {@link VisibilityConfig.Builder} avoiding the need to
         * create one manually via {@link VisibilityConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link VisibilityConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #visibilityConfig(VisibilityConfig)}.
         * 
         * @param visibilityConfig
         *        a consumer that will call methods on {@link VisibilityConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #visibilityConfig(VisibilityConfig)
         */
        default Builder visibilityConfig(Consumer<VisibilityConfig.Builder> visibilityConfig) {
            return visibilityConfig(VisibilityConfig.builder().applyMutation(visibilityConfig).build());
        }

        /**
         * <p>
         * The label namespace prefix for this rule group. All labels added by rules in this rule group have this
         * prefix.
         * </p>
         * <ul>
         * <li>
         * <p>
         * The syntax for the label namespace prefix for your rule groups is the following:
         * </p>
         * <p>
         * <code>awswaf:&lt;account ID&gt;:rulegroup:&lt;rule group name&gt;:</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * When a rule with a label matches a web request, WAF adds the fully qualified label to the request. A fully
         * qualified label is made up of the label namespace from the rule group or web ACL where the rule is defined
         * and the label from the rule, separated by a colon:
         * </p>
         * <p>
         * <code>&lt;label namespace&gt;:&lt;label from rule&gt;</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param labelNamespace
         *        The label namespace prefix for this rule group. All labels added by rules in this rule group have this
         *        prefix. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The syntax for the label namespace prefix for your rule groups is the following:
         *        </p>
         *        <p>
         *        <code>awswaf:&lt;account ID&gt;:rulegroup:&lt;rule group name&gt;:</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        When a rule with a label matches a web request, WAF adds the fully qualified label to the request. A
         *        fully qualified label is made up of the label namespace from the rule group or web ACL where the rule
         *        is defined and the label from the rule, separated by a colon:
         *        </p>
         *        <p>
         *        <code>&lt;label namespace&gt;:&lt;label from rule&gt;</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder labelNamespace(String labelNamespace);

        /**
         * <p>
         * A map of custom response keys and content bodies. When you create a rule with a block action, you can send a
         * custom response to the web request. You define these for the rule group, and then use them in the rules that
         * you define in the rule group.
         * </p>
         * <p>
         * For information about customizing web requests and responses, see <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-custom-request-response.html">Customizing web
         * requests and responses in WAF</a> in the <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">WAF Developer Guide</a>.
         * </p>
         * <p>
         * For information about the limits on count and size for custom request and response settings, see <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">WAF quotas</a> in the <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">WAF Developer Guide</a>.
         * </p>
         * 
         * @param customResponseBodies
         *        A map of custom response keys and content bodies. When you create a rule with a block action, you can
         *        send a custom response to the web request. You define these for the rule group, and then use them in
         *        the rules that you define in the rule group. </p>
         *        <p>
         *        For information about customizing web requests and responses, see <a
         *        href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-custom-request-response.html"
         *        >Customizing web requests and responses in WAF</a> in the <a
         *        href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">WAF Developer Guide</a>.
         *        </p>
         *        <p>
         *        For information about the limits on count and size for custom request and response settings, see <a
         *        href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">WAF quotas</a> in the <a
         *        href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">WAF Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customResponseBodies(Map<String, CustomResponseBody> customResponseBodies);

        /**
         * <p>
         * The labels that one or more rules in this rule group add to matching web requests. These labels are defined
         * in the <code>RuleLabels</code> for a <a>Rule</a>.
         * </p>
         * 
         * @param availableLabels
         *        The labels that one or more rules in this rule group add to matching web requests. These labels are
         *        defined in the <code>RuleLabels</code> for a <a>Rule</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availableLabels(Collection<LabelSummary> availableLabels);

        /**
         * <p>
         * The labels that one or more rules in this rule group add to matching web requests. These labels are defined
         * in the <code>RuleLabels</code> for a <a>Rule</a>.
         * </p>
         * 
         * @param availableLabels
         *        The labels that one or more rules in this rule group add to matching web requests. These labels are
         *        defined in the <code>RuleLabels</code> for a <a>Rule</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availableLabels(LabelSummary... availableLabels);

        /**
         * <p>
         * The labels that one or more rules in this rule group add to matching web requests. These labels are defined
         * in the <code>RuleLabels</code> for a <a>Rule</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<LabelSummary>.Builder} avoiding the need to
         * create one manually via {@link List<LabelSummary>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<LabelSummary>.Builder#build()} is called immediately and its
         * result is passed to {@link #availableLabels(List<LabelSummary>)}.
         * 
         * @param availableLabels
         *        a consumer that will call methods on {@link List<LabelSummary>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #availableLabels(List<LabelSummary>)
         */
        Builder availableLabels(Consumer<LabelSummary.Builder>... availableLabels);

        /**
         * <p>
         * The labels that one or more rules in this rule group match against in label match statements. These labels
         * are defined in a <code>LabelMatchStatement</code> specification, in the <a>Statement</a> definition of a
         * rule.
         * </p>
         * 
         * @param consumedLabels
         *        The labels that one or more rules in this rule group match against in label match statements. These
         *        labels are defined in a <code>LabelMatchStatement</code> specification, in the <a>Statement</a>
         *        definition of a rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder consumedLabels(Collection<LabelSummary> consumedLabels);

        /**
         * <p>
         * The labels that one or more rules in this rule group match against in label match statements. These labels
         * are defined in a <code>LabelMatchStatement</code> specification, in the <a>Statement</a> definition of a
         * rule.
         * </p>
         * 
         * @param consumedLabels
         *        The labels that one or more rules in this rule group match against in label match statements. These
         *        labels are defined in a <code>LabelMatchStatement</code> specification, in the <a>Statement</a>
         *        definition of a rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder consumedLabels(LabelSummary... consumedLabels);

        /**
         * <p>
         * The labels that one or more rules in this rule group match against in label match statements. These labels
         * are defined in a <code>LabelMatchStatement</code> specification, in the <a>Statement</a> definition of a
         * rule.
         * </p>
         * This is a convenience that creates an instance of the {@link List<LabelSummary>.Builder} avoiding the need to
         * create one manually via {@link List<LabelSummary>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<LabelSummary>.Builder#build()} is called immediately and its
         * result is passed to {@link #consumedLabels(List<LabelSummary>)}.
         * 
         * @param consumedLabels
         *        a consumer that will call methods on {@link List<LabelSummary>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #consumedLabels(List<LabelSummary>)
         */
        Builder consumedLabels(Consumer<LabelSummary.Builder>... consumedLabels);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String id;

        private Long capacity;

        private String arn;

        private String description;

        private List<Rule> rules = DefaultSdkAutoConstructList.getInstance();

        private VisibilityConfig visibilityConfig;

        private String labelNamespace;

        private Map<String, CustomResponseBody> customResponseBodies = DefaultSdkAutoConstructMap.getInstance();

        private List<LabelSummary> availableLabels = DefaultSdkAutoConstructList.getInstance();

        private List<LabelSummary> consumedLabels = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(RuleGroup model) {
            name(model.name);
            id(model.id);
            capacity(model.capacity);
            arn(model.arn);
            description(model.description);
            rules(model.rules);
            visibilityConfig(model.visibilityConfig);
            labelNamespace(model.labelNamespace);
            customResponseBodies(model.customResponseBodies);
            availableLabels(model.availableLabels);
            consumedLabels(model.consumedLabels);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        @Transient
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final Long getCapacity() {
            return capacity;
        }

        public final void setCapacity(Long capacity) {
            this.capacity = capacity;
        }

        @Override
        @Transient
        public final Builder capacity(Long capacity) {
            this.capacity = capacity;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        @Transient
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        @Transient
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final List<Rule.Builder> getRules() {
            List<Rule.Builder> result = RulesCopier.copyToBuilder(this.rules);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRules(Collection<Rule.BuilderImpl> rules) {
            this.rules = RulesCopier.copyFromBuilder(rules);
        }

        @Override
        @Transient
        public final Builder rules(Collection<Rule> rules) {
            this.rules = RulesCopier.copy(rules);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder rules(Rule... rules) {
            rules(Arrays.asList(rules));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder rules(Consumer<Rule.Builder>... rules) {
            rules(Stream.of(rules).map(c -> Rule.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final VisibilityConfig.Builder getVisibilityConfig() {
            return visibilityConfig != null ? visibilityConfig.toBuilder() : null;
        }

        public final void setVisibilityConfig(VisibilityConfig.BuilderImpl visibilityConfig) {
            this.visibilityConfig = visibilityConfig != null ? visibilityConfig.build() : null;
        }

        @Override
        @Transient
        public final Builder visibilityConfig(VisibilityConfig visibilityConfig) {
            this.visibilityConfig = visibilityConfig;
            return this;
        }

        public final String getLabelNamespace() {
            return labelNamespace;
        }

        public final void setLabelNamespace(String labelNamespace) {
            this.labelNamespace = labelNamespace;
        }

        @Override
        @Transient
        public final Builder labelNamespace(String labelNamespace) {
            this.labelNamespace = labelNamespace;
            return this;
        }

        public final Map<String, CustomResponseBody.Builder> getCustomResponseBodies() {
            Map<String, CustomResponseBody.Builder> result = CustomResponseBodiesCopier.copyToBuilder(this.customResponseBodies);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setCustomResponseBodies(Map<String, CustomResponseBody.BuilderImpl> customResponseBodies) {
            this.customResponseBodies = CustomResponseBodiesCopier.copyFromBuilder(customResponseBodies);
        }

        @Override
        @Transient
        public final Builder customResponseBodies(Map<String, CustomResponseBody> customResponseBodies) {
            this.customResponseBodies = CustomResponseBodiesCopier.copy(customResponseBodies);
            return this;
        }

        public final List<LabelSummary.Builder> getAvailableLabels() {
            List<LabelSummary.Builder> result = LabelSummariesCopier.copyToBuilder(this.availableLabels);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAvailableLabels(Collection<LabelSummary.BuilderImpl> availableLabels) {
            this.availableLabels = LabelSummariesCopier.copyFromBuilder(availableLabels);
        }

        @Override
        @Transient
        public final Builder availableLabels(Collection<LabelSummary> availableLabels) {
            this.availableLabels = LabelSummariesCopier.copy(availableLabels);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder availableLabels(LabelSummary... availableLabels) {
            availableLabels(Arrays.asList(availableLabels));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder availableLabels(Consumer<LabelSummary.Builder>... availableLabels) {
            availableLabels(Stream.of(availableLabels).map(c -> LabelSummary.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<LabelSummary.Builder> getConsumedLabels() {
            List<LabelSummary.Builder> result = LabelSummariesCopier.copyToBuilder(this.consumedLabels);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setConsumedLabels(Collection<LabelSummary.BuilderImpl> consumedLabels) {
            this.consumedLabels = LabelSummariesCopier.copyFromBuilder(consumedLabels);
        }

        @Override
        @Transient
        public final Builder consumedLabels(Collection<LabelSummary> consumedLabels) {
            this.consumedLabels = LabelSummariesCopier.copy(consumedLabels);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder consumedLabels(LabelSummary... consumedLabels) {
            consumedLabels(Arrays.asList(consumedLabels));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder consumedLabels(Consumer<LabelSummary.Builder>... consumedLabels) {
            consumedLabels(Stream.of(consumedLabels).map(c -> LabelSummary.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public RuleGroup build() {
            return new RuleGroup(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
