/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A rate-based rule tracks the rate of requests for each originating IP address, and triggers the rule action when the
 * rate exceeds a limit that you specify on the number of requests in any 5-minute time span. You can use this to put a
 * temporary block on requests from an IP address that is sending excessive requests.
 * </p>
 * <p>
 * WAF tracks and manages web requests separately for each instance of a rate-based rule that you use. For example, if
 * you provide the same rate-based rule settings in two web ACLs, each of the two rule statements represents a separate
 * instance of the rate-based rule and gets its own tracking and management by WAF. If you define a rate-based rule
 * inside a rule group, and then use that rule group in multiple places, each use creates a separate instance of the
 * rate-based rule that gets its own tracking and management by WAF.
 * </p>
 * <p>
 * When the rule action triggers, WAF blocks additional requests from the IP address until the request rate falls below
 * the limit.
 * </p>
 * <p>
 * You can optionally nest another statement inside the rate-based statement, to narrow the scope of the rule so that it
 * only counts requests that match the nested statement. For example, based on recent requests that you have seen from
 * an attacker, you might create a rate-based rule with a nested AND rule statement that contains the following nested
 * statements:
 * </p>
 * <ul>
 * <li>
 * <p>
 * An IP match statement with an IP set that specified the address 192.0.2.44.
 * </p>
 * </li>
 * <li>
 * <p>
 * A string match statement that searches in the User-Agent header for the string BadBot.
 * </p>
 * </li>
 * </ul>
 * <p>
 * In this rate-based rule, you also define a rate limit. For this example, the rate limit is 1,000. Requests that meet
 * both of the conditions in the statements are counted. If the count exceeds 1,000 requests per five minutes, the rule
 * action triggers. Requests that do not meet both conditions are not counted towards the rate limit and are not
 * affected by this rule.
 * </p>
 * <p>
 * You cannot nest a <code>RateBasedStatement</code> inside another statement, for example inside a
 * <code>NotStatement</code> or <code>OrStatement</code>. You can define a <code>RateBasedStatement</code> inside a web
 * ACL and inside a rule group.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RateBasedStatement implements SdkPojo, Serializable,
        ToCopyableBuilder<RateBasedStatement.Builder, RateBasedStatement> {
    private static final SdkField<Long> LIMIT_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("Limit")
            .getter(getter(RateBasedStatement::limit)).setter(setter(Builder::limit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Limit").build()).build();

    private static final SdkField<String> AGGREGATE_KEY_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AggregateKeyType").getter(getter(RateBasedStatement::aggregateKeyTypeAsString))
            .setter(setter(Builder::aggregateKeyType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AggregateKeyType").build()).build();

    private static final SdkField<Statement> SCOPE_DOWN_STATEMENT_FIELD = SdkField.<Statement> builder(MarshallingType.SDK_POJO)
            .memberName("ScopeDownStatement").getter(getter(RateBasedStatement::scopeDownStatement))
            .setter(setter(Builder::scopeDownStatement)).constructor(Statement::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScopeDownStatement").build())
            .build();

    private static final SdkField<ForwardedIPConfig> FORWARDED_IP_CONFIG_FIELD = SdkField
            .<ForwardedIPConfig> builder(MarshallingType.SDK_POJO).memberName("ForwardedIPConfig")
            .getter(getter(RateBasedStatement::forwardedIPConfig)).setter(setter(Builder::forwardedIPConfig))
            .constructor(ForwardedIPConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ForwardedIPConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LIMIT_FIELD,
            AGGREGATE_KEY_TYPE_FIELD, SCOPE_DOWN_STATEMENT_FIELD, FORWARDED_IP_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final Long limit;

    private final String aggregateKeyType;

    private final Statement scopeDownStatement;

    private final ForwardedIPConfig forwardedIPConfig;

    private RateBasedStatement(BuilderImpl builder) {
        this.limit = builder.limit;
        this.aggregateKeyType = builder.aggregateKeyType;
        this.scopeDownStatement = builder.scopeDownStatement;
        this.forwardedIPConfig = builder.forwardedIPConfig;
    }

    /**
     * <p>
     * The limit on requests per 5-minute period for a single originating IP address. If the statement includes a
     * <code>ScopeDownStatement</code>, this limit is applied only to the requests that match the statement.
     * </p>
     * 
     * @return The limit on requests per 5-minute period for a single originating IP address. If the statement includes
     *         a <code>ScopeDownStatement</code>, this limit is applied only to the requests that match the statement.
     */
    public final Long limit() {
        return limit;
    }

    /**
     * <p>
     * Setting that indicates how to aggregate the request counts. The options are the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * IP - Aggregate the request counts on the IP address from the web request origin.
     * </p>
     * </li>
     * <li>
     * <p>
     * FORWARDED_IP - Aggregate the request counts on the first IP address in an HTTP header. If you use this, configure
     * the <code>ForwardedIPConfig</code>, to specify the header to use.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #aggregateKeyType}
     * will return {@link RateBasedStatementAggregateKeyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #aggregateKeyTypeAsString}.
     * </p>
     * 
     * @return Setting that indicates how to aggregate the request counts. The options are the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         IP - Aggregate the request counts on the IP address from the web request origin.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FORWARDED_IP - Aggregate the request counts on the first IP address in an HTTP header. If you use this,
     *         configure the <code>ForwardedIPConfig</code>, to specify the header to use.
     *         </p>
     *         </li>
     * @see RateBasedStatementAggregateKeyType
     */
    public final RateBasedStatementAggregateKeyType aggregateKeyType() {
        return RateBasedStatementAggregateKeyType.fromValue(aggregateKeyType);
    }

    /**
     * <p>
     * Setting that indicates how to aggregate the request counts. The options are the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * IP - Aggregate the request counts on the IP address from the web request origin.
     * </p>
     * </li>
     * <li>
     * <p>
     * FORWARDED_IP - Aggregate the request counts on the first IP address in an HTTP header. If you use this, configure
     * the <code>ForwardedIPConfig</code>, to specify the header to use.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #aggregateKeyType}
     * will return {@link RateBasedStatementAggregateKeyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #aggregateKeyTypeAsString}.
     * </p>
     * 
     * @return Setting that indicates how to aggregate the request counts. The options are the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         IP - Aggregate the request counts on the IP address from the web request origin.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FORWARDED_IP - Aggregate the request counts on the first IP address in an HTTP header. If you use this,
     *         configure the <code>ForwardedIPConfig</code>, to specify the header to use.
     *         </p>
     *         </li>
     * @see RateBasedStatementAggregateKeyType
     */
    public final String aggregateKeyTypeAsString() {
        return aggregateKeyType;
    }

    /**
     * <p>
     * An optional nested statement that narrows the scope of the web requests that are evaluated by the rate-based
     * statement. Requests are only tracked by the rate-based statement if they match the scope-down statement. You can
     * use any nestable <a>Statement</a> in the scope-down statement, and you can nest statements at any level, the same
     * as you can for a rule statement.
     * </p>
     * 
     * @return An optional nested statement that narrows the scope of the web requests that are evaluated by the
     *         rate-based statement. Requests are only tracked by the rate-based statement if they match the scope-down
     *         statement. You can use any nestable <a>Statement</a> in the scope-down statement, and you can nest
     *         statements at any level, the same as you can for a rule statement.
     */
    public final Statement scopeDownStatement() {
        return scopeDownStatement;
    }

    /**
     * <p>
     * The configuration for inspecting IP addresses in an HTTP header that you specify, instead of using the IP address
     * that's reported by the web request origin. Commonly, this is the X-Forwarded-For (XFF) header, but you can
     * specify any header name.
     * </p>
     * <note>
     * <p>
     * If the specified header isn't present in the request, WAF doesn't apply the rule to the web request at all.
     * </p>
     * </note>
     * <p>
     * This is required if <code>AggregateKeyType</code> is set to <code>FORWARDED_IP</code>.
     * </p>
     * 
     * @return The configuration for inspecting IP addresses in an HTTP header that you specify, instead of using the IP
     *         address that's reported by the web request origin. Commonly, this is the X-Forwarded-For (XFF) header,
     *         but you can specify any header name. </p> <note>
     *         <p>
     *         If the specified header isn't present in the request, WAF doesn't apply the rule to the web request at
     *         all.
     *         </p>
     *         </note>
     *         <p>
     *         This is required if <code>AggregateKeyType</code> is set to <code>FORWARDED_IP</code>.
     */
    public final ForwardedIPConfig forwardedIPConfig() {
        return forwardedIPConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(limit());
        hashCode = 31 * hashCode + Objects.hashCode(aggregateKeyTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(scopeDownStatement());
        hashCode = 31 * hashCode + Objects.hashCode(forwardedIPConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RateBasedStatement)) {
            return false;
        }
        RateBasedStatement other = (RateBasedStatement) obj;
        return Objects.equals(limit(), other.limit())
                && Objects.equals(aggregateKeyTypeAsString(), other.aggregateKeyTypeAsString())
                && Objects.equals(scopeDownStatement(), other.scopeDownStatement())
                && Objects.equals(forwardedIPConfig(), other.forwardedIPConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RateBasedStatement").add("Limit", limit()).add("AggregateKeyType", aggregateKeyTypeAsString())
                .add("ScopeDownStatement", scopeDownStatement()).add("ForwardedIPConfig", forwardedIPConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Limit":
            return Optional.ofNullable(clazz.cast(limit()));
        case "AggregateKeyType":
            return Optional.ofNullable(clazz.cast(aggregateKeyTypeAsString()));
        case "ScopeDownStatement":
            return Optional.ofNullable(clazz.cast(scopeDownStatement()));
        case "ForwardedIPConfig":
            return Optional.ofNullable(clazz.cast(forwardedIPConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RateBasedStatement, T> g) {
        return obj -> g.apply((RateBasedStatement) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RateBasedStatement> {
        /**
         * <p>
         * The limit on requests per 5-minute period for a single originating IP address. If the statement includes a
         * <code>ScopeDownStatement</code>, this limit is applied only to the requests that match the statement.
         * </p>
         * 
         * @param limit
         *        The limit on requests per 5-minute period for a single originating IP address. If the statement
         *        includes a <code>ScopeDownStatement</code>, this limit is applied only to the requests that match the
         *        statement.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder limit(Long limit);

        /**
         * <p>
         * Setting that indicates how to aggregate the request counts. The options are the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * IP - Aggregate the request counts on the IP address from the web request origin.
         * </p>
         * </li>
         * <li>
         * <p>
         * FORWARDED_IP - Aggregate the request counts on the first IP address in an HTTP header. If you use this,
         * configure the <code>ForwardedIPConfig</code>, to specify the header to use.
         * </p>
         * </li>
         * </ul>
         * 
         * @param aggregateKeyType
         *        Setting that indicates how to aggregate the request counts. The options are the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        IP - Aggregate the request counts on the IP address from the web request origin.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FORWARDED_IP - Aggregate the request counts on the first IP address in an HTTP header. If you use
         *        this, configure the <code>ForwardedIPConfig</code>, to specify the header to use.
         *        </p>
         *        </li>
         * @see RateBasedStatementAggregateKeyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RateBasedStatementAggregateKeyType
         */
        Builder aggregateKeyType(String aggregateKeyType);

        /**
         * <p>
         * Setting that indicates how to aggregate the request counts. The options are the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * IP - Aggregate the request counts on the IP address from the web request origin.
         * </p>
         * </li>
         * <li>
         * <p>
         * FORWARDED_IP - Aggregate the request counts on the first IP address in an HTTP header. If you use this,
         * configure the <code>ForwardedIPConfig</code>, to specify the header to use.
         * </p>
         * </li>
         * </ul>
         * 
         * @param aggregateKeyType
         *        Setting that indicates how to aggregate the request counts. The options are the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        IP - Aggregate the request counts on the IP address from the web request origin.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FORWARDED_IP - Aggregate the request counts on the first IP address in an HTTP header. If you use
         *        this, configure the <code>ForwardedIPConfig</code>, to specify the header to use.
         *        </p>
         *        </li>
         * @see RateBasedStatementAggregateKeyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RateBasedStatementAggregateKeyType
         */
        Builder aggregateKeyType(RateBasedStatementAggregateKeyType aggregateKeyType);

        /**
         * <p>
         * An optional nested statement that narrows the scope of the web requests that are evaluated by the rate-based
         * statement. Requests are only tracked by the rate-based statement if they match the scope-down statement. You
         * can use any nestable <a>Statement</a> in the scope-down statement, and you can nest statements at any level,
         * the same as you can for a rule statement.
         * </p>
         * 
         * @param scopeDownStatement
         *        An optional nested statement that narrows the scope of the web requests that are evaluated by the
         *        rate-based statement. Requests are only tracked by the rate-based statement if they match the
         *        scope-down statement. You can use any nestable <a>Statement</a> in the scope-down statement, and you
         *        can nest statements at any level, the same as you can for a rule statement.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scopeDownStatement(Statement scopeDownStatement);

        /**
         * <p>
         * An optional nested statement that narrows the scope of the web requests that are evaluated by the rate-based
         * statement. Requests are only tracked by the rate-based statement if they match the scope-down statement. You
         * can use any nestable <a>Statement</a> in the scope-down statement, and you can nest statements at any level,
         * the same as you can for a rule statement.
         * </p>
         * This is a convenience method that creates an instance of the {@link Statement.Builder} avoiding the need to
         * create one manually via {@link Statement#builder()}.
         *
         * When the {@link Consumer} completes, {@link Statement.Builder#build()} is called immediately and its result
         * is passed to {@link #scopeDownStatement(Statement)}.
         * 
         * @param scopeDownStatement
         *        a consumer that will call methods on {@link Statement.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scopeDownStatement(Statement)
         */
        default Builder scopeDownStatement(Consumer<Statement.Builder> scopeDownStatement) {
            return scopeDownStatement(Statement.builder().applyMutation(scopeDownStatement).build());
        }

        /**
         * <p>
         * The configuration for inspecting IP addresses in an HTTP header that you specify, instead of using the IP
         * address that's reported by the web request origin. Commonly, this is the X-Forwarded-For (XFF) header, but
         * you can specify any header name.
         * </p>
         * <note>
         * <p>
         * If the specified header isn't present in the request, WAF doesn't apply the rule to the web request at all.
         * </p>
         * </note>
         * <p>
         * This is required if <code>AggregateKeyType</code> is set to <code>FORWARDED_IP</code>.
         * </p>
         * 
         * @param forwardedIPConfig
         *        The configuration for inspecting IP addresses in an HTTP header that you specify, instead of using the
         *        IP address that's reported by the web request origin. Commonly, this is the X-Forwarded-For (XFF)
         *        header, but you can specify any header name. </p> <note>
         *        <p>
         *        If the specified header isn't present in the request, WAF doesn't apply the rule to the web request at
         *        all.
         *        </p>
         *        </note>
         *        <p>
         *        This is required if <code>AggregateKeyType</code> is set to <code>FORWARDED_IP</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forwardedIPConfig(ForwardedIPConfig forwardedIPConfig);

        /**
         * <p>
         * The configuration for inspecting IP addresses in an HTTP header that you specify, instead of using the IP
         * address that's reported by the web request origin. Commonly, this is the X-Forwarded-For (XFF) header, but
         * you can specify any header name.
         * </p>
         * <note>
         * <p>
         * If the specified header isn't present in the request, WAF doesn't apply the rule to the web request at all.
         * </p>
         * </note>
         * <p>
         * This is required if <code>AggregateKeyType</code> is set to <code>FORWARDED_IP</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link ForwardedIPConfig.Builder} avoiding the
         * need to create one manually via {@link ForwardedIPConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link ForwardedIPConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #forwardedIPConfig(ForwardedIPConfig)}.
         * 
         * @param forwardedIPConfig
         *        a consumer that will call methods on {@link ForwardedIPConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #forwardedIPConfig(ForwardedIPConfig)
         */
        default Builder forwardedIPConfig(Consumer<ForwardedIPConfig.Builder> forwardedIPConfig) {
            return forwardedIPConfig(ForwardedIPConfig.builder().applyMutation(forwardedIPConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Long limit;

        private String aggregateKeyType;

        private Statement scopeDownStatement;

        private ForwardedIPConfig forwardedIPConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(RateBasedStatement model) {
            limit(model.limit);
            aggregateKeyType(model.aggregateKeyType);
            scopeDownStatement(model.scopeDownStatement);
            forwardedIPConfig(model.forwardedIPConfig);
        }

        public final Long getLimit() {
            return limit;
        }

        public final void setLimit(Long limit) {
            this.limit = limit;
        }

        @Override
        public final Builder limit(Long limit) {
            this.limit = limit;
            return this;
        }

        public final String getAggregateKeyType() {
            return aggregateKeyType;
        }

        public final void setAggregateKeyType(String aggregateKeyType) {
            this.aggregateKeyType = aggregateKeyType;
        }

        @Override
        public final Builder aggregateKeyType(String aggregateKeyType) {
            this.aggregateKeyType = aggregateKeyType;
            return this;
        }

        @Override
        public final Builder aggregateKeyType(RateBasedStatementAggregateKeyType aggregateKeyType) {
            this.aggregateKeyType(aggregateKeyType == null ? null : aggregateKeyType.toString());
            return this;
        }

        public final Statement.Builder getScopeDownStatement() {
            return scopeDownStatement != null ? scopeDownStatement.toBuilder() : null;
        }

        public final void setScopeDownStatement(Statement.BuilderImpl scopeDownStatement) {
            this.scopeDownStatement = scopeDownStatement != null ? scopeDownStatement.build() : null;
        }

        @Override
        public final Builder scopeDownStatement(Statement scopeDownStatement) {
            this.scopeDownStatement = scopeDownStatement;
            return this;
        }

        public final ForwardedIPConfig.Builder getForwardedIPConfig() {
            return forwardedIPConfig != null ? forwardedIPConfig.toBuilder() : null;
        }

        public final void setForwardedIPConfig(ForwardedIPConfig.BuilderImpl forwardedIPConfig) {
            this.forwardedIPConfig = forwardedIPConfig != null ? forwardedIPConfig.build() : null;
        }

        @Override
        public final Builder forwardedIPConfig(ForwardedIPConfig forwardedIPConfig) {
            this.forwardedIPConfig = forwardedIPConfig;
            return this;
        }

        @Override
        public RateBasedStatement build() {
            return new RateBasedStatement(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
