/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A rule statement that inspects for malicious SQL code. Attackers insert malicious SQL code into web requests to do
 * things like modify your database or extract data from it.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SqliMatchStatement implements SdkPojo, Serializable,
        ToCopyableBuilder<SqliMatchStatement.Builder, SqliMatchStatement> {
    private static final SdkField<FieldToMatch> FIELD_TO_MATCH_FIELD = SdkField.<FieldToMatch> builder(MarshallingType.SDK_POJO)
            .memberName("FieldToMatch").getter(getter(SqliMatchStatement::fieldToMatch)).setter(setter(Builder::fieldToMatch))
            .constructor(FieldToMatch::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FieldToMatch").build()).build();

    private static final SdkField<List<TextTransformation>> TEXT_TRANSFORMATIONS_FIELD = SdkField
            .<List<TextTransformation>> builder(MarshallingType.LIST)
            .memberName("TextTransformations")
            .getter(getter(SqliMatchStatement::textTransformations))
            .setter(setter(Builder::textTransformations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TextTransformations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<TextTransformation> builder(MarshallingType.SDK_POJO)
                                            .constructor(TextTransformation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SENSITIVITY_LEVEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SensitivityLevel").getter(getter(SqliMatchStatement::sensitivityLevelAsString))
            .setter(setter(Builder::sensitivityLevel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SensitivityLevel").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FIELD_TO_MATCH_FIELD,
            TEXT_TRANSFORMATIONS_FIELD, SENSITIVITY_LEVEL_FIELD));

    private static final long serialVersionUID = 1L;

    private final FieldToMatch fieldToMatch;

    private final List<TextTransformation> textTransformations;

    private final String sensitivityLevel;

    private SqliMatchStatement(BuilderImpl builder) {
        this.fieldToMatch = builder.fieldToMatch;
        this.textTransformations = builder.textTransformations;
        this.sensitivityLevel = builder.sensitivityLevel;
    }

    /**
     * <p>
     * The part of the web request that you want WAF to inspect.
     * </p>
     * 
     * @return The part of the web request that you want WAF to inspect.
     */
    public final FieldToMatch fieldToMatch() {
        return fieldToMatch;
    }

    /**
     * For responses, this returns true if the service returned a value for the TextTransformations property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTextTransformations() {
        return textTransformations != null && !(textTransformations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Text transformations eliminate some of the unusual formatting that attackers use in web requests in an effort to
     * bypass detection. If you specify one or more transformations in a rule statement, WAF performs all
     * transformations on the content of the request component identified by <code>FieldToMatch</code>, starting from
     * the lowest priority setting, before inspecting the content for a match.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTextTransformations} method.
     * </p>
     * 
     * @return Text transformations eliminate some of the unusual formatting that attackers use in web requests in an
     *         effort to bypass detection. If you specify one or more transformations in a rule statement, WAF performs
     *         all transformations on the content of the request component identified by <code>FieldToMatch</code>,
     *         starting from the lowest priority setting, before inspecting the content for a match.
     */
    public final List<TextTransformation> textTransformations() {
        return textTransformations;
    }

    /**
     * <p>
     * The sensitivity that you want WAF to use to inspect for SQL injection attacks.
     * </p>
     * <p>
     * <code>HIGH</code> detects more attacks, but might generate more false positives, especially if your web requests
     * frequently contain unusual strings. For information about identifying and mitigating false positives, see <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/web-acl-testing.html">Testing and tuning</a> in the
     * <i>WAF Developer Guide</i>.
     * </p>
     * <p>
     * <code>LOW</code> is generally a better choice for resources that already have other protections against SQL
     * injection attacks or that have a low tolerance for false positives.
     * </p>
     * <p>
     * Default: <code>LOW</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sensitivityLevel}
     * will return {@link SensitivityLevel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #sensitivityLevelAsString}.
     * </p>
     * 
     * @return The sensitivity that you want WAF to use to inspect for SQL injection attacks. </p>
     *         <p>
     *         <code>HIGH</code> detects more attacks, but might generate more false positives, especially if your web
     *         requests frequently contain unusual strings. For information about identifying and mitigating false
     *         positives, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/web-acl-testing.html">Testing and tuning</a>
     *         in the <i>WAF Developer Guide</i>.
     *         </p>
     *         <p>
     *         <code>LOW</code> is generally a better choice for resources that already have other protections against
     *         SQL injection attacks or that have a low tolerance for false positives.
     *         </p>
     *         <p>
     *         Default: <code>LOW</code>
     * @see SensitivityLevel
     */
    public final SensitivityLevel sensitivityLevel() {
        return SensitivityLevel.fromValue(sensitivityLevel);
    }

    /**
     * <p>
     * The sensitivity that you want WAF to use to inspect for SQL injection attacks.
     * </p>
     * <p>
     * <code>HIGH</code> detects more attacks, but might generate more false positives, especially if your web requests
     * frequently contain unusual strings. For information about identifying and mitigating false positives, see <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/web-acl-testing.html">Testing and tuning</a> in the
     * <i>WAF Developer Guide</i>.
     * </p>
     * <p>
     * <code>LOW</code> is generally a better choice for resources that already have other protections against SQL
     * injection attacks or that have a low tolerance for false positives.
     * </p>
     * <p>
     * Default: <code>LOW</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sensitivityLevel}
     * will return {@link SensitivityLevel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #sensitivityLevelAsString}.
     * </p>
     * 
     * @return The sensitivity that you want WAF to use to inspect for SQL injection attacks. </p>
     *         <p>
     *         <code>HIGH</code> detects more attacks, but might generate more false positives, especially if your web
     *         requests frequently contain unusual strings. For information about identifying and mitigating false
     *         positives, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/web-acl-testing.html">Testing and tuning</a>
     *         in the <i>WAF Developer Guide</i>.
     *         </p>
     *         <p>
     *         <code>LOW</code> is generally a better choice for resources that already have other protections against
     *         SQL injection attacks or that have a low tolerance for false positives.
     *         </p>
     *         <p>
     *         Default: <code>LOW</code>
     * @see SensitivityLevel
     */
    public final String sensitivityLevelAsString() {
        return sensitivityLevel;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(fieldToMatch());
        hashCode = 31 * hashCode + Objects.hashCode(hasTextTransformations() ? textTransformations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(sensitivityLevelAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SqliMatchStatement)) {
            return false;
        }
        SqliMatchStatement other = (SqliMatchStatement) obj;
        return Objects.equals(fieldToMatch(), other.fieldToMatch()) && hasTextTransformations() == other.hasTextTransformations()
                && Objects.equals(textTransformations(), other.textTransformations())
                && Objects.equals(sensitivityLevelAsString(), other.sensitivityLevelAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SqliMatchStatement").add("FieldToMatch", fieldToMatch())
                .add("TextTransformations", hasTextTransformations() ? textTransformations() : null)
                .add("SensitivityLevel", sensitivityLevelAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FieldToMatch":
            return Optional.ofNullable(clazz.cast(fieldToMatch()));
        case "TextTransformations":
            return Optional.ofNullable(clazz.cast(textTransformations()));
        case "SensitivityLevel":
            return Optional.ofNullable(clazz.cast(sensitivityLevelAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SqliMatchStatement, T> g) {
        return obj -> g.apply((SqliMatchStatement) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SqliMatchStatement> {
        /**
         * <p>
         * The part of the web request that you want WAF to inspect.
         * </p>
         * 
         * @param fieldToMatch
         *        The part of the web request that you want WAF to inspect.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fieldToMatch(FieldToMatch fieldToMatch);

        /**
         * <p>
         * The part of the web request that you want WAF to inspect.
         * </p>
         * This is a convenience method that creates an instance of the {@link FieldToMatch.Builder} avoiding the need
         * to create one manually via {@link FieldToMatch#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FieldToMatch.Builder#build()} is called immediately and its
         * result is passed to {@link #fieldToMatch(FieldToMatch)}.
         * 
         * @param fieldToMatch
         *        a consumer that will call methods on {@link FieldToMatch.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #fieldToMatch(FieldToMatch)
         */
        default Builder fieldToMatch(Consumer<FieldToMatch.Builder> fieldToMatch) {
            return fieldToMatch(FieldToMatch.builder().applyMutation(fieldToMatch).build());
        }

        /**
         * <p>
         * Text transformations eliminate some of the unusual formatting that attackers use in web requests in an effort
         * to bypass detection. If you specify one or more transformations in a rule statement, WAF performs all
         * transformations on the content of the request component identified by <code>FieldToMatch</code>, starting
         * from the lowest priority setting, before inspecting the content for a match.
         * </p>
         * 
         * @param textTransformations
         *        Text transformations eliminate some of the unusual formatting that attackers use in web requests in an
         *        effort to bypass detection. If you specify one or more transformations in a rule statement, WAF
         *        performs all transformations on the content of the request component identified by
         *        <code>FieldToMatch</code>, starting from the lowest priority setting, before inspecting the content
         *        for a match.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder textTransformations(Collection<TextTransformation> textTransformations);

        /**
         * <p>
         * Text transformations eliminate some of the unusual formatting that attackers use in web requests in an effort
         * to bypass detection. If you specify one or more transformations in a rule statement, WAF performs all
         * transformations on the content of the request component identified by <code>FieldToMatch</code>, starting
         * from the lowest priority setting, before inspecting the content for a match.
         * </p>
         * 
         * @param textTransformations
         *        Text transformations eliminate some of the unusual formatting that attackers use in web requests in an
         *        effort to bypass detection. If you specify one or more transformations in a rule statement, WAF
         *        performs all transformations on the content of the request component identified by
         *        <code>FieldToMatch</code>, starting from the lowest priority setting, before inspecting the content
         *        for a match.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder textTransformations(TextTransformation... textTransformations);

        /**
         * <p>
         * Text transformations eliminate some of the unusual formatting that attackers use in web requests in an effort
         * to bypass detection. If you specify one or more transformations in a rule statement, WAF performs all
         * transformations on the content of the request component identified by <code>FieldToMatch</code>, starting
         * from the lowest priority setting, before inspecting the content for a match.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.wafv2.model.TextTransformation.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.wafv2.model.TextTransformation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.wafv2.model.TextTransformation.Builder#build()} is called immediately
         * and its result is passed to {@link #textTransformations(List<TextTransformation>)}.
         * 
         * @param textTransformations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.wafv2.model.TextTransformation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #textTransformations(java.util.Collection<TextTransformation>)
         */
        Builder textTransformations(Consumer<TextTransformation.Builder>... textTransformations);

        /**
         * <p>
         * The sensitivity that you want WAF to use to inspect for SQL injection attacks.
         * </p>
         * <p>
         * <code>HIGH</code> detects more attacks, but might generate more false positives, especially if your web
         * requests frequently contain unusual strings. For information about identifying and mitigating false
         * positives, see <a href="https://docs.aws.amazon.com/waf/latest/developerguide/web-acl-testing.html">Testing
         * and tuning</a> in the <i>WAF Developer Guide</i>.
         * </p>
         * <p>
         * <code>LOW</code> is generally a better choice for resources that already have other protections against SQL
         * injection attacks or that have a low tolerance for false positives.
         * </p>
         * <p>
         * Default: <code>LOW</code>
         * </p>
         * 
         * @param sensitivityLevel
         *        The sensitivity that you want WAF to use to inspect for SQL injection attacks. </p>
         *        <p>
         *        <code>HIGH</code> detects more attacks, but might generate more false positives, especially if your
         *        web requests frequently contain unusual strings. For information about identifying and mitigating
         *        false positives, see <a
         *        href="https://docs.aws.amazon.com/waf/latest/developerguide/web-acl-testing.html">Testing and
         *        tuning</a> in the <i>WAF Developer Guide</i>.
         *        </p>
         *        <p>
         *        <code>LOW</code> is generally a better choice for resources that already have other protections
         *        against SQL injection attacks or that have a low tolerance for false positives.
         *        </p>
         *        <p>
         *        Default: <code>LOW</code>
         * @see SensitivityLevel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SensitivityLevel
         */
        Builder sensitivityLevel(String sensitivityLevel);

        /**
         * <p>
         * The sensitivity that you want WAF to use to inspect for SQL injection attacks.
         * </p>
         * <p>
         * <code>HIGH</code> detects more attacks, but might generate more false positives, especially if your web
         * requests frequently contain unusual strings. For information about identifying and mitigating false
         * positives, see <a href="https://docs.aws.amazon.com/waf/latest/developerguide/web-acl-testing.html">Testing
         * and tuning</a> in the <i>WAF Developer Guide</i>.
         * </p>
         * <p>
         * <code>LOW</code> is generally a better choice for resources that already have other protections against SQL
         * injection attacks or that have a low tolerance for false positives.
         * </p>
         * <p>
         * Default: <code>LOW</code>
         * </p>
         * 
         * @param sensitivityLevel
         *        The sensitivity that you want WAF to use to inspect for SQL injection attacks. </p>
         *        <p>
         *        <code>HIGH</code> detects more attacks, but might generate more false positives, especially if your
         *        web requests frequently contain unusual strings. For information about identifying and mitigating
         *        false positives, see <a
         *        href="https://docs.aws.amazon.com/waf/latest/developerguide/web-acl-testing.html">Testing and
         *        tuning</a> in the <i>WAF Developer Guide</i>.
         *        </p>
         *        <p>
         *        <code>LOW</code> is generally a better choice for resources that already have other protections
         *        against SQL injection attacks or that have a low tolerance for false positives.
         *        </p>
         *        <p>
         *        Default: <code>LOW</code>
         * @see SensitivityLevel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SensitivityLevel
         */
        Builder sensitivityLevel(SensitivityLevel sensitivityLevel);
    }

    static final class BuilderImpl implements Builder {
        private FieldToMatch fieldToMatch;

        private List<TextTransformation> textTransformations = DefaultSdkAutoConstructList.getInstance();

        private String sensitivityLevel;

        private BuilderImpl() {
        }

        private BuilderImpl(SqliMatchStatement model) {
            fieldToMatch(model.fieldToMatch);
            textTransformations(model.textTransformations);
            sensitivityLevel(model.sensitivityLevel);
        }

        public final FieldToMatch.Builder getFieldToMatch() {
            return fieldToMatch != null ? fieldToMatch.toBuilder() : null;
        }

        public final void setFieldToMatch(FieldToMatch.BuilderImpl fieldToMatch) {
            this.fieldToMatch = fieldToMatch != null ? fieldToMatch.build() : null;
        }

        @Override
        public final Builder fieldToMatch(FieldToMatch fieldToMatch) {
            this.fieldToMatch = fieldToMatch;
            return this;
        }

        public final List<TextTransformation.Builder> getTextTransformations() {
            List<TextTransformation.Builder> result = TextTransformationsCopier.copyToBuilder(this.textTransformations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTextTransformations(Collection<TextTransformation.BuilderImpl> textTransformations) {
            this.textTransformations = TextTransformationsCopier.copyFromBuilder(textTransformations);
        }

        @Override
        public final Builder textTransformations(Collection<TextTransformation> textTransformations) {
            this.textTransformations = TextTransformationsCopier.copy(textTransformations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder textTransformations(TextTransformation... textTransformations) {
            textTransformations(Arrays.asList(textTransformations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder textTransformations(Consumer<TextTransformation.Builder>... textTransformations) {
            textTransformations(Stream.of(textTransformations).map(c -> TextTransformation.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getSensitivityLevel() {
            return sensitivityLevel;
        }

        public final void setSensitivityLevel(String sensitivityLevel) {
            this.sensitivityLevel = sensitivityLevel;
        }

        @Override
        public final Builder sensitivityLevel(String sensitivityLevel) {
            this.sensitivityLevel = sensitivityLevel;
            return this;
        }

        @Override
        public final Builder sensitivityLevel(SensitivityLevel sensitivityLevel) {
            this.sensitivityLevel(sensitivityLevel == null ? null : sensitivityLevel.toString());
            return this;
        }

        @Override
        public SqliMatchStatement build() {
            return new SqliMatchStatement(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
