/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A rule statement that labels web requests by country and region and that matches against web requests based on
 * country code. A geo match rule labels every request that it inspects regardless of whether it finds a match.
 * </p>
 * <ul>
 * <li>
 * <p>
 * To manage requests only by country, you can use this statement by itself and specify the countries that you want to
 * match against in the <code>CountryCodes</code> array.
 * </p>
 * </li>
 * <li>
 * <p>
 * Otherwise, configure your geo match rule with Count action so that it only labels requests. Then, add one or more
 * label match rules to run after the geo match rule and configure them to match against the geographic labels and
 * handle the requests as needed.
 * </p>
 * </li>
 * </ul>
 * <p>
 * WAF labels requests using the alpha-2 country and region codes from the International Organization for
 * Standardization (ISO) 3166 standard. WAF determines the codes using either the IP address in the web request origin
 * or, if you specify it, the address in the geo match <code>ForwardedIPConfig</code>.
 * </p>
 * <p>
 * If you use the web request origin, the label formats are
 * <code>awswaf:clientip:geo:region:&lt;ISO country code&gt;-&lt;ISO region code&gt;</code> and
 * <code>awswaf:clientip:geo:country:&lt;ISO country code&gt;</code>.
 * </p>
 * <p>
 * If you use a forwarded IP address, the label formats are
 * <code>awswaf:forwardedip:geo:region:&lt;ISO country code&gt;-&lt;ISO region code&gt;</code> and
 * <code>awswaf:forwardedip:geo:country:&lt;ISO country code&gt;</code>.
 * </p>
 * <p>
 * For additional details, see <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-rule-statement-type-geo-match.html">Geographic match
 * rule statement</a> in the <a href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">WAF
 * Developer Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GeoMatchStatement implements SdkPojo, Serializable,
        ToCopyableBuilder<GeoMatchStatement.Builder, GeoMatchStatement> {
    private static final SdkField<List<String>> COUNTRY_CODES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("CountryCodes")
            .getter(getter(GeoMatchStatement::countryCodesAsStrings))
            .setter(setter(Builder::countryCodesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CountryCodes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ForwardedIPConfig> FORWARDED_IP_CONFIG_FIELD = SdkField
            .<ForwardedIPConfig> builder(MarshallingType.SDK_POJO).memberName("ForwardedIPConfig")
            .getter(getter(GeoMatchStatement::forwardedIPConfig)).setter(setter(Builder::forwardedIPConfig))
            .constructor(ForwardedIPConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ForwardedIPConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COUNTRY_CODES_FIELD,
            FORWARDED_IP_CONFIG_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<String> countryCodes;

    private final ForwardedIPConfig forwardedIPConfig;

    private GeoMatchStatement(BuilderImpl builder) {
        this.countryCodes = builder.countryCodes;
        this.forwardedIPConfig = builder.forwardedIPConfig;
    }

    /**
     * <p>
     * An array of two-character country codes that you want to match against, for example, <code>[ "US", "CN" ]</code>,
     * from the alpha-2 country ISO codes of the ISO 3166 international standard.
     * </p>
     * <p>
     * When you use a geo match statement just for the region and country labels that it adds to requests, you still
     * have to supply a country code for the rule to evaluate. In this case, you configure the rule to only count
     * matching requests, but it will still generate logging and count metrics for any matches. You can reduce the
     * logging and metrics that the rule produces by specifying a country that's unlikely to be a source of traffic to
     * your site.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCountryCodes} method.
     * </p>
     * 
     * @return An array of two-character country codes that you want to match against, for example,
     *         <code>[ "US", "CN" ]</code>, from the alpha-2 country ISO codes of the ISO 3166 international standard.
     *         </p>
     *         <p>
     *         When you use a geo match statement just for the region and country labels that it adds to requests, you
     *         still have to supply a country code for the rule to evaluate. In this case, you configure the rule to
     *         only count matching requests, but it will still generate logging and count metrics for any matches. You
     *         can reduce the logging and metrics that the rule produces by specifying a country that's unlikely to be a
     *         source of traffic to your site.
     */
    public final List<CountryCode> countryCodes() {
        return CountryCodesCopier.copyStringToEnum(countryCodes);
    }

    /**
     * For responses, this returns true if the service returned a value for the CountryCodes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCountryCodes() {
        return countryCodes != null && !(countryCodes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of two-character country codes that you want to match against, for example, <code>[ "US", "CN" ]</code>,
     * from the alpha-2 country ISO codes of the ISO 3166 international standard.
     * </p>
     * <p>
     * When you use a geo match statement just for the region and country labels that it adds to requests, you still
     * have to supply a country code for the rule to evaluate. In this case, you configure the rule to only count
     * matching requests, but it will still generate logging and count metrics for any matches. You can reduce the
     * logging and metrics that the rule produces by specifying a country that's unlikely to be a source of traffic to
     * your site.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCountryCodes} method.
     * </p>
     * 
     * @return An array of two-character country codes that you want to match against, for example,
     *         <code>[ "US", "CN" ]</code>, from the alpha-2 country ISO codes of the ISO 3166 international standard.
     *         </p>
     *         <p>
     *         When you use a geo match statement just for the region and country labels that it adds to requests, you
     *         still have to supply a country code for the rule to evaluate. In this case, you configure the rule to
     *         only count matching requests, but it will still generate logging and count metrics for any matches. You
     *         can reduce the logging and metrics that the rule produces by specifying a country that's unlikely to be a
     *         source of traffic to your site.
     */
    public final List<String> countryCodesAsStrings() {
        return countryCodes;
    }

    /**
     * <p>
     * The configuration for inspecting IP addresses in an HTTP header that you specify, instead of using the IP address
     * that's reported by the web request origin. Commonly, this is the X-Forwarded-For (XFF) header, but you can
     * specify any header name.
     * </p>
     * <note>
     * <p>
     * If the specified header isn't present in the request, WAF doesn't apply the rule to the web request at all.
     * </p>
     * </note>
     * 
     * @return The configuration for inspecting IP addresses in an HTTP header that you specify, instead of using the IP
     *         address that's reported by the web request origin. Commonly, this is the X-Forwarded-For (XFF) header,
     *         but you can specify any header name. </p> <note>
     *         <p>
     *         If the specified header isn't present in the request, WAF doesn't apply the rule to the web request at
     *         all.
     *         </p>
     */
    public final ForwardedIPConfig forwardedIPConfig() {
        return forwardedIPConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasCountryCodes() ? countryCodesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(forwardedIPConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GeoMatchStatement)) {
            return false;
        }
        GeoMatchStatement other = (GeoMatchStatement) obj;
        return hasCountryCodes() == other.hasCountryCodes()
                && Objects.equals(countryCodesAsStrings(), other.countryCodesAsStrings())
                && Objects.equals(forwardedIPConfig(), other.forwardedIPConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GeoMatchStatement").add("CountryCodes", hasCountryCodes() ? countryCodesAsStrings() : null)
                .add("ForwardedIPConfig", forwardedIPConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CountryCodes":
            return Optional.ofNullable(clazz.cast(countryCodesAsStrings()));
        case "ForwardedIPConfig":
            return Optional.ofNullable(clazz.cast(forwardedIPConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("CountryCodes", COUNTRY_CODES_FIELD);
        map.put("ForwardedIPConfig", FORWARDED_IP_CONFIG_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<GeoMatchStatement, T> g) {
        return obj -> g.apply((GeoMatchStatement) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GeoMatchStatement> {
        /**
         * <p>
         * An array of two-character country codes that you want to match against, for example,
         * <code>[ "US", "CN" ]</code>, from the alpha-2 country ISO codes of the ISO 3166 international standard.
         * </p>
         * <p>
         * When you use a geo match statement just for the region and country labels that it adds to requests, you still
         * have to supply a country code for the rule to evaluate. In this case, you configure the rule to only count
         * matching requests, but it will still generate logging and count metrics for any matches. You can reduce the
         * logging and metrics that the rule produces by specifying a country that's unlikely to be a source of traffic
         * to your site.
         * </p>
         * 
         * @param countryCodes
         *        An array of two-character country codes that you want to match against, for example,
         *        <code>[ "US", "CN" ]</code>, from the alpha-2 country ISO codes of the ISO 3166 international
         *        standard. </p>
         *        <p>
         *        When you use a geo match statement just for the region and country labels that it adds to requests,
         *        you still have to supply a country code for the rule to evaluate. In this case, you configure the rule
         *        to only count matching requests, but it will still generate logging and count metrics for any matches.
         *        You can reduce the logging and metrics that the rule produces by specifying a country that's unlikely
         *        to be a source of traffic to your site.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder countryCodesWithStrings(Collection<String> countryCodes);

        /**
         * <p>
         * An array of two-character country codes that you want to match against, for example,
         * <code>[ "US", "CN" ]</code>, from the alpha-2 country ISO codes of the ISO 3166 international standard.
         * </p>
         * <p>
         * When you use a geo match statement just for the region and country labels that it adds to requests, you still
         * have to supply a country code for the rule to evaluate. In this case, you configure the rule to only count
         * matching requests, but it will still generate logging and count metrics for any matches. You can reduce the
         * logging and metrics that the rule produces by specifying a country that's unlikely to be a source of traffic
         * to your site.
         * </p>
         * 
         * @param countryCodes
         *        An array of two-character country codes that you want to match against, for example,
         *        <code>[ "US", "CN" ]</code>, from the alpha-2 country ISO codes of the ISO 3166 international
         *        standard. </p>
         *        <p>
         *        When you use a geo match statement just for the region and country labels that it adds to requests,
         *        you still have to supply a country code for the rule to evaluate. In this case, you configure the rule
         *        to only count matching requests, but it will still generate logging and count metrics for any matches.
         *        You can reduce the logging and metrics that the rule produces by specifying a country that's unlikely
         *        to be a source of traffic to your site.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder countryCodesWithStrings(String... countryCodes);

        /**
         * <p>
         * An array of two-character country codes that you want to match against, for example,
         * <code>[ "US", "CN" ]</code>, from the alpha-2 country ISO codes of the ISO 3166 international standard.
         * </p>
         * <p>
         * When you use a geo match statement just for the region and country labels that it adds to requests, you still
         * have to supply a country code for the rule to evaluate. In this case, you configure the rule to only count
         * matching requests, but it will still generate logging and count metrics for any matches. You can reduce the
         * logging and metrics that the rule produces by specifying a country that's unlikely to be a source of traffic
         * to your site.
         * </p>
         * 
         * @param countryCodes
         *        An array of two-character country codes that you want to match against, for example,
         *        <code>[ "US", "CN" ]</code>, from the alpha-2 country ISO codes of the ISO 3166 international
         *        standard. </p>
         *        <p>
         *        When you use a geo match statement just for the region and country labels that it adds to requests,
         *        you still have to supply a country code for the rule to evaluate. In this case, you configure the rule
         *        to only count matching requests, but it will still generate logging and count metrics for any matches.
         *        You can reduce the logging and metrics that the rule produces by specifying a country that's unlikely
         *        to be a source of traffic to your site.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder countryCodes(Collection<CountryCode> countryCodes);

        /**
         * <p>
         * An array of two-character country codes that you want to match against, for example,
         * <code>[ "US", "CN" ]</code>, from the alpha-2 country ISO codes of the ISO 3166 international standard.
         * </p>
         * <p>
         * When you use a geo match statement just for the region and country labels that it adds to requests, you still
         * have to supply a country code for the rule to evaluate. In this case, you configure the rule to only count
         * matching requests, but it will still generate logging and count metrics for any matches. You can reduce the
         * logging and metrics that the rule produces by specifying a country that's unlikely to be a source of traffic
         * to your site.
         * </p>
         * 
         * @param countryCodes
         *        An array of two-character country codes that you want to match against, for example,
         *        <code>[ "US", "CN" ]</code>, from the alpha-2 country ISO codes of the ISO 3166 international
         *        standard. </p>
         *        <p>
         *        When you use a geo match statement just for the region and country labels that it adds to requests,
         *        you still have to supply a country code for the rule to evaluate. In this case, you configure the rule
         *        to only count matching requests, but it will still generate logging and count metrics for any matches.
         *        You can reduce the logging and metrics that the rule produces by specifying a country that's unlikely
         *        to be a source of traffic to your site.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder countryCodes(CountryCode... countryCodes);

        /**
         * <p>
         * The configuration for inspecting IP addresses in an HTTP header that you specify, instead of using the IP
         * address that's reported by the web request origin. Commonly, this is the X-Forwarded-For (XFF) header, but
         * you can specify any header name.
         * </p>
         * <note>
         * <p>
         * If the specified header isn't present in the request, WAF doesn't apply the rule to the web request at all.
         * </p>
         * </note>
         * 
         * @param forwardedIPConfig
         *        The configuration for inspecting IP addresses in an HTTP header that you specify, instead of using the
         *        IP address that's reported by the web request origin. Commonly, this is the X-Forwarded-For (XFF)
         *        header, but you can specify any header name. </p> <note>
         *        <p>
         *        If the specified header isn't present in the request, WAF doesn't apply the rule to the web request at
         *        all.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forwardedIPConfig(ForwardedIPConfig forwardedIPConfig);

        /**
         * <p>
         * The configuration for inspecting IP addresses in an HTTP header that you specify, instead of using the IP
         * address that's reported by the web request origin. Commonly, this is the X-Forwarded-For (XFF) header, but
         * you can specify any header name.
         * </p>
         * <note>
         * <p>
         * If the specified header isn't present in the request, WAF doesn't apply the rule to the web request at all.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link ForwardedIPConfig.Builder}
         * avoiding the need to create one manually via {@link ForwardedIPConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ForwardedIPConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #forwardedIPConfig(ForwardedIPConfig)}.
         * 
         * @param forwardedIPConfig
         *        a consumer that will call methods on {@link ForwardedIPConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #forwardedIPConfig(ForwardedIPConfig)
         */
        default Builder forwardedIPConfig(Consumer<ForwardedIPConfig.Builder> forwardedIPConfig) {
            return forwardedIPConfig(ForwardedIPConfig.builder().applyMutation(forwardedIPConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private List<String> countryCodes = DefaultSdkAutoConstructList.getInstance();

        private ForwardedIPConfig forwardedIPConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(GeoMatchStatement model) {
            countryCodesWithStrings(model.countryCodes);
            forwardedIPConfig(model.forwardedIPConfig);
        }

        public final Collection<String> getCountryCodes() {
            if (countryCodes instanceof SdkAutoConstructList) {
                return null;
            }
            return countryCodes;
        }

        public final void setCountryCodes(Collection<String> countryCodes) {
            this.countryCodes = CountryCodesCopier.copy(countryCodes);
        }

        @Override
        public final Builder countryCodesWithStrings(Collection<String> countryCodes) {
            this.countryCodes = CountryCodesCopier.copy(countryCodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder countryCodesWithStrings(String... countryCodes) {
            countryCodesWithStrings(Arrays.asList(countryCodes));
            return this;
        }

        @Override
        public final Builder countryCodes(Collection<CountryCode> countryCodes) {
            this.countryCodes = CountryCodesCopier.copyEnumToString(countryCodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder countryCodes(CountryCode... countryCodes) {
            countryCodes(Arrays.asList(countryCodes));
            return this;
        }

        public final ForwardedIPConfig.Builder getForwardedIPConfig() {
            return forwardedIPConfig != null ? forwardedIPConfig.toBuilder() : null;
        }

        public final void setForwardedIPConfig(ForwardedIPConfig.BuilderImpl forwardedIPConfig) {
            this.forwardedIPConfig = forwardedIPConfig != null ? forwardedIPConfig.build() : null;
        }

        @Override
        public final Builder forwardedIPConfig(ForwardedIPConfig forwardedIPConfig) {
            this.forwardedIPConfig = forwardedIPConfig;
            return this;
        }

        @Override
        public GeoMatchStatement build() {
            return new GeoMatchStatement(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
