/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines an association between logging destinations and a web ACL resource, for logging from WAF. As part of the
 * association, you can specify parts of the standard logging fields to keep out of the logs and you can specify filters
 * so that you log only a subset of the logging records.
 * </p>
 * <note>
 * <p>
 * You can define one logging destination per web ACL.
 * </p>
 * </note>
 * <p>
 * You can access information about the traffic that WAF inspects using the following steps:
 * </p>
 * <ol>
 * <li>
 * <p>
 * Create your logging destination. You can use an Amazon CloudWatch Logs log group, an Amazon Simple Storage Service
 * (Amazon S3) bucket, or an Amazon Kinesis Data Firehose.
 * </p>
 * <p>
 * The name that you give the destination must start with <code>aws-waf-logs-</code>. Depending on the type of
 * destination, you might need to configure additional settings or permissions.
 * </p>
 * <p>
 * For configuration requirements and pricing information for each destination type, see <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/logging.html">Logging web ACL traffic</a> in the <i>WAF
 * Developer Guide</i>.
 * </p>
 * </li>
 * <li>
 * <p>
 * Associate your logging destination to your web ACL using a <code>PutLoggingConfiguration</code> request.
 * </p>
 * </li>
 * </ol>
 * <p>
 * When you successfully enable logging using a <code>PutLoggingConfiguration</code> request, WAF creates an additional
 * role or policy that is required to write logs to the logging destination. For an Amazon CloudWatch Logs log group,
 * WAF creates a resource policy on the log group. For an Amazon S3 bucket, WAF creates a bucket policy. For an Amazon
 * Kinesis Data Firehose, WAF creates a service-linked role.
 * </p>
 * <p>
 * For additional information about web ACL logging, see <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/logging.html">Logging web ACL traffic information</a> in
 * the <i>WAF Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LoggingConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<LoggingConfiguration.Builder, LoggingConfiguration> {
    private static final SdkField<String> RESOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceArn").getter(getter(LoggingConfiguration::resourceArn)).setter(setter(Builder::resourceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceArn").build()).build();

    private static final SdkField<List<String>> LOG_DESTINATION_CONFIGS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("LogDestinationConfigs")
            .getter(getter(LoggingConfiguration::logDestinationConfigs))
            .setter(setter(Builder::logDestinationConfigs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogDestinationConfigs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<FieldToMatch>> REDACTED_FIELDS_FIELD = SdkField
            .<List<FieldToMatch>> builder(MarshallingType.LIST)
            .memberName("RedactedFields")
            .getter(getter(LoggingConfiguration::redactedFields))
            .setter(setter(Builder::redactedFields))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RedactedFields").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<FieldToMatch> builder(MarshallingType.SDK_POJO)
                                            .constructor(FieldToMatch::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> MANAGED_BY_FIREWALL_MANAGER_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN).memberName("ManagedByFirewallManager")
            .getter(getter(LoggingConfiguration::managedByFirewallManager)).setter(setter(Builder::managedByFirewallManager))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ManagedByFirewallManager").build())
            .build();

    private static final SdkField<LoggingFilter> LOGGING_FILTER_FIELD = SdkField
            .<LoggingFilter> builder(MarshallingType.SDK_POJO).memberName("LoggingFilter")
            .getter(getter(LoggingConfiguration::loggingFilter)).setter(setter(Builder::loggingFilter))
            .constructor(LoggingFilter::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LoggingFilter").build()).build();

    private static final SdkField<String> LOG_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LogType").getter(getter(LoggingConfiguration::logTypeAsString)).setter(setter(Builder::logType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogType").build()).build();

    private static final SdkField<String> LOG_SCOPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LogScope").getter(getter(LoggingConfiguration::logScopeAsString)).setter(setter(Builder::logScope))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogScope").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESOURCE_ARN_FIELD,
            LOG_DESTINATION_CONFIGS_FIELD, REDACTED_FIELDS_FIELD, MANAGED_BY_FIREWALL_MANAGER_FIELD, LOGGING_FILTER_FIELD,
            LOG_TYPE_FIELD, LOG_SCOPE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String resourceArn;

    private final List<String> logDestinationConfigs;

    private final List<FieldToMatch> redactedFields;

    private final Boolean managedByFirewallManager;

    private final LoggingFilter loggingFilter;

    private final String logType;

    private final String logScope;

    private LoggingConfiguration(BuilderImpl builder) {
        this.resourceArn = builder.resourceArn;
        this.logDestinationConfigs = builder.logDestinationConfigs;
        this.redactedFields = builder.redactedFields;
        this.managedByFirewallManager = builder.managedByFirewallManager;
        this.loggingFilter = builder.loggingFilter;
        this.logType = builder.logType;
        this.logScope = builder.logScope;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the web ACL that you want to associate with <code>LogDestinationConfigs</code>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the web ACL that you want to associate with
     *         <code>LogDestinationConfigs</code>.
     */
    public final String resourceArn() {
        return resourceArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the LogDestinationConfigs property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasLogDestinationConfigs() {
        return logDestinationConfigs != null && !(logDestinationConfigs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The logging destination configuration that you want to associate with the web ACL.
     * </p>
     * <note>
     * <p>
     * You can associate one logging destination to a web ACL.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLogDestinationConfigs} method.
     * </p>
     * 
     * @return The logging destination configuration that you want to associate with the web ACL.</p> <note>
     *         <p>
     *         You can associate one logging destination to a web ACL.
     *         </p>
     */
    public final List<String> logDestinationConfigs() {
        return logDestinationConfigs;
    }

    /**
     * For responses, this returns true if the service returned a value for the RedactedFields property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRedactedFields() {
        return redactedFields != null && !(redactedFields instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The parts of the request that you want to keep out of the logs.
     * </p>
     * <p>
     * For example, if you redact the <code>SingleHeader</code> field, the <code>HEADER</code> field in the logs will be
     * <code>REDACTED</code> for all rules that use the <code>SingleHeader</code> <code>FieldToMatch</code> setting.
     * </p>
     * <p>
     * Redaction applies only to the component that's specified in the rule's <code>FieldToMatch</code> setting, so the
     * <code>SingleHeader</code> redaction doesn't apply to rules that use the <code>Headers</code>
     * <code>FieldToMatch</code>.
     * </p>
     * <note>
     * <p>
     * You can specify only the following fields for redaction: <code>UriPath</code>, <code>QueryString</code>,
     * <code>SingleHeader</code>, and <code>Method</code>.
     * </p>
     * </note> <note>
     * <p>
     * This setting has no impact on request sampling. With request sampling, the only way to exclude fields is by
     * disabling sampling in the web ACL visibility configuration.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRedactedFields} method.
     * </p>
     * 
     * @return The parts of the request that you want to keep out of the logs.</p>
     *         <p>
     *         For example, if you redact the <code>SingleHeader</code> field, the <code>HEADER</code> field in the logs
     *         will be <code>REDACTED</code> for all rules that use the <code>SingleHeader</code>
     *         <code>FieldToMatch</code> setting.
     *         </p>
     *         <p>
     *         Redaction applies only to the component that's specified in the rule's <code>FieldToMatch</code> setting,
     *         so the <code>SingleHeader</code> redaction doesn't apply to rules that use the <code>Headers</code>
     *         <code>FieldToMatch</code>.
     *         </p>
     *         <note>
     *         <p>
     *         You can specify only the following fields for redaction: <code>UriPath</code>, <code>QueryString</code>,
     *         <code>SingleHeader</code>, and <code>Method</code>.
     *         </p>
     *         </note> <note>
     *         <p>
     *         This setting has no impact on request sampling. With request sampling, the only way to exclude fields is
     *         by disabling sampling in the web ACL visibility configuration.
     *         </p>
     */
    public final List<FieldToMatch> redactedFields() {
        return redactedFields;
    }

    /**
     * <p>
     * Indicates whether the logging configuration was created by Firewall Manager, as part of an WAF policy
     * configuration. If true, only Firewall Manager can modify or delete the configuration.
     * </p>
     * <p>
     * The logging configuration can be created by Firewall Manager for use with any web ACL that Firewall Manager is
     * using for an WAF policy. Web ACLs that Firewall Manager creates and uses have their
     * <code>ManagedByFirewallManager</code> property set to true. Web ACLs that were created by a customer account and
     * then retrofitted by Firewall Manager for use by a policy have their <code>RetrofittedByFirewallManager</code>
     * property set to true. For either case, any corresponding logging configuration will indicate
     * <code>ManagedByFirewallManager</code>.
     * </p>
     * 
     * @return Indicates whether the logging configuration was created by Firewall Manager, as part of an WAF policy
     *         configuration. If true, only Firewall Manager can modify or delete the configuration. </p>
     *         <p>
     *         The logging configuration can be created by Firewall Manager for use with any web ACL that Firewall
     *         Manager is using for an WAF policy. Web ACLs that Firewall Manager creates and uses have their
     *         <code>ManagedByFirewallManager</code> property set to true. Web ACLs that were created by a customer
     *         account and then retrofitted by Firewall Manager for use by a policy have their
     *         <code>RetrofittedByFirewallManager</code> property set to true. For either case, any corresponding
     *         logging configuration will indicate <code>ManagedByFirewallManager</code>.
     */
    public final Boolean managedByFirewallManager() {
        return managedByFirewallManager;
    }

    /**
     * <p>
     * Filtering that specifies which web requests are kept in the logs and which are dropped. You can filter on the
     * rule action and on the web request labels that were applied by matching rules during web ACL evaluation.
     * </p>
     * 
     * @return Filtering that specifies which web requests are kept in the logs and which are dropped. You can filter on
     *         the rule action and on the web request labels that were applied by matching rules during web ACL
     *         evaluation.
     */
    public final LoggingFilter loggingFilter() {
        return loggingFilter;
    }

    /**
     * <p>
     * Used to distinguish between various logging options. Currently, there is one option.
     * </p>
     * <p>
     * Default: <code>WAF_LOGS</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #logType} will
     * return {@link LogType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #logTypeAsString}.
     * </p>
     * 
     * @return Used to distinguish between various logging options. Currently, there is one option.</p>
     *         <p>
     *         Default: <code>WAF_LOGS</code>
     * @see LogType
     */
    public final LogType logType() {
        return LogType.fromValue(logType);
    }

    /**
     * <p>
     * Used to distinguish between various logging options. Currently, there is one option.
     * </p>
     * <p>
     * Default: <code>WAF_LOGS</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #logType} will
     * return {@link LogType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #logTypeAsString}.
     * </p>
     * 
     * @return Used to distinguish between various logging options. Currently, there is one option.</p>
     *         <p>
     *         Default: <code>WAF_LOGS</code>
     * @see LogType
     */
    public final String logTypeAsString() {
        return logType;
    }

    /**
     * <p>
     * The owner of the logging configuration, which must be set to <code>CUSTOMER</code> for the configurations that
     * you manage.
     * </p>
     * <p>
     * The log scope <code>SECURITY_LAKE</code> indicates a configuration that is managed through Amazon Security Lake.
     * You can use Security Lake to collect log and event data from various sources for normalization, analysis, and
     * management. For information, see <a
     * href="https://docs.aws.amazon.com/security-lake/latest/userguide/internal-sources.html">Collecting data from
     * Amazon Web Services services</a> in the <i>Amazon Security Lake user guide</i>.
     * </p>
     * <p>
     * Default: <code>CUSTOMER</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #logScope} will
     * return {@link LogScope#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #logScopeAsString}.
     * </p>
     * 
     * @return The owner of the logging configuration, which must be set to <code>CUSTOMER</code> for the configurations
     *         that you manage. </p>
     *         <p>
     *         The log scope <code>SECURITY_LAKE</code> indicates a configuration that is managed through Amazon
     *         Security Lake. You can use Security Lake to collect log and event data from various sources for
     *         normalization, analysis, and management. For information, see <a
     *         href="https://docs.aws.amazon.com/security-lake/latest/userguide/internal-sources.html">Collecting data
     *         from Amazon Web Services services</a> in the <i>Amazon Security Lake user guide</i>.
     *         </p>
     *         <p>
     *         Default: <code>CUSTOMER</code>
     * @see LogScope
     */
    public final LogScope logScope() {
        return LogScope.fromValue(logScope);
    }

    /**
     * <p>
     * The owner of the logging configuration, which must be set to <code>CUSTOMER</code> for the configurations that
     * you manage.
     * </p>
     * <p>
     * The log scope <code>SECURITY_LAKE</code> indicates a configuration that is managed through Amazon Security Lake.
     * You can use Security Lake to collect log and event data from various sources for normalization, analysis, and
     * management. For information, see <a
     * href="https://docs.aws.amazon.com/security-lake/latest/userguide/internal-sources.html">Collecting data from
     * Amazon Web Services services</a> in the <i>Amazon Security Lake user guide</i>.
     * </p>
     * <p>
     * Default: <code>CUSTOMER</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #logScope} will
     * return {@link LogScope#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #logScopeAsString}.
     * </p>
     * 
     * @return The owner of the logging configuration, which must be set to <code>CUSTOMER</code> for the configurations
     *         that you manage. </p>
     *         <p>
     *         The log scope <code>SECURITY_LAKE</code> indicates a configuration that is managed through Amazon
     *         Security Lake. You can use Security Lake to collect log and event data from various sources for
     *         normalization, analysis, and management. For information, see <a
     *         href="https://docs.aws.amazon.com/security-lake/latest/userguide/internal-sources.html">Collecting data
     *         from Amazon Web Services services</a> in the <i>Amazon Security Lake user guide</i>.
     *         </p>
     *         <p>
     *         Default: <code>CUSTOMER</code>
     * @see LogScope
     */
    public final String logScopeAsString() {
        return logScope;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resourceArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasLogDestinationConfigs() ? logDestinationConfigs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasRedactedFields() ? redactedFields() : null);
        hashCode = 31 * hashCode + Objects.hashCode(managedByFirewallManager());
        hashCode = 31 * hashCode + Objects.hashCode(loggingFilter());
        hashCode = 31 * hashCode + Objects.hashCode(logTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(logScopeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LoggingConfiguration)) {
            return false;
        }
        LoggingConfiguration other = (LoggingConfiguration) obj;
        return Objects.equals(resourceArn(), other.resourceArn())
                && hasLogDestinationConfigs() == other.hasLogDestinationConfigs()
                && Objects.equals(logDestinationConfigs(), other.logDestinationConfigs())
                && hasRedactedFields() == other.hasRedactedFields() && Objects.equals(redactedFields(), other.redactedFields())
                && Objects.equals(managedByFirewallManager(), other.managedByFirewallManager())
                && Objects.equals(loggingFilter(), other.loggingFilter())
                && Objects.equals(logTypeAsString(), other.logTypeAsString())
                && Objects.equals(logScopeAsString(), other.logScopeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LoggingConfiguration").add("ResourceArn", resourceArn())
                .add("LogDestinationConfigs", hasLogDestinationConfigs() ? logDestinationConfigs() : null)
                .add("RedactedFields", hasRedactedFields() ? redactedFields() : null)
                .add("ManagedByFirewallManager", managedByFirewallManager()).add("LoggingFilter", loggingFilter())
                .add("LogType", logTypeAsString()).add("LogScope", logScopeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ResourceArn":
            return Optional.ofNullable(clazz.cast(resourceArn()));
        case "LogDestinationConfigs":
            return Optional.ofNullable(clazz.cast(logDestinationConfigs()));
        case "RedactedFields":
            return Optional.ofNullable(clazz.cast(redactedFields()));
        case "ManagedByFirewallManager":
            return Optional.ofNullable(clazz.cast(managedByFirewallManager()));
        case "LoggingFilter":
            return Optional.ofNullable(clazz.cast(loggingFilter()));
        case "LogType":
            return Optional.ofNullable(clazz.cast(logTypeAsString()));
        case "LogScope":
            return Optional.ofNullable(clazz.cast(logScopeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ResourceArn", RESOURCE_ARN_FIELD);
        map.put("LogDestinationConfigs", LOG_DESTINATION_CONFIGS_FIELD);
        map.put("RedactedFields", REDACTED_FIELDS_FIELD);
        map.put("ManagedByFirewallManager", MANAGED_BY_FIREWALL_MANAGER_FIELD);
        map.put("LoggingFilter", LOGGING_FILTER_FIELD);
        map.put("LogType", LOG_TYPE_FIELD);
        map.put("LogScope", LOG_SCOPE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<LoggingConfiguration, T> g) {
        return obj -> g.apply((LoggingConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LoggingConfiguration> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the web ACL that you want to associate with
         * <code>LogDestinationConfigs</code>.
         * </p>
         * 
         * @param resourceArn
         *        The Amazon Resource Name (ARN) of the web ACL that you want to associate with
         *        <code>LogDestinationConfigs</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceArn(String resourceArn);

        /**
         * <p>
         * The logging destination configuration that you want to associate with the web ACL.
         * </p>
         * <note>
         * <p>
         * You can associate one logging destination to a web ACL.
         * </p>
         * </note>
         * 
         * @param logDestinationConfigs
         *        The logging destination configuration that you want to associate with the web ACL.</p> <note>
         *        <p>
         *        You can associate one logging destination to a web ACL.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logDestinationConfigs(Collection<String> logDestinationConfigs);

        /**
         * <p>
         * The logging destination configuration that you want to associate with the web ACL.
         * </p>
         * <note>
         * <p>
         * You can associate one logging destination to a web ACL.
         * </p>
         * </note>
         * 
         * @param logDestinationConfigs
         *        The logging destination configuration that you want to associate with the web ACL.</p> <note>
         *        <p>
         *        You can associate one logging destination to a web ACL.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logDestinationConfigs(String... logDestinationConfigs);

        /**
         * <p>
         * The parts of the request that you want to keep out of the logs.
         * </p>
         * <p>
         * For example, if you redact the <code>SingleHeader</code> field, the <code>HEADER</code> field in the logs
         * will be <code>REDACTED</code> for all rules that use the <code>SingleHeader</code> <code>FieldToMatch</code>
         * setting.
         * </p>
         * <p>
         * Redaction applies only to the component that's specified in the rule's <code>FieldToMatch</code> setting, so
         * the <code>SingleHeader</code> redaction doesn't apply to rules that use the <code>Headers</code>
         * <code>FieldToMatch</code>.
         * </p>
         * <note>
         * <p>
         * You can specify only the following fields for redaction: <code>UriPath</code>, <code>QueryString</code>,
         * <code>SingleHeader</code>, and <code>Method</code>.
         * </p>
         * </note> <note>
         * <p>
         * This setting has no impact on request sampling. With request sampling, the only way to exclude fields is by
         * disabling sampling in the web ACL visibility configuration.
         * </p>
         * </note>
         * 
         * @param redactedFields
         *        The parts of the request that you want to keep out of the logs.</p>
         *        <p>
         *        For example, if you redact the <code>SingleHeader</code> field, the <code>HEADER</code> field in the
         *        logs will be <code>REDACTED</code> for all rules that use the <code>SingleHeader</code>
         *        <code>FieldToMatch</code> setting.
         *        </p>
         *        <p>
         *        Redaction applies only to the component that's specified in the rule's <code>FieldToMatch</code>
         *        setting, so the <code>SingleHeader</code> redaction doesn't apply to rules that use the
         *        <code>Headers</code> <code>FieldToMatch</code>.
         *        </p>
         *        <note>
         *        <p>
         *        You can specify only the following fields for redaction: <code>UriPath</code>,
         *        <code>QueryString</code>, <code>SingleHeader</code>, and <code>Method</code>.
         *        </p>
         *        </note> <note>
         *        <p>
         *        This setting has no impact on request sampling. With request sampling, the only way to exclude fields
         *        is by disabling sampling in the web ACL visibility configuration.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder redactedFields(Collection<FieldToMatch> redactedFields);

        /**
         * <p>
         * The parts of the request that you want to keep out of the logs.
         * </p>
         * <p>
         * For example, if you redact the <code>SingleHeader</code> field, the <code>HEADER</code> field in the logs
         * will be <code>REDACTED</code> for all rules that use the <code>SingleHeader</code> <code>FieldToMatch</code>
         * setting.
         * </p>
         * <p>
         * Redaction applies only to the component that's specified in the rule's <code>FieldToMatch</code> setting, so
         * the <code>SingleHeader</code> redaction doesn't apply to rules that use the <code>Headers</code>
         * <code>FieldToMatch</code>.
         * </p>
         * <note>
         * <p>
         * You can specify only the following fields for redaction: <code>UriPath</code>, <code>QueryString</code>,
         * <code>SingleHeader</code>, and <code>Method</code>.
         * </p>
         * </note> <note>
         * <p>
         * This setting has no impact on request sampling. With request sampling, the only way to exclude fields is by
         * disabling sampling in the web ACL visibility configuration.
         * </p>
         * </note>
         * 
         * @param redactedFields
         *        The parts of the request that you want to keep out of the logs.</p>
         *        <p>
         *        For example, if you redact the <code>SingleHeader</code> field, the <code>HEADER</code> field in the
         *        logs will be <code>REDACTED</code> for all rules that use the <code>SingleHeader</code>
         *        <code>FieldToMatch</code> setting.
         *        </p>
         *        <p>
         *        Redaction applies only to the component that's specified in the rule's <code>FieldToMatch</code>
         *        setting, so the <code>SingleHeader</code> redaction doesn't apply to rules that use the
         *        <code>Headers</code> <code>FieldToMatch</code>.
         *        </p>
         *        <note>
         *        <p>
         *        You can specify only the following fields for redaction: <code>UriPath</code>,
         *        <code>QueryString</code>, <code>SingleHeader</code>, and <code>Method</code>.
         *        </p>
         *        </note> <note>
         *        <p>
         *        This setting has no impact on request sampling. With request sampling, the only way to exclude fields
         *        is by disabling sampling in the web ACL visibility configuration.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder redactedFields(FieldToMatch... redactedFields);

        /**
         * <p>
         * The parts of the request that you want to keep out of the logs.
         * </p>
         * <p>
         * For example, if you redact the <code>SingleHeader</code> field, the <code>HEADER</code> field in the logs
         * will be <code>REDACTED</code> for all rules that use the <code>SingleHeader</code> <code>FieldToMatch</code>
         * setting.
         * </p>
         * <p>
         * Redaction applies only to the component that's specified in the rule's <code>FieldToMatch</code> setting, so
         * the <code>SingleHeader</code> redaction doesn't apply to rules that use the <code>Headers</code>
         * <code>FieldToMatch</code>.
         * </p>
         * <note>
         * <p>
         * You can specify only the following fields for redaction: <code>UriPath</code>, <code>QueryString</code>,
         * <code>SingleHeader</code>, and <code>Method</code>.
         * </p>
         * </note> <note>
         * <p>
         * This setting has no impact on request sampling. With request sampling, the only way to exclude fields is by
         * disabling sampling in the web ACL visibility configuration.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.wafv2.model.FieldToMatch.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.wafv2.model.FieldToMatch#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.wafv2.model.FieldToMatch.Builder#build()} is called immediately and
         * its result is passed to {@link #redactedFields(List<FieldToMatch>)}.
         * 
         * @param redactedFields
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.wafv2.model.FieldToMatch.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #redactedFields(java.util.Collection<FieldToMatch>)
         */
        Builder redactedFields(Consumer<FieldToMatch.Builder>... redactedFields);

        /**
         * <p>
         * Indicates whether the logging configuration was created by Firewall Manager, as part of an WAF policy
         * configuration. If true, only Firewall Manager can modify or delete the configuration.
         * </p>
         * <p>
         * The logging configuration can be created by Firewall Manager for use with any web ACL that Firewall Manager
         * is using for an WAF policy. Web ACLs that Firewall Manager creates and uses have their
         * <code>ManagedByFirewallManager</code> property set to true. Web ACLs that were created by a customer account
         * and then retrofitted by Firewall Manager for use by a policy have their
         * <code>RetrofittedByFirewallManager</code> property set to true. For either case, any corresponding logging
         * configuration will indicate <code>ManagedByFirewallManager</code>.
         * </p>
         * 
         * @param managedByFirewallManager
         *        Indicates whether the logging configuration was created by Firewall Manager, as part of an WAF policy
         *        configuration. If true, only Firewall Manager can modify or delete the configuration. </p>
         *        <p>
         *        The logging configuration can be created by Firewall Manager for use with any web ACL that Firewall
         *        Manager is using for an WAF policy. Web ACLs that Firewall Manager creates and uses have their
         *        <code>ManagedByFirewallManager</code> property set to true. Web ACLs that were created by a customer
         *        account and then retrofitted by Firewall Manager for use by a policy have their
         *        <code>RetrofittedByFirewallManager</code> property set to true. For either case, any corresponding
         *        logging configuration will indicate <code>ManagedByFirewallManager</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder managedByFirewallManager(Boolean managedByFirewallManager);

        /**
         * <p>
         * Filtering that specifies which web requests are kept in the logs and which are dropped. You can filter on the
         * rule action and on the web request labels that were applied by matching rules during web ACL evaluation.
         * </p>
         * 
         * @param loggingFilter
         *        Filtering that specifies which web requests are kept in the logs and which are dropped. You can filter
         *        on the rule action and on the web request labels that were applied by matching rules during web ACL
         *        evaluation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loggingFilter(LoggingFilter loggingFilter);

        /**
         * <p>
         * Filtering that specifies which web requests are kept in the logs and which are dropped. You can filter on the
         * rule action and on the web request labels that were applied by matching rules during web ACL evaluation.
         * </p>
         * This is a convenience method that creates an instance of the {@link LoggingFilter.Builder} avoiding the need
         * to create one manually via {@link LoggingFilter#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LoggingFilter.Builder#build()} is called immediately and its
         * result is passed to {@link #loggingFilter(LoggingFilter)}.
         * 
         * @param loggingFilter
         *        a consumer that will call methods on {@link LoggingFilter.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #loggingFilter(LoggingFilter)
         */
        default Builder loggingFilter(Consumer<LoggingFilter.Builder> loggingFilter) {
            return loggingFilter(LoggingFilter.builder().applyMutation(loggingFilter).build());
        }

        /**
         * <p>
         * Used to distinguish between various logging options. Currently, there is one option.
         * </p>
         * <p>
         * Default: <code>WAF_LOGS</code>
         * </p>
         * 
         * @param logType
         *        Used to distinguish between various logging options. Currently, there is one option.</p>
         *        <p>
         *        Default: <code>WAF_LOGS</code>
         * @see LogType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogType
         */
        Builder logType(String logType);

        /**
         * <p>
         * Used to distinguish between various logging options. Currently, there is one option.
         * </p>
         * <p>
         * Default: <code>WAF_LOGS</code>
         * </p>
         * 
         * @param logType
         *        Used to distinguish between various logging options. Currently, there is one option.</p>
         *        <p>
         *        Default: <code>WAF_LOGS</code>
         * @see LogType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogType
         */
        Builder logType(LogType logType);

        /**
         * <p>
         * The owner of the logging configuration, which must be set to <code>CUSTOMER</code> for the configurations
         * that you manage.
         * </p>
         * <p>
         * The log scope <code>SECURITY_LAKE</code> indicates a configuration that is managed through Amazon Security
         * Lake. You can use Security Lake to collect log and event data from various sources for normalization,
         * analysis, and management. For information, see <a
         * href="https://docs.aws.amazon.com/security-lake/latest/userguide/internal-sources.html">Collecting data from
         * Amazon Web Services services</a> in the <i>Amazon Security Lake user guide</i>.
         * </p>
         * <p>
         * Default: <code>CUSTOMER</code>
         * </p>
         * 
         * @param logScope
         *        The owner of the logging configuration, which must be set to <code>CUSTOMER</code> for the
         *        configurations that you manage. </p>
         *        <p>
         *        The log scope <code>SECURITY_LAKE</code> indicates a configuration that is managed through Amazon
         *        Security Lake. You can use Security Lake to collect log and event data from various sources for
         *        normalization, analysis, and management. For information, see <a
         *        href="https://docs.aws.amazon.com/security-lake/latest/userguide/internal-sources.html">Collecting
         *        data from Amazon Web Services services</a> in the <i>Amazon Security Lake user guide</i>.
         *        </p>
         *        <p>
         *        Default: <code>CUSTOMER</code>
         * @see LogScope
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogScope
         */
        Builder logScope(String logScope);

        /**
         * <p>
         * The owner of the logging configuration, which must be set to <code>CUSTOMER</code> for the configurations
         * that you manage.
         * </p>
         * <p>
         * The log scope <code>SECURITY_LAKE</code> indicates a configuration that is managed through Amazon Security
         * Lake. You can use Security Lake to collect log and event data from various sources for normalization,
         * analysis, and management. For information, see <a
         * href="https://docs.aws.amazon.com/security-lake/latest/userguide/internal-sources.html">Collecting data from
         * Amazon Web Services services</a> in the <i>Amazon Security Lake user guide</i>.
         * </p>
         * <p>
         * Default: <code>CUSTOMER</code>
         * </p>
         * 
         * @param logScope
         *        The owner of the logging configuration, which must be set to <code>CUSTOMER</code> for the
         *        configurations that you manage. </p>
         *        <p>
         *        The log scope <code>SECURITY_LAKE</code> indicates a configuration that is managed through Amazon
         *        Security Lake. You can use Security Lake to collect log and event data from various sources for
         *        normalization, analysis, and management. For information, see <a
         *        href="https://docs.aws.amazon.com/security-lake/latest/userguide/internal-sources.html">Collecting
         *        data from Amazon Web Services services</a> in the <i>Amazon Security Lake user guide</i>.
         *        </p>
         *        <p>
         *        Default: <code>CUSTOMER</code>
         * @see LogScope
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogScope
         */
        Builder logScope(LogScope logScope);
    }

    static final class BuilderImpl implements Builder {
        private String resourceArn;

        private List<String> logDestinationConfigs = DefaultSdkAutoConstructList.getInstance();

        private List<FieldToMatch> redactedFields = DefaultSdkAutoConstructList.getInstance();

        private Boolean managedByFirewallManager;

        private LoggingFilter loggingFilter;

        private String logType;

        private String logScope;

        private BuilderImpl() {
        }

        private BuilderImpl(LoggingConfiguration model) {
            resourceArn(model.resourceArn);
            logDestinationConfigs(model.logDestinationConfigs);
            redactedFields(model.redactedFields);
            managedByFirewallManager(model.managedByFirewallManager);
            loggingFilter(model.loggingFilter);
            logType(model.logType);
            logScope(model.logScope);
        }

        public final String getResourceArn() {
            return resourceArn;
        }

        public final void setResourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
        }

        @Override
        public final Builder resourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
            return this;
        }

        public final Collection<String> getLogDestinationConfigs() {
            if (logDestinationConfigs instanceof SdkAutoConstructList) {
                return null;
            }
            return logDestinationConfigs;
        }

        public final void setLogDestinationConfigs(Collection<String> logDestinationConfigs) {
            this.logDestinationConfigs = LogDestinationConfigsCopier.copy(logDestinationConfigs);
        }

        @Override
        public final Builder logDestinationConfigs(Collection<String> logDestinationConfigs) {
            this.logDestinationConfigs = LogDestinationConfigsCopier.copy(logDestinationConfigs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder logDestinationConfigs(String... logDestinationConfigs) {
            logDestinationConfigs(Arrays.asList(logDestinationConfigs));
            return this;
        }

        public final List<FieldToMatch.Builder> getRedactedFields() {
            List<FieldToMatch.Builder> result = RedactedFieldsCopier.copyToBuilder(this.redactedFields);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRedactedFields(Collection<FieldToMatch.BuilderImpl> redactedFields) {
            this.redactedFields = RedactedFieldsCopier.copyFromBuilder(redactedFields);
        }

        @Override
        public final Builder redactedFields(Collection<FieldToMatch> redactedFields) {
            this.redactedFields = RedactedFieldsCopier.copy(redactedFields);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder redactedFields(FieldToMatch... redactedFields) {
            redactedFields(Arrays.asList(redactedFields));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder redactedFields(Consumer<FieldToMatch.Builder>... redactedFields) {
            redactedFields(Stream.of(redactedFields).map(c -> FieldToMatch.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Boolean getManagedByFirewallManager() {
            return managedByFirewallManager;
        }

        public final void setManagedByFirewallManager(Boolean managedByFirewallManager) {
            this.managedByFirewallManager = managedByFirewallManager;
        }

        @Override
        public final Builder managedByFirewallManager(Boolean managedByFirewallManager) {
            this.managedByFirewallManager = managedByFirewallManager;
            return this;
        }

        public final LoggingFilter.Builder getLoggingFilter() {
            return loggingFilter != null ? loggingFilter.toBuilder() : null;
        }

        public final void setLoggingFilter(LoggingFilter.BuilderImpl loggingFilter) {
            this.loggingFilter = loggingFilter != null ? loggingFilter.build() : null;
        }

        @Override
        public final Builder loggingFilter(LoggingFilter loggingFilter) {
            this.loggingFilter = loggingFilter;
            return this;
        }

        public final String getLogType() {
            return logType;
        }

        public final void setLogType(String logType) {
            this.logType = logType;
        }

        @Override
        public final Builder logType(String logType) {
            this.logType = logType;
            return this;
        }

        @Override
        public final Builder logType(LogType logType) {
            this.logType(logType == null ? null : logType.toString());
            return this;
        }

        public final String getLogScope() {
            return logScope;
        }

        public final void setLogScope(String logScope) {
            this.logScope = logScope;
        }

        @Override
        public final Builder logScope(String logScope) {
            this.logScope = logScope;
            return this;
        }

        @Override
        public final Builder logScope(LogScope logScope) {
            this.logScope(logScope == null ? null : logScope.toString());
            return this;
        }

        @Override
        public LoggingConfiguration build() {
            return new LoggingConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
