/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The criteria for inspecting responses to login requests and account creation requests, used by the ATP and ACFP rule
 * groups to track login and account creation success and failure rates.
 * </p>
 * <note>
 * <p>
 * Response inspection is available only in web ACLs that protect Amazon CloudFront distributions.
 * </p>
 * </note>
 * <p>
 * The rule groups evaluates the responses that your protected resources send back to client login and account creation
 * attempts, keeping count of successful and failed attempts from each IP address and client session. Using this
 * information, the rule group labels and mitigates requests from client sessions and IP addresses with too much
 * suspicious activity in a short amount of time.
 * </p>
 * <p>
 * This is part of the <code>AWSManagedRulesATPRuleSet</code> and <code>AWSManagedRulesACFPRuleSet</code> configurations
 * in <code>ManagedRuleGroupConfig</code>.
 * </p>
 * <p>
 * Enable response inspection by configuring exactly one component of the response to inspect, for example,
 * <code>Header</code> or <code>StatusCode</code>. You can't configure more than one component for inspection. If you
 * don't configure any of the response inspection options, response inspection is disabled.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResponseInspection implements SdkPojo, Serializable,
        ToCopyableBuilder<ResponseInspection.Builder, ResponseInspection> {
    private static final SdkField<ResponseInspectionStatusCode> STATUS_CODE_FIELD = SdkField
            .<ResponseInspectionStatusCode> builder(MarshallingType.SDK_POJO).memberName("StatusCode")
            .getter(getter(ResponseInspection::statusCode)).setter(setter(Builder::statusCode))
            .constructor(ResponseInspectionStatusCode::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusCode").build()).build();

    private static final SdkField<ResponseInspectionHeader> HEADER_FIELD = SdkField
            .<ResponseInspectionHeader> builder(MarshallingType.SDK_POJO).memberName("Header")
            .getter(getter(ResponseInspection::header)).setter(setter(Builder::header))
            .constructor(ResponseInspectionHeader::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Header").build()).build();

    private static final SdkField<ResponseInspectionBodyContains> BODY_CONTAINS_FIELD = SdkField
            .<ResponseInspectionBodyContains> builder(MarshallingType.SDK_POJO).memberName("BodyContains")
            .getter(getter(ResponseInspection::bodyContains)).setter(setter(Builder::bodyContains))
            .constructor(ResponseInspectionBodyContains::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BodyContains").build()).build();

    private static final SdkField<ResponseInspectionJson> JSON_FIELD = SdkField
            .<ResponseInspectionJson> builder(MarshallingType.SDK_POJO).memberName("Json")
            .getter(getter(ResponseInspection::json)).setter(setter(Builder::json)).constructor(ResponseInspectionJson::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Json").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATUS_CODE_FIELD,
            HEADER_FIELD, BODY_CONTAINS_FIELD, JSON_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final ResponseInspectionStatusCode statusCode;

    private final ResponseInspectionHeader header;

    private final ResponseInspectionBodyContains bodyContains;

    private final ResponseInspectionJson json;

    private ResponseInspection(BuilderImpl builder) {
        this.statusCode = builder.statusCode;
        this.header = builder.header;
        this.bodyContains = builder.bodyContains;
        this.json = builder.json;
    }

    /**
     * <p>
     * Configures inspection of the response status code for success and failure indicators.
     * </p>
     * 
     * @return Configures inspection of the response status code for success and failure indicators.
     */
    public final ResponseInspectionStatusCode statusCode() {
        return statusCode;
    }

    /**
     * <p>
     * Configures inspection of the response header for success and failure indicators.
     * </p>
     * 
     * @return Configures inspection of the response header for success and failure indicators.
     */
    public final ResponseInspectionHeader header() {
        return header;
    }

    /**
     * <p>
     * Configures inspection of the response body for success and failure indicators. WAF can inspect the first 65,536
     * bytes (64 KB) of the response body.
     * </p>
     * 
     * @return Configures inspection of the response body for success and failure indicators. WAF can inspect the first
     *         65,536 bytes (64 KB) of the response body.
     */
    public final ResponseInspectionBodyContains bodyContains() {
        return bodyContains;
    }

    /**
     * <p>
     * Configures inspection of the response JSON for success and failure indicators. WAF can inspect the first 65,536
     * bytes (64 KB) of the response JSON.
     * </p>
     * 
     * @return Configures inspection of the response JSON for success and failure indicators. WAF can inspect the first
     *         65,536 bytes (64 KB) of the response JSON.
     */
    public final ResponseInspectionJson json() {
        return json;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(statusCode());
        hashCode = 31 * hashCode + Objects.hashCode(header());
        hashCode = 31 * hashCode + Objects.hashCode(bodyContains());
        hashCode = 31 * hashCode + Objects.hashCode(json());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResponseInspection)) {
            return false;
        }
        ResponseInspection other = (ResponseInspection) obj;
        return Objects.equals(statusCode(), other.statusCode()) && Objects.equals(header(), other.header())
                && Objects.equals(bodyContains(), other.bodyContains()) && Objects.equals(json(), other.json());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResponseInspection").add("StatusCode", statusCode()).add("Header", header())
                .add("BodyContains", bodyContains()).add("Json", json()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StatusCode":
            return Optional.ofNullable(clazz.cast(statusCode()));
        case "Header":
            return Optional.ofNullable(clazz.cast(header()));
        case "BodyContains":
            return Optional.ofNullable(clazz.cast(bodyContains()));
        case "Json":
            return Optional.ofNullable(clazz.cast(json()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("StatusCode", STATUS_CODE_FIELD);
        map.put("Header", HEADER_FIELD);
        map.put("BodyContains", BODY_CONTAINS_FIELD);
        map.put("Json", JSON_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ResponseInspection, T> g) {
        return obj -> g.apply((ResponseInspection) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResponseInspection> {
        /**
         * <p>
         * Configures inspection of the response status code for success and failure indicators.
         * </p>
         * 
         * @param statusCode
         *        Configures inspection of the response status code for success and failure indicators.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusCode(ResponseInspectionStatusCode statusCode);

        /**
         * <p>
         * Configures inspection of the response status code for success and failure indicators.
         * </p>
         * This is a convenience method that creates an instance of the {@link ResponseInspectionStatusCode.Builder}
         * avoiding the need to create one manually via {@link ResponseInspectionStatusCode#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResponseInspectionStatusCode.Builder#build()} is called
         * immediately and its result is passed to {@link #statusCode(ResponseInspectionStatusCode)}.
         * 
         * @param statusCode
         *        a consumer that will call methods on {@link ResponseInspectionStatusCode.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #statusCode(ResponseInspectionStatusCode)
         */
        default Builder statusCode(Consumer<ResponseInspectionStatusCode.Builder> statusCode) {
            return statusCode(ResponseInspectionStatusCode.builder().applyMutation(statusCode).build());
        }

        /**
         * <p>
         * Configures inspection of the response header for success and failure indicators.
         * </p>
         * 
         * @param header
         *        Configures inspection of the response header for success and failure indicators.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder header(ResponseInspectionHeader header);

        /**
         * <p>
         * Configures inspection of the response header for success and failure indicators.
         * </p>
         * This is a convenience method that creates an instance of the {@link ResponseInspectionHeader.Builder}
         * avoiding the need to create one manually via {@link ResponseInspectionHeader#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResponseInspectionHeader.Builder#build()} is called immediately
         * and its result is passed to {@link #header(ResponseInspectionHeader)}.
         * 
         * @param header
         *        a consumer that will call methods on {@link ResponseInspectionHeader.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #header(ResponseInspectionHeader)
         */
        default Builder header(Consumer<ResponseInspectionHeader.Builder> header) {
            return header(ResponseInspectionHeader.builder().applyMutation(header).build());
        }

        /**
         * <p>
         * Configures inspection of the response body for success and failure indicators. WAF can inspect the first
         * 65,536 bytes (64 KB) of the response body.
         * </p>
         * 
         * @param bodyContains
         *        Configures inspection of the response body for success and failure indicators. WAF can inspect the
         *        first 65,536 bytes (64 KB) of the response body.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bodyContains(ResponseInspectionBodyContains bodyContains);

        /**
         * <p>
         * Configures inspection of the response body for success and failure indicators. WAF can inspect the first
         * 65,536 bytes (64 KB) of the response body.
         * </p>
         * This is a convenience method that creates an instance of the {@link ResponseInspectionBodyContains.Builder}
         * avoiding the need to create one manually via {@link ResponseInspectionBodyContains#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResponseInspectionBodyContains.Builder#build()} is called
         * immediately and its result is passed to {@link #bodyContains(ResponseInspectionBodyContains)}.
         * 
         * @param bodyContains
         *        a consumer that will call methods on {@link ResponseInspectionBodyContains.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #bodyContains(ResponseInspectionBodyContains)
         */
        default Builder bodyContains(Consumer<ResponseInspectionBodyContains.Builder> bodyContains) {
            return bodyContains(ResponseInspectionBodyContains.builder().applyMutation(bodyContains).build());
        }

        /**
         * <p>
         * Configures inspection of the response JSON for success and failure indicators. WAF can inspect the first
         * 65,536 bytes (64 KB) of the response JSON.
         * </p>
         * 
         * @param json
         *        Configures inspection of the response JSON for success and failure indicators. WAF can inspect the
         *        first 65,536 bytes (64 KB) of the response JSON.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder json(ResponseInspectionJson json);

        /**
         * <p>
         * Configures inspection of the response JSON for success and failure indicators. WAF can inspect the first
         * 65,536 bytes (64 KB) of the response JSON.
         * </p>
         * This is a convenience method that creates an instance of the {@link ResponseInspectionJson.Builder} avoiding
         * the need to create one manually via {@link ResponseInspectionJson#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResponseInspectionJson.Builder#build()} is called immediately and
         * its result is passed to {@link #json(ResponseInspectionJson)}.
         * 
         * @param json
         *        a consumer that will call methods on {@link ResponseInspectionJson.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #json(ResponseInspectionJson)
         */
        default Builder json(Consumer<ResponseInspectionJson.Builder> json) {
            return json(ResponseInspectionJson.builder().applyMutation(json).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private ResponseInspectionStatusCode statusCode;

        private ResponseInspectionHeader header;

        private ResponseInspectionBodyContains bodyContains;

        private ResponseInspectionJson json;

        private BuilderImpl() {
        }

        private BuilderImpl(ResponseInspection model) {
            statusCode(model.statusCode);
            header(model.header);
            bodyContains(model.bodyContains);
            json(model.json);
        }

        public final ResponseInspectionStatusCode.Builder getStatusCode() {
            return statusCode != null ? statusCode.toBuilder() : null;
        }

        public final void setStatusCode(ResponseInspectionStatusCode.BuilderImpl statusCode) {
            this.statusCode = statusCode != null ? statusCode.build() : null;
        }

        @Override
        public final Builder statusCode(ResponseInspectionStatusCode statusCode) {
            this.statusCode = statusCode;
            return this;
        }

        public final ResponseInspectionHeader.Builder getHeader() {
            return header != null ? header.toBuilder() : null;
        }

        public final void setHeader(ResponseInspectionHeader.BuilderImpl header) {
            this.header = header != null ? header.build() : null;
        }

        @Override
        public final Builder header(ResponseInspectionHeader header) {
            this.header = header;
            return this;
        }

        public final ResponseInspectionBodyContains.Builder getBodyContains() {
            return bodyContains != null ? bodyContains.toBuilder() : null;
        }

        public final void setBodyContains(ResponseInspectionBodyContains.BuilderImpl bodyContains) {
            this.bodyContains = bodyContains != null ? bodyContains.build() : null;
        }

        @Override
        public final Builder bodyContains(ResponseInspectionBodyContains bodyContains) {
            this.bodyContains = bodyContains;
            return this;
        }

        public final ResponseInspectionJson.Builder getJson() {
            return json != null ? json.toBuilder() : null;
        }

        public final void setJson(ResponseInspectionJson.BuilderImpl json) {
            this.json = json != null ? json.build() : null;
        }

        @Override
        public final Builder json(ResponseInspectionJson json) {
            this.json = json;
            return this;
        }

        @Override
        public ResponseInspection build() {
            return new ResponseInspection(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
