/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A web ACL defines a collection of rules to use to inspect and control web requests. Each rule has a statement that
 * defines what to look for in web requests and an action that WAF applies to requests that match the statement. In the
 * web ACL, you assign a default action to take (allow, block) for any request that does not match any of the rules. The
 * rules in a web ACL can be a combination of the types <a>Rule</a>, <a>RuleGroup</a>, and managed rule group. You can
 * associate a web ACL with one or more Amazon Web Services resources to protect. The resources can be an Amazon
 * CloudFront distribution, an Amazon API Gateway REST API, an Application Load Balancer, an AppSync GraphQL API, an
 * Amazon Cognito user pool, an App Runner service, or an Amazon Web Services Verified Access instance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class WebACL implements SdkPojo, Serializable, ToCopyableBuilder<WebACL.Builder, WebACL> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(WebACL::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(WebACL::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("ARN")
            .getter(getter(WebACL::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ARN").build()).build();

    private static final SdkField<DefaultAction> DEFAULT_ACTION_FIELD = SdkField
            .<DefaultAction> builder(MarshallingType.SDK_POJO).memberName("DefaultAction").getter(getter(WebACL::defaultAction))
            .setter(setter(Builder::defaultAction)).constructor(DefaultAction::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultAction").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(WebACL::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<List<Rule>> RULES_FIELD = SdkField
            .<List<Rule>> builder(MarshallingType.LIST)
            .memberName("Rules")
            .getter(getter(WebACL::rules))
            .setter(setter(Builder::rules))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Rules").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Rule> builder(MarshallingType.SDK_POJO)
                                            .constructor(Rule::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<VisibilityConfig> VISIBILITY_CONFIG_FIELD = SdkField
            .<VisibilityConfig> builder(MarshallingType.SDK_POJO).memberName("VisibilityConfig")
            .getter(getter(WebACL::visibilityConfig)).setter(setter(Builder::visibilityConfig))
            .constructor(VisibilityConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VisibilityConfig").build()).build();

    private static final SdkField<Long> CAPACITY_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("Capacity")
            .getter(getter(WebACL::capacity)).setter(setter(Builder::capacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Capacity").build()).build();

    private static final SdkField<List<FirewallManagerRuleGroup>> PRE_PROCESS_FIREWALL_MANAGER_RULE_GROUPS_FIELD = SdkField
            .<List<FirewallManagerRuleGroup>> builder(MarshallingType.LIST)
            .memberName("PreProcessFirewallManagerRuleGroups")
            .getter(getter(WebACL::preProcessFirewallManagerRuleGroups))
            .setter(setter(Builder::preProcessFirewallManagerRuleGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("PreProcessFirewallManagerRuleGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<FirewallManagerRuleGroup> builder(MarshallingType.SDK_POJO)
                                            .constructor(FirewallManagerRuleGroup::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<FirewallManagerRuleGroup>> POST_PROCESS_FIREWALL_MANAGER_RULE_GROUPS_FIELD = SdkField
            .<List<FirewallManagerRuleGroup>> builder(MarshallingType.LIST)
            .memberName("PostProcessFirewallManagerRuleGroups")
            .getter(getter(WebACL::postProcessFirewallManagerRuleGroups))
            .setter(setter(Builder::postProcessFirewallManagerRuleGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("PostProcessFirewallManagerRuleGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<FirewallManagerRuleGroup> builder(MarshallingType.SDK_POJO)
                                            .constructor(FirewallManagerRuleGroup::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> MANAGED_BY_FIREWALL_MANAGER_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN).memberName("ManagedByFirewallManager")
            .getter(getter(WebACL::managedByFirewallManager)).setter(setter(Builder::managedByFirewallManager))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ManagedByFirewallManager").build())
            .build();

    private static final SdkField<String> LABEL_NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LabelNamespace").getter(getter(WebACL::labelNamespace)).setter(setter(Builder::labelNamespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LabelNamespace").build()).build();

    private static final SdkField<Map<String, CustomResponseBody>> CUSTOM_RESPONSE_BODIES_FIELD = SdkField
            .<Map<String, CustomResponseBody>> builder(MarshallingType.MAP)
            .memberName("CustomResponseBodies")
            .getter(getter(WebACL::customResponseBodies))
            .setter(setter(Builder::customResponseBodies))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomResponseBodies").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<CustomResponseBody> builder(MarshallingType.SDK_POJO)
                                            .constructor(CustomResponseBody::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<CaptchaConfig> CAPTCHA_CONFIG_FIELD = SdkField
            .<CaptchaConfig> builder(MarshallingType.SDK_POJO).memberName("CaptchaConfig").getter(getter(WebACL::captchaConfig))
            .setter(setter(Builder::captchaConfig)).constructor(CaptchaConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CaptchaConfig").build()).build();

    private static final SdkField<ChallengeConfig> CHALLENGE_CONFIG_FIELD = SdkField
            .<ChallengeConfig> builder(MarshallingType.SDK_POJO).memberName("ChallengeConfig")
            .getter(getter(WebACL::challengeConfig)).setter(setter(Builder::challengeConfig))
            .constructor(ChallengeConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ChallengeConfig").build()).build();

    private static final SdkField<List<String>> TOKEN_DOMAINS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("TokenDomains")
            .getter(getter(WebACL::tokenDomains))
            .setter(setter(Builder::tokenDomains))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TokenDomains").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<AssociationConfig> ASSOCIATION_CONFIG_FIELD = SdkField
            .<AssociationConfig> builder(MarshallingType.SDK_POJO).memberName("AssociationConfig")
            .getter(getter(WebACL::associationConfig)).setter(setter(Builder::associationConfig))
            .constructor(AssociationConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociationConfig").build()).build();

    private static final SdkField<Boolean> RETROFITTED_BY_FIREWALL_MANAGER_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("RetrofittedByFirewallManager")
            .getter(getter(WebACL::retrofittedByFirewallManager))
            .setter(setter(Builder::retrofittedByFirewallManager))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RetrofittedByFirewallManager")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ID_FIELD,
            ARN_FIELD, DEFAULT_ACTION_FIELD, DESCRIPTION_FIELD, RULES_FIELD, VISIBILITY_CONFIG_FIELD, CAPACITY_FIELD,
            PRE_PROCESS_FIREWALL_MANAGER_RULE_GROUPS_FIELD, POST_PROCESS_FIREWALL_MANAGER_RULE_GROUPS_FIELD,
            MANAGED_BY_FIREWALL_MANAGER_FIELD, LABEL_NAMESPACE_FIELD, CUSTOM_RESPONSE_BODIES_FIELD, CAPTCHA_CONFIG_FIELD,
            CHALLENGE_CONFIG_FIELD, TOKEN_DOMAINS_FIELD, ASSOCIATION_CONFIG_FIELD, RETROFITTED_BY_FIREWALL_MANAGER_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String id;

    private final String arn;

    private final DefaultAction defaultAction;

    private final String description;

    private final List<Rule> rules;

    private final VisibilityConfig visibilityConfig;

    private final Long capacity;

    private final List<FirewallManagerRuleGroup> preProcessFirewallManagerRuleGroups;

    private final List<FirewallManagerRuleGroup> postProcessFirewallManagerRuleGroups;

    private final Boolean managedByFirewallManager;

    private final String labelNamespace;

    private final Map<String, CustomResponseBody> customResponseBodies;

    private final CaptchaConfig captchaConfig;

    private final ChallengeConfig challengeConfig;

    private final List<String> tokenDomains;

    private final AssociationConfig associationConfig;

    private final Boolean retrofittedByFirewallManager;

    private WebACL(BuilderImpl builder) {
        this.name = builder.name;
        this.id = builder.id;
        this.arn = builder.arn;
        this.defaultAction = builder.defaultAction;
        this.description = builder.description;
        this.rules = builder.rules;
        this.visibilityConfig = builder.visibilityConfig;
        this.capacity = builder.capacity;
        this.preProcessFirewallManagerRuleGroups = builder.preProcessFirewallManagerRuleGroups;
        this.postProcessFirewallManagerRuleGroups = builder.postProcessFirewallManagerRuleGroups;
        this.managedByFirewallManager = builder.managedByFirewallManager;
        this.labelNamespace = builder.labelNamespace;
        this.customResponseBodies = builder.customResponseBodies;
        this.captchaConfig = builder.captchaConfig;
        this.challengeConfig = builder.challengeConfig;
        this.tokenDomains = builder.tokenDomains;
        this.associationConfig = builder.associationConfig;
        this.retrofittedByFirewallManager = builder.retrofittedByFirewallManager;
    }

    /**
     * <p>
     * The name of the web ACL. You cannot change the name of a web ACL after you create it.
     * </p>
     * 
     * @return The name of the web ACL. You cannot change the name of a web ACL after you create it.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * A unique identifier for the <code>WebACL</code>. This ID is returned in the responses to create and list
     * commands. You use this ID to do things like get, update, and delete a <code>WebACL</code>.
     * </p>
     * 
     * @return A unique identifier for the <code>WebACL</code>. This ID is returned in the responses to create and list
     *         commands. You use this ID to do things like get, update, and delete a <code>WebACL</code>.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the web ACL that you want to associate with the resource.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the web ACL that you want to associate with the resource.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The action to perform if none of the <code>Rules</code> contained in the <code>WebACL</code> match.
     * </p>
     * 
     * @return The action to perform if none of the <code>Rules</code> contained in the <code>WebACL</code> match.
     */
    public final DefaultAction defaultAction() {
        return defaultAction;
    }

    /**
     * <p>
     * A description of the web ACL that helps with identification.
     * </p>
     * 
     * @return A description of the web ACL that helps with identification.
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the Rules property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasRules() {
        return rules != null && !(rules instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The <a>Rule</a> statements used to identify the web requests that you want to manage. Each rule includes one
     * top-level statement that WAF uses to identify matching web requests, and parameters that govern how WAF handles
     * them.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRules} method.
     * </p>
     * 
     * @return The <a>Rule</a> statements used to identify the web requests that you want to manage. Each rule includes
     *         one top-level statement that WAF uses to identify matching web requests, and parameters that govern how
     *         WAF handles them.
     */
    public final List<Rule> rules() {
        return rules;
    }

    /**
     * <p>
     * Defines and enables Amazon CloudWatch metrics and web request sample collection.
     * </p>
     * 
     * @return Defines and enables Amazon CloudWatch metrics and web request sample collection.
     */
    public final VisibilityConfig visibilityConfig() {
        return visibilityConfig;
    }

    /**
     * <p>
     * The web ACL capacity units (WCUs) currently being used by this web ACL.
     * </p>
     * <p>
     * WAF uses WCUs to calculate and control the operating resources that are used to run your rules, rule groups, and
     * web ACLs. WAF calculates capacity differently for each rule type, to reflect the relative cost of each rule.
     * Simple rules that cost little to run use fewer WCUs than more complex rules that use more processing power. Rule
     * group capacity is fixed at creation, which helps users plan their web ACL WCU usage when they use a rule group.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/aws-waf-capacity-units.html">WAF web ACL capacity
     * units (WCU)</a> in the <i>WAF Developer Guide</i>.
     * </p>
     * 
     * @return The web ACL capacity units (WCUs) currently being used by this web ACL. </p>
     *         <p>
     *         WAF uses WCUs to calculate and control the operating resources that are used to run your rules, rule
     *         groups, and web ACLs. WAF calculates capacity differently for each rule type, to reflect the relative
     *         cost of each rule. Simple rules that cost little to run use fewer WCUs than more complex rules that use
     *         more processing power. Rule group capacity is fixed at creation, which helps users plan their web ACL WCU
     *         usage when they use a rule group. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/aws-waf-capacity-units.html">WAF web ACL
     *         capacity units (WCU)</a> in the <i>WAF Developer Guide</i>.
     */
    public final Long capacity() {
        return capacity;
    }

    /**
     * For responses, this returns true if the service returned a value for the PreProcessFirewallManagerRuleGroups
     * property. This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()}
     * method on the property). This is useful because the SDK will never return a null collection or map, but you may
     * need to differentiate between the service returning nothing (or null) and the service returning an empty
     * collection or map. For requests, this returns true if a value for the property was specified in the request
     * builder, and false if a value was not specified.
     */
    public final boolean hasPreProcessFirewallManagerRuleGroups() {
        return preProcessFirewallManagerRuleGroups != null
                && !(preProcessFirewallManagerRuleGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The first set of rules for WAF to process in the web ACL. This is defined in an Firewall Manager WAF policy and
     * contains only rule group references. You can't alter these. Any rules and rule groups that you define for the web
     * ACL are prioritized after these.
     * </p>
     * <p>
     * In the Firewall Manager WAF policy, the Firewall Manager administrator can define a set of rule groups to run
     * first in the web ACL and a set of rule groups to run last. Within each set, the administrator prioritizes the
     * rule groups, to determine their relative processing order.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPreProcessFirewallManagerRuleGroups}
     * method.
     * </p>
     * 
     * @return The first set of rules for WAF to process in the web ACL. This is defined in an Firewall Manager WAF
     *         policy and contains only rule group references. You can't alter these. Any rules and rule groups that you
     *         define for the web ACL are prioritized after these. </p>
     *         <p>
     *         In the Firewall Manager WAF policy, the Firewall Manager administrator can define a set of rule groups to
     *         run first in the web ACL and a set of rule groups to run last. Within each set, the administrator
     *         prioritizes the rule groups, to determine their relative processing order.
     */
    public final List<FirewallManagerRuleGroup> preProcessFirewallManagerRuleGroups() {
        return preProcessFirewallManagerRuleGroups;
    }

    /**
     * For responses, this returns true if the service returned a value for the PostProcessFirewallManagerRuleGroups
     * property. This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()}
     * method on the property). This is useful because the SDK will never return a null collection or map, but you may
     * need to differentiate between the service returning nothing (or null) and the service returning an empty
     * collection or map. For requests, this returns true if a value for the property was specified in the request
     * builder, and false if a value was not specified.
     */
    public final boolean hasPostProcessFirewallManagerRuleGroups() {
        return postProcessFirewallManagerRuleGroups != null
                && !(postProcessFirewallManagerRuleGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The last set of rules for WAF to process in the web ACL. This is defined in an Firewall Manager WAF policy and
     * contains only rule group references. You can't alter these. Any rules and rule groups that you define for the web
     * ACL are prioritized before these.
     * </p>
     * <p>
     * In the Firewall Manager WAF policy, the Firewall Manager administrator can define a set of rule groups to run
     * first in the web ACL and a set of rule groups to run last. Within each set, the administrator prioritizes the
     * rule groups, to determine their relative processing order.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPostProcessFirewallManagerRuleGroups}
     * method.
     * </p>
     * 
     * @return The last set of rules for WAF to process in the web ACL. This is defined in an Firewall Manager WAF
     *         policy and contains only rule group references. You can't alter these. Any rules and rule groups that you
     *         define for the web ACL are prioritized before these. </p>
     *         <p>
     *         In the Firewall Manager WAF policy, the Firewall Manager administrator can define a set of rule groups to
     *         run first in the web ACL and a set of rule groups to run last. Within each set, the administrator
     *         prioritizes the rule groups, to determine their relative processing order.
     */
    public final List<FirewallManagerRuleGroup> postProcessFirewallManagerRuleGroups() {
        return postProcessFirewallManagerRuleGroups;
    }

    /**
     * <p>
     * Indicates whether this web ACL was created by Firewall Manager and is being managed by Firewall Manager. If true,
     * then only Firewall Manager can delete the web ACL or any Firewall Manager rule groups in the web ACL. See also
     * the properties <code>RetrofittedByFirewallManager</code>, <code>PreProcessFirewallManagerRuleGroups</code>, and
     * <code>PostProcessFirewallManagerRuleGroups</code>.
     * </p>
     * 
     * @return Indicates whether this web ACL was created by Firewall Manager and is being managed by Firewall Manager.
     *         If true, then only Firewall Manager can delete the web ACL or any Firewall Manager rule groups in the web
     *         ACL. See also the properties <code>RetrofittedByFirewallManager</code>,
     *         <code>PreProcessFirewallManagerRuleGroups</code>, and <code>PostProcessFirewallManagerRuleGroups</code>.
     */
    public final Boolean managedByFirewallManager() {
        return managedByFirewallManager;
    }

    /**
     * <p>
     * The label namespace prefix for this web ACL. All labels added by rules in this web ACL have this prefix.
     * </p>
     * <ul>
     * <li>
     * <p>
     * The syntax for the label namespace prefix for a web ACL is the following:
     * </p>
     * <p>
     * <code>awswaf:&lt;account ID&gt;:webacl:&lt;web ACL name&gt;:</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * When a rule with a label matches a web request, WAF adds the fully qualified label to the request. A fully
     * qualified label is made up of the label namespace from the rule group or web ACL where the rule is defined and
     * the label from the rule, separated by a colon:
     * </p>
     * <p>
     * <code>&lt;label namespace&gt;:&lt;label from rule&gt;</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The label namespace prefix for this web ACL. All labels added by rules in this web ACL have this prefix.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The syntax for the label namespace prefix for a web ACL is the following:
     *         </p>
     *         <p>
     *         <code>awswaf:&lt;account ID&gt;:webacl:&lt;web ACL name&gt;:</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         When a rule with a label matches a web request, WAF adds the fully qualified label to the request. A
     *         fully qualified label is made up of the label namespace from the rule group or web ACL where the rule is
     *         defined and the label from the rule, separated by a colon:
     *         </p>
     *         <p>
     *         <code>&lt;label namespace&gt;:&lt;label from rule&gt;</code>
     *         </p>
     *         </li>
     */
    public final String labelNamespace() {
        return labelNamespace;
    }

    /**
     * For responses, this returns true if the service returned a value for the CustomResponseBodies property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCustomResponseBodies() {
        return customResponseBodies != null && !(customResponseBodies instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of custom response keys and content bodies. When you create a rule with a block action, you can send a
     * custom response to the web request. You define these for the web ACL, and then use them in the rules and default
     * actions that you define in the web ACL.
     * </p>
     * <p>
     * For information about customizing web requests and responses, see <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-custom-request-response.html">Customizing web
     * requests and responses in WAF</a> in the <i>WAF Developer Guide</i>.
     * </p>
     * <p>
     * For information about the limits on count and size for custom request and response settings, see <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">WAF quotas</a> in the <i>WAF Developer
     * Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCustomResponseBodies} method.
     * </p>
     * 
     * @return A map of custom response keys and content bodies. When you create a rule with a block action, you can
     *         send a custom response to the web request. You define these for the web ACL, and then use them in the
     *         rules and default actions that you define in the web ACL. </p>
     *         <p>
     *         For information about customizing web requests and responses, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-custom-request-response.html">Customizing
     *         web requests and responses in WAF</a> in the <i>WAF Developer Guide</i>.
     *         </p>
     *         <p>
     *         For information about the limits on count and size for custom request and response settings, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">WAF quotas</a> in the <i>WAF
     *         Developer Guide</i>.
     */
    public final Map<String, CustomResponseBody> customResponseBodies() {
        return customResponseBodies;
    }

    /**
     * <p>
     * Specifies how WAF should handle <code>CAPTCHA</code> evaluations for rules that don't have their own
     * <code>CaptchaConfig</code> settings. If you don't specify this, WAF uses its default settings for
     * <code>CaptchaConfig</code>.
     * </p>
     * 
     * @return Specifies how WAF should handle <code>CAPTCHA</code> evaluations for rules that don't have their own
     *         <code>CaptchaConfig</code> settings. If you don't specify this, WAF uses its default settings for
     *         <code>CaptchaConfig</code>.
     */
    public final CaptchaConfig captchaConfig() {
        return captchaConfig;
    }

    /**
     * <p>
     * Specifies how WAF should handle challenge evaluations for rules that don't have their own
     * <code>ChallengeConfig</code> settings. If you don't specify this, WAF uses its default settings for
     * <code>ChallengeConfig</code>.
     * </p>
     * 
     * @return Specifies how WAF should handle challenge evaluations for rules that don't have their own
     *         <code>ChallengeConfig</code> settings. If you don't specify this, WAF uses its default settings for
     *         <code>ChallengeConfig</code>.
     */
    public final ChallengeConfig challengeConfig() {
        return challengeConfig;
    }

    /**
     * For responses, this returns true if the service returned a value for the TokenDomains property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTokenDomains() {
        return tokenDomains != null && !(tokenDomains instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the domains that WAF should accept in a web request token. This enables the use of tokens across
     * multiple protected websites. When WAF provides a token, it uses the domain of the Amazon Web Services resource
     * that the web ACL is protecting. If you don't specify a list of token domains, WAF accepts tokens only for the
     * domain of the protected resource. With a token domain list, WAF accepts the resource's host domain plus all
     * domains in the token domain list, including their prefixed subdomains.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTokenDomains} method.
     * </p>
     * 
     * @return Specifies the domains that WAF should accept in a web request token. This enables the use of tokens
     *         across multiple protected websites. When WAF provides a token, it uses the domain of the Amazon Web
     *         Services resource that the web ACL is protecting. If you don't specify a list of token domains, WAF
     *         accepts tokens only for the domain of the protected resource. With a token domain list, WAF accepts the
     *         resource's host domain plus all domains in the token domain list, including their prefixed subdomains.
     */
    public final List<String> tokenDomains() {
        return tokenDomains;
    }

    /**
     * <p>
     * Specifies custom configurations for the associations between the web ACL and protected resources.
     * </p>
     * <p>
     * Use this to customize the maximum size of the request body that your protected resources forward to WAF for
     * inspection. You can customize this setting for CloudFront, API Gateway, Amazon Cognito, App Runner, or Verified
     * Access resources. The default setting is 16 KB (16,384 bytes).
     * </p>
     * <note>
     * <p>
     * You are charged additional fees when your protected resources forward body sizes that are larger than the
     * default. For more information, see <a href="http://aws.amazon.com/waf/pricing/">WAF Pricing</a>.
     * </p>
     * </note>
     * <p>
     * For Application Load Balancer and AppSync, the limit is fixed at 8 KB (8,192 bytes).
     * </p>
     * 
     * @return Specifies custom configurations for the associations between the web ACL and protected resources. </p>
     *         <p>
     *         Use this to customize the maximum size of the request body that your protected resources forward to WAF
     *         for inspection. You can customize this setting for CloudFront, API Gateway, Amazon Cognito, App Runner,
     *         or Verified Access resources. The default setting is 16 KB (16,384 bytes).
     *         </p>
     *         <note>
     *         <p>
     *         You are charged additional fees when your protected resources forward body sizes that are larger than the
     *         default. For more information, see <a href="http://aws.amazon.com/waf/pricing/">WAF Pricing</a>.
     *         </p>
     *         </note>
     *         <p>
     *         For Application Load Balancer and AppSync, the limit is fixed at 8 KB (8,192 bytes).
     */
    public final AssociationConfig associationConfig() {
        return associationConfig;
    }

    /**
     * <p>
     * Indicates whether this web ACL was created by a customer account and then retrofitted by Firewall Manager. If
     * true, then the web ACL is currently being managed by a Firewall Manager WAF policy, and only Firewall Manager can
     * manage any Firewall Manager rule groups in the web ACL. See also the properties
     * <code>ManagedByFirewallManager</code>, <code>PreProcessFirewallManagerRuleGroups</code>, and
     * <code>PostProcessFirewallManagerRuleGroups</code>.
     * </p>
     * 
     * @return Indicates whether this web ACL was created by a customer account and then retrofitted by Firewall
     *         Manager. If true, then the web ACL is currently being managed by a Firewall Manager WAF policy, and only
     *         Firewall Manager can manage any Firewall Manager rule groups in the web ACL. See also the properties
     *         <code>ManagedByFirewallManager</code>, <code>PreProcessFirewallManagerRuleGroups</code>, and
     *         <code>PostProcessFirewallManagerRuleGroups</code>.
     */
    public final Boolean retrofittedByFirewallManager() {
        return retrofittedByFirewallManager;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(defaultAction());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasRules() ? rules() : null);
        hashCode = 31 * hashCode + Objects.hashCode(visibilityConfig());
        hashCode = 31 * hashCode + Objects.hashCode(capacity());
        hashCode = 31 * hashCode
                + Objects.hashCode(hasPreProcessFirewallManagerRuleGroups() ? preProcessFirewallManagerRuleGroups() : null);
        hashCode = 31 * hashCode
                + Objects.hashCode(hasPostProcessFirewallManagerRuleGroups() ? postProcessFirewallManagerRuleGroups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(managedByFirewallManager());
        hashCode = 31 * hashCode + Objects.hashCode(labelNamespace());
        hashCode = 31 * hashCode + Objects.hashCode(hasCustomResponseBodies() ? customResponseBodies() : null);
        hashCode = 31 * hashCode + Objects.hashCode(captchaConfig());
        hashCode = 31 * hashCode + Objects.hashCode(challengeConfig());
        hashCode = 31 * hashCode + Objects.hashCode(hasTokenDomains() ? tokenDomains() : null);
        hashCode = 31 * hashCode + Objects.hashCode(associationConfig());
        hashCode = 31 * hashCode + Objects.hashCode(retrofittedByFirewallManager());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof WebACL)) {
            return false;
        }
        WebACL other = (WebACL) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(id(), other.id()) && Objects.equals(arn(), other.arn())
                && Objects.equals(defaultAction(), other.defaultAction()) && Objects.equals(description(), other.description())
                && hasRules() == other.hasRules() && Objects.equals(rules(), other.rules())
                && Objects.equals(visibilityConfig(), other.visibilityConfig()) && Objects.equals(capacity(), other.capacity())
                && hasPreProcessFirewallManagerRuleGroups() == other.hasPreProcessFirewallManagerRuleGroups()
                && Objects.equals(preProcessFirewallManagerRuleGroups(), other.preProcessFirewallManagerRuleGroups())
                && hasPostProcessFirewallManagerRuleGroups() == other.hasPostProcessFirewallManagerRuleGroups()
                && Objects.equals(postProcessFirewallManagerRuleGroups(), other.postProcessFirewallManagerRuleGroups())
                && Objects.equals(managedByFirewallManager(), other.managedByFirewallManager())
                && Objects.equals(labelNamespace(), other.labelNamespace())
                && hasCustomResponseBodies() == other.hasCustomResponseBodies()
                && Objects.equals(customResponseBodies(), other.customResponseBodies())
                && Objects.equals(captchaConfig(), other.captchaConfig())
                && Objects.equals(challengeConfig(), other.challengeConfig()) && hasTokenDomains() == other.hasTokenDomains()
                && Objects.equals(tokenDomains(), other.tokenDomains())
                && Objects.equals(associationConfig(), other.associationConfig())
                && Objects.equals(retrofittedByFirewallManager(), other.retrofittedByFirewallManager());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("WebACL")
                .add("Name", name())
                .add("Id", id())
                .add("ARN", arn())
                .add("DefaultAction", defaultAction())
                .add("Description", description())
                .add("Rules", hasRules() ? rules() : null)
                .add("VisibilityConfig", visibilityConfig())
                .add("Capacity", capacity())
                .add("PreProcessFirewallManagerRuleGroups",
                        hasPreProcessFirewallManagerRuleGroups() ? preProcessFirewallManagerRuleGroups() : null)
                .add("PostProcessFirewallManagerRuleGroups",
                        hasPostProcessFirewallManagerRuleGroups() ? postProcessFirewallManagerRuleGroups() : null)
                .add("ManagedByFirewallManager", managedByFirewallManager()).add("LabelNamespace", labelNamespace())
                .add("CustomResponseBodies", hasCustomResponseBodies() ? customResponseBodies() : null)
                .add("CaptchaConfig", captchaConfig()).add("ChallengeConfig", challengeConfig())
                .add("TokenDomains", hasTokenDomains() ? tokenDomains() : null).add("AssociationConfig", associationConfig())
                .add("RetrofittedByFirewallManager", retrofittedByFirewallManager()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "ARN":
            return Optional.ofNullable(clazz.cast(arn()));
        case "DefaultAction":
            return Optional.ofNullable(clazz.cast(defaultAction()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Rules":
            return Optional.ofNullable(clazz.cast(rules()));
        case "VisibilityConfig":
            return Optional.ofNullable(clazz.cast(visibilityConfig()));
        case "Capacity":
            return Optional.ofNullable(clazz.cast(capacity()));
        case "PreProcessFirewallManagerRuleGroups":
            return Optional.ofNullable(clazz.cast(preProcessFirewallManagerRuleGroups()));
        case "PostProcessFirewallManagerRuleGroups":
            return Optional.ofNullable(clazz.cast(postProcessFirewallManagerRuleGroups()));
        case "ManagedByFirewallManager":
            return Optional.ofNullable(clazz.cast(managedByFirewallManager()));
        case "LabelNamespace":
            return Optional.ofNullable(clazz.cast(labelNamespace()));
        case "CustomResponseBodies":
            return Optional.ofNullable(clazz.cast(customResponseBodies()));
        case "CaptchaConfig":
            return Optional.ofNullable(clazz.cast(captchaConfig()));
        case "ChallengeConfig":
            return Optional.ofNullable(clazz.cast(challengeConfig()));
        case "TokenDomains":
            return Optional.ofNullable(clazz.cast(tokenDomains()));
        case "AssociationConfig":
            return Optional.ofNullable(clazz.cast(associationConfig()));
        case "RetrofittedByFirewallManager":
            return Optional.ofNullable(clazz.cast(retrofittedByFirewallManager()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("Id", ID_FIELD);
        map.put("ARN", ARN_FIELD);
        map.put("DefaultAction", DEFAULT_ACTION_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("Rules", RULES_FIELD);
        map.put("VisibilityConfig", VISIBILITY_CONFIG_FIELD);
        map.put("Capacity", CAPACITY_FIELD);
        map.put("PreProcessFirewallManagerRuleGroups", PRE_PROCESS_FIREWALL_MANAGER_RULE_GROUPS_FIELD);
        map.put("PostProcessFirewallManagerRuleGroups", POST_PROCESS_FIREWALL_MANAGER_RULE_GROUPS_FIELD);
        map.put("ManagedByFirewallManager", MANAGED_BY_FIREWALL_MANAGER_FIELD);
        map.put("LabelNamespace", LABEL_NAMESPACE_FIELD);
        map.put("CustomResponseBodies", CUSTOM_RESPONSE_BODIES_FIELD);
        map.put("CaptchaConfig", CAPTCHA_CONFIG_FIELD);
        map.put("ChallengeConfig", CHALLENGE_CONFIG_FIELD);
        map.put("TokenDomains", TOKEN_DOMAINS_FIELD);
        map.put("AssociationConfig", ASSOCIATION_CONFIG_FIELD);
        map.put("RetrofittedByFirewallManager", RETROFITTED_BY_FIREWALL_MANAGER_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<WebACL, T> g) {
        return obj -> g.apply((WebACL) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, WebACL> {
        /**
         * <p>
         * The name of the web ACL. You cannot change the name of a web ACL after you create it.
         * </p>
         * 
         * @param name
         *        The name of the web ACL. You cannot change the name of a web ACL after you create it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A unique identifier for the <code>WebACL</code>. This ID is returned in the responses to create and list
         * commands. You use this ID to do things like get, update, and delete a <code>WebACL</code>.
         * </p>
         * 
         * @param id
         *        A unique identifier for the <code>WebACL</code>. This ID is returned in the responses to create and
         *        list commands. You use this ID to do things like get, update, and delete a <code>WebACL</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the web ACL that you want to associate with the resource.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the web ACL that you want to associate with the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The action to perform if none of the <code>Rules</code> contained in the <code>WebACL</code> match.
         * </p>
         * 
         * @param defaultAction
         *        The action to perform if none of the <code>Rules</code> contained in the <code>WebACL</code> match.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultAction(DefaultAction defaultAction);

        /**
         * <p>
         * The action to perform if none of the <code>Rules</code> contained in the <code>WebACL</code> match.
         * </p>
         * This is a convenience method that creates an instance of the {@link DefaultAction.Builder} avoiding the need
         * to create one manually via {@link DefaultAction#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DefaultAction.Builder#build()} is called immediately and its
         * result is passed to {@link #defaultAction(DefaultAction)}.
         * 
         * @param defaultAction
         *        a consumer that will call methods on {@link DefaultAction.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #defaultAction(DefaultAction)
         */
        default Builder defaultAction(Consumer<DefaultAction.Builder> defaultAction) {
            return defaultAction(DefaultAction.builder().applyMutation(defaultAction).build());
        }

        /**
         * <p>
         * A description of the web ACL that helps with identification.
         * </p>
         * 
         * @param description
         *        A description of the web ACL that helps with identification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The <a>Rule</a> statements used to identify the web requests that you want to manage. Each rule includes one
         * top-level statement that WAF uses to identify matching web requests, and parameters that govern how WAF
         * handles them.
         * </p>
         * 
         * @param rules
         *        The <a>Rule</a> statements used to identify the web requests that you want to manage. Each rule
         *        includes one top-level statement that WAF uses to identify matching web requests, and parameters that
         *        govern how WAF handles them.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rules(Collection<Rule> rules);

        /**
         * <p>
         * The <a>Rule</a> statements used to identify the web requests that you want to manage. Each rule includes one
         * top-level statement that WAF uses to identify matching web requests, and parameters that govern how WAF
         * handles them.
         * </p>
         * 
         * @param rules
         *        The <a>Rule</a> statements used to identify the web requests that you want to manage. Each rule
         *        includes one top-level statement that WAF uses to identify matching web requests, and parameters that
         *        govern how WAF handles them.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rules(Rule... rules);

        /**
         * <p>
         * The <a>Rule</a> statements used to identify the web requests that you want to manage. Each rule includes one
         * top-level statement that WAF uses to identify matching web requests, and parameters that govern how WAF
         * handles them.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.wafv2.model.Rule.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.wafv2.model.Rule#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.wafv2.model.Rule.Builder#build()}
         * is called immediately and its result is passed to {@link #rules(List<Rule>)}.
         * 
         * @param rules
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.wafv2.model.Rule.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rules(java.util.Collection<Rule>)
         */
        Builder rules(Consumer<Rule.Builder>... rules);

        /**
         * <p>
         * Defines and enables Amazon CloudWatch metrics and web request sample collection.
         * </p>
         * 
         * @param visibilityConfig
         *        Defines and enables Amazon CloudWatch metrics and web request sample collection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder visibilityConfig(VisibilityConfig visibilityConfig);

        /**
         * <p>
         * Defines and enables Amazon CloudWatch metrics and web request sample collection.
         * </p>
         * This is a convenience method that creates an instance of the {@link VisibilityConfig.Builder} avoiding the
         * need to create one manually via {@link VisibilityConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VisibilityConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #visibilityConfig(VisibilityConfig)}.
         * 
         * @param visibilityConfig
         *        a consumer that will call methods on {@link VisibilityConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #visibilityConfig(VisibilityConfig)
         */
        default Builder visibilityConfig(Consumer<VisibilityConfig.Builder> visibilityConfig) {
            return visibilityConfig(VisibilityConfig.builder().applyMutation(visibilityConfig).build());
        }

        /**
         * <p>
         * The web ACL capacity units (WCUs) currently being used by this web ACL.
         * </p>
         * <p>
         * WAF uses WCUs to calculate and control the operating resources that are used to run your rules, rule groups,
         * and web ACLs. WAF calculates capacity differently for each rule type, to reflect the relative cost of each
         * rule. Simple rules that cost little to run use fewer WCUs than more complex rules that use more processing
         * power. Rule group capacity is fixed at creation, which helps users plan their web ACL WCU usage when they use
         * a rule group. For more information, see <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/aws-waf-capacity-units.html">WAF web ACL capacity
         * units (WCU)</a> in the <i>WAF Developer Guide</i>.
         * </p>
         * 
         * @param capacity
         *        The web ACL capacity units (WCUs) currently being used by this web ACL. </p>
         *        <p>
         *        WAF uses WCUs to calculate and control the operating resources that are used to run your rules, rule
         *        groups, and web ACLs. WAF calculates capacity differently for each rule type, to reflect the relative
         *        cost of each rule. Simple rules that cost little to run use fewer WCUs than more complex rules that
         *        use more processing power. Rule group capacity is fixed at creation, which helps users plan their web
         *        ACL WCU usage when they use a rule group. For more information, see <a
         *        href="https://docs.aws.amazon.com/waf/latest/developerguide/aws-waf-capacity-units.html">WAF web ACL
         *        capacity units (WCU)</a> in the <i>WAF Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacity(Long capacity);

        /**
         * <p>
         * The first set of rules for WAF to process in the web ACL. This is defined in an Firewall Manager WAF policy
         * and contains only rule group references. You can't alter these. Any rules and rule groups that you define for
         * the web ACL are prioritized after these.
         * </p>
         * <p>
         * In the Firewall Manager WAF policy, the Firewall Manager administrator can define a set of rule groups to run
         * first in the web ACL and a set of rule groups to run last. Within each set, the administrator prioritizes the
         * rule groups, to determine their relative processing order.
         * </p>
         * 
         * @param preProcessFirewallManagerRuleGroups
         *        The first set of rules for WAF to process in the web ACL. This is defined in an Firewall Manager WAF
         *        policy and contains only rule group references. You can't alter these. Any rules and rule groups that
         *        you define for the web ACL are prioritized after these. </p>
         *        <p>
         *        In the Firewall Manager WAF policy, the Firewall Manager administrator can define a set of rule groups
         *        to run first in the web ACL and a set of rule groups to run last. Within each set, the administrator
         *        prioritizes the rule groups, to determine their relative processing order.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preProcessFirewallManagerRuleGroups(Collection<FirewallManagerRuleGroup> preProcessFirewallManagerRuleGroups);

        /**
         * <p>
         * The first set of rules for WAF to process in the web ACL. This is defined in an Firewall Manager WAF policy
         * and contains only rule group references. You can't alter these. Any rules and rule groups that you define for
         * the web ACL are prioritized after these.
         * </p>
         * <p>
         * In the Firewall Manager WAF policy, the Firewall Manager administrator can define a set of rule groups to run
         * first in the web ACL and a set of rule groups to run last. Within each set, the administrator prioritizes the
         * rule groups, to determine their relative processing order.
         * </p>
         * 
         * @param preProcessFirewallManagerRuleGroups
         *        The first set of rules for WAF to process in the web ACL. This is defined in an Firewall Manager WAF
         *        policy and contains only rule group references. You can't alter these. Any rules and rule groups that
         *        you define for the web ACL are prioritized after these. </p>
         *        <p>
         *        In the Firewall Manager WAF policy, the Firewall Manager administrator can define a set of rule groups
         *        to run first in the web ACL and a set of rule groups to run last. Within each set, the administrator
         *        prioritizes the rule groups, to determine their relative processing order.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preProcessFirewallManagerRuleGroups(FirewallManagerRuleGroup... preProcessFirewallManagerRuleGroups);

        /**
         * <p>
         * The first set of rules for WAF to process in the web ACL. This is defined in an Firewall Manager WAF policy
         * and contains only rule group references. You can't alter these. Any rules and rule groups that you define for
         * the web ACL are prioritized after these.
         * </p>
         * <p>
         * In the Firewall Manager WAF policy, the Firewall Manager administrator can define a set of rule groups to run
         * first in the web ACL and a set of rule groups to run last. Within each set, the administrator prioritizes the
         * rule groups, to determine their relative processing order.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.wafv2.model.FirewallManagerRuleGroup.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.wafv2.model.FirewallManagerRuleGroup#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.wafv2.model.FirewallManagerRuleGroup.Builder#build()} is called
         * immediately and its result is passed to {@link
         * #preProcessFirewallManagerRuleGroups(List<FirewallManagerRuleGroup>)}.
         * 
         * @param preProcessFirewallManagerRuleGroups
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.wafv2.model.FirewallManagerRuleGroup.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #preProcessFirewallManagerRuleGroups(java.util.Collection<FirewallManagerRuleGroup>)
         */
        Builder preProcessFirewallManagerRuleGroups(
                Consumer<FirewallManagerRuleGroup.Builder>... preProcessFirewallManagerRuleGroups);

        /**
         * <p>
         * The last set of rules for WAF to process in the web ACL. This is defined in an Firewall Manager WAF policy
         * and contains only rule group references. You can't alter these. Any rules and rule groups that you define for
         * the web ACL are prioritized before these.
         * </p>
         * <p>
         * In the Firewall Manager WAF policy, the Firewall Manager administrator can define a set of rule groups to run
         * first in the web ACL and a set of rule groups to run last. Within each set, the administrator prioritizes the
         * rule groups, to determine their relative processing order.
         * </p>
         * 
         * @param postProcessFirewallManagerRuleGroups
         *        The last set of rules for WAF to process in the web ACL. This is defined in an Firewall Manager WAF
         *        policy and contains only rule group references. You can't alter these. Any rules and rule groups that
         *        you define for the web ACL are prioritized before these. </p>
         *        <p>
         *        In the Firewall Manager WAF policy, the Firewall Manager administrator can define a set of rule groups
         *        to run first in the web ACL and a set of rule groups to run last. Within each set, the administrator
         *        prioritizes the rule groups, to determine their relative processing order.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder postProcessFirewallManagerRuleGroups(Collection<FirewallManagerRuleGroup> postProcessFirewallManagerRuleGroups);

        /**
         * <p>
         * The last set of rules for WAF to process in the web ACL. This is defined in an Firewall Manager WAF policy
         * and contains only rule group references. You can't alter these. Any rules and rule groups that you define for
         * the web ACL are prioritized before these.
         * </p>
         * <p>
         * In the Firewall Manager WAF policy, the Firewall Manager administrator can define a set of rule groups to run
         * first in the web ACL and a set of rule groups to run last. Within each set, the administrator prioritizes the
         * rule groups, to determine their relative processing order.
         * </p>
         * 
         * @param postProcessFirewallManagerRuleGroups
         *        The last set of rules for WAF to process in the web ACL. This is defined in an Firewall Manager WAF
         *        policy and contains only rule group references. You can't alter these. Any rules and rule groups that
         *        you define for the web ACL are prioritized before these. </p>
         *        <p>
         *        In the Firewall Manager WAF policy, the Firewall Manager administrator can define a set of rule groups
         *        to run first in the web ACL and a set of rule groups to run last. Within each set, the administrator
         *        prioritizes the rule groups, to determine their relative processing order.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder postProcessFirewallManagerRuleGroups(FirewallManagerRuleGroup... postProcessFirewallManagerRuleGroups);

        /**
         * <p>
         * The last set of rules for WAF to process in the web ACL. This is defined in an Firewall Manager WAF policy
         * and contains only rule group references. You can't alter these. Any rules and rule groups that you define for
         * the web ACL are prioritized before these.
         * </p>
         * <p>
         * In the Firewall Manager WAF policy, the Firewall Manager administrator can define a set of rule groups to run
         * first in the web ACL and a set of rule groups to run last. Within each set, the administrator prioritizes the
         * rule groups, to determine their relative processing order.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.wafv2.model.FirewallManagerRuleGroup.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.wafv2.model.FirewallManagerRuleGroup#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.wafv2.model.FirewallManagerRuleGroup.Builder#build()} is called
         * immediately and its result is passed to {@link
         * #postProcessFirewallManagerRuleGroups(List<FirewallManagerRuleGroup>)}.
         * 
         * @param postProcessFirewallManagerRuleGroups
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.wafv2.model.FirewallManagerRuleGroup.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #postProcessFirewallManagerRuleGroups(java.util.Collection<FirewallManagerRuleGroup>)
         */
        Builder postProcessFirewallManagerRuleGroups(
                Consumer<FirewallManagerRuleGroup.Builder>... postProcessFirewallManagerRuleGroups);

        /**
         * <p>
         * Indicates whether this web ACL was created by Firewall Manager and is being managed by Firewall Manager. If
         * true, then only Firewall Manager can delete the web ACL or any Firewall Manager rule groups in the web ACL.
         * See also the properties <code>RetrofittedByFirewallManager</code>,
         * <code>PreProcessFirewallManagerRuleGroups</code>, and <code>PostProcessFirewallManagerRuleGroups</code>.
         * </p>
         * 
         * @param managedByFirewallManager
         *        Indicates whether this web ACL was created by Firewall Manager and is being managed by Firewall
         *        Manager. If true, then only Firewall Manager can delete the web ACL or any Firewall Manager rule
         *        groups in the web ACL. See also the properties <code>RetrofittedByFirewallManager</code>,
         *        <code>PreProcessFirewallManagerRuleGroups</code>, and
         *        <code>PostProcessFirewallManagerRuleGroups</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder managedByFirewallManager(Boolean managedByFirewallManager);

        /**
         * <p>
         * The label namespace prefix for this web ACL. All labels added by rules in this web ACL have this prefix.
         * </p>
         * <ul>
         * <li>
         * <p>
         * The syntax for the label namespace prefix for a web ACL is the following:
         * </p>
         * <p>
         * <code>awswaf:&lt;account ID&gt;:webacl:&lt;web ACL name&gt;:</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * When a rule with a label matches a web request, WAF adds the fully qualified label to the request. A fully
         * qualified label is made up of the label namespace from the rule group or web ACL where the rule is defined
         * and the label from the rule, separated by a colon:
         * </p>
         * <p>
         * <code>&lt;label namespace&gt;:&lt;label from rule&gt;</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param labelNamespace
         *        The label namespace prefix for this web ACL. All labels added by rules in this web ACL have this
         *        prefix. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The syntax for the label namespace prefix for a web ACL is the following:
         *        </p>
         *        <p>
         *        <code>awswaf:&lt;account ID&gt;:webacl:&lt;web ACL name&gt;:</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        When a rule with a label matches a web request, WAF adds the fully qualified label to the request. A
         *        fully qualified label is made up of the label namespace from the rule group or web ACL where the rule
         *        is defined and the label from the rule, separated by a colon:
         *        </p>
         *        <p>
         *        <code>&lt;label namespace&gt;:&lt;label from rule&gt;</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder labelNamespace(String labelNamespace);

        /**
         * <p>
         * A map of custom response keys and content bodies. When you create a rule with a block action, you can send a
         * custom response to the web request. You define these for the web ACL, and then use them in the rules and
         * default actions that you define in the web ACL.
         * </p>
         * <p>
         * For information about customizing web requests and responses, see <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-custom-request-response.html">Customizing web
         * requests and responses in WAF</a> in the <i>WAF Developer Guide</i>.
         * </p>
         * <p>
         * For information about the limits on count and size for custom request and response settings, see <a
         * href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">WAF quotas</a> in the <i>WAF
         * Developer Guide</i>.
         * </p>
         * 
         * @param customResponseBodies
         *        A map of custom response keys and content bodies. When you create a rule with a block action, you can
         *        send a custom response to the web request. You define these for the web ACL, and then use them in the
         *        rules and default actions that you define in the web ACL. </p>
         *        <p>
         *        For information about customizing web requests and responses, see <a
         *        href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-custom-request-response.html"
         *        >Customizing web requests and responses in WAF</a> in the <i>WAF Developer Guide</i>.
         *        </p>
         *        <p>
         *        For information about the limits on count and size for custom request and response settings, see <a
         *        href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">WAF quotas</a> in the <i>WAF
         *        Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customResponseBodies(Map<String, CustomResponseBody> customResponseBodies);

        /**
         * <p>
         * Specifies how WAF should handle <code>CAPTCHA</code> evaluations for rules that don't have their own
         * <code>CaptchaConfig</code> settings. If you don't specify this, WAF uses its default settings for
         * <code>CaptchaConfig</code>.
         * </p>
         * 
         * @param captchaConfig
         *        Specifies how WAF should handle <code>CAPTCHA</code> evaluations for rules that don't have their own
         *        <code>CaptchaConfig</code> settings. If you don't specify this, WAF uses its default settings for
         *        <code>CaptchaConfig</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder captchaConfig(CaptchaConfig captchaConfig);

        /**
         * <p>
         * Specifies how WAF should handle <code>CAPTCHA</code> evaluations for rules that don't have their own
         * <code>CaptchaConfig</code> settings. If you don't specify this, WAF uses its default settings for
         * <code>CaptchaConfig</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link CaptchaConfig.Builder} avoiding the need
         * to create one manually via {@link CaptchaConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CaptchaConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #captchaConfig(CaptchaConfig)}.
         * 
         * @param captchaConfig
         *        a consumer that will call methods on {@link CaptchaConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #captchaConfig(CaptchaConfig)
         */
        default Builder captchaConfig(Consumer<CaptchaConfig.Builder> captchaConfig) {
            return captchaConfig(CaptchaConfig.builder().applyMutation(captchaConfig).build());
        }

        /**
         * <p>
         * Specifies how WAF should handle challenge evaluations for rules that don't have their own
         * <code>ChallengeConfig</code> settings. If you don't specify this, WAF uses its default settings for
         * <code>ChallengeConfig</code>.
         * </p>
         * 
         * @param challengeConfig
         *        Specifies how WAF should handle challenge evaluations for rules that don't have their own
         *        <code>ChallengeConfig</code> settings. If you don't specify this, WAF uses its default settings for
         *        <code>ChallengeConfig</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder challengeConfig(ChallengeConfig challengeConfig);

        /**
         * <p>
         * Specifies how WAF should handle challenge evaluations for rules that don't have their own
         * <code>ChallengeConfig</code> settings. If you don't specify this, WAF uses its default settings for
         * <code>ChallengeConfig</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link ChallengeConfig.Builder} avoiding the
         * need to create one manually via {@link ChallengeConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ChallengeConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #challengeConfig(ChallengeConfig)}.
         * 
         * @param challengeConfig
         *        a consumer that will call methods on {@link ChallengeConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #challengeConfig(ChallengeConfig)
         */
        default Builder challengeConfig(Consumer<ChallengeConfig.Builder> challengeConfig) {
            return challengeConfig(ChallengeConfig.builder().applyMutation(challengeConfig).build());
        }

        /**
         * <p>
         * Specifies the domains that WAF should accept in a web request token. This enables the use of tokens across
         * multiple protected websites. When WAF provides a token, it uses the domain of the Amazon Web Services
         * resource that the web ACL is protecting. If you don't specify a list of token domains, WAF accepts tokens
         * only for the domain of the protected resource. With a token domain list, WAF accepts the resource's host
         * domain plus all domains in the token domain list, including their prefixed subdomains.
         * </p>
         * 
         * @param tokenDomains
         *        Specifies the domains that WAF should accept in a web request token. This enables the use of tokens
         *        across multiple protected websites. When WAF provides a token, it uses the domain of the Amazon Web
         *        Services resource that the web ACL is protecting. If you don't specify a list of token domains, WAF
         *        accepts tokens only for the domain of the protected resource. With a token domain list, WAF accepts
         *        the resource's host domain plus all domains in the token domain list, including their prefixed
         *        subdomains.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tokenDomains(Collection<String> tokenDomains);

        /**
         * <p>
         * Specifies the domains that WAF should accept in a web request token. This enables the use of tokens across
         * multiple protected websites. When WAF provides a token, it uses the domain of the Amazon Web Services
         * resource that the web ACL is protecting. If you don't specify a list of token domains, WAF accepts tokens
         * only for the domain of the protected resource. With a token domain list, WAF accepts the resource's host
         * domain plus all domains in the token domain list, including their prefixed subdomains.
         * </p>
         * 
         * @param tokenDomains
         *        Specifies the domains that WAF should accept in a web request token. This enables the use of tokens
         *        across multiple protected websites. When WAF provides a token, it uses the domain of the Amazon Web
         *        Services resource that the web ACL is protecting. If you don't specify a list of token domains, WAF
         *        accepts tokens only for the domain of the protected resource. With a token domain list, WAF accepts
         *        the resource's host domain plus all domains in the token domain list, including their prefixed
         *        subdomains.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tokenDomains(String... tokenDomains);

        /**
         * <p>
         * Specifies custom configurations for the associations between the web ACL and protected resources.
         * </p>
         * <p>
         * Use this to customize the maximum size of the request body that your protected resources forward to WAF for
         * inspection. You can customize this setting for CloudFront, API Gateway, Amazon Cognito, App Runner, or
         * Verified Access resources. The default setting is 16 KB (16,384 bytes).
         * </p>
         * <note>
         * <p>
         * You are charged additional fees when your protected resources forward body sizes that are larger than the
         * default. For more information, see <a href="http://aws.amazon.com/waf/pricing/">WAF Pricing</a>.
         * </p>
         * </note>
         * <p>
         * For Application Load Balancer and AppSync, the limit is fixed at 8 KB (8,192 bytes).
         * </p>
         * 
         * @param associationConfig
         *        Specifies custom configurations for the associations between the web ACL and protected resources. </p>
         *        <p>
         *        Use this to customize the maximum size of the request body that your protected resources forward to
         *        WAF for inspection. You can customize this setting for CloudFront, API Gateway, Amazon Cognito, App
         *        Runner, or Verified Access resources. The default setting is 16 KB (16,384 bytes).
         *        </p>
         *        <note>
         *        <p>
         *        You are charged additional fees when your protected resources forward body sizes that are larger than
         *        the default. For more information, see <a href="http://aws.amazon.com/waf/pricing/">WAF Pricing</a>.
         *        </p>
         *        </note>
         *        <p>
         *        For Application Load Balancer and AppSync, the limit is fixed at 8 KB (8,192 bytes).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationConfig(AssociationConfig associationConfig);

        /**
         * <p>
         * Specifies custom configurations for the associations between the web ACL and protected resources.
         * </p>
         * <p>
         * Use this to customize the maximum size of the request body that your protected resources forward to WAF for
         * inspection. You can customize this setting for CloudFront, API Gateway, Amazon Cognito, App Runner, or
         * Verified Access resources. The default setting is 16 KB (16,384 bytes).
         * </p>
         * <note>
         * <p>
         * You are charged additional fees when your protected resources forward body sizes that are larger than the
         * default. For more information, see <a href="http://aws.amazon.com/waf/pricing/">WAF Pricing</a>.
         * </p>
         * </note>
         * <p>
         * For Application Load Balancer and AppSync, the limit is fixed at 8 KB (8,192 bytes).
         * </p>
         * This is a convenience method that creates an instance of the {@link AssociationConfig.Builder} avoiding the
         * need to create one manually via {@link AssociationConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AssociationConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #associationConfig(AssociationConfig)}.
         * 
         * @param associationConfig
         *        a consumer that will call methods on {@link AssociationConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #associationConfig(AssociationConfig)
         */
        default Builder associationConfig(Consumer<AssociationConfig.Builder> associationConfig) {
            return associationConfig(AssociationConfig.builder().applyMutation(associationConfig).build());
        }

        /**
         * <p>
         * Indicates whether this web ACL was created by a customer account and then retrofitted by Firewall Manager. If
         * true, then the web ACL is currently being managed by a Firewall Manager WAF policy, and only Firewall Manager
         * can manage any Firewall Manager rule groups in the web ACL. See also the properties
         * <code>ManagedByFirewallManager</code>, <code>PreProcessFirewallManagerRuleGroups</code>, and
         * <code>PostProcessFirewallManagerRuleGroups</code>.
         * </p>
         * 
         * @param retrofittedByFirewallManager
         *        Indicates whether this web ACL was created by a customer account and then retrofitted by Firewall
         *        Manager. If true, then the web ACL is currently being managed by a Firewall Manager WAF policy, and
         *        only Firewall Manager can manage any Firewall Manager rule groups in the web ACL. See also the
         *        properties <code>ManagedByFirewallManager</code>, <code>PreProcessFirewallManagerRuleGroups</code>,
         *        and <code>PostProcessFirewallManagerRuleGroups</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retrofittedByFirewallManager(Boolean retrofittedByFirewallManager);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String id;

        private String arn;

        private DefaultAction defaultAction;

        private String description;

        private List<Rule> rules = DefaultSdkAutoConstructList.getInstance();

        private VisibilityConfig visibilityConfig;

        private Long capacity;

        private List<FirewallManagerRuleGroup> preProcessFirewallManagerRuleGroups = DefaultSdkAutoConstructList.getInstance();

        private List<FirewallManagerRuleGroup> postProcessFirewallManagerRuleGroups = DefaultSdkAutoConstructList.getInstance();

        private Boolean managedByFirewallManager;

        private String labelNamespace;

        private Map<String, CustomResponseBody> customResponseBodies = DefaultSdkAutoConstructMap.getInstance();

        private CaptchaConfig captchaConfig;

        private ChallengeConfig challengeConfig;

        private List<String> tokenDomains = DefaultSdkAutoConstructList.getInstance();

        private AssociationConfig associationConfig;

        private Boolean retrofittedByFirewallManager;

        private BuilderImpl() {
        }

        private BuilderImpl(WebACL model) {
            name(model.name);
            id(model.id);
            arn(model.arn);
            defaultAction(model.defaultAction);
            description(model.description);
            rules(model.rules);
            visibilityConfig(model.visibilityConfig);
            capacity(model.capacity);
            preProcessFirewallManagerRuleGroups(model.preProcessFirewallManagerRuleGroups);
            postProcessFirewallManagerRuleGroups(model.postProcessFirewallManagerRuleGroups);
            managedByFirewallManager(model.managedByFirewallManager);
            labelNamespace(model.labelNamespace);
            customResponseBodies(model.customResponseBodies);
            captchaConfig(model.captchaConfig);
            challengeConfig(model.challengeConfig);
            tokenDomains(model.tokenDomains);
            associationConfig(model.associationConfig);
            retrofittedByFirewallManager(model.retrofittedByFirewallManager);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final DefaultAction.Builder getDefaultAction() {
            return defaultAction != null ? defaultAction.toBuilder() : null;
        }

        public final void setDefaultAction(DefaultAction.BuilderImpl defaultAction) {
            this.defaultAction = defaultAction != null ? defaultAction.build() : null;
        }

        @Override
        public final Builder defaultAction(DefaultAction defaultAction) {
            this.defaultAction = defaultAction;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final List<Rule.Builder> getRules() {
            List<Rule.Builder> result = RulesCopier.copyToBuilder(this.rules);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRules(Collection<Rule.BuilderImpl> rules) {
            this.rules = RulesCopier.copyFromBuilder(rules);
        }

        @Override
        public final Builder rules(Collection<Rule> rules) {
            this.rules = RulesCopier.copy(rules);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder rules(Rule... rules) {
            rules(Arrays.asList(rules));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder rules(Consumer<Rule.Builder>... rules) {
            rules(Stream.of(rules).map(c -> Rule.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final VisibilityConfig.Builder getVisibilityConfig() {
            return visibilityConfig != null ? visibilityConfig.toBuilder() : null;
        }

        public final void setVisibilityConfig(VisibilityConfig.BuilderImpl visibilityConfig) {
            this.visibilityConfig = visibilityConfig != null ? visibilityConfig.build() : null;
        }

        @Override
        public final Builder visibilityConfig(VisibilityConfig visibilityConfig) {
            this.visibilityConfig = visibilityConfig;
            return this;
        }

        public final Long getCapacity() {
            return capacity;
        }

        public final void setCapacity(Long capacity) {
            this.capacity = capacity;
        }

        @Override
        public final Builder capacity(Long capacity) {
            this.capacity = capacity;
            return this;
        }

        public final List<FirewallManagerRuleGroup.Builder> getPreProcessFirewallManagerRuleGroups() {
            List<FirewallManagerRuleGroup.Builder> result = FirewallManagerRuleGroupsCopier
                    .copyToBuilder(this.preProcessFirewallManagerRuleGroups);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPreProcessFirewallManagerRuleGroups(
                Collection<FirewallManagerRuleGroup.BuilderImpl> preProcessFirewallManagerRuleGroups) {
            this.preProcessFirewallManagerRuleGroups = FirewallManagerRuleGroupsCopier
                    .copyFromBuilder(preProcessFirewallManagerRuleGroups);
        }

        @Override
        public final Builder preProcessFirewallManagerRuleGroups(
                Collection<FirewallManagerRuleGroup> preProcessFirewallManagerRuleGroups) {
            this.preProcessFirewallManagerRuleGroups = FirewallManagerRuleGroupsCopier.copy(preProcessFirewallManagerRuleGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder preProcessFirewallManagerRuleGroups(FirewallManagerRuleGroup... preProcessFirewallManagerRuleGroups) {
            preProcessFirewallManagerRuleGroups(Arrays.asList(preProcessFirewallManagerRuleGroups));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder preProcessFirewallManagerRuleGroups(
                Consumer<FirewallManagerRuleGroup.Builder>... preProcessFirewallManagerRuleGroups) {
            preProcessFirewallManagerRuleGroups(Stream.of(preProcessFirewallManagerRuleGroups)
                    .map(c -> FirewallManagerRuleGroup.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<FirewallManagerRuleGroup.Builder> getPostProcessFirewallManagerRuleGroups() {
            List<FirewallManagerRuleGroup.Builder> result = FirewallManagerRuleGroupsCopier
                    .copyToBuilder(this.postProcessFirewallManagerRuleGroups);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPostProcessFirewallManagerRuleGroups(
                Collection<FirewallManagerRuleGroup.BuilderImpl> postProcessFirewallManagerRuleGroups) {
            this.postProcessFirewallManagerRuleGroups = FirewallManagerRuleGroupsCopier
                    .copyFromBuilder(postProcessFirewallManagerRuleGroups);
        }

        @Override
        public final Builder postProcessFirewallManagerRuleGroups(
                Collection<FirewallManagerRuleGroup> postProcessFirewallManagerRuleGroups) {
            this.postProcessFirewallManagerRuleGroups = FirewallManagerRuleGroupsCopier
                    .copy(postProcessFirewallManagerRuleGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder postProcessFirewallManagerRuleGroups(
                FirewallManagerRuleGroup... postProcessFirewallManagerRuleGroups) {
            postProcessFirewallManagerRuleGroups(Arrays.asList(postProcessFirewallManagerRuleGroups));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder postProcessFirewallManagerRuleGroups(
                Consumer<FirewallManagerRuleGroup.Builder>... postProcessFirewallManagerRuleGroups) {
            postProcessFirewallManagerRuleGroups(Stream.of(postProcessFirewallManagerRuleGroups)
                    .map(c -> FirewallManagerRuleGroup.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Boolean getManagedByFirewallManager() {
            return managedByFirewallManager;
        }

        public final void setManagedByFirewallManager(Boolean managedByFirewallManager) {
            this.managedByFirewallManager = managedByFirewallManager;
        }

        @Override
        public final Builder managedByFirewallManager(Boolean managedByFirewallManager) {
            this.managedByFirewallManager = managedByFirewallManager;
            return this;
        }

        public final String getLabelNamespace() {
            return labelNamespace;
        }

        public final void setLabelNamespace(String labelNamespace) {
            this.labelNamespace = labelNamespace;
        }

        @Override
        public final Builder labelNamespace(String labelNamespace) {
            this.labelNamespace = labelNamespace;
            return this;
        }

        public final Map<String, CustomResponseBody.Builder> getCustomResponseBodies() {
            Map<String, CustomResponseBody.Builder> result = CustomResponseBodiesCopier.copyToBuilder(this.customResponseBodies);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setCustomResponseBodies(Map<String, CustomResponseBody.BuilderImpl> customResponseBodies) {
            this.customResponseBodies = CustomResponseBodiesCopier.copyFromBuilder(customResponseBodies);
        }

        @Override
        public final Builder customResponseBodies(Map<String, CustomResponseBody> customResponseBodies) {
            this.customResponseBodies = CustomResponseBodiesCopier.copy(customResponseBodies);
            return this;
        }

        public final CaptchaConfig.Builder getCaptchaConfig() {
            return captchaConfig != null ? captchaConfig.toBuilder() : null;
        }

        public final void setCaptchaConfig(CaptchaConfig.BuilderImpl captchaConfig) {
            this.captchaConfig = captchaConfig != null ? captchaConfig.build() : null;
        }

        @Override
        public final Builder captchaConfig(CaptchaConfig captchaConfig) {
            this.captchaConfig = captchaConfig;
            return this;
        }

        public final ChallengeConfig.Builder getChallengeConfig() {
            return challengeConfig != null ? challengeConfig.toBuilder() : null;
        }

        public final void setChallengeConfig(ChallengeConfig.BuilderImpl challengeConfig) {
            this.challengeConfig = challengeConfig != null ? challengeConfig.build() : null;
        }

        @Override
        public final Builder challengeConfig(ChallengeConfig challengeConfig) {
            this.challengeConfig = challengeConfig;
            return this;
        }

        public final Collection<String> getTokenDomains() {
            if (tokenDomains instanceof SdkAutoConstructList) {
                return null;
            }
            return tokenDomains;
        }

        public final void setTokenDomains(Collection<String> tokenDomains) {
            this.tokenDomains = TokenDomainsCopier.copy(tokenDomains);
        }

        @Override
        public final Builder tokenDomains(Collection<String> tokenDomains) {
            this.tokenDomains = TokenDomainsCopier.copy(tokenDomains);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tokenDomains(String... tokenDomains) {
            tokenDomains(Arrays.asList(tokenDomains));
            return this;
        }

        public final AssociationConfig.Builder getAssociationConfig() {
            return associationConfig != null ? associationConfig.toBuilder() : null;
        }

        public final void setAssociationConfig(AssociationConfig.BuilderImpl associationConfig) {
            this.associationConfig = associationConfig != null ? associationConfig.build() : null;
        }

        @Override
        public final Builder associationConfig(AssociationConfig associationConfig) {
            this.associationConfig = associationConfig;
            return this;
        }

        public final Boolean getRetrofittedByFirewallManager() {
            return retrofittedByFirewallManager;
        }

        public final void setRetrofittedByFirewallManager(Boolean retrofittedByFirewallManager) {
            this.retrofittedByFirewallManager = retrofittedByFirewallManager;
        }

        @Override
        public final Builder retrofittedByFirewallManager(Boolean retrofittedByFirewallManager) {
            this.retrofittedByFirewallManager = retrofittedByFirewallManager;
            return this;
        }

        @Override
        public WebACL build() {
            return new WebACL(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
