/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The filter to use to identify the subset of headers to inspect in a web request.
 * </p>
 * <p>
 * You must specify exactly one setting: either <code>All</code>, <code>IncludedHeaders</code>, or
 * <code>ExcludedHeaders</code>.
 * </p>
 * <p>
 * Example JSON: <code>"MatchPattern": { "ExcludedHeaders": [ "KeyToExclude1", "KeyToExclude2" ] }</code>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class HeaderMatchPattern implements SdkPojo, Serializable,
        ToCopyableBuilder<HeaderMatchPattern.Builder, HeaderMatchPattern> {
    private static final SdkField<All> ALL_FIELD = SdkField.<All> builder(MarshallingType.SDK_POJO).memberName("All")
            .getter(getter(HeaderMatchPattern::all)).setter(setter(Builder::all)).constructor(All::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("All").build()).build();

    private static final SdkField<List<String>> INCLUDED_HEADERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("IncludedHeaders")
            .getter(getter(HeaderMatchPattern::includedHeaders))
            .setter(setter(Builder::includedHeaders))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IncludedHeaders").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> EXCLUDED_HEADERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ExcludedHeaders")
            .getter(getter(HeaderMatchPattern::excludedHeaders))
            .setter(setter(Builder::excludedHeaders))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExcludedHeaders").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALL_FIELD,
            INCLUDED_HEADERS_FIELD, EXCLUDED_HEADERS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final All all;

    private final List<String> includedHeaders;

    private final List<String> excludedHeaders;

    private HeaderMatchPattern(BuilderImpl builder) {
        this.all = builder.all;
        this.includedHeaders = builder.includedHeaders;
        this.excludedHeaders = builder.excludedHeaders;
    }

    /**
     * <p>
     * Inspect all headers.
     * </p>
     * 
     * @return Inspect all headers.
     */
    public final All all() {
        return all;
    }

    /**
     * For responses, this returns true if the service returned a value for the IncludedHeaders property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIncludedHeaders() {
        return includedHeaders != null && !(includedHeaders instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Inspect only the headers that have a key that matches one of the strings specified here.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIncludedHeaders} method.
     * </p>
     * 
     * @return Inspect only the headers that have a key that matches one of the strings specified here.
     */
    public final List<String> includedHeaders() {
        return includedHeaders;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExcludedHeaders property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasExcludedHeaders() {
        return excludedHeaders != null && !(excludedHeaders instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Inspect only the headers whose keys don't match any of the strings specified here.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExcludedHeaders} method.
     * </p>
     * 
     * @return Inspect only the headers whose keys don't match any of the strings specified here.
     */
    public final List<String> excludedHeaders() {
        return excludedHeaders;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(all());
        hashCode = 31 * hashCode + Objects.hashCode(hasIncludedHeaders() ? includedHeaders() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasExcludedHeaders() ? excludedHeaders() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof HeaderMatchPattern)) {
            return false;
        }
        HeaderMatchPattern other = (HeaderMatchPattern) obj;
        return Objects.equals(all(), other.all()) && hasIncludedHeaders() == other.hasIncludedHeaders()
                && Objects.equals(includedHeaders(), other.includedHeaders())
                && hasExcludedHeaders() == other.hasExcludedHeaders()
                && Objects.equals(excludedHeaders(), other.excludedHeaders());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("HeaderMatchPattern").add("All", all())
                .add("IncludedHeaders", hasIncludedHeaders() ? includedHeaders() : null)
                .add("ExcludedHeaders", hasExcludedHeaders() ? excludedHeaders() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "All":
            return Optional.ofNullable(clazz.cast(all()));
        case "IncludedHeaders":
            return Optional.ofNullable(clazz.cast(includedHeaders()));
        case "ExcludedHeaders":
            return Optional.ofNullable(clazz.cast(excludedHeaders()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("All", ALL_FIELD);
        map.put("IncludedHeaders", INCLUDED_HEADERS_FIELD);
        map.put("ExcludedHeaders", EXCLUDED_HEADERS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<HeaderMatchPattern, T> g) {
        return obj -> g.apply((HeaderMatchPattern) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, HeaderMatchPattern> {
        /**
         * <p>
         * Inspect all headers.
         * </p>
         * 
         * @param all
         *        Inspect all headers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder all(All all);

        /**
         * <p>
         * Inspect all headers.
         * </p>
         * This is a convenience method that creates an instance of the {@link All.Builder} avoiding the need to create
         * one manually via {@link All#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link All.Builder#build()} is called immediately and its result is
         * passed to {@link #all(All)}.
         * 
         * @param all
         *        a consumer that will call methods on {@link All.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #all(All)
         */
        default Builder all(Consumer<All.Builder> all) {
            return all(All.builder().applyMutation(all).build());
        }

        /**
         * <p>
         * Inspect only the headers that have a key that matches one of the strings specified here.
         * </p>
         * 
         * @param includedHeaders
         *        Inspect only the headers that have a key that matches one of the strings specified here.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includedHeaders(Collection<String> includedHeaders);

        /**
         * <p>
         * Inspect only the headers that have a key that matches one of the strings specified here.
         * </p>
         * 
         * @param includedHeaders
         *        Inspect only the headers that have a key that matches one of the strings specified here.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includedHeaders(String... includedHeaders);

        /**
         * <p>
         * Inspect only the headers whose keys don't match any of the strings specified here.
         * </p>
         * 
         * @param excludedHeaders
         *        Inspect only the headers whose keys don't match any of the strings specified here.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludedHeaders(Collection<String> excludedHeaders);

        /**
         * <p>
         * Inspect only the headers whose keys don't match any of the strings specified here.
         * </p>
         * 
         * @param excludedHeaders
         *        Inspect only the headers whose keys don't match any of the strings specified here.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludedHeaders(String... excludedHeaders);
    }

    static final class BuilderImpl implements Builder {
        private All all;

        private List<String> includedHeaders = DefaultSdkAutoConstructList.getInstance();

        private List<String> excludedHeaders = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(HeaderMatchPattern model) {
            all(model.all);
            includedHeaders(model.includedHeaders);
            excludedHeaders(model.excludedHeaders);
        }

        public final All.Builder getAll() {
            return all != null ? all.toBuilder() : null;
        }

        public final void setAll(All.BuilderImpl all) {
            this.all = all != null ? all.build() : null;
        }

        @Override
        public final Builder all(All all) {
            this.all = all;
            return this;
        }

        public final Collection<String> getIncludedHeaders() {
            if (includedHeaders instanceof SdkAutoConstructList) {
                return null;
            }
            return includedHeaders;
        }

        public final void setIncludedHeaders(Collection<String> includedHeaders) {
            this.includedHeaders = HeaderNamesCopier.copy(includedHeaders);
        }

        @Override
        public final Builder includedHeaders(Collection<String> includedHeaders) {
            this.includedHeaders = HeaderNamesCopier.copy(includedHeaders);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder includedHeaders(String... includedHeaders) {
            includedHeaders(Arrays.asList(includedHeaders));
            return this;
        }

        public final Collection<String> getExcludedHeaders() {
            if (excludedHeaders instanceof SdkAutoConstructList) {
                return null;
            }
            return excludedHeaders;
        }

        public final void setExcludedHeaders(Collection<String> excludedHeaders) {
            this.excludedHeaders = HeaderNamesCopier.copy(excludedHeaders);
        }

        @Override
        public final Builder excludedHeaders(Collection<String> excludedHeaders) {
            this.excludedHeaders = HeaderNamesCopier.copy(excludedHeaders);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder excludedHeaders(String... excludedHeaders) {
            excludedHeaders(Arrays.asList(excludedHeaders));
            return this;
        }

        @Override
        public HeaderMatchPattern build() {
            return new HeaderMatchPattern(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
