/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wellarchitected;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.wellarchitected.model.AccessDeniedException;
import software.amazon.awssdk.services.wellarchitected.model.AssociateLensesRequest;
import software.amazon.awssdk.services.wellarchitected.model.AssociateLensesResponse;
import software.amazon.awssdk.services.wellarchitected.model.ConflictException;
import software.amazon.awssdk.services.wellarchitected.model.CreateLensShareRequest;
import software.amazon.awssdk.services.wellarchitected.model.CreateLensShareResponse;
import software.amazon.awssdk.services.wellarchitected.model.CreateLensVersionRequest;
import software.amazon.awssdk.services.wellarchitected.model.CreateLensVersionResponse;
import software.amazon.awssdk.services.wellarchitected.model.CreateMilestoneRequest;
import software.amazon.awssdk.services.wellarchitected.model.CreateMilestoneResponse;
import software.amazon.awssdk.services.wellarchitected.model.CreateWorkloadRequest;
import software.amazon.awssdk.services.wellarchitected.model.CreateWorkloadResponse;
import software.amazon.awssdk.services.wellarchitected.model.CreateWorkloadShareRequest;
import software.amazon.awssdk.services.wellarchitected.model.CreateWorkloadShareResponse;
import software.amazon.awssdk.services.wellarchitected.model.DeleteLensRequest;
import software.amazon.awssdk.services.wellarchitected.model.DeleteLensResponse;
import software.amazon.awssdk.services.wellarchitected.model.DeleteLensShareRequest;
import software.amazon.awssdk.services.wellarchitected.model.DeleteLensShareResponse;
import software.amazon.awssdk.services.wellarchitected.model.DeleteWorkloadRequest;
import software.amazon.awssdk.services.wellarchitected.model.DeleteWorkloadResponse;
import software.amazon.awssdk.services.wellarchitected.model.DeleteWorkloadShareRequest;
import software.amazon.awssdk.services.wellarchitected.model.DeleteWorkloadShareResponse;
import software.amazon.awssdk.services.wellarchitected.model.DisassociateLensesRequest;
import software.amazon.awssdk.services.wellarchitected.model.DisassociateLensesResponse;
import software.amazon.awssdk.services.wellarchitected.model.ExportLensRequest;
import software.amazon.awssdk.services.wellarchitected.model.ExportLensResponse;
import software.amazon.awssdk.services.wellarchitected.model.GetAnswerRequest;
import software.amazon.awssdk.services.wellarchitected.model.GetAnswerResponse;
import software.amazon.awssdk.services.wellarchitected.model.GetLensRequest;
import software.amazon.awssdk.services.wellarchitected.model.GetLensResponse;
import software.amazon.awssdk.services.wellarchitected.model.GetLensReviewReportRequest;
import software.amazon.awssdk.services.wellarchitected.model.GetLensReviewReportResponse;
import software.amazon.awssdk.services.wellarchitected.model.GetLensReviewRequest;
import software.amazon.awssdk.services.wellarchitected.model.GetLensReviewResponse;
import software.amazon.awssdk.services.wellarchitected.model.GetLensVersionDifferenceRequest;
import software.amazon.awssdk.services.wellarchitected.model.GetLensVersionDifferenceResponse;
import software.amazon.awssdk.services.wellarchitected.model.GetMilestoneRequest;
import software.amazon.awssdk.services.wellarchitected.model.GetMilestoneResponse;
import software.amazon.awssdk.services.wellarchitected.model.GetWorkloadRequest;
import software.amazon.awssdk.services.wellarchitected.model.GetWorkloadResponse;
import software.amazon.awssdk.services.wellarchitected.model.ImportLensRequest;
import software.amazon.awssdk.services.wellarchitected.model.ImportLensResponse;
import software.amazon.awssdk.services.wellarchitected.model.InternalServerException;
import software.amazon.awssdk.services.wellarchitected.model.ListAnswersRequest;
import software.amazon.awssdk.services.wellarchitected.model.ListAnswersResponse;
import software.amazon.awssdk.services.wellarchitected.model.ListLensReviewImprovementsRequest;
import software.amazon.awssdk.services.wellarchitected.model.ListLensReviewImprovementsResponse;
import software.amazon.awssdk.services.wellarchitected.model.ListLensReviewsRequest;
import software.amazon.awssdk.services.wellarchitected.model.ListLensReviewsResponse;
import software.amazon.awssdk.services.wellarchitected.model.ListLensSharesRequest;
import software.amazon.awssdk.services.wellarchitected.model.ListLensSharesResponse;
import software.amazon.awssdk.services.wellarchitected.model.ListLensesRequest;
import software.amazon.awssdk.services.wellarchitected.model.ListLensesResponse;
import software.amazon.awssdk.services.wellarchitected.model.ListMilestonesRequest;
import software.amazon.awssdk.services.wellarchitected.model.ListMilestonesResponse;
import software.amazon.awssdk.services.wellarchitected.model.ListNotificationsRequest;
import software.amazon.awssdk.services.wellarchitected.model.ListNotificationsResponse;
import software.amazon.awssdk.services.wellarchitected.model.ListShareInvitationsRequest;
import software.amazon.awssdk.services.wellarchitected.model.ListShareInvitationsResponse;
import software.amazon.awssdk.services.wellarchitected.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.wellarchitected.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.wellarchitected.model.ListWorkloadSharesRequest;
import software.amazon.awssdk.services.wellarchitected.model.ListWorkloadSharesResponse;
import software.amazon.awssdk.services.wellarchitected.model.ListWorkloadsRequest;
import software.amazon.awssdk.services.wellarchitected.model.ListWorkloadsResponse;
import software.amazon.awssdk.services.wellarchitected.model.ResourceNotFoundException;
import software.amazon.awssdk.services.wellarchitected.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.wellarchitected.model.TagResourceRequest;
import software.amazon.awssdk.services.wellarchitected.model.TagResourceResponse;
import software.amazon.awssdk.services.wellarchitected.model.ThrottlingException;
import software.amazon.awssdk.services.wellarchitected.model.UntagResourceRequest;
import software.amazon.awssdk.services.wellarchitected.model.UntagResourceResponse;
import software.amazon.awssdk.services.wellarchitected.model.UpdateAnswerRequest;
import software.amazon.awssdk.services.wellarchitected.model.UpdateAnswerResponse;
import software.amazon.awssdk.services.wellarchitected.model.UpdateGlobalSettingsRequest;
import software.amazon.awssdk.services.wellarchitected.model.UpdateGlobalSettingsResponse;
import software.amazon.awssdk.services.wellarchitected.model.UpdateLensReviewRequest;
import software.amazon.awssdk.services.wellarchitected.model.UpdateLensReviewResponse;
import software.amazon.awssdk.services.wellarchitected.model.UpdateShareInvitationRequest;
import software.amazon.awssdk.services.wellarchitected.model.UpdateShareInvitationResponse;
import software.amazon.awssdk.services.wellarchitected.model.UpdateWorkloadRequest;
import software.amazon.awssdk.services.wellarchitected.model.UpdateWorkloadResponse;
import software.amazon.awssdk.services.wellarchitected.model.UpdateWorkloadShareRequest;
import software.amazon.awssdk.services.wellarchitected.model.UpdateWorkloadShareResponse;
import software.amazon.awssdk.services.wellarchitected.model.UpgradeLensReviewRequest;
import software.amazon.awssdk.services.wellarchitected.model.UpgradeLensReviewResponse;
import software.amazon.awssdk.services.wellarchitected.model.ValidationException;
import software.amazon.awssdk.services.wellarchitected.model.WellArchitectedException;
import software.amazon.awssdk.services.wellarchitected.model.WellArchitectedRequest;
import software.amazon.awssdk.services.wellarchitected.paginators.ListAnswersPublisher;
import software.amazon.awssdk.services.wellarchitected.paginators.ListLensReviewImprovementsPublisher;
import software.amazon.awssdk.services.wellarchitected.paginators.ListLensReviewsPublisher;
import software.amazon.awssdk.services.wellarchitected.paginators.ListLensSharesPublisher;
import software.amazon.awssdk.services.wellarchitected.paginators.ListLensesPublisher;
import software.amazon.awssdk.services.wellarchitected.paginators.ListMilestonesPublisher;
import software.amazon.awssdk.services.wellarchitected.paginators.ListNotificationsPublisher;
import software.amazon.awssdk.services.wellarchitected.paginators.ListShareInvitationsPublisher;
import software.amazon.awssdk.services.wellarchitected.paginators.ListWorkloadSharesPublisher;
import software.amazon.awssdk.services.wellarchitected.paginators.ListWorkloadsPublisher;
import software.amazon.awssdk.services.wellarchitected.transform.AssociateLensesRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.CreateLensShareRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.CreateLensVersionRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.CreateMilestoneRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.CreateWorkloadRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.CreateWorkloadShareRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.DeleteLensRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.DeleteLensShareRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.DeleteWorkloadRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.DeleteWorkloadShareRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.DisassociateLensesRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.ExportLensRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.GetAnswerRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.GetLensRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.GetLensReviewReportRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.GetLensReviewRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.GetLensVersionDifferenceRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.GetMilestoneRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.GetWorkloadRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.ImportLensRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.ListAnswersRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.ListLensReviewImprovementsRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.ListLensReviewsRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.ListLensSharesRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.ListLensesRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.ListMilestonesRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.ListNotificationsRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.ListShareInvitationsRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.ListWorkloadSharesRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.ListWorkloadsRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.UpdateAnswerRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.UpdateGlobalSettingsRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.UpdateLensReviewRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.UpdateShareInvitationRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.UpdateWorkloadRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.UpdateWorkloadShareRequestMarshaller;
import software.amazon.awssdk.services.wellarchitected.transform.UpgradeLensReviewRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link WellArchitectedAsyncClient}.
 *
 * @see WellArchitectedAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultWellArchitectedAsyncClient implements WellArchitectedAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultWellArchitectedAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultWellArchitectedAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Associate a lens to a workload.
     * </p>
     * <p>
     * Up to 10 lenses can be associated with a workload in a single API operation. A maximum of 20 lenses can be
     * associated with a workload.
     * </p>
     * <note>
     * <p>
     * <b>Disclaimer</b>
     * </p>
     * <p>
     * By accessing and/or applying custom lenses created by another Amazon Web Services user or account, you
     * acknowledge that custom lenses created by other users and shared with you are Third Party Content as defined in
     * the Amazon Web Services Customer Agreement.
     * </p>
     * </note>
     *
     * @param associateLensesRequest
     *        Input to associate lens reviews.
     * @return A Java Future containing the result of the AssociateLenses operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ConflictException The resource already exists.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.AssociateLenses
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/AssociateLenses"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateLensesResponse> associateLenses(AssociateLensesRequest associateLensesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateLensesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateLenses");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateLensesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateLensesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateLensesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateLensesRequest, AssociateLensesResponse>()
                            .withOperationName("AssociateLenses")
                            .withMarshaller(new AssociateLensesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(associateLensesRequest));
            CompletableFuture<AssociateLensesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create a lens share.
     * </p>
     * <p>
     * The owner of a lens can share it with other Amazon Web Services accounts and IAM users in the same Amazon Web
     * Services Region. Shared access to a lens is not removed until the lens invitation is deleted.
     * </p>
     * <note>
     * <p>
     * <b>Disclaimer</b>
     * </p>
     * <p>
     * By sharing your custom lenses with other Amazon Web Services accounts, you acknowledge that Amazon Web Services
     * will make your custom lenses available to those other accounts. Those other accounts may continue to access and
     * use your shared custom lenses even if you delete the custom lenses from your own Amazon Web Services account or
     * terminate your Amazon Web Services account.
     * </p>
     * </note>
     *
     * @param createLensShareRequest
     * @return A Java Future containing the result of the CreateLensShare operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>ConflictException The resource already exists.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ServiceQuotaExceededException The user has reached their resource quota.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.CreateLensShare
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/CreateLensShare"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLensShareResponse> createLensShare(CreateLensShareRequest createLensShareRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLensShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLensShare");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLensShareResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLensShareResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLensShareResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLensShareRequest, CreateLensShareResponse>()
                            .withOperationName("CreateLensShare")
                            .withMarshaller(new CreateLensShareRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLensShareRequest));
            CompletableFuture<CreateLensShareResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create a new lens version.
     * </p>
     * <p>
     * A lens can have up to 100 versions.
     * </p>
     * <p>
     * After a lens has been imported, create a new lens version to publish it. The owner of a lens can share the lens
     * with other Amazon Web Services accounts and IAM users in the same Amazon Web Services Region. Only the owner of a
     * lens can delete it.
     * </p>
     *
     * @param createLensVersionRequest
     * @return A Java Future containing the result of the CreateLensVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ServiceQuotaExceededException The user has reached their resource quota.</li>
     *         <li>ConflictException The resource already exists.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.CreateLensVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/CreateLensVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLensVersionResponse> createLensVersion(CreateLensVersionRequest createLensVersionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLensVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLensVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLensVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLensVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLensVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLensVersionRequest, CreateLensVersionResponse>()
                            .withOperationName("CreateLensVersion")
                            .withMarshaller(new CreateLensVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLensVersionRequest));
            CompletableFuture<CreateLensVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create a milestone for an existing workload.
     * </p>
     *
     * @param createMilestoneRequest
     *        Input for milestone creation.
     * @return A Java Future containing the result of the CreateMilestone operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>ConflictException The resource already exists.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ServiceQuotaExceededException The user has reached their resource quota.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.CreateMilestone
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/CreateMilestone"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMilestoneResponse> createMilestone(CreateMilestoneRequest createMilestoneRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMilestoneRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMilestone");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMilestoneResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateMilestoneResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMilestoneResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMilestoneRequest, CreateMilestoneResponse>()
                            .withOperationName("CreateMilestone")
                            .withMarshaller(new CreateMilestoneRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createMilestoneRequest));
            CompletableFuture<CreateMilestoneResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create a new workload.
     * </p>
     * <p>
     * The owner of a workload can share the workload with other Amazon Web Services accounts and IAM users in the same
     * Amazon Web Services Region. Only the owner of a workload can delete it.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/wellarchitected/latest/userguide/define-workload.html">Defining a Workload</a>
     * in the <i>Well-Architected Tool User Guide</i>.
     * </p>
     *
     * @param createWorkloadRequest
     *        Input for workload creation.
     * @return A Java Future containing the result of the CreateWorkload operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>ConflictException The resource already exists.</li>
     *         <li>ServiceQuotaExceededException The user has reached their resource quota.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.CreateWorkload
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/CreateWorkload"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateWorkloadResponse> createWorkload(CreateWorkloadRequest createWorkloadRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWorkloadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWorkload");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateWorkloadResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateWorkloadResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateWorkloadResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateWorkloadRequest, CreateWorkloadResponse>()
                            .withOperationName("CreateWorkload")
                            .withMarshaller(new CreateWorkloadRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createWorkloadRequest));
            CompletableFuture<CreateWorkloadResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create a workload share.
     * </p>
     * <p>
     * The owner of a workload can share it with other Amazon Web Services accounts and IAM users in the same Amazon Web
     * Services Region. Shared access to a workload is not removed until the workload invitation is deleted.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/wellarchitected/latest/userguide/workloads-sharing.html">Sharing a Workload</a>
     * in the <i>Well-Architected Tool User Guide</i>.
     * </p>
     *
     * @param createWorkloadShareRequest
     *        Input for Create Workload Share
     * @return A Java Future containing the result of the CreateWorkloadShare operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>ConflictException The resource already exists.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ServiceQuotaExceededException The user has reached their resource quota.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.CreateWorkloadShare
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/CreateWorkloadShare"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateWorkloadShareResponse> createWorkloadShare(
            CreateWorkloadShareRequest createWorkloadShareRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWorkloadShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWorkloadShare");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateWorkloadShareResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateWorkloadShareResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateWorkloadShareResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateWorkloadShareRequest, CreateWorkloadShareResponse>()
                            .withOperationName("CreateWorkloadShare")
                            .withMarshaller(new CreateWorkloadShareRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createWorkloadShareRequest));
            CompletableFuture<CreateWorkloadShareResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete an existing lens.
     * </p>
     * <p>
     * Only the owner of a lens can delete it. After the lens is deleted, Amazon Web Services accounts and IAM users
     * that you shared the lens with can continue to use it, but they will no longer be able to apply it to new
     * workloads.
     * </p>
     * <note>
     * <p>
     * <b>Disclaimer</b>
     * </p>
     * <p>
     * By sharing your custom lenses with other Amazon Web Services accounts, you acknowledge that Amazon Web Services
     * will make your custom lenses available to those other accounts. Those other accounts may continue to access and
     * use your shared custom lenses even if you delete the custom lenses from your own Amazon Web Services account or
     * terminate your Amazon Web Services account.
     * </p>
     * </note>
     *
     * @param deleteLensRequest
     * @return A Java Future containing the result of the DeleteLens operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ConflictException The resource already exists.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.DeleteLens
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/DeleteLens" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLensResponse> deleteLens(DeleteLensRequest deleteLensRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLensRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLens");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLensResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteLensResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLensResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLensRequest, DeleteLensResponse>().withOperationName("DeleteLens")
                            .withMarshaller(new DeleteLensRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteLensRequest));
            CompletableFuture<DeleteLensResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete a lens share.
     * </p>
     * <p>
     * After the lens share is deleted, Amazon Web Services accounts and IAM users that you shared the lens with can
     * continue to use it, but they will no longer be able to apply it to new workloads.
     * </p>
     * <note>
     * <p>
     * <b>Disclaimer</b>
     * </p>
     * <p>
     * By sharing your custom lenses with other Amazon Web Services accounts, you acknowledge that Amazon Web Services
     * will make your custom lenses available to those other accounts. Those other accounts may continue to access and
     * use your shared custom lenses even if you delete the custom lenses from your own Amazon Web Services account or
     * terminate your Amazon Web Services account.
     * </p>
     * </note>
     *
     * @param deleteLensShareRequest
     * @return A Java Future containing the result of the DeleteLensShare operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ConflictException The resource already exists.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.DeleteLensShare
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/DeleteLensShare"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLensShareResponse> deleteLensShare(DeleteLensShareRequest deleteLensShareRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLensShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLensShare");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLensShareResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteLensShareResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLensShareResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLensShareRequest, DeleteLensShareResponse>()
                            .withOperationName("DeleteLensShare")
                            .withMarshaller(new DeleteLensShareRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteLensShareRequest));
            CompletableFuture<DeleteLensShareResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete an existing workload.
     * </p>
     *
     * @param deleteWorkloadRequest
     *        Input for workload deletion.
     * @return A Java Future containing the result of the DeleteWorkload operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ConflictException The resource already exists.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.DeleteWorkload
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/DeleteWorkload"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWorkloadResponse> deleteWorkload(DeleteWorkloadRequest deleteWorkloadRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWorkloadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWorkload");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteWorkloadResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteWorkloadResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteWorkloadResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWorkloadRequest, DeleteWorkloadResponse>()
                            .withOperationName("DeleteWorkload")
                            .withMarshaller(new DeleteWorkloadRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteWorkloadRequest));
            CompletableFuture<DeleteWorkloadResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete a workload share.
     * </p>
     *
     * @param deleteWorkloadShareRequest
     *        Input for Delete Workload Share
     * @return A Java Future containing the result of the DeleteWorkloadShare operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ConflictException The resource already exists.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.DeleteWorkloadShare
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/DeleteWorkloadShare"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWorkloadShareResponse> deleteWorkloadShare(
            DeleteWorkloadShareRequest deleteWorkloadShareRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWorkloadShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWorkloadShare");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteWorkloadShareResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteWorkloadShareResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteWorkloadShareResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWorkloadShareRequest, DeleteWorkloadShareResponse>()
                            .withOperationName("DeleteWorkloadShare")
                            .withMarshaller(new DeleteWorkloadShareRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteWorkloadShareRequest));
            CompletableFuture<DeleteWorkloadShareResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociate a lens from a workload.
     * </p>
     * <p>
     * Up to 10 lenses can be disassociated from a workload in a single API operation.
     * </p>
     * <note>
     * <p>
     * The Amazon Web Services Well-Architected Framework lens (<code>wellarchitected</code>) cannot be removed from a
     * workload.
     * </p>
     * </note>
     *
     * @param disassociateLensesRequest
     *        Input to disassociate lens reviews.
     * @return A Java Future containing the result of the DisassociateLenses operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ConflictException The resource already exists.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.DisassociateLenses
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/DisassociateLenses"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateLensesResponse> disassociateLenses(DisassociateLensesRequest disassociateLensesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateLensesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateLenses");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateLensesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateLensesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateLensesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateLensesRequest, DisassociateLensesResponse>()
                            .withOperationName("DisassociateLenses")
                            .withMarshaller(new DisassociateLensesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disassociateLensesRequest));
            CompletableFuture<DisassociateLensesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Export an existing lens.
     * </p>
     * <p>
     * Lenses are defined in JSON. For more information, see <a
     * href="https://docs.aws.amazon.com/wellarchitected/latest/userguide/lenses-format-specification.html">JSON format
     * specification</a> in the <i>Well-Architected Tool User Guide</i>. Only the owner of a lens can export it.
     * </p>
     * <note>
     * <p>
     * <b>Disclaimer</b>
     * </p>
     * <p>
     * Do not include or gather personal identifiable information (PII) of end users or other identifiable individuals
     * in or via your custom lenses. If your custom lens or those shared with you and used in your account do include or
     * collect PII you are responsible for: ensuring that the included PII is processed in accordance with applicable
     * law, providing adequate privacy notices, and obtaining necessary consents for processing such data.
     * </p>
     * </note>
     *
     * @param exportLensRequest
     * @return A Java Future containing the result of the ExportLens operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.ExportLens
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ExportLens" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ExportLensResponse> exportLens(ExportLensRequest exportLensRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportLensRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportLens");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportLensResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ExportLensResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExportLensResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportLensRequest, ExportLensResponse>().withOperationName("ExportLens")
                            .withMarshaller(new ExportLensRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(exportLensRequest));
            CompletableFuture<ExportLensResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the answer to a specific question in a workload review.
     * </p>
     *
     * @param getAnswerRequest
     *        Input to get answer.
     * @return A Java Future containing the result of the GetAnswer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.GetAnswer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/GetAnswer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetAnswerResponse> getAnswer(GetAnswerRequest getAnswerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAnswerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAnswer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAnswerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetAnswerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAnswerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAnswerRequest, GetAnswerResponse>().withOperationName("GetAnswer")
                            .withMarshaller(new GetAnswerRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAnswerRequest));
            CompletableFuture<GetAnswerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get an existing lens.
     * </p>
     *
     * @param getLensRequest
     * @return A Java Future containing the result of the GetLens operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.GetLens
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/GetLens" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetLensResponse> getLens(GetLensRequest getLensRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLensRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLens");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLensResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetLensResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLensResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLensRequest, GetLensResponse>().withOperationName("GetLens")
                            .withMarshaller(new GetLensRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getLensRequest));
            CompletableFuture<GetLensResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get lens review.
     * </p>
     *
     * @param getLensReviewRequest
     *        Input to get lens review.
     * @return A Java Future containing the result of the GetLensReview operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.GetLensReview
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/GetLensReview" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetLensReviewResponse> getLensReview(GetLensReviewRequest getLensReviewRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLensReviewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLensReview");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLensReviewResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetLensReviewResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLensReviewResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLensReviewRequest, GetLensReviewResponse>()
                            .withOperationName("GetLensReview")
                            .withMarshaller(new GetLensReviewRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getLensReviewRequest));
            CompletableFuture<GetLensReviewResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get lens review report.
     * </p>
     *
     * @param getLensReviewReportRequest
     *        Input to get lens review report.
     * @return A Java Future containing the result of the GetLensReviewReport operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.GetLensReviewReport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/GetLensReviewReport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetLensReviewReportResponse> getLensReviewReport(
            GetLensReviewReportRequest getLensReviewReportRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLensReviewReportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLensReviewReport");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLensReviewReportResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetLensReviewReportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLensReviewReportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLensReviewReportRequest, GetLensReviewReportResponse>()
                            .withOperationName("GetLensReviewReport")
                            .withMarshaller(new GetLensReviewReportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getLensReviewReportRequest));
            CompletableFuture<GetLensReviewReportResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get lens version differences.
     * </p>
     *
     * @param getLensVersionDifferenceRequest
     * @return A Java Future containing the result of the GetLensVersionDifference operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.GetLensVersionDifference
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/GetLensVersionDifference"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetLensVersionDifferenceResponse> getLensVersionDifference(
            GetLensVersionDifferenceRequest getLensVersionDifferenceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLensVersionDifferenceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLensVersionDifference");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLensVersionDifferenceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetLensVersionDifferenceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLensVersionDifferenceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLensVersionDifferenceRequest, GetLensVersionDifferenceResponse>()
                            .withOperationName("GetLensVersionDifference")
                            .withMarshaller(new GetLensVersionDifferenceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getLensVersionDifferenceRequest));
            CompletableFuture<GetLensVersionDifferenceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get a milestone for an existing workload.
     * </p>
     *
     * @param getMilestoneRequest
     *        Input to get a milestone.
     * @return A Java Future containing the result of the GetMilestone operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.GetMilestone
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/GetMilestone" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMilestoneResponse> getMilestone(GetMilestoneRequest getMilestoneRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMilestoneRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMilestone");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMilestoneResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetMilestoneResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMilestoneResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMilestoneRequest, GetMilestoneResponse>()
                            .withOperationName("GetMilestone").withMarshaller(new GetMilestoneRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getMilestoneRequest));
            CompletableFuture<GetMilestoneResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get an existing workload.
     * </p>
     *
     * @param getWorkloadRequest
     *        Input to get a workload.
     * @return A Java Future containing the result of the GetWorkload operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.GetWorkload
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/GetWorkload" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetWorkloadResponse> getWorkload(GetWorkloadRequest getWorkloadRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getWorkloadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetWorkload");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetWorkloadResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetWorkloadResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetWorkloadResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetWorkloadRequest, GetWorkloadResponse>()
                            .withOperationName("GetWorkload").withMarshaller(new GetWorkloadRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getWorkloadRequest));
            CompletableFuture<GetWorkloadResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Import a new lens.
     * </p>
     * <p>
     * The lens cannot be applied to workloads or shared with other Amazon Web Services accounts until it's published
     * with <a>CreateLensVersion</a>
     * </p>
     * <p>
     * Lenses are defined in JSON. For more information, see <a
     * href="https://docs.aws.amazon.com/wellarchitected/latest/userguide/lenses-format-specification.html">JSON format
     * specification</a> in the <i>Well-Architected Tool User Guide</i>.
     * </p>
     * <p>
     * A custom lens cannot exceed 500 KB in size.
     * </p>
     * <note>
     * <p>
     * <b>Disclaimer</b>
     * </p>
     * <p>
     * Do not include or gather personal identifiable information (PII) of end users or other identifiable individuals
     * in or via your custom lenses. If your custom lens or those shared with you and used in your account do include or
     * collect PII you are responsible for: ensuring that the included PII is processed in accordance with applicable
     * law, providing adequate privacy notices, and obtaining necessary consents for processing such data.
     * </p>
     * </note>
     *
     * @param importLensRequest
     * @return A Java Future containing the result of the ImportLens operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>ConflictException The resource already exists.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ServiceQuotaExceededException The user has reached their resource quota.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.ImportLens
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ImportLens" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ImportLensResponse> importLens(ImportLensRequest importLensRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, importLensRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportLens");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ImportLensResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ImportLensResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ImportLensResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ImportLensRequest, ImportLensResponse>().withOperationName("ImportLens")
                            .withMarshaller(new ImportLensRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(importLensRequest));
            CompletableFuture<ImportLensResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List of answers.
     * </p>
     *
     * @param listAnswersRequest
     *        Input to list answers.
     * @return A Java Future containing the result of the ListAnswers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.ListAnswers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListAnswers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAnswersResponse> listAnswers(ListAnswersRequest listAnswersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAnswersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAnswers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAnswersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAnswersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAnswersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAnswersRequest, ListAnswersResponse>()
                            .withOperationName("ListAnswers").withMarshaller(new ListAnswersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAnswersRequest));
            CompletableFuture<ListAnswersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List of answers.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAnswers(software.amazon.awssdk.services.wellarchitected.model.ListAnswersRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListAnswersPublisher publisher = client.listAnswersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListAnswersPublisher publisher = client.listAnswersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.wellarchitected.model.ListAnswersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.wellarchitected.model.ListAnswersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAnswers(software.amazon.awssdk.services.wellarchitected.model.ListAnswersRequest)} operation.</b>
     * </p>
     *
     * @param listAnswersRequest
     *        Input to list answers.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.ListAnswers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListAnswers" target="_top">AWS
     *      API Documentation</a>
     */
    public ListAnswersPublisher listAnswersPaginator(ListAnswersRequest listAnswersRequest) {
        return new ListAnswersPublisher(this, applyPaginatorUserAgent(listAnswersRequest));
    }

    /**
     * <p>
     * List lens review improvements.
     * </p>
     *
     * @param listLensReviewImprovementsRequest
     *        Input to list lens review improvements.
     * @return A Java Future containing the result of the ListLensReviewImprovements operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.ListLensReviewImprovements
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListLensReviewImprovements"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLensReviewImprovementsResponse> listLensReviewImprovements(
            ListLensReviewImprovementsRequest listLensReviewImprovementsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLensReviewImprovementsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLensReviewImprovements");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLensReviewImprovementsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListLensReviewImprovementsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLensReviewImprovementsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLensReviewImprovementsRequest, ListLensReviewImprovementsResponse>()
                            .withOperationName("ListLensReviewImprovements")
                            .withMarshaller(new ListLensReviewImprovementsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLensReviewImprovementsRequest));
            CompletableFuture<ListLensReviewImprovementsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List lens review improvements.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listLensReviewImprovements(software.amazon.awssdk.services.wellarchitected.model.ListLensReviewImprovementsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListLensReviewImprovementsPublisher publisher = client.listLensReviewImprovementsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListLensReviewImprovementsPublisher publisher = client.listLensReviewImprovementsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.wellarchitected.model.ListLensReviewImprovementsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.wellarchitected.model.ListLensReviewImprovementsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLensReviewImprovements(software.amazon.awssdk.services.wellarchitected.model.ListLensReviewImprovementsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listLensReviewImprovementsRequest
     *        Input to list lens review improvements.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.ListLensReviewImprovements
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListLensReviewImprovements"
     *      target="_top">AWS API Documentation</a>
     */
    public ListLensReviewImprovementsPublisher listLensReviewImprovementsPaginator(
            ListLensReviewImprovementsRequest listLensReviewImprovementsRequest) {
        return new ListLensReviewImprovementsPublisher(this, applyPaginatorUserAgent(listLensReviewImprovementsRequest));
    }

    /**
     * <p>
     * List lens reviews.
     * </p>
     *
     * @param listLensReviewsRequest
     *        Input to list lens reviews.
     * @return A Java Future containing the result of the ListLensReviews operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.ListLensReviews
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListLensReviews"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLensReviewsResponse> listLensReviews(ListLensReviewsRequest listLensReviewsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLensReviewsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLensReviews");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLensReviewsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListLensReviewsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLensReviewsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLensReviewsRequest, ListLensReviewsResponse>()
                            .withOperationName("ListLensReviews")
                            .withMarshaller(new ListLensReviewsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLensReviewsRequest));
            CompletableFuture<ListLensReviewsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List lens reviews.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listLensReviews(software.amazon.awssdk.services.wellarchitected.model.ListLensReviewsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListLensReviewsPublisher publisher = client.listLensReviewsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListLensReviewsPublisher publisher = client.listLensReviewsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.wellarchitected.model.ListLensReviewsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.wellarchitected.model.ListLensReviewsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLensReviews(software.amazon.awssdk.services.wellarchitected.model.ListLensReviewsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listLensReviewsRequest
     *        Input to list lens reviews.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.ListLensReviews
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListLensReviews"
     *      target="_top">AWS API Documentation</a>
     */
    public ListLensReviewsPublisher listLensReviewsPaginator(ListLensReviewsRequest listLensReviewsRequest) {
        return new ListLensReviewsPublisher(this, applyPaginatorUserAgent(listLensReviewsRequest));
    }

    /**
     * <p>
     * List the lens shares associated with the lens.
     * </p>
     *
     * @param listLensSharesRequest
     * @return A Java Future containing the result of the ListLensShares operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.ListLensShares
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListLensShares"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLensSharesResponse> listLensShares(ListLensSharesRequest listLensSharesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLensSharesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLensShares");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLensSharesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListLensSharesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLensSharesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLensSharesRequest, ListLensSharesResponse>()
                            .withOperationName("ListLensShares")
                            .withMarshaller(new ListLensSharesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLensSharesRequest));
            CompletableFuture<ListLensSharesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the lens shares associated with the lens.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listLensShares(software.amazon.awssdk.services.wellarchitected.model.ListLensSharesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListLensSharesPublisher publisher = client.listLensSharesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListLensSharesPublisher publisher = client.listLensSharesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.wellarchitected.model.ListLensSharesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.wellarchitected.model.ListLensSharesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLensShares(software.amazon.awssdk.services.wellarchitected.model.ListLensSharesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listLensSharesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.ListLensShares
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListLensShares"
     *      target="_top">AWS API Documentation</a>
     */
    public ListLensSharesPublisher listLensSharesPaginator(ListLensSharesRequest listLensSharesRequest) {
        return new ListLensSharesPublisher(this, applyPaginatorUserAgent(listLensSharesRequest));
    }

    /**
     * <p>
     * List the available lenses.
     * </p>
     *
     * @param listLensesRequest
     *        Input to list lenses.
     * @return A Java Future containing the result of the ListLenses operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.ListLenses
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListLenses" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLensesResponse> listLenses(ListLensesRequest listLensesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLensesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLenses");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLensesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListLensesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLensesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLensesRequest, ListLensesResponse>().withOperationName("ListLenses")
                            .withMarshaller(new ListLensesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLensesRequest));
            CompletableFuture<ListLensesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the available lenses.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listLenses(software.amazon.awssdk.services.wellarchitected.model.ListLensesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListLensesPublisher publisher = client.listLensesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListLensesPublisher publisher = client.listLensesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.wellarchitected.model.ListLensesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.wellarchitected.model.ListLensesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLenses(software.amazon.awssdk.services.wellarchitected.model.ListLensesRequest)} operation.</b>
     * </p>
     *
     * @param listLensesRequest
     *        Input to list lenses.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.ListLenses
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListLenses" target="_top">AWS
     *      API Documentation</a>
     */
    public ListLensesPublisher listLensesPaginator(ListLensesRequest listLensesRequest) {
        return new ListLensesPublisher(this, applyPaginatorUserAgent(listLensesRequest));
    }

    /**
     * <p>
     * List all milestones for an existing workload.
     * </p>
     *
     * @param listMilestonesRequest
     *        Input to list all milestones for a workload.
     * @return A Java Future containing the result of the ListMilestones operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.ListMilestones
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListMilestones"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListMilestonesResponse> listMilestones(ListMilestonesRequest listMilestonesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMilestonesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMilestones");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMilestonesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListMilestonesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMilestonesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMilestonesRequest, ListMilestonesResponse>()
                            .withOperationName("ListMilestones")
                            .withMarshaller(new ListMilestonesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listMilestonesRequest));
            CompletableFuture<ListMilestonesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all milestones for an existing workload.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listMilestones(software.amazon.awssdk.services.wellarchitected.model.ListMilestonesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListMilestonesPublisher publisher = client.listMilestonesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListMilestonesPublisher publisher = client.listMilestonesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.wellarchitected.model.ListMilestonesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.wellarchitected.model.ListMilestonesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMilestones(software.amazon.awssdk.services.wellarchitected.model.ListMilestonesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listMilestonesRequest
     *        Input to list all milestones for a workload.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.ListMilestones
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListMilestones"
     *      target="_top">AWS API Documentation</a>
     */
    public ListMilestonesPublisher listMilestonesPaginator(ListMilestonesRequest listMilestonesRequest) {
        return new ListMilestonesPublisher(this, applyPaginatorUserAgent(listMilestonesRequest));
    }

    /**
     * <p>
     * List lens notifications.
     * </p>
     *
     * @param listNotificationsRequest
     * @return A Java Future containing the result of the ListNotifications operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.ListNotifications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListNotifications"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListNotificationsResponse> listNotifications(ListNotificationsRequest listNotificationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNotificationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNotifications");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListNotificationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListNotificationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListNotificationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListNotificationsRequest, ListNotificationsResponse>()
                            .withOperationName("ListNotifications")
                            .withMarshaller(new ListNotificationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listNotificationsRequest));
            CompletableFuture<ListNotificationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List lens notifications.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listNotifications(software.amazon.awssdk.services.wellarchitected.model.ListNotificationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListNotificationsPublisher publisher = client.listNotificationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListNotificationsPublisher publisher = client.listNotificationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.wellarchitected.model.ListNotificationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.wellarchitected.model.ListNotificationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNotifications(software.amazon.awssdk.services.wellarchitected.model.ListNotificationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listNotificationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.ListNotifications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListNotifications"
     *      target="_top">AWS API Documentation</a>
     */
    public ListNotificationsPublisher listNotificationsPaginator(ListNotificationsRequest listNotificationsRequest) {
        return new ListNotificationsPublisher(this, applyPaginatorUserAgent(listNotificationsRequest));
    }

    /**
     * <p>
     * List the workload invitations.
     * </p>
     *
     * @param listShareInvitationsRequest
     *        Input for List Share Invitations
     * @return A Java Future containing the result of the ListShareInvitations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.ListShareInvitations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListShareInvitations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListShareInvitationsResponse> listShareInvitations(
            ListShareInvitationsRequest listShareInvitationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listShareInvitationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListShareInvitations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListShareInvitationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListShareInvitationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListShareInvitationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListShareInvitationsRequest, ListShareInvitationsResponse>()
                            .withOperationName("ListShareInvitations")
                            .withMarshaller(new ListShareInvitationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listShareInvitationsRequest));
            CompletableFuture<ListShareInvitationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the workload invitations.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listShareInvitations(software.amazon.awssdk.services.wellarchitected.model.ListShareInvitationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListShareInvitationsPublisher publisher = client.listShareInvitationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListShareInvitationsPublisher publisher = client.listShareInvitationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.wellarchitected.model.ListShareInvitationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.wellarchitected.model.ListShareInvitationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listShareInvitations(software.amazon.awssdk.services.wellarchitected.model.ListShareInvitationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listShareInvitationsRequest
     *        Input for List Share Invitations
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.ListShareInvitations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListShareInvitations"
     *      target="_top">AWS API Documentation</a>
     */
    public ListShareInvitationsPublisher listShareInvitationsPaginator(ListShareInvitationsRequest listShareInvitationsRequest) {
        return new ListShareInvitationsPublisher(this, applyPaginatorUserAgent(listShareInvitationsRequest));
    }

    /**
     * <p>
     * List the tags for a resource.
     * </p>
     * <note>
     * <p>
     * The WorkloadArn parameter can be either a workload ARN or a custom lens ARN.
     * </p>
     * </note>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the workload shares associated with the workload.
     * </p>
     *
     * @param listWorkloadSharesRequest
     *        Input for List Workload Share
     * @return A Java Future containing the result of the ListWorkloadShares operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.ListWorkloadShares
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListWorkloadShares"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListWorkloadSharesResponse> listWorkloadShares(ListWorkloadSharesRequest listWorkloadSharesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWorkloadSharesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkloadShares");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWorkloadSharesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListWorkloadSharesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListWorkloadSharesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWorkloadSharesRequest, ListWorkloadSharesResponse>()
                            .withOperationName("ListWorkloadShares")
                            .withMarshaller(new ListWorkloadSharesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listWorkloadSharesRequest));
            CompletableFuture<ListWorkloadSharesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the workload shares associated with the workload.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listWorkloadShares(software.amazon.awssdk.services.wellarchitected.model.ListWorkloadSharesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListWorkloadSharesPublisher publisher = client.listWorkloadSharesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListWorkloadSharesPublisher publisher = client.listWorkloadSharesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.wellarchitected.model.ListWorkloadSharesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.wellarchitected.model.ListWorkloadSharesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listWorkloadShares(software.amazon.awssdk.services.wellarchitected.model.ListWorkloadSharesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listWorkloadSharesRequest
     *        Input for List Workload Share
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.ListWorkloadShares
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListWorkloadShares"
     *      target="_top">AWS API Documentation</a>
     */
    public ListWorkloadSharesPublisher listWorkloadSharesPaginator(ListWorkloadSharesRequest listWorkloadSharesRequest) {
        return new ListWorkloadSharesPublisher(this, applyPaginatorUserAgent(listWorkloadSharesRequest));
    }

    /**
     * <p>
     * List workloads. Paginated.
     * </p>
     *
     * @param listWorkloadsRequest
     *        Input to list all workloads.
     * @return A Java Future containing the result of the ListWorkloads operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.ListWorkloads
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListWorkloads" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListWorkloadsResponse> listWorkloads(ListWorkloadsRequest listWorkloadsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWorkloadsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkloads");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWorkloadsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListWorkloadsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListWorkloadsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWorkloadsRequest, ListWorkloadsResponse>()
                            .withOperationName("ListWorkloads")
                            .withMarshaller(new ListWorkloadsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listWorkloadsRequest));
            CompletableFuture<ListWorkloadsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List workloads. Paginated.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listWorkloads(software.amazon.awssdk.services.wellarchitected.model.ListWorkloadsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListWorkloadsPublisher publisher = client.listWorkloadsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.wellarchitected.paginators.ListWorkloadsPublisher publisher = client.listWorkloadsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.wellarchitected.model.ListWorkloadsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.wellarchitected.model.ListWorkloadsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listWorkloads(software.amazon.awssdk.services.wellarchitected.model.ListWorkloadsRequest)} operation.</b>
     * </p>
     *
     * @param listWorkloadsRequest
     *        Input to list all workloads.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.ListWorkloads
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/ListWorkloads" target="_top">AWS
     *      API Documentation</a>
     */
    public ListWorkloadsPublisher listWorkloadsPaginator(ListWorkloadsRequest listWorkloadsRequest) {
        return new ListWorkloadsPublisher(this, applyPaginatorUserAgent(listWorkloadsRequest));
    }

    /**
     * <p>
     * Adds one or more tags to the specified resource.
     * </p>
     * <note>
     * <p>
     * The WorkloadArn parameter can be either a workload ARN or a custom lens ARN.
     * </p>
     * </note>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes specified tags from a resource.
     * </p>
     * <note>
     * <p>
     * The WorkloadArn parameter can be either a workload ARN or a custom lens ARN.
     * </p>
     * </note>
     * <p>
     * To specify multiple tags, use separate <b>tagKeys</b> parameters, for example:
     * </p>
     * <p>
     * <code>DELETE /tags/WorkloadArn?tagKeys=key1&amp;tagKeys=key2</code>
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update the answer to a specific question in a workload review.
     * </p>
     *
     * @param updateAnswerRequest
     *        Input to update answer.
     * @return A Java Future containing the result of the UpdateAnswer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ConflictException The resource already exists.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.UpdateAnswer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/UpdateAnswer" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAnswerResponse> updateAnswer(UpdateAnswerRequest updateAnswerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAnswerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAnswer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAnswerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateAnswerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAnswerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAnswerRequest, UpdateAnswerResponse>()
                            .withOperationName("UpdateAnswer").withMarshaller(new UpdateAnswerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateAnswerRequest));
            CompletableFuture<UpdateAnswerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates whether the Amazon Web Services account is opted into organization sharing features.
     * </p>
     *
     * @param updateGlobalSettingsRequest
     * @return A Java Future containing the result of the UpdateGlobalSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>ConflictException The resource already exists.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.UpdateGlobalSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/UpdateGlobalSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateGlobalSettingsResponse> updateGlobalSettings(
            UpdateGlobalSettingsRequest updateGlobalSettingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGlobalSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGlobalSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateGlobalSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateGlobalSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateGlobalSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateGlobalSettingsRequest, UpdateGlobalSettingsResponse>()
                            .withOperationName("UpdateGlobalSettings")
                            .withMarshaller(new UpdateGlobalSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateGlobalSettingsRequest));
            CompletableFuture<UpdateGlobalSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update lens review.
     * </p>
     *
     * @param updateLensReviewRequest
     *        Input for update lens review.
     * @return A Java Future containing the result of the UpdateLensReview operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ConflictException The resource already exists.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.UpdateLensReview
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/UpdateLensReview"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLensReviewResponse> updateLensReview(UpdateLensReviewRequest updateLensReviewRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLensReviewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLensReview");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLensReviewResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateLensReviewResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateLensReviewResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLensReviewRequest, UpdateLensReviewResponse>()
                            .withOperationName("UpdateLensReview")
                            .withMarshaller(new UpdateLensReviewRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateLensReviewRequest));
            CompletableFuture<UpdateLensReviewResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update a workload invitation.
     * </p>
     *
     * @param updateShareInvitationRequest
     *        Input for Update Share Invitation
     * @return A Java Future containing the result of the UpdateShareInvitation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ConflictException The resource already exists.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.UpdateShareInvitation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/UpdateShareInvitation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateShareInvitationResponse> updateShareInvitation(
            UpdateShareInvitationRequest updateShareInvitationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateShareInvitationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateShareInvitation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateShareInvitationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateShareInvitationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateShareInvitationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateShareInvitationRequest, UpdateShareInvitationResponse>()
                            .withOperationName("UpdateShareInvitation")
                            .withMarshaller(new UpdateShareInvitationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateShareInvitationRequest));
            CompletableFuture<UpdateShareInvitationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update an existing workload.
     * </p>
     *
     * @param updateWorkloadRequest
     *        Input to update a workload.
     * @return A Java Future containing the result of the UpdateWorkload operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ConflictException The resource already exists.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.UpdateWorkload
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/UpdateWorkload"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateWorkloadResponse> updateWorkload(UpdateWorkloadRequest updateWorkloadRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateWorkloadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateWorkload");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateWorkloadResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateWorkloadResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateWorkloadResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateWorkloadRequest, UpdateWorkloadResponse>()
                            .withOperationName("UpdateWorkload")
                            .withMarshaller(new UpdateWorkloadRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateWorkloadRequest));
            CompletableFuture<UpdateWorkloadResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update a workload share.
     * </p>
     *
     * @param updateWorkloadShareRequest
     *        Input for Update Workload Share
     * @return A Java Future containing the result of the UpdateWorkloadShare operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ConflictException The resource already exists.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.UpdateWorkloadShare
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/UpdateWorkloadShare"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateWorkloadShareResponse> updateWorkloadShare(
            UpdateWorkloadShareRequest updateWorkloadShareRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateWorkloadShareRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateWorkloadShare");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateWorkloadShareResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateWorkloadShareResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateWorkloadShareResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateWorkloadShareRequest, UpdateWorkloadShareResponse>()
                            .withOperationName("UpdateWorkloadShare")
                            .withMarshaller(new UpdateWorkloadShareRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateWorkloadShareRequest));
            CompletableFuture<UpdateWorkloadShareResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Upgrade lens review.
     * </p>
     *
     * @param upgradeLensReviewRequest
     * @return A Java Future containing the result of the UpgradeLensReview operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The user input is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ConflictException The resource already exists.</li>
     *         <li>InternalServerException There is a problem with the Well-Architected Tool API service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WellArchitectedException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample WellArchitectedAsyncClient.UpgradeLensReview
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wellarchitected-2020-03-31/UpgradeLensReview"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpgradeLensReviewResponse> upgradeLensReview(UpgradeLensReviewRequest upgradeLensReviewRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, upgradeLensReviewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WellArchitected");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpgradeLensReview");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpgradeLensReviewResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpgradeLensReviewResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpgradeLensReviewResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpgradeLensReviewRequest, UpgradeLensReviewResponse>()
                            .withOperationName("UpgradeLensReview")
                            .withMarshaller(new UpgradeLensReviewRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(upgradeLensReviewRequest));
            CompletableFuture<UpgradeLensReviewResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(WellArchitectedException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .contentType("application/json")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends WellArchitectedRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
