/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wisdom;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.wisdom.model.AccessDeniedException;
import software.amazon.awssdk.services.wisdom.model.ConflictException;
import software.amazon.awssdk.services.wisdom.model.CreateAssistantAssociationRequest;
import software.amazon.awssdk.services.wisdom.model.CreateAssistantAssociationResponse;
import software.amazon.awssdk.services.wisdom.model.CreateAssistantRequest;
import software.amazon.awssdk.services.wisdom.model.CreateAssistantResponse;
import software.amazon.awssdk.services.wisdom.model.CreateContentRequest;
import software.amazon.awssdk.services.wisdom.model.CreateContentResponse;
import software.amazon.awssdk.services.wisdom.model.CreateKnowledgeBaseRequest;
import software.amazon.awssdk.services.wisdom.model.CreateKnowledgeBaseResponse;
import software.amazon.awssdk.services.wisdom.model.CreateSessionRequest;
import software.amazon.awssdk.services.wisdom.model.CreateSessionResponse;
import software.amazon.awssdk.services.wisdom.model.DeleteAssistantAssociationRequest;
import software.amazon.awssdk.services.wisdom.model.DeleteAssistantAssociationResponse;
import software.amazon.awssdk.services.wisdom.model.DeleteAssistantRequest;
import software.amazon.awssdk.services.wisdom.model.DeleteAssistantResponse;
import software.amazon.awssdk.services.wisdom.model.DeleteContentRequest;
import software.amazon.awssdk.services.wisdom.model.DeleteContentResponse;
import software.amazon.awssdk.services.wisdom.model.DeleteKnowledgeBaseRequest;
import software.amazon.awssdk.services.wisdom.model.DeleteKnowledgeBaseResponse;
import software.amazon.awssdk.services.wisdom.model.GetAssistantAssociationRequest;
import software.amazon.awssdk.services.wisdom.model.GetAssistantAssociationResponse;
import software.amazon.awssdk.services.wisdom.model.GetAssistantRequest;
import software.amazon.awssdk.services.wisdom.model.GetAssistantResponse;
import software.amazon.awssdk.services.wisdom.model.GetContentRequest;
import software.amazon.awssdk.services.wisdom.model.GetContentResponse;
import software.amazon.awssdk.services.wisdom.model.GetContentSummaryRequest;
import software.amazon.awssdk.services.wisdom.model.GetContentSummaryResponse;
import software.amazon.awssdk.services.wisdom.model.GetKnowledgeBaseRequest;
import software.amazon.awssdk.services.wisdom.model.GetKnowledgeBaseResponse;
import software.amazon.awssdk.services.wisdom.model.GetRecommendationsRequest;
import software.amazon.awssdk.services.wisdom.model.GetRecommendationsResponse;
import software.amazon.awssdk.services.wisdom.model.GetSessionRequest;
import software.amazon.awssdk.services.wisdom.model.GetSessionResponse;
import software.amazon.awssdk.services.wisdom.model.ListAssistantAssociationsRequest;
import software.amazon.awssdk.services.wisdom.model.ListAssistantAssociationsResponse;
import software.amazon.awssdk.services.wisdom.model.ListAssistantsRequest;
import software.amazon.awssdk.services.wisdom.model.ListAssistantsResponse;
import software.amazon.awssdk.services.wisdom.model.ListContentsRequest;
import software.amazon.awssdk.services.wisdom.model.ListContentsResponse;
import software.amazon.awssdk.services.wisdom.model.ListKnowledgeBasesRequest;
import software.amazon.awssdk.services.wisdom.model.ListKnowledgeBasesResponse;
import software.amazon.awssdk.services.wisdom.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.wisdom.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.wisdom.model.NotifyRecommendationsReceivedRequest;
import software.amazon.awssdk.services.wisdom.model.NotifyRecommendationsReceivedResponse;
import software.amazon.awssdk.services.wisdom.model.PreconditionFailedException;
import software.amazon.awssdk.services.wisdom.model.QueryAssistantRequest;
import software.amazon.awssdk.services.wisdom.model.QueryAssistantResponse;
import software.amazon.awssdk.services.wisdom.model.RemoveKnowledgeBaseTemplateUriRequest;
import software.amazon.awssdk.services.wisdom.model.RemoveKnowledgeBaseTemplateUriResponse;
import software.amazon.awssdk.services.wisdom.model.ResourceNotFoundException;
import software.amazon.awssdk.services.wisdom.model.SearchContentRequest;
import software.amazon.awssdk.services.wisdom.model.SearchContentResponse;
import software.amazon.awssdk.services.wisdom.model.SearchSessionsRequest;
import software.amazon.awssdk.services.wisdom.model.SearchSessionsResponse;
import software.amazon.awssdk.services.wisdom.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.wisdom.model.StartContentUploadRequest;
import software.amazon.awssdk.services.wisdom.model.StartContentUploadResponse;
import software.amazon.awssdk.services.wisdom.model.TagResourceRequest;
import software.amazon.awssdk.services.wisdom.model.TagResourceResponse;
import software.amazon.awssdk.services.wisdom.model.TooManyTagsException;
import software.amazon.awssdk.services.wisdom.model.UntagResourceRequest;
import software.amazon.awssdk.services.wisdom.model.UntagResourceResponse;
import software.amazon.awssdk.services.wisdom.model.UpdateContentRequest;
import software.amazon.awssdk.services.wisdom.model.UpdateContentResponse;
import software.amazon.awssdk.services.wisdom.model.UpdateKnowledgeBaseTemplateUriRequest;
import software.amazon.awssdk.services.wisdom.model.UpdateKnowledgeBaseTemplateUriResponse;
import software.amazon.awssdk.services.wisdom.model.ValidationException;
import software.amazon.awssdk.services.wisdom.model.WisdomException;
import software.amazon.awssdk.services.wisdom.transform.CreateAssistantAssociationRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.CreateAssistantRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.CreateContentRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.CreateKnowledgeBaseRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.CreateSessionRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.DeleteAssistantAssociationRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.DeleteAssistantRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.DeleteContentRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.DeleteKnowledgeBaseRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.GetAssistantAssociationRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.GetAssistantRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.GetContentRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.GetContentSummaryRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.GetKnowledgeBaseRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.GetRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.GetSessionRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.ListAssistantAssociationsRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.ListAssistantsRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.ListContentsRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.ListKnowledgeBasesRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.NotifyRecommendationsReceivedRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.QueryAssistantRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.RemoveKnowledgeBaseTemplateUriRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.SearchContentRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.SearchSessionsRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.StartContentUploadRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.UpdateContentRequestMarshaller;
import software.amazon.awssdk.services.wisdom.transform.UpdateKnowledgeBaseTemplateUriRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link WisdomClient}.
 *
 * @see WisdomClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultWisdomClient implements WisdomClient {
    private static final Logger log = Logger.loggerFor(DefaultWisdomClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final WisdomServiceClientConfiguration serviceClientConfiguration;

    protected DefaultWisdomClient(WisdomServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an Amazon Connect Wisdom assistant.
     * </p>
     *
     * @param createAssistantRequest
     * @return Result of the CreateAssistant operation returned by the service.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource. For example,
     *         if you're using a <code>Create</code> API (such as <code>CreateAssistant</code>) that accepts name, a
     *         conflicting resource (usually with the same name) is being created or mutated.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws ServiceQuotaExceededException
     *         You've exceeded your service quota. To perform the requested action, remove some of the relevant
     *         resources, or use service quotas to request a service quota increase.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.CreateAssistant
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/CreateAssistant" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateAssistantResponse createAssistant(CreateAssistantRequest createAssistantRequest) throws ConflictException,
            ValidationException, ServiceQuotaExceededException, AccessDeniedException, AwsServiceException, SdkClientException,
            WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAssistantResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateAssistantResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAssistantRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAssistant");

            return clientHandler.execute(new ClientExecutionParams<CreateAssistantRequest, CreateAssistantResponse>()
                    .withOperationName("CreateAssistant").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createAssistantRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateAssistantRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an association between an Amazon Connect Wisdom assistant and another resource. Currently, the only
     * supported association is with a knowledge base. An assistant can have only a single association.
     * </p>
     *
     * @param createAssistantAssociationRequest
     * @return Result of the CreateAssistantAssociation operation returned by the service.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource. For example,
     *         if you're using a <code>Create</code> API (such as <code>CreateAssistant</code>) that accepts name, a
     *         conflicting resource (usually with the same name) is being created or mutated.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws ServiceQuotaExceededException
     *         You've exceeded your service quota. To perform the requested action, remove some of the relevant
     *         resources, or use service quotas to request a service quota increase.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.CreateAssistantAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/CreateAssistantAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateAssistantAssociationResponse createAssistantAssociation(
            CreateAssistantAssociationRequest createAssistantAssociationRequest) throws ConflictException, ValidationException,
            ServiceQuotaExceededException, AccessDeniedException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAssistantAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateAssistantAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAssistantAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAssistantAssociation");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateAssistantAssociationRequest, CreateAssistantAssociationResponse>()
                            .withOperationName("CreateAssistantAssociation").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createAssistantAssociationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateAssistantAssociationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates Wisdom content. Before to calling this API, use <a
     * href="https://docs.aws.amazon.com/wisdom/latest/APIReference/API_StartContentUpload.html">StartContentUpload</a>
     * to upload an asset.
     * </p>
     *
     * @param createContentRequest
     * @return Result of the CreateContent operation returned by the service.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource. For example,
     *         if you're using a <code>Create</code> API (such as <code>CreateAssistant</code>) that accepts name, a
     *         conflicting resource (usually with the same name) is being created or mutated.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws ServiceQuotaExceededException
     *         You've exceeded your service quota. To perform the requested action, remove some of the relevant
     *         resources, or use service quotas to request a service quota increase.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.CreateContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/CreateContent" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateContentResponse createContent(CreateContentRequest createContentRequest) throws ConflictException,
            ValidationException, ServiceQuotaExceededException, AccessDeniedException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateContentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateContentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createContentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateContent");

            return clientHandler.execute(new ClientExecutionParams<CreateContentRequest, CreateContentResponse>()
                    .withOperationName("CreateContent").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createContentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateContentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a knowledge base.
     * </p>
     * <note>
     * <p>
     * When using this API, you cannot reuse <a
     * href="https://docs.aws.amazon.com/appintegrations/latest/APIReference/Welcome.html">Amazon AppIntegrations</a>
     * DataIntegrations with external knowledge bases such as Salesforce and ServiceNow. If you do, you'll get an
     * <code>InvalidRequestException</code> error.
     * </p>
     * <p>
     * For example, you're programmatically managing your external knowledge base, and you want to add or remove one of
     * the fields that is being ingested from Salesforce. Do the following:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Call <a
     * href="https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteKnowledgeBase.html">DeleteKnowledgeBase
     * </a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Call <a href="https://docs.aws.amazon.com/appintegrations/latest/APIReference/API_DeleteDataIntegration.html">
     * DeleteDataIntegration</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Call <a href="https://docs.aws.amazon.com/appintegrations/latest/APIReference/API_CreateDataIntegration.html">
     * CreateDataIntegration</a> to recreate the DataIntegration or a create different one.
     * </p>
     * </li>
     * <li>
     * <p>
     * Call CreateKnowledgeBase.
     * </p>
     * </li>
     * </ol>
     * </note>
     *
     * @param createKnowledgeBaseRequest
     * @return Result of the CreateKnowledgeBase operation returned by the service.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource. For example,
     *         if you're using a <code>Create</code> API (such as <code>CreateAssistant</code>) that accepts name, a
     *         conflicting resource (usually with the same name) is being created or mutated.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws ServiceQuotaExceededException
     *         You've exceeded your service quota. To perform the requested action, remove some of the relevant
     *         resources, or use service quotas to request a service quota increase.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.CreateKnowledgeBase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/CreateKnowledgeBase" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateKnowledgeBaseResponse createKnowledgeBase(CreateKnowledgeBaseRequest createKnowledgeBaseRequest)
            throws ConflictException, ValidationException, ServiceQuotaExceededException, AccessDeniedException,
            AwsServiceException, SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateKnowledgeBaseResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateKnowledgeBaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createKnowledgeBaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateKnowledgeBase");

            return clientHandler.execute(new ClientExecutionParams<CreateKnowledgeBaseRequest, CreateKnowledgeBaseResponse>()
                    .withOperationName("CreateKnowledgeBase").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createKnowledgeBaseRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateKnowledgeBaseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a session. A session is a contextual container used for generating recommendations. Amazon Connect
     * creates a new Wisdom session for each contact on which Wisdom is enabled.
     * </p>
     *
     * @param createSessionRequest
     * @return Result of the CreateSession operation returned by the service.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource. For example,
     *         if you're using a <code>Create</code> API (such as <code>CreateAssistant</code>) that accepts name, a
     *         conflicting resource (usually with the same name) is being created or mutated.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.CreateSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/CreateSession" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateSessionResponse createSession(CreateSessionRequest createSessionRequest) throws ConflictException,
            ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateSessionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSession");

            return clientHandler.execute(new ClientExecutionParams<CreateSessionRequest, CreateSessionResponse>()
                    .withOperationName("CreateSession").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createSessionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateSessionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an assistant.
     * </p>
     *
     * @param deleteAssistantRequest
     * @return Result of the DeleteAssistant operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.DeleteAssistant
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/DeleteAssistant" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteAssistantResponse deleteAssistant(DeleteAssistantRequest deleteAssistantRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, AwsServiceException, SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAssistantResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteAssistantResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAssistantRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAssistant");

            return clientHandler.execute(new ClientExecutionParams<DeleteAssistantRequest, DeleteAssistantResponse>()
                    .withOperationName("DeleteAssistant").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteAssistantRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAssistantRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an assistant association.
     * </p>
     *
     * @param deleteAssistantAssociationRequest
     * @return Result of the DeleteAssistantAssociation operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.DeleteAssistantAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/DeleteAssistantAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteAssistantAssociationResponse deleteAssistantAssociation(
            DeleteAssistantAssociationRequest deleteAssistantAssociationRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, AwsServiceException, SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAssistantAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteAssistantAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAssistantAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAssistantAssociation");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteAssistantAssociationRequest, DeleteAssistantAssociationResponse>()
                            .withOperationName("DeleteAssistantAssociation").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteAssistantAssociationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteAssistantAssociationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the content.
     * </p>
     *
     * @param deleteContentRequest
     * @return Result of the DeleteContent operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.DeleteContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/DeleteContent" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteContentResponse deleteContent(DeleteContentRequest deleteContentRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, AwsServiceException, SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteContentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteContentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteContentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteContent");

            return clientHandler.execute(new ClientExecutionParams<DeleteContentRequest, DeleteContentResponse>()
                    .withOperationName("DeleteContent").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteContentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteContentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the knowledge base.
     * </p>
     * <note>
     * <p>
     * When you use this API to delete an external knowledge base such as Salesforce or ServiceNow, you must also delete
     * the <a href="https://docs.aws.amazon.com/appintegrations/latest/APIReference/Welcome.html">Amazon
     * AppIntegrations</a> DataIntegration. This is because you can't reuse the DataIntegration after it's been
     * associated with an external knowledge base. However, you can delete and recreate it. See <a
     * href="https://docs.aws.amazon.com/appintegrations/latest/APIReference/API_DeleteDataIntegration.html"
     * >DeleteDataIntegration</a> and <a
     * href="https://docs.aws.amazon.com/appintegrations/latest/APIReference/API_CreateDataIntegration.html"
     * >CreateDataIntegration</a> in the <i>Amazon AppIntegrations API Reference</i>.
     * </p>
     * </note>
     *
     * @param deleteKnowledgeBaseRequest
     * @return Result of the DeleteKnowledgeBase operation returned by the service.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource. For example,
     *         if you're using a <code>Create</code> API (such as <code>CreateAssistant</code>) that accepts name, a
     *         conflicting resource (usually with the same name) is being created or mutated.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.DeleteKnowledgeBase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/DeleteKnowledgeBase" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteKnowledgeBaseResponse deleteKnowledgeBase(DeleteKnowledgeBaseRequest deleteKnowledgeBaseRequest)
            throws ConflictException, ValidationException, AccessDeniedException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteKnowledgeBaseResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteKnowledgeBaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteKnowledgeBaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteKnowledgeBase");

            return clientHandler.execute(new ClientExecutionParams<DeleteKnowledgeBaseRequest, DeleteKnowledgeBaseResponse>()
                    .withOperationName("DeleteKnowledgeBase").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteKnowledgeBaseRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteKnowledgeBaseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about an assistant.
     * </p>
     *
     * @param getAssistantRequest
     * @return Result of the GetAssistant operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.GetAssistant
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/GetAssistant" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetAssistantResponse getAssistant(GetAssistantRequest getAssistantRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, AwsServiceException, SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAssistantResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAssistantResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAssistantRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAssistant");

            return clientHandler.execute(new ClientExecutionParams<GetAssistantRequest, GetAssistantResponse>()
                    .withOperationName("GetAssistant").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getAssistantRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAssistantRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about an assistant association.
     * </p>
     *
     * @param getAssistantAssociationRequest
     * @return Result of the GetAssistantAssociation operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.GetAssistantAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/GetAssistantAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAssistantAssociationResponse getAssistantAssociation(GetAssistantAssociationRequest getAssistantAssociationRequest)
            throws ValidationException, AccessDeniedException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAssistantAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetAssistantAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAssistantAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAssistantAssociation");

            return clientHandler
                    .execute(new ClientExecutionParams<GetAssistantAssociationRequest, GetAssistantAssociationResponse>()
                            .withOperationName("GetAssistantAssociation").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getAssistantAssociationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetAssistantAssociationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves content, including a pre-signed URL to download the content.
     * </p>
     *
     * @param getContentRequest
     * @return Result of the GetContent operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.GetContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/GetContent" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetContentResponse getContent(GetContentRequest getContentRequest) throws ValidationException, AccessDeniedException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetContentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetContentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getContentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetContent");

            return clientHandler
                    .execute(new ClientExecutionParams<GetContentRequest, GetContentResponse>().withOperationName("GetContent")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getContentRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetContentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves summary information about the content.
     * </p>
     *
     * @param getContentSummaryRequest
     * @return Result of the GetContentSummary operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.GetContentSummary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/GetContentSummary" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetContentSummaryResponse getContentSummary(GetContentSummaryRequest getContentSummaryRequest)
            throws ValidationException, AccessDeniedException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetContentSummaryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetContentSummaryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getContentSummaryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetContentSummary");

            return clientHandler.execute(new ClientExecutionParams<GetContentSummaryRequest, GetContentSummaryResponse>()
                    .withOperationName("GetContentSummary").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getContentSummaryRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetContentSummaryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about the knowledge base.
     * </p>
     *
     * @param getKnowledgeBaseRequest
     * @return Result of the GetKnowledgeBase operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.GetKnowledgeBase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/GetKnowledgeBase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetKnowledgeBaseResponse getKnowledgeBase(GetKnowledgeBaseRequest getKnowledgeBaseRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, AwsServiceException, SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetKnowledgeBaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetKnowledgeBaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getKnowledgeBaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetKnowledgeBase");

            return clientHandler.execute(new ClientExecutionParams<GetKnowledgeBaseRequest, GetKnowledgeBaseResponse>()
                    .withOperationName("GetKnowledgeBase").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getKnowledgeBaseRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetKnowledgeBaseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves recommendations for the specified session. To avoid retrieving the same recommendations in subsequent
     * calls, use <a
     * href="https://docs.aws.amazon.com/wisdom/latest/APIReference/API_NotifyRecommendationsReceived.html"
     * >NotifyRecommendationsReceived</a>. This API supports long-polling behavior with the <code>waitTimeSeconds</code>
     * parameter. Short poll is the default behavior and only returns recommendations already available. To perform a
     * manual query against an assistant, use <a
     * href="https://docs.aws.amazon.com/wisdom/latest/APIReference/API_QueryAssistant.html">QueryAssistant</a>.
     * </p>
     *
     * @param getRecommendationsRequest
     * @return Result of the GetRecommendations operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.GetRecommendations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/GetRecommendations" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetRecommendationsResponse getRecommendations(GetRecommendationsRequest getRecommendationsRequest)
            throws ValidationException, AccessDeniedException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetRecommendationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRecommendationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRecommendations");

            return clientHandler.execute(new ClientExecutionParams<GetRecommendationsRequest, GetRecommendationsResponse>()
                    .withOperationName("GetRecommendations").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getRecommendationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetRecommendationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information for a specified session.
     * </p>
     *
     * @param getSessionRequest
     * @return Result of the GetSession operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.GetSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/GetSession" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSessionResponse getSession(GetSessionRequest getSessionRequest) throws ValidationException, AccessDeniedException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSessionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSession");

            return clientHandler
                    .execute(new ClientExecutionParams<GetSessionRequest, GetSessionResponse>().withOperationName("GetSession")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getSessionRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetSessionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists information about assistant associations.
     * </p>
     *
     * @param listAssistantAssociationsRequest
     * @return Result of the ListAssistantAssociations operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.ListAssistantAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/ListAssistantAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAssistantAssociationsResponse listAssistantAssociations(
            ListAssistantAssociationsRequest listAssistantAssociationsRequest) throws ValidationException, AccessDeniedException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAssistantAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAssistantAssociationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAssistantAssociationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAssistantAssociations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListAssistantAssociationsRequest, ListAssistantAssociationsResponse>()
                            .withOperationName("ListAssistantAssociations").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listAssistantAssociationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListAssistantAssociationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists information about assistants.
     * </p>
     *
     * @param listAssistantsRequest
     * @return Result of the ListAssistants operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.ListAssistants
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/ListAssistants" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAssistantsResponse listAssistants(ListAssistantsRequest listAssistantsRequest) throws ValidationException,
            AccessDeniedException, AwsServiceException, SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAssistantsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAssistantsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAssistantsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAssistants");

            return clientHandler.execute(new ClientExecutionParams<ListAssistantsRequest, ListAssistantsResponse>()
                    .withOperationName("ListAssistants").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listAssistantsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAssistantsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the content.
     * </p>
     *
     * @param listContentsRequest
     * @return Result of the ListContents operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.ListContents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/ListContents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListContentsResponse listContents(ListContentsRequest listContentsRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, AwsServiceException, SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListContentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListContentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listContentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListContents");

            return clientHandler.execute(new ClientExecutionParams<ListContentsRequest, ListContentsResponse>()
                    .withOperationName("ListContents").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listContentsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListContentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the knowledge bases.
     * </p>
     *
     * @param listKnowledgeBasesRequest
     * @return Result of the ListKnowledgeBases operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.ListKnowledgeBases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/ListKnowledgeBases" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListKnowledgeBasesResponse listKnowledgeBases(ListKnowledgeBasesRequest listKnowledgeBasesRequest)
            throws ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListKnowledgeBasesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListKnowledgeBasesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listKnowledgeBasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListKnowledgeBases");

            return clientHandler.execute(new ClientExecutionParams<ListKnowledgeBasesRequest, ListKnowledgeBasesResponse>()
                    .withOperationName("ListKnowledgeBases").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listKnowledgeBasesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListKnowledgeBasesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags for the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, AwsServiceException, SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the specified recommendations from the specified assistant's queue of newly available recommendations.
     * You can use this API in conjunction with <a
     * href="https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetRecommendations.html">GetRecommendations</a>
     * and a <code>waitTimeSeconds</code> input for long-polling behavior and avoiding duplicate recommendations.
     * </p>
     *
     * @param notifyRecommendationsReceivedRequest
     * @return Result of the NotifyRecommendationsReceived operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.NotifyRecommendationsReceived
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/NotifyRecommendationsReceived"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public NotifyRecommendationsReceivedResponse notifyRecommendationsReceived(
            NotifyRecommendationsReceivedRequest notifyRecommendationsReceivedRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, AwsServiceException, SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<NotifyRecommendationsReceivedResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, NotifyRecommendationsReceivedResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                notifyRecommendationsReceivedRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "NotifyRecommendationsReceived");

            return clientHandler
                    .execute(new ClientExecutionParams<NotifyRecommendationsReceivedRequest, NotifyRecommendationsReceivedResponse>()
                            .withOperationName("NotifyRecommendationsReceived").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(notifyRecommendationsReceivedRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new NotifyRecommendationsReceivedRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Performs a manual search against the specified assistant. To retrieve recommendations for an assistant, use <a
     * href="https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetRecommendations.html">GetRecommendations</a>.
     * </p>
     *
     * @param queryAssistantRequest
     * @return Result of the QueryAssistant operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.QueryAssistant
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/QueryAssistant" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public QueryAssistantResponse queryAssistant(QueryAssistantRequest queryAssistantRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, AwsServiceException, SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<QueryAssistantResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                QueryAssistantResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, queryAssistantRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "QueryAssistant");

            return clientHandler.execute(new ClientExecutionParams<QueryAssistantRequest, QueryAssistantResponse>()
                    .withOperationName("QueryAssistant").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(queryAssistantRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new QueryAssistantRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a URI template from a knowledge base.
     * </p>
     *
     * @param removeKnowledgeBaseTemplateUriRequest
     * @return Result of the RemoveKnowledgeBaseTemplateUri operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.RemoveKnowledgeBaseTemplateUri
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/RemoveKnowledgeBaseTemplateUri"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RemoveKnowledgeBaseTemplateUriResponse removeKnowledgeBaseTemplateUri(
            RemoveKnowledgeBaseTemplateUriRequest removeKnowledgeBaseTemplateUriRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, AwsServiceException, SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemoveKnowledgeBaseTemplateUriResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RemoveKnowledgeBaseTemplateUriResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                removeKnowledgeBaseTemplateUriRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveKnowledgeBaseTemplateUri");

            return clientHandler
                    .execute(new ClientExecutionParams<RemoveKnowledgeBaseTemplateUriRequest, RemoveKnowledgeBaseTemplateUriResponse>()
                            .withOperationName("RemoveKnowledgeBaseTemplateUri").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(removeKnowledgeBaseTemplateUriRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RemoveKnowledgeBaseTemplateUriRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Searches for content in a specified knowledge base. Can be used to get a specific content resource by its name.
     * </p>
     *
     * @param searchContentRequest
     * @return Result of the SearchContent operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.SearchContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/SearchContent" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SearchContentResponse searchContent(SearchContentRequest searchContentRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, AwsServiceException, SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SearchContentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SearchContentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchContentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchContent");

            return clientHandler.execute(new ClientExecutionParams<SearchContentRequest, SearchContentResponse>()
                    .withOperationName("SearchContent").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(searchContentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SearchContentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Searches for sessions.
     * </p>
     *
     * @param searchSessionsRequest
     * @return Result of the SearchSessions operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.SearchSessions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/SearchSessions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SearchSessionsResponse searchSessions(SearchSessionsRequest searchSessionsRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, AwsServiceException, SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SearchSessionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SearchSessionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchSessionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchSessions");

            return clientHandler.execute(new ClientExecutionParams<SearchSessionsRequest, SearchSessionsResponse>()
                    .withOperationName("SearchSessions").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(searchSessionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SearchSessionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get a URL to upload content to a knowledge base. To upload content, first make a PUT request to the returned URL
     * with your file, making sure to include the required headers. Then use <a
     * href="https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateContent.html">CreateContent</a> to
     * finalize the content creation process or <a
     * href="https://docs.aws.amazon.com/wisdom/latest/APIReference/API_UpdateContent.html">UpdateContent</a> to modify
     * an existing resource. You can only upload content to a knowledge base of type CUSTOM.
     * </p>
     *
     * @param startContentUploadRequest
     * @return Result of the StartContentUpload operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.StartContentUpload
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/StartContentUpload" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartContentUploadResponse startContentUpload(StartContentUploadRequest startContentUploadRequest)
            throws ValidationException, AccessDeniedException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartContentUploadResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartContentUploadResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startContentUploadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartContentUpload");

            return clientHandler.execute(new ClientExecutionParams<StartContentUploadRequest, StartContentUploadResponse>()
                    .withOperationName("StartContentUpload").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startContentUploadRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartContentUploadRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds the specified tags to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws TooManyTagsException
     *         Amazon Connect Wisdom throws this exception if you have too many tags in your tag set.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws TooManyTagsException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the specified tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            AwsServiceException, SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates information about the content.
     * </p>
     *
     * @param updateContentRequest
     * @return Result of the UpdateContent operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws PreconditionFailedException
     *         The provided <code>revisionId</code> does not match, indicating the content has been modified since it
     *         was last read.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.UpdateContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/UpdateContent" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateContentResponse updateContent(UpdateContentRequest updateContentRequest) throws ValidationException,
            AccessDeniedException, PreconditionFailedException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateContentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateContentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateContentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateContent");

            return clientHandler.execute(new ClientExecutionParams<UpdateContentRequest, UpdateContentResponse>()
                    .withOperationName("UpdateContent").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateContentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateContentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the template URI of a knowledge base. This is only supported for knowledge bases of type EXTERNAL.
     * Include a single variable in <code>&#36{variable}</code> format; this interpolated by Wisdom using ingested
     * content. For example, if you ingest a Salesforce article, it has an <code>Id</code> value, and you can set the
     * template URI to
     * <code>https://myInstanceName.lightning.force.com/lightning/r/Knowledge__kav/*&#36{Id}*&#47;view</code>.
     * </p>
     *
     * @param updateKnowledgeBaseTemplateUriRequest
     * @return Result of the UpdateKnowledgeBaseTemplateUri operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WisdomException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WisdomClient.UpdateKnowledgeBaseTemplateUri
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/wisdom-2020-10-19/UpdateKnowledgeBaseTemplateUri"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateKnowledgeBaseTemplateUriResponse updateKnowledgeBaseTemplateUri(
            UpdateKnowledgeBaseTemplateUriRequest updateKnowledgeBaseTemplateUriRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, AwsServiceException, SdkClientException, WisdomException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateKnowledgeBaseTemplateUriResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateKnowledgeBaseTemplateUriResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateKnowledgeBaseTemplateUriRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Wisdom");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateKnowledgeBaseTemplateUri");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateKnowledgeBaseTemplateUriRequest, UpdateKnowledgeBaseTemplateUriResponse>()
                            .withOperationName("UpdateKnowledgeBaseTemplateUri").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateKnowledgeBaseTemplateUriRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateKnowledgeBaseTemplateUriRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(WisdomException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PreconditionFailedException")
                                .exceptionBuilderSupplier(PreconditionFailedException::builder).httpStatusCode(412).build());
    }

    @Override
    public final WisdomServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
