/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.worklink;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.worklink.internal.WorkLinkServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.worklink.model.AssociateDomainRequest;
import software.amazon.awssdk.services.worklink.model.AssociateDomainResponse;
import software.amazon.awssdk.services.worklink.model.AssociateWebsiteAuthorizationProviderRequest;
import software.amazon.awssdk.services.worklink.model.AssociateWebsiteAuthorizationProviderResponse;
import software.amazon.awssdk.services.worklink.model.AssociateWebsiteCertificateAuthorityRequest;
import software.amazon.awssdk.services.worklink.model.AssociateWebsiteCertificateAuthorityResponse;
import software.amazon.awssdk.services.worklink.model.CreateFleetRequest;
import software.amazon.awssdk.services.worklink.model.CreateFleetResponse;
import software.amazon.awssdk.services.worklink.model.DeleteFleetRequest;
import software.amazon.awssdk.services.worklink.model.DeleteFleetResponse;
import software.amazon.awssdk.services.worklink.model.DescribeAuditStreamConfigurationRequest;
import software.amazon.awssdk.services.worklink.model.DescribeAuditStreamConfigurationResponse;
import software.amazon.awssdk.services.worklink.model.DescribeCompanyNetworkConfigurationRequest;
import software.amazon.awssdk.services.worklink.model.DescribeCompanyNetworkConfigurationResponse;
import software.amazon.awssdk.services.worklink.model.DescribeDevicePolicyConfigurationRequest;
import software.amazon.awssdk.services.worklink.model.DescribeDevicePolicyConfigurationResponse;
import software.amazon.awssdk.services.worklink.model.DescribeDeviceRequest;
import software.amazon.awssdk.services.worklink.model.DescribeDeviceResponse;
import software.amazon.awssdk.services.worklink.model.DescribeDomainRequest;
import software.amazon.awssdk.services.worklink.model.DescribeDomainResponse;
import software.amazon.awssdk.services.worklink.model.DescribeFleetMetadataRequest;
import software.amazon.awssdk.services.worklink.model.DescribeFleetMetadataResponse;
import software.amazon.awssdk.services.worklink.model.DescribeIdentityProviderConfigurationRequest;
import software.amazon.awssdk.services.worklink.model.DescribeIdentityProviderConfigurationResponse;
import software.amazon.awssdk.services.worklink.model.DescribeWebsiteCertificateAuthorityRequest;
import software.amazon.awssdk.services.worklink.model.DescribeWebsiteCertificateAuthorityResponse;
import software.amazon.awssdk.services.worklink.model.DisassociateDomainRequest;
import software.amazon.awssdk.services.worklink.model.DisassociateDomainResponse;
import software.amazon.awssdk.services.worklink.model.DisassociateWebsiteAuthorizationProviderRequest;
import software.amazon.awssdk.services.worklink.model.DisassociateWebsiteAuthorizationProviderResponse;
import software.amazon.awssdk.services.worklink.model.DisassociateWebsiteCertificateAuthorityRequest;
import software.amazon.awssdk.services.worklink.model.DisassociateWebsiteCertificateAuthorityResponse;
import software.amazon.awssdk.services.worklink.model.InternalServerErrorException;
import software.amazon.awssdk.services.worklink.model.InvalidRequestException;
import software.amazon.awssdk.services.worklink.model.ListDevicesRequest;
import software.amazon.awssdk.services.worklink.model.ListDevicesResponse;
import software.amazon.awssdk.services.worklink.model.ListDomainsRequest;
import software.amazon.awssdk.services.worklink.model.ListDomainsResponse;
import software.amazon.awssdk.services.worklink.model.ListFleetsRequest;
import software.amazon.awssdk.services.worklink.model.ListFleetsResponse;
import software.amazon.awssdk.services.worklink.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.worklink.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.worklink.model.ListWebsiteAuthorizationProvidersRequest;
import software.amazon.awssdk.services.worklink.model.ListWebsiteAuthorizationProvidersResponse;
import software.amazon.awssdk.services.worklink.model.ListWebsiteCertificateAuthoritiesRequest;
import software.amazon.awssdk.services.worklink.model.ListWebsiteCertificateAuthoritiesResponse;
import software.amazon.awssdk.services.worklink.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.worklink.model.ResourceNotFoundException;
import software.amazon.awssdk.services.worklink.model.RestoreDomainAccessRequest;
import software.amazon.awssdk.services.worklink.model.RestoreDomainAccessResponse;
import software.amazon.awssdk.services.worklink.model.RevokeDomainAccessRequest;
import software.amazon.awssdk.services.worklink.model.RevokeDomainAccessResponse;
import software.amazon.awssdk.services.worklink.model.SignOutUserRequest;
import software.amazon.awssdk.services.worklink.model.SignOutUserResponse;
import software.amazon.awssdk.services.worklink.model.TagResourceRequest;
import software.amazon.awssdk.services.worklink.model.TagResourceResponse;
import software.amazon.awssdk.services.worklink.model.TooManyRequestsException;
import software.amazon.awssdk.services.worklink.model.UnauthorizedException;
import software.amazon.awssdk.services.worklink.model.UntagResourceRequest;
import software.amazon.awssdk.services.worklink.model.UntagResourceResponse;
import software.amazon.awssdk.services.worklink.model.UpdateAuditStreamConfigurationRequest;
import software.amazon.awssdk.services.worklink.model.UpdateAuditStreamConfigurationResponse;
import software.amazon.awssdk.services.worklink.model.UpdateCompanyNetworkConfigurationRequest;
import software.amazon.awssdk.services.worklink.model.UpdateCompanyNetworkConfigurationResponse;
import software.amazon.awssdk.services.worklink.model.UpdateDevicePolicyConfigurationRequest;
import software.amazon.awssdk.services.worklink.model.UpdateDevicePolicyConfigurationResponse;
import software.amazon.awssdk.services.worklink.model.UpdateDomainMetadataRequest;
import software.amazon.awssdk.services.worklink.model.UpdateDomainMetadataResponse;
import software.amazon.awssdk.services.worklink.model.UpdateFleetMetadataRequest;
import software.amazon.awssdk.services.worklink.model.UpdateFleetMetadataResponse;
import software.amazon.awssdk.services.worklink.model.UpdateIdentityProviderConfigurationRequest;
import software.amazon.awssdk.services.worklink.model.UpdateIdentityProviderConfigurationResponse;
import software.amazon.awssdk.services.worklink.model.WorkLinkException;
import software.amazon.awssdk.services.worklink.transform.AssociateDomainRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.AssociateWebsiteAuthorizationProviderRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.AssociateWebsiteCertificateAuthorityRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.CreateFleetRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.DeleteFleetRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.DescribeAuditStreamConfigurationRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.DescribeCompanyNetworkConfigurationRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.DescribeDevicePolicyConfigurationRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.DescribeDeviceRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.DescribeDomainRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.DescribeFleetMetadataRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.DescribeIdentityProviderConfigurationRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.DescribeWebsiteCertificateAuthorityRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.DisassociateDomainRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.DisassociateWebsiteAuthorizationProviderRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.DisassociateWebsiteCertificateAuthorityRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.ListDevicesRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.ListDomainsRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.ListFleetsRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.ListWebsiteAuthorizationProvidersRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.ListWebsiteCertificateAuthoritiesRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.RestoreDomainAccessRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.RevokeDomainAccessRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.SignOutUserRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.UpdateAuditStreamConfigurationRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.UpdateCompanyNetworkConfigurationRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.UpdateDevicePolicyConfigurationRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.UpdateDomainMetadataRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.UpdateFleetMetadataRequestMarshaller;
import software.amazon.awssdk.services.worklink.transform.UpdateIdentityProviderConfigurationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link WorkLinkAsyncClient}.
 *
 * @see WorkLinkAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultWorkLinkAsyncClient implements WorkLinkAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultWorkLinkAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultWorkLinkAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Specifies a domain to be associated to Amazon WorkLink.
     * </p>
     *
     * @param associateDomainRequest
     * @return A Java Future containing the result of the AssociateDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ResourceAlreadyExistsException The resource already exists.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.AssociateDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/AssociateDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateDomainResponse> associateDomain(AssociateDomainRequest associateDomainRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateDomainRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateDomainRequest, AssociateDomainResponse>()
                            .withOperationName("AssociateDomain").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateDomainRequest));
            CompletableFuture<AssociateDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates a website authorization provider with a specified fleet. This is used to authorize users against
     * associated websites in the company network.
     * </p>
     *
     * @param associateWebsiteAuthorizationProviderRequest
     * @return A Java Future containing the result of the AssociateWebsiteAuthorizationProvider operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ResourceAlreadyExistsException The resource already exists.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.AssociateWebsiteAuthorizationProvider
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/AssociateWebsiteAuthorizationProvider"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateWebsiteAuthorizationProviderResponse> associateWebsiteAuthorizationProvider(
            AssociateWebsiteAuthorizationProviderRequest associateWebsiteAuthorizationProviderRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateWebsiteAuthorizationProviderRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateWebsiteAuthorizationProviderRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateWebsiteAuthorizationProvider");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateWebsiteAuthorizationProviderResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, AssociateWebsiteAuthorizationProviderResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateWebsiteAuthorizationProviderResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateWebsiteAuthorizationProviderRequest, AssociateWebsiteAuthorizationProviderResponse>()
                            .withOperationName("AssociateWebsiteAuthorizationProvider").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateWebsiteAuthorizationProviderRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateWebsiteAuthorizationProviderRequest));
            CompletableFuture<AssociateWebsiteAuthorizationProviderResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Imports the root certificate of a certificate authority (CA) used to obtain TLS certificates used by associated
     * websites within the company network.
     * </p>
     *
     * @param associateWebsiteCertificateAuthorityRequest
     * @return A Java Future containing the result of the AssociateWebsiteCertificateAuthority operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ResourceAlreadyExistsException The resource already exists.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.AssociateWebsiteCertificateAuthority
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/AssociateWebsiteCertificateAuthority"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateWebsiteCertificateAuthorityResponse> associateWebsiteCertificateAuthority(
            AssociateWebsiteCertificateAuthorityRequest associateWebsiteCertificateAuthorityRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateWebsiteCertificateAuthorityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateWebsiteCertificateAuthorityRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateWebsiteCertificateAuthority");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateWebsiteCertificateAuthorityResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, AssociateWebsiteCertificateAuthorityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateWebsiteCertificateAuthorityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateWebsiteCertificateAuthorityRequest, AssociateWebsiteCertificateAuthorityResponse>()
                            .withOperationName("AssociateWebsiteCertificateAuthority").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateWebsiteCertificateAuthorityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateWebsiteCertificateAuthorityRequest));
            CompletableFuture<AssociateWebsiteCertificateAuthorityResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a fleet. A fleet consists of resources and the configuration that delivers associated websites to
     * authorized users who download and set up the Amazon WorkLink app.
     * </p>
     *
     * @param createFleetRequest
     * @return A Java Future containing the result of the CreateFleet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ResourceAlreadyExistsException The resource already exists.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.CreateFleet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/CreateFleet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateFleetResponse> createFleet(CreateFleetRequest createFleetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createFleetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFleetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFleet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateFleetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateFleetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateFleetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateFleetRequest, CreateFleetResponse>()
                            .withOperationName("CreateFleet").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateFleetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createFleetRequest));
            CompletableFuture<CreateFleetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a fleet. Prevents users from accessing previously associated websites.
     * </p>
     *
     * @param deleteFleetRequest
     * @return A Java Future containing the result of the DeleteFleet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.DeleteFleet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/DeleteFleet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFleetResponse> deleteFleet(DeleteFleetRequest deleteFleetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteFleetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFleetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFleet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteFleetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteFleetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteFleetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteFleetRequest, DeleteFleetResponse>()
                            .withOperationName("DeleteFleet").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteFleetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteFleetRequest));
            CompletableFuture<DeleteFleetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the configuration for delivering audit streams to the customer account.
     * </p>
     *
     * @param describeAuditStreamConfigurationRequest
     * @return A Java Future containing the result of the DescribeAuditStreamConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.DescribeAuditStreamConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/DescribeAuditStreamConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAuditStreamConfigurationResponse> describeAuditStreamConfiguration(
            DescribeAuditStreamConfigurationRequest describeAuditStreamConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAuditStreamConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeAuditStreamConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAuditStreamConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAuditStreamConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeAuditStreamConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAuditStreamConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAuditStreamConfigurationRequest, DescribeAuditStreamConfigurationResponse>()
                            .withOperationName("DescribeAuditStreamConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeAuditStreamConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeAuditStreamConfigurationRequest));
            CompletableFuture<DescribeAuditStreamConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the networking configuration to access the internal websites associated with the specified fleet.
     * </p>
     *
     * @param describeCompanyNetworkConfigurationRequest
     * @return A Java Future containing the result of the DescribeCompanyNetworkConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.DescribeCompanyNetworkConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/DescribeCompanyNetworkConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeCompanyNetworkConfigurationResponse> describeCompanyNetworkConfiguration(
            DescribeCompanyNetworkConfigurationRequest describeCompanyNetworkConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeCompanyNetworkConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeCompanyNetworkConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeCompanyNetworkConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeCompanyNetworkConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeCompanyNetworkConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeCompanyNetworkConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeCompanyNetworkConfigurationRequest, DescribeCompanyNetworkConfigurationResponse>()
                            .withOperationName("DescribeCompanyNetworkConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeCompanyNetworkConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeCompanyNetworkConfigurationRequest));
            CompletableFuture<DescribeCompanyNetworkConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about a user's device.
     * </p>
     *
     * @param describeDeviceRequest
     * @return A Java Future containing the result of the DescribeDevice operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.DescribeDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/DescribeDevice" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDeviceResponse> describeDevice(DescribeDeviceRequest describeDeviceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDeviceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDevice");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDeviceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDeviceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDeviceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDeviceRequest, DescribeDeviceResponse>()
                            .withOperationName("DescribeDevice").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeDeviceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeDeviceRequest));
            CompletableFuture<DescribeDeviceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the device policy configuration for the specified fleet.
     * </p>
     *
     * @param describeDevicePolicyConfigurationRequest
     * @return A Java Future containing the result of the DescribeDevicePolicyConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.DescribeDevicePolicyConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/DescribeDevicePolicyConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDevicePolicyConfigurationResponse> describeDevicePolicyConfiguration(
            DescribeDevicePolicyConfigurationRequest describeDevicePolicyConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDevicePolicyConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeDevicePolicyConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDevicePolicyConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDevicePolicyConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeDevicePolicyConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDevicePolicyConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDevicePolicyConfigurationRequest, DescribeDevicePolicyConfigurationResponse>()
                            .withOperationName("DescribeDevicePolicyConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeDevicePolicyConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeDevicePolicyConfigurationRequest));
            CompletableFuture<DescribeDevicePolicyConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about the domain.
     * </p>
     *
     * @param describeDomainRequest
     * @return A Java Future containing the result of the DescribeDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.DescribeDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/DescribeDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDomainResponse> describeDomain(DescribeDomainRequest describeDomainRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDomainRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDomainRequest, DescribeDomainResponse>()
                            .withOperationName("DescribeDomain").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeDomainRequest));
            CompletableFuture<DescribeDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides basic information for the specified fleet, excluding identity provider, networking, and device
     * configuration details.
     * </p>
     *
     * @param describeFleetMetadataRequest
     * @return A Java Future containing the result of the DescribeFleetMetadata operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.DescribeFleetMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/DescribeFleetMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFleetMetadataResponse> describeFleetMetadata(
            DescribeFleetMetadataRequest describeFleetMetadataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeFleetMetadataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFleetMetadataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFleetMetadata");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFleetMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeFleetMetadataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeFleetMetadataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFleetMetadataRequest, DescribeFleetMetadataResponse>()
                            .withOperationName("DescribeFleetMetadata").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeFleetMetadataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeFleetMetadataRequest));
            CompletableFuture<DescribeFleetMetadataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the identity provider configuration of the specified fleet.
     * </p>
     *
     * @param describeIdentityProviderConfigurationRequest
     * @return A Java Future containing the result of the DescribeIdentityProviderConfiguration operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.DescribeIdentityProviderConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/DescribeIdentityProviderConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeIdentityProviderConfigurationResponse> describeIdentityProviderConfiguration(
            DescribeIdentityProviderConfigurationRequest describeIdentityProviderConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeIdentityProviderConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeIdentityProviderConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeIdentityProviderConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeIdentityProviderConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeIdentityProviderConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeIdentityProviderConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeIdentityProviderConfigurationRequest, DescribeIdentityProviderConfigurationResponse>()
                            .withOperationName("DescribeIdentityProviderConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeIdentityProviderConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeIdentityProviderConfigurationRequest));
            CompletableFuture<DescribeIdentityProviderConfigurationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about the certificate authority.
     * </p>
     *
     * @param describeWebsiteCertificateAuthorityRequest
     * @return A Java Future containing the result of the DescribeWebsiteCertificateAuthority operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.DescribeWebsiteCertificateAuthority
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/DescribeWebsiteCertificateAuthority"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeWebsiteCertificateAuthorityResponse> describeWebsiteCertificateAuthority(
            DescribeWebsiteCertificateAuthorityRequest describeWebsiteCertificateAuthorityRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeWebsiteCertificateAuthorityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeWebsiteCertificateAuthorityRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeWebsiteCertificateAuthority");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeWebsiteCertificateAuthorityResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeWebsiteCertificateAuthorityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeWebsiteCertificateAuthorityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeWebsiteCertificateAuthorityRequest, DescribeWebsiteCertificateAuthorityResponse>()
                            .withOperationName("DescribeWebsiteCertificateAuthority").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeWebsiteCertificateAuthorityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeWebsiteCertificateAuthorityRequest));
            CompletableFuture<DescribeWebsiteCertificateAuthorityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates a domain from Amazon WorkLink. End users lose the ability to access the domain with Amazon
     * WorkLink.
     * </p>
     *
     * @param disassociateDomainRequest
     * @return A Java Future containing the result of the DisassociateDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.DisassociateDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/DisassociateDomain" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateDomainResponse> disassociateDomain(DisassociateDomainRequest disassociateDomainRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateDomainRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateDomainRequest, DisassociateDomainResponse>()
                            .withOperationName("DisassociateDomain").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateDomainRequest));
            CompletableFuture<DisassociateDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates a website authorization provider from a specified fleet. After the disassociation, users can't load
     * any associated websites that require this authorization provider.
     * </p>
     *
     * @param disassociateWebsiteAuthorizationProviderRequest
     * @return A Java Future containing the result of the DisassociateWebsiteAuthorizationProvider operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ResourceAlreadyExistsException The resource already exists.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.DisassociateWebsiteAuthorizationProvider
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/DisassociateWebsiteAuthorizationProvider"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateWebsiteAuthorizationProviderResponse> disassociateWebsiteAuthorizationProvider(
            DisassociateWebsiteAuthorizationProviderRequest disassociateWebsiteAuthorizationProviderRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                disassociateWebsiteAuthorizationProviderRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateWebsiteAuthorizationProviderRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateWebsiteAuthorizationProvider");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateWebsiteAuthorizationProviderResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DisassociateWebsiteAuthorizationProviderResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateWebsiteAuthorizationProviderResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateWebsiteAuthorizationProviderRequest, DisassociateWebsiteAuthorizationProviderResponse>()
                            .withOperationName("DisassociateWebsiteAuthorizationProvider").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateWebsiteAuthorizationProviderRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateWebsiteAuthorizationProviderRequest));
            CompletableFuture<DisassociateWebsiteAuthorizationProviderResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a certificate authority (CA).
     * </p>
     *
     * @param disassociateWebsiteCertificateAuthorityRequest
     * @return A Java Future containing the result of the DisassociateWebsiteCertificateAuthority operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.DisassociateWebsiteCertificateAuthority
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/DisassociateWebsiteCertificateAuthority"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateWebsiteCertificateAuthorityResponse> disassociateWebsiteCertificateAuthority(
            DisassociateWebsiteCertificateAuthorityRequest disassociateWebsiteCertificateAuthorityRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateWebsiteCertificateAuthorityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateWebsiteCertificateAuthorityRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateWebsiteCertificateAuthority");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateWebsiteCertificateAuthorityResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DisassociateWebsiteCertificateAuthorityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateWebsiteCertificateAuthorityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateWebsiteCertificateAuthorityRequest, DisassociateWebsiteCertificateAuthorityResponse>()
                            .withOperationName("DisassociateWebsiteCertificateAuthority").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateWebsiteCertificateAuthorityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateWebsiteCertificateAuthorityRequest));
            CompletableFuture<DisassociateWebsiteCertificateAuthorityResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of devices registered with the specified fleet.
     * </p>
     *
     * @param listDevicesRequest
     * @return A Java Future containing the result of the ListDevices operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.ListDevices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/ListDevices" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDevicesResponse> listDevices(ListDevicesRequest listDevicesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDevicesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDevicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDevices");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDevicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDevicesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDevicesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDevicesRequest, ListDevicesResponse>()
                            .withOperationName("ListDevices").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDevicesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDevicesRequest));
            CompletableFuture<ListDevicesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of domains associated to a specified fleet.
     * </p>
     *
     * @param listDomainsRequest
     * @return A Java Future containing the result of the ListDomains operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.ListDomains
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/ListDomains" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDomainsResponse> listDomains(ListDomainsRequest listDomainsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDomainsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDomainsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDomains");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDomainsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDomainsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDomainsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDomainsRequest, ListDomainsResponse>()
                            .withOperationName("ListDomains").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDomainsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDomainsRequest));
            CompletableFuture<ListDomainsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of fleets for the current account and Region.
     * </p>
     *
     * @param listFleetsRequest
     * @return A Java Future containing the result of the ListFleets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.ListFleets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/ListFleets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListFleetsResponse> listFleets(ListFleetsRequest listFleetsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listFleetsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFleetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFleets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFleetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListFleetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFleetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFleetsRequest, ListFleetsResponse>().withOperationName("ListFleets")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListFleetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listFleetsRequest));
            CompletableFuture<ListFleetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of tags for the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of website authorization providers associated with a specified fleet.
     * </p>
     *
     * @param listWebsiteAuthorizationProvidersRequest
     * @return A Java Future containing the result of the ListWebsiteAuthorizationProviders operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.ListWebsiteAuthorizationProviders
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/ListWebsiteAuthorizationProviders"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListWebsiteAuthorizationProvidersResponse> listWebsiteAuthorizationProviders(
            ListWebsiteAuthorizationProvidersRequest listWebsiteAuthorizationProvidersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listWebsiteAuthorizationProvidersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listWebsiteAuthorizationProvidersRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWebsiteAuthorizationProviders");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWebsiteAuthorizationProvidersResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListWebsiteAuthorizationProvidersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListWebsiteAuthorizationProvidersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWebsiteAuthorizationProvidersRequest, ListWebsiteAuthorizationProvidersResponse>()
                            .withOperationName("ListWebsiteAuthorizationProviders").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListWebsiteAuthorizationProvidersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listWebsiteAuthorizationProvidersRequest));
            CompletableFuture<ListWebsiteAuthorizationProvidersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of certificate authorities added for the current account and Region.
     * </p>
     *
     * @param listWebsiteCertificateAuthoritiesRequest
     * @return A Java Future containing the result of the ListWebsiteCertificateAuthorities operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.ListWebsiteCertificateAuthorities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/ListWebsiteCertificateAuthorities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListWebsiteCertificateAuthoritiesResponse> listWebsiteCertificateAuthorities(
            ListWebsiteCertificateAuthoritiesRequest listWebsiteCertificateAuthoritiesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listWebsiteCertificateAuthoritiesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listWebsiteCertificateAuthoritiesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWebsiteCertificateAuthorities");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWebsiteCertificateAuthoritiesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListWebsiteCertificateAuthoritiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListWebsiteCertificateAuthoritiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWebsiteCertificateAuthoritiesRequest, ListWebsiteCertificateAuthoritiesResponse>()
                            .withOperationName("ListWebsiteCertificateAuthorities").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListWebsiteCertificateAuthoritiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listWebsiteCertificateAuthoritiesRequest));
            CompletableFuture<ListWebsiteCertificateAuthoritiesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Moves a domain to ACTIVE status if it was in the INACTIVE status.
     * </p>
     *
     * @param restoreDomainAccessRequest
     * @return A Java Future containing the result of the RestoreDomainAccess operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.RestoreDomainAccess
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/RestoreDomainAccess" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RestoreDomainAccessResponse> restoreDomainAccess(
            RestoreDomainAccessRequest restoreDomainAccessRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(restoreDomainAccessRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restoreDomainAccessRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestoreDomainAccess");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RestoreDomainAccessResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RestoreDomainAccessResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RestoreDomainAccessResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RestoreDomainAccessRequest, RestoreDomainAccessResponse>()
                            .withOperationName("RestoreDomainAccess").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RestoreDomainAccessRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(restoreDomainAccessRequest));
            CompletableFuture<RestoreDomainAccessResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Moves a domain to INACTIVE status if it was in the ACTIVE status.
     * </p>
     *
     * @param revokeDomainAccessRequest
     * @return A Java Future containing the result of the RevokeDomainAccess operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.RevokeDomainAccess
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/RevokeDomainAccess" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RevokeDomainAccessResponse> revokeDomainAccess(RevokeDomainAccessRequest revokeDomainAccessRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(revokeDomainAccessRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, revokeDomainAccessRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RevokeDomainAccess");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RevokeDomainAccessResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RevokeDomainAccessResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RevokeDomainAccessResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RevokeDomainAccessRequest, RevokeDomainAccessResponse>()
                            .withOperationName("RevokeDomainAccess").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RevokeDomainAccessRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(revokeDomainAccessRequest));
            CompletableFuture<RevokeDomainAccessResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Signs the user out from all of their devices. The user can sign in again if they have valid credentials.
     * </p>
     *
     * @param signOutUserRequest
     * @return A Java Future containing the result of the SignOutUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.SignOutUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/SignOutUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SignOutUserResponse> signOutUser(SignOutUserRequest signOutUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(signOutUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, signOutUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SignOutUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SignOutUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SignOutUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SignOutUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SignOutUserRequest, SignOutUserResponse>()
                            .withOperationName("SignOutUser").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SignOutUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(signOutUserRequest));
            CompletableFuture<SignOutUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or overwrites one or more tags for the specified resource, such as a fleet. Each tag consists of a key and
     * an optional value. If a resource already has a tag with the same key, this operation updates its value.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the audit stream configuration for the fleet.
     * </p>
     *
     * @param updateAuditStreamConfigurationRequest
     * @return A Java Future containing the result of the UpdateAuditStreamConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.UpdateAuditStreamConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/UpdateAuditStreamConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAuditStreamConfigurationResponse> updateAuditStreamConfiguration(
            UpdateAuditStreamConfigurationRequest updateAuditStreamConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAuditStreamConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateAuditStreamConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAuditStreamConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAuditStreamConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateAuditStreamConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAuditStreamConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAuditStreamConfigurationRequest, UpdateAuditStreamConfigurationResponse>()
                            .withOperationName("UpdateAuditStreamConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateAuditStreamConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateAuditStreamConfigurationRequest));
            CompletableFuture<UpdateAuditStreamConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the company network configuration for the fleet.
     * </p>
     *
     * @param updateCompanyNetworkConfigurationRequest
     * @return A Java Future containing the result of the UpdateCompanyNetworkConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.UpdateCompanyNetworkConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/UpdateCompanyNetworkConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateCompanyNetworkConfigurationResponse> updateCompanyNetworkConfiguration(
            UpdateCompanyNetworkConfigurationRequest updateCompanyNetworkConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateCompanyNetworkConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateCompanyNetworkConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateCompanyNetworkConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateCompanyNetworkConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateCompanyNetworkConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateCompanyNetworkConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateCompanyNetworkConfigurationRequest, UpdateCompanyNetworkConfigurationResponse>()
                            .withOperationName("UpdateCompanyNetworkConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateCompanyNetworkConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateCompanyNetworkConfigurationRequest));
            CompletableFuture<UpdateCompanyNetworkConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the device policy configuration for the fleet.
     * </p>
     *
     * @param updateDevicePolicyConfigurationRequest
     * @return A Java Future containing the result of the UpdateDevicePolicyConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.UpdateDevicePolicyConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/UpdateDevicePolicyConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDevicePolicyConfigurationResponse> updateDevicePolicyConfiguration(
            UpdateDevicePolicyConfigurationRequest updateDevicePolicyConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDevicePolicyConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateDevicePolicyConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDevicePolicyConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDevicePolicyConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDevicePolicyConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDevicePolicyConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDevicePolicyConfigurationRequest, UpdateDevicePolicyConfigurationResponse>()
                            .withOperationName("UpdateDevicePolicyConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateDevicePolicyConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateDevicePolicyConfigurationRequest));
            CompletableFuture<UpdateDevicePolicyConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates domain metadata, such as DisplayName.
     * </p>
     *
     * @param updateDomainMetadataRequest
     * @return A Java Future containing the result of the UpdateDomainMetadata operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.UpdateDomainMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/UpdateDomainMetadata" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDomainMetadataResponse> updateDomainMetadata(
            UpdateDomainMetadataRequest updateDomainMetadataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDomainMetadataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDomainMetadataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDomainMetadata");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDomainMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDomainMetadataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDomainMetadataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDomainMetadataRequest, UpdateDomainMetadataResponse>()
                            .withOperationName("UpdateDomainMetadata").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateDomainMetadataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateDomainMetadataRequest));
            CompletableFuture<UpdateDomainMetadataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates fleet metadata, such as DisplayName.
     * </p>
     *
     * @param updateFleetMetadataRequest
     * @return A Java Future containing the result of the UpdateFleetMetadata operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.UpdateFleetMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/UpdateFleetMetadata" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFleetMetadataResponse> updateFleetMetadata(
            UpdateFleetMetadataRequest updateFleetMetadataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateFleetMetadataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFleetMetadataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFleetMetadata");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFleetMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateFleetMetadataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFleetMetadataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFleetMetadataRequest, UpdateFleetMetadataResponse>()
                            .withOperationName("UpdateFleetMetadata").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateFleetMetadataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateFleetMetadataRequest));
            CompletableFuture<UpdateFleetMetadataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the identity provider configuration for the fleet.
     * </p>
     *
     * @param updateIdentityProviderConfigurationRequest
     * @return A Java Future containing the result of the UpdateIdentityProviderConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You are not authorized to perform this action.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>TooManyRequestsException The number of requests exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkLinkException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample WorkLinkAsyncClient.UpdateIdentityProviderConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/worklink-2018-09-25/UpdateIdentityProviderConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateIdentityProviderConfigurationResponse> updateIdentityProviderConfiguration(
            UpdateIdentityProviderConfigurationRequest updateIdentityProviderConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateIdentityProviderConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateIdentityProviderConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkLink");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateIdentityProviderConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateIdentityProviderConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateIdentityProviderConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateIdentityProviderConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateIdentityProviderConfigurationRequest, UpdateIdentityProviderConfigurationResponse>()
                            .withOperationName("UpdateIdentityProviderConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateIdentityProviderConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateIdentityProviderConfigurationRequest));
            CompletableFuture<UpdateIdentityProviderConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final WorkLinkServiceClientConfiguration serviceClientConfiguration() {
        return new WorkLinkServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(WorkLinkException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        WorkLinkServiceClientConfigurationBuilder serviceConfigBuilder = new WorkLinkServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
