/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.workmailmessageflow;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.sync.ResponseTransformer;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.workmailmessageflow.model.GetRawMessageContentRequest;
import software.amazon.awssdk.services.workmailmessageflow.model.GetRawMessageContentResponse;
import software.amazon.awssdk.services.workmailmessageflow.model.InvalidContentLocationException;
import software.amazon.awssdk.services.workmailmessageflow.model.MessageFrozenException;
import software.amazon.awssdk.services.workmailmessageflow.model.MessageRejectedException;
import software.amazon.awssdk.services.workmailmessageflow.model.PutRawMessageContentRequest;
import software.amazon.awssdk.services.workmailmessageflow.model.PutRawMessageContentResponse;
import software.amazon.awssdk.services.workmailmessageflow.model.ResourceNotFoundException;
import software.amazon.awssdk.services.workmailmessageflow.model.WorkMailMessageFlowException;
import software.amazon.awssdk.services.workmailmessageflow.transform.GetRawMessageContentRequestMarshaller;
import software.amazon.awssdk.services.workmailmessageflow.transform.PutRawMessageContentRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link WorkMailMessageFlowClient}.
 *
 * @see WorkMailMessageFlowClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultWorkMailMessageFlowClient implements WorkMailMessageFlowClient {
    private static final Logger log = Logger.loggerFor(DefaultWorkMailMessageFlowClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultWorkMailMessageFlowClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Retrieves the raw content of an in-transit email message, in MIME format.
     * </p>
     *
     * @param getRawMessageContentRequest
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled
     *        GetRawMessageContentResponse and an InputStream to the response content are provided as parameters to the
     *        callback. The callback may return a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        The raw content of the email message, in MIME format.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ResourceNotFoundException
     *         The requested email message is not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkMailMessageFlowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkMailMessageFlowClient.GetRawMessageContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workmailmessageflow-2019-05-01/GetRawMessageContent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public <ReturnT> ReturnT getRawMessageContent(GetRawMessageContentRequest getRawMessageContentRequest,
            ResponseTransformer<GetRawMessageContentResponse, ReturnT> responseTransformer) throws ResourceNotFoundException,
            AwsServiceException, SdkClientException, WorkMailMessageFlowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetRawMessageContentResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetRawMessageContentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRawMessageContentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkMailMessageFlow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRawMessageContent");

            return clientHandler.execute(
                    new ClientExecutionParams<GetRawMessageContentRequest, GetRawMessageContentResponse>()
                            .withOperationName("GetRawMessageContent").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getRawMessageContentRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetRawMessageContentRequestMarshaller(protocolFactory)), responseTransformer);
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the raw content of an in-transit email message, in MIME format.
     * </p>
     * <p>
     * This example describes how to update in-transit email message. For more information and examples for using this
     * API, see <a href="https://docs.aws.amazon.com/workmail/latest/adminguide/update-with-lambda.html"> Updating
     * message content with AWS Lambda</a>.
     * </p>
     * <note>
     * <p>
     * Updates to an in-transit message only appear when you call <code>PutRawMessageContent</code> from an AWS Lambda
     * function configured with a synchronous <a
     * href="https://docs.aws.amazon.com/workmail/latest/adminguide/lambda.html#synchronous-rules"> Run Lambda</a> rule.
     * If you call <code>PutRawMessageContent</code> on a delivered or sent message, the message remains unchanged, even
     * though <a
     * href="https://docs.aws.amazon.com/workmail/latest/APIReference/API_messageflow_GetRawMessageContent.html"
     * >GetRawMessageContent</a> returns an updated message.
     * </p>
     * </note>
     *
     * @param putRawMessageContentRequest
     * @return Result of the PutRawMessageContent operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested email message is not found.
     * @throws InvalidContentLocationException
     *         WorkMail could not access the updated email content. Possible reasons:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         You made the request in a region other than your S3 bucket region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/bucket-owner-condition.html">S3 bucket
     *         owner</a> is not the same as the calling AWS account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You have an incomplete or missing S3 bucket policy. For more information about policies, see <a
     *         href="https://docs.aws.amazon.com/workmail/latest/adminguide/update-with-lambda.html"> Updating message
     *         content with AWS Lambda </a> in the <i>WorkMail Administrator Guide</i>.
     *         </p>
     *         </li>
     * @throws MessageRejectedException
     *         The requested email could not be updated due to an error in the MIME content. Check the error message for
     *         more information about what caused the error.
     * @throws MessageFrozenException
     *         The requested email is not eligible for update. This is usually the case for a redirected email.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws WorkMailMessageFlowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample WorkMailMessageFlowClient.PutRawMessageContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workmailmessageflow-2019-05-01/PutRawMessageContent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutRawMessageContentResponse putRawMessageContent(PutRawMessageContentRequest putRawMessageContentRequest)
            throws ResourceNotFoundException, InvalidContentLocationException, MessageRejectedException, MessageFrozenException,
            AwsServiceException, SdkClientException, WorkMailMessageFlowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutRawMessageContentResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutRawMessageContentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRawMessageContentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkMailMessageFlow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRawMessageContent");

            return clientHandler.execute(new ClientExecutionParams<PutRawMessageContentRequest, PutRawMessageContentResponse>()
                    .withOperationName("PutRawMessageContent").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putRawMessageContentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutRawMessageContentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(WorkMailMessageFlowException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidContentLocation")
                                .exceptionBuilderSupplier(InvalidContentLocationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MessageFrozen")
                                .exceptionBuilderSupplier(MessageFrozenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MessageRejected")
                                .exceptionBuilderSupplier(MessageRejectedException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
