/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.workmailmessageflow;

import static software.amazon.awssdk.utils.FunctionalUtils.runAndLogError;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.core.async.AsyncResponseTransformerUtils;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.workmailmessageflow.model.GetRawMessageContentRequest;
import software.amazon.awssdk.services.workmailmessageflow.model.GetRawMessageContentResponse;
import software.amazon.awssdk.services.workmailmessageflow.model.InvalidContentLocationException;
import software.amazon.awssdk.services.workmailmessageflow.model.MessageFrozenException;
import software.amazon.awssdk.services.workmailmessageflow.model.MessageRejectedException;
import software.amazon.awssdk.services.workmailmessageflow.model.PutRawMessageContentRequest;
import software.amazon.awssdk.services.workmailmessageflow.model.PutRawMessageContentResponse;
import software.amazon.awssdk.services.workmailmessageflow.model.ResourceNotFoundException;
import software.amazon.awssdk.services.workmailmessageflow.model.WorkMailMessageFlowException;
import software.amazon.awssdk.services.workmailmessageflow.transform.GetRawMessageContentRequestMarshaller;
import software.amazon.awssdk.services.workmailmessageflow.transform.PutRawMessageContentRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Pair;

/**
 * Internal implementation of {@link WorkMailMessageFlowAsyncClient}.
 *
 * @see WorkMailMessageFlowAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultWorkMailMessageFlowAsyncClient implements WorkMailMessageFlowAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultWorkMailMessageFlowAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultWorkMailMessageFlowAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Retrieves the raw content of an in-transit email message, in MIME format.
     * </p>
     *
     * @param getRawMessageContentRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        The raw content of the email message, in MIME format.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested email message is not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkMailMessageFlowException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample WorkMailMessageFlowAsyncClient.GetRawMessageContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workmailmessageflow-2019-05-01/GetRawMessageContent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public <ReturnT> CompletableFuture<ReturnT> getRawMessageContent(GetRawMessageContentRequest getRawMessageContentRequest,
            AsyncResponseTransformer<GetRawMessageContentResponse, ReturnT> asyncResponseTransformer) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRawMessageContentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkMailMessageFlow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRawMessageContent");
            Pair<AsyncResponseTransformer<GetRawMessageContentResponse, ReturnT>, CompletableFuture<Void>> pair = AsyncResponseTransformerUtils
                    .wrapWithEndOfStreamFuture(asyncResponseTransformer);
            asyncResponseTransformer = pair.left();
            CompletableFuture<Void> endOfStreamFuture = pair.right();
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetRawMessageContentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRawMessageContentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ReturnT> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<GetRawMessageContentRequest, GetRawMessageContentResponse>()
                            .withOperationName("GetRawMessageContent")
                            .withMarshaller(new GetRawMessageContentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getRawMessageContentRequest),
                    asyncResponseTransformer);
            AsyncResponseTransformer<GetRawMessageContentResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            CompletableFuture<ReturnT> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                            () -> finalAsyncResponseTransformer.exceptionOccurred(e));
                }
                endOfStreamFuture.whenComplete((r2, e2) -> {
                    metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                });
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            AsyncResponseTransformer<GetRawMessageContentResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> finalAsyncResponseTransformer.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the raw content of an in-transit email message, in MIME format.
     * </p>
     * <p>
     * This example describes how to update in-transit email message. For more information and examples for using this
     * API, see <a href="https://docs.aws.amazon.com/workmail/latest/adminguide/update-with-lambda.html"> Updating
     * message content with AWS Lambda</a>.
     * </p>
     * <note>
     * <p>
     * Updates to an in-transit message only appear when you call <code>PutRawMessageContent</code> from an AWS Lambda
     * function configured with a synchronous <a
     * href="https://docs.aws.amazon.com/workmail/latest/adminguide/lambda.html#synchronous-rules"> Run Lambda</a> rule.
     * If you call <code>PutRawMessageContent</code> on a delivered or sent message, the message remains unchanged, even
     * though <a
     * href="https://docs.aws.amazon.com/workmail/latest/APIReference/API_messageflow_GetRawMessageContent.html"
     * >GetRawMessageContent</a> returns an updated message.
     * </p>
     * </note>
     *
     * @param putRawMessageContentRequest
     * @return A Java Future containing the result of the PutRawMessageContent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested email message is not found.</li>
     *         <li>InvalidContentLocationException WorkMail could not access the updated email content. Possible
     *         reasons:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         You made the request in a region other than your S3 bucket region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/bucket-owner-condition.html">S3 bucket
     *         owner</a> is not the same as the calling AWS account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You have an incomplete or missing S3 bucket policy. For more information about policies, see <a
     *         href="https://docs.aws.amazon.com/workmail/latest/adminguide/update-with-lambda.html"> Updating message
     *         content with AWS Lambda </a> in the <i>WorkMail Administrator Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>MessageRejectedException The requested email could not be updated due to an error in the MIME
     *         content. Check the error message for more information about what caused the error.</li>
     *         <li>MessageFrozenException The requested email is not eligible for update. This is usually the case for a
     *         redirected email.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkMailMessageFlowException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample WorkMailMessageFlowAsyncClient.PutRawMessageContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workmailmessageflow-2019-05-01/PutRawMessageContent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutRawMessageContentResponse> putRawMessageContent(
            PutRawMessageContentRequest putRawMessageContentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRawMessageContentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkMailMessageFlow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRawMessageContent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutRawMessageContentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutRawMessageContentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutRawMessageContentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutRawMessageContentRequest, PutRawMessageContentResponse>()
                            .withOperationName("PutRawMessageContent")
                            .withMarshaller(new PutRawMessageContentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putRawMessageContentRequest));
            CompletableFuture<PutRawMessageContentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(WorkMailMessageFlowException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidContentLocation")
                                .exceptionBuilderSupplier(InvalidContentLocationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MessageFrozen")
                                .exceptionBuilderSupplier(MessageFrozenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MessageRejected")
                                .exceptionBuilderSupplier(MessageRejectedException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
