/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.workspacesthinclient;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.workspacesthinclient.internal.WorkSpacesThinClientServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.workspacesthinclient.model.AccessDeniedException;
import software.amazon.awssdk.services.workspacesthinclient.model.ConflictException;
import software.amazon.awssdk.services.workspacesthinclient.model.CreateEnvironmentRequest;
import software.amazon.awssdk.services.workspacesthinclient.model.CreateEnvironmentResponse;
import software.amazon.awssdk.services.workspacesthinclient.model.DeleteDeviceRequest;
import software.amazon.awssdk.services.workspacesthinclient.model.DeleteDeviceResponse;
import software.amazon.awssdk.services.workspacesthinclient.model.DeleteEnvironmentRequest;
import software.amazon.awssdk.services.workspacesthinclient.model.DeleteEnvironmentResponse;
import software.amazon.awssdk.services.workspacesthinclient.model.DeregisterDeviceRequest;
import software.amazon.awssdk.services.workspacesthinclient.model.DeregisterDeviceResponse;
import software.amazon.awssdk.services.workspacesthinclient.model.GetDeviceRequest;
import software.amazon.awssdk.services.workspacesthinclient.model.GetDeviceResponse;
import software.amazon.awssdk.services.workspacesthinclient.model.GetEnvironmentRequest;
import software.amazon.awssdk.services.workspacesthinclient.model.GetEnvironmentResponse;
import software.amazon.awssdk.services.workspacesthinclient.model.GetSoftwareSetRequest;
import software.amazon.awssdk.services.workspacesthinclient.model.GetSoftwareSetResponse;
import software.amazon.awssdk.services.workspacesthinclient.model.InternalServerException;
import software.amazon.awssdk.services.workspacesthinclient.model.InternalServiceException;
import software.amazon.awssdk.services.workspacesthinclient.model.ListDevicesRequest;
import software.amazon.awssdk.services.workspacesthinclient.model.ListDevicesResponse;
import software.amazon.awssdk.services.workspacesthinclient.model.ListEnvironmentsRequest;
import software.amazon.awssdk.services.workspacesthinclient.model.ListEnvironmentsResponse;
import software.amazon.awssdk.services.workspacesthinclient.model.ListSoftwareSetsRequest;
import software.amazon.awssdk.services.workspacesthinclient.model.ListSoftwareSetsResponse;
import software.amazon.awssdk.services.workspacesthinclient.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.workspacesthinclient.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.workspacesthinclient.model.ResourceNotFoundException;
import software.amazon.awssdk.services.workspacesthinclient.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.workspacesthinclient.model.TagResourceRequest;
import software.amazon.awssdk.services.workspacesthinclient.model.TagResourceResponse;
import software.amazon.awssdk.services.workspacesthinclient.model.ThrottlingException;
import software.amazon.awssdk.services.workspacesthinclient.model.UntagResourceRequest;
import software.amazon.awssdk.services.workspacesthinclient.model.UntagResourceResponse;
import software.amazon.awssdk.services.workspacesthinclient.model.UpdateDeviceRequest;
import software.amazon.awssdk.services.workspacesthinclient.model.UpdateDeviceResponse;
import software.amazon.awssdk.services.workspacesthinclient.model.UpdateEnvironmentRequest;
import software.amazon.awssdk.services.workspacesthinclient.model.UpdateEnvironmentResponse;
import software.amazon.awssdk.services.workspacesthinclient.model.UpdateSoftwareSetRequest;
import software.amazon.awssdk.services.workspacesthinclient.model.UpdateSoftwareSetResponse;
import software.amazon.awssdk.services.workspacesthinclient.model.ValidationException;
import software.amazon.awssdk.services.workspacesthinclient.model.WorkSpacesThinClientException;
import software.amazon.awssdk.services.workspacesthinclient.transform.CreateEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.workspacesthinclient.transform.DeleteDeviceRequestMarshaller;
import software.amazon.awssdk.services.workspacesthinclient.transform.DeleteEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.workspacesthinclient.transform.DeregisterDeviceRequestMarshaller;
import software.amazon.awssdk.services.workspacesthinclient.transform.GetDeviceRequestMarshaller;
import software.amazon.awssdk.services.workspacesthinclient.transform.GetEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.workspacesthinclient.transform.GetSoftwareSetRequestMarshaller;
import software.amazon.awssdk.services.workspacesthinclient.transform.ListDevicesRequestMarshaller;
import software.amazon.awssdk.services.workspacesthinclient.transform.ListEnvironmentsRequestMarshaller;
import software.amazon.awssdk.services.workspacesthinclient.transform.ListSoftwareSetsRequestMarshaller;
import software.amazon.awssdk.services.workspacesthinclient.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.workspacesthinclient.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.workspacesthinclient.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.workspacesthinclient.transform.UpdateDeviceRequestMarshaller;
import software.amazon.awssdk.services.workspacesthinclient.transform.UpdateEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.workspacesthinclient.transform.UpdateSoftwareSetRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link WorkSpacesThinClientAsyncClient}.
 *
 * @see WorkSpacesThinClientAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultWorkSpacesThinClientAsyncClient implements WorkSpacesThinClientAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultWorkSpacesThinClientAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultWorkSpacesThinClientAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an environment for your thin client devices.
     * </p>
     *
     * @param createEnvironmentRequest
     * @return A Java Future containing the result of the CreateEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException Your request exceeds a service quota.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkSpacesThinClientException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample WorkSpacesThinClientAsyncClient.CreateEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-thin-client-2023-08-22/CreateEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateEnvironmentResponse> createEnvironment(CreateEnvironmentRequest createEnvironmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkSpaces Thin Client");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            CompletableFuture<CreateEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateEnvironmentRequest, CreateEnvironmentResponse>()
                            .withOperationName("CreateEnvironment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(createEnvironmentRequest));
            CompletableFuture<CreateEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a thin client device.
     * </p>
     *
     * @param deleteDeviceRequest
     * @return A Java Future containing the result of the DeleteDevice operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkSpacesThinClientException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample WorkSpacesThinClientAsyncClient.DeleteDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-thin-client-2023-08-22/DeleteDevice"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDeviceResponse> deleteDevice(DeleteDeviceRequest deleteDeviceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDeviceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkSpaces Thin Client");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDevice");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDeviceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteDeviceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            CompletableFuture<DeleteDeviceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDeviceRequest, DeleteDeviceResponse>()
                            .withOperationName("DeleteDevice").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteDeviceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(deleteDeviceRequest));
            CompletableFuture<DeleteDeviceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an environment.
     * </p>
     *
     * @param deleteEnvironmentRequest
     * @return A Java Future containing the result of the DeleteEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkSpacesThinClientException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample WorkSpacesThinClientAsyncClient.DeleteEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-thin-client-2023-08-22/DeleteEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEnvironmentResponse> deleteEnvironment(DeleteEnvironmentRequest deleteEnvironmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkSpaces Thin Client");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            CompletableFuture<DeleteEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEnvironmentRequest, DeleteEnvironmentResponse>()
                            .withOperationName("DeleteEnvironment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(deleteEnvironmentRequest));
            CompletableFuture<DeleteEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deregisters a thin client device.
     * </p>
     *
     * @param deregisterDeviceRequest
     * @return A Java Future containing the result of the DeregisterDevice operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkSpacesThinClientException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample WorkSpacesThinClientAsyncClient.DeregisterDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-thin-client-2023-08-22/DeregisterDevice"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeregisterDeviceResponse> deregisterDevice(DeregisterDeviceRequest deregisterDeviceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deregisterDeviceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deregisterDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkSpaces Thin Client");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterDevice");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeregisterDeviceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeregisterDeviceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            CompletableFuture<DeregisterDeviceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeregisterDeviceRequest, DeregisterDeviceResponse>()
                            .withOperationName("DeregisterDevice").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeregisterDeviceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(deregisterDeviceRequest));
            CompletableFuture<DeregisterDeviceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information for a thin client device.
     * </p>
     *
     * @param getDeviceRequest
     * @return A Java Future containing the result of the GetDevice operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkSpacesThinClientException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample WorkSpacesThinClientAsyncClient.GetDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-thin-client-2023-08-22/GetDevice"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDeviceResponse> getDevice(GetDeviceRequest getDeviceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDeviceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkSpaces Thin Client");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDevice");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDeviceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetDeviceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            CompletableFuture<GetDeviceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDeviceRequest, GetDeviceResponse>().withOperationName("GetDevice")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDeviceRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(getDeviceRequest));
            CompletableFuture<GetDeviceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information for an environment.
     * </p>
     *
     * @param getEnvironmentRequest
     * @return A Java Future containing the result of the GetEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkSpacesThinClientException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample WorkSpacesThinClientAsyncClient.GetEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-thin-client-2023-08-22/GetEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEnvironmentResponse> getEnvironment(GetEnvironmentRequest getEnvironmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEnvironmentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkSpaces Thin Client");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            CompletableFuture<GetEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEnvironmentRequest, GetEnvironmentResponse>()
                            .withOperationName("GetEnvironment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(getEnvironmentRequest));
            CompletableFuture<GetEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information for a software set.
     * </p>
     *
     * @param getSoftwareSetRequest
     * @return A Java Future containing the result of the GetSoftwareSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkSpacesThinClientException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample WorkSpacesThinClientAsyncClient.GetSoftwareSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-thin-client-2023-08-22/GetSoftwareSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSoftwareSetResponse> getSoftwareSet(GetSoftwareSetRequest getSoftwareSetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSoftwareSetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSoftwareSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkSpaces Thin Client");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSoftwareSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSoftwareSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSoftwareSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            CompletableFuture<GetSoftwareSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSoftwareSetRequest, GetSoftwareSetResponse>()
                            .withOperationName("GetSoftwareSet").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSoftwareSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(getSoftwareSetRequest));
            CompletableFuture<GetSoftwareSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of thin client devices.
     * </p>
     *
     * @param listDevicesRequest
     * @return A Java Future containing the result of the ListDevices operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkSpacesThinClientException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample WorkSpacesThinClientAsyncClient.ListDevices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-thin-client-2023-08-22/ListDevices"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDevicesResponse> listDevices(ListDevicesRequest listDevicesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDevicesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDevicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkSpaces Thin Client");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDevices");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDevicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDevicesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            CompletableFuture<ListDevicesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDevicesRequest, ListDevicesResponse>()
                            .withOperationName("ListDevices").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDevicesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(listDevicesRequest));
            CompletableFuture<ListDevicesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of environments.
     * </p>
     *
     * @param listEnvironmentsRequest
     * @return A Java Future containing the result of the ListEnvironments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkSpacesThinClientException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample WorkSpacesThinClientAsyncClient.ListEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-thin-client-2023-08-22/ListEnvironments"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEnvironmentsResponse> listEnvironments(ListEnvironmentsRequest listEnvironmentsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEnvironmentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEnvironmentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkSpaces Thin Client");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEnvironments");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEnvironmentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEnvironmentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            CompletableFuture<ListEnvironmentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEnvironmentsRequest, ListEnvironmentsResponse>()
                            .withOperationName("ListEnvironments").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListEnvironmentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(listEnvironmentsRequest));
            CompletableFuture<ListEnvironmentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of software sets.
     * </p>
     *
     * @param listSoftwareSetsRequest
     * @return A Java Future containing the result of the ListSoftwareSets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkSpacesThinClientException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample WorkSpacesThinClientAsyncClient.ListSoftwareSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-thin-client-2023-08-22/ListSoftwareSets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSoftwareSetsResponse> listSoftwareSets(ListSoftwareSetsRequest listSoftwareSetsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSoftwareSetsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSoftwareSetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkSpaces Thin Client");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSoftwareSets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSoftwareSetsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSoftwareSetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            CompletableFuture<ListSoftwareSetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSoftwareSetsRequest, ListSoftwareSetsResponse>()
                            .withOperationName("ListSoftwareSets").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSoftwareSetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(listSoftwareSetsRequest));
            CompletableFuture<ListSoftwareSetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of tags for a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found.</li>
     *         <li>InternalServiceException Request processing failed due to some unknown error, exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkSpacesThinClientException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample WorkSpacesThinClientAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-thin-client-2023-08-22/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkSpaces Thin Client");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found.</li>
     *         <li>InternalServiceException Request processing failed due to some unknown error, exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkSpacesThinClientException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample WorkSpacesThinClientAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-thin-client-2023-08-22/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkSpaces Thin Client");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a tag or tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found.</li>
     *         <li>InternalServiceException Request processing failed due to some unknown error, exception, or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkSpacesThinClientException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample WorkSpacesThinClientAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-thin-client-2023-08-22/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkSpaces Thin Client");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a thin client device.
     * </p>
     *
     * @param updateDeviceRequest
     * @return A Java Future containing the result of the UpdateDevice operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkSpacesThinClientException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample WorkSpacesThinClientAsyncClient.UpdateDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-thin-client-2023-08-22/UpdateDevice"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDeviceResponse> updateDevice(UpdateDeviceRequest updateDeviceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDeviceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkSpaces Thin Client");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDevice");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDeviceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateDeviceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            CompletableFuture<UpdateDeviceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDeviceRequest, UpdateDeviceResponse>()
                            .withOperationName("UpdateDevice").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateDeviceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(updateDeviceRequest));
            CompletableFuture<UpdateDeviceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an environment.
     * </p>
     *
     * @param updateEnvironmentRequest
     * @return A Java Future containing the result of the UpdateEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkSpacesThinClientException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample WorkSpacesThinClientAsyncClient.UpdateEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-thin-client-2023-08-22/UpdateEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateEnvironmentResponse> updateEnvironment(UpdateEnvironmentRequest updateEnvironmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkSpaces Thin Client");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            CompletableFuture<UpdateEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateEnvironmentRequest, UpdateEnvironmentResponse>()
                            .withOperationName("UpdateEnvironment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(updateEnvironmentRequest));
            CompletableFuture<UpdateEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a software set.
     * </p>
     *
     * @param updateSoftwareSetRequest
     * @return A Java Future containing the result of the UpdateSoftwareSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkSpacesThinClientException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample WorkSpacesThinClientAsyncClient.UpdateSoftwareSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/workspaces-thin-client-2023-08-22/UpdateSoftwareSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSoftwareSetResponse> updateSoftwareSet(UpdateSoftwareSetRequest updateSoftwareSetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateSoftwareSetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSoftwareSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "WorkSpaces Thin Client");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSoftwareSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSoftwareSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSoftwareSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            CompletableFuture<UpdateSoftwareSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSoftwareSetRequest, UpdateSoftwareSetResponse>()
                            .withOperationName("UpdateSoftwareSet").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateSoftwareSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(updateSoftwareSetRequest));
            CompletableFuture<UpdateSoftwareSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final WorkSpacesThinClientServiceClientConfiguration serviceClientConfiguration() {
        return new WorkSpacesThinClientServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(WorkSpacesThinClientException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceException")
                                .exceptionBuilderSupplier(InternalServiceException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        WorkSpacesThinClientServiceClientConfigurationBuilder serviceConfigBuilder = new WorkSpacesThinClientServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
