/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.xray.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A collection of segment documents with matching trace IDs.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Trace implements SdkPojo, Serializable, ToCopyableBuilder<Trace.Builder, Trace> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(Trace::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<Double> DURATION_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("Duration").getter(getter(Trace::duration)).setter(setter(Builder::duration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Duration").build()).build();

    private static final SdkField<Boolean> LIMIT_EXCEEDED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("LimitExceeded").getter(getter(Trace::limitExceeded)).setter(setter(Builder::limitExceeded))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LimitExceeded").build()).build();

    private static final SdkField<List<Segment>> SEGMENTS_FIELD = SdkField
            .<List<Segment>> builder(MarshallingType.LIST)
            .memberName("Segments")
            .getter(getter(Trace::segments))
            .setter(setter(Builder::segments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Segments").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Segment> builder(MarshallingType.SDK_POJO)
                                            .constructor(Segment::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, DURATION_FIELD,
            LIMIT_EXCEEDED_FIELD, SEGMENTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final Double duration;

    private final Boolean limitExceeded;

    private final List<Segment> segments;

    private Trace(BuilderImpl builder) {
        this.id = builder.id;
        this.duration = builder.duration;
        this.limitExceeded = builder.limitExceeded;
        this.segments = builder.segments;
    }

    /**
     * <p>
     * The unique identifier for the request that generated the trace's segments and subsegments.
     * </p>
     * 
     * @return The unique identifier for the request that generated the trace's segments and subsegments.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The length of time in seconds between the start time of the root segment and the end time of the last segment
     * that completed.
     * </p>
     * 
     * @return The length of time in seconds between the start time of the root segment and the end time of the last
     *         segment that completed.
     */
    public final Double duration() {
        return duration;
    }

    /**
     * <p>
     * LimitExceeded is set to true when the trace has exceeded one of the defined quotas. For more information about
     * quotas, see <a href="https://docs.aws.amazon.com/general/latest/gr/xray.html">AWS X-Ray endpoints and quotas</a>.
     * </p>
     * 
     * @return LimitExceeded is set to true when the trace has exceeded one of the defined quotas. For more information
     *         about quotas, see <a href="https://docs.aws.amazon.com/general/latest/gr/xray.html">AWS X-Ray endpoints
     *         and quotas</a>.
     */
    public final Boolean limitExceeded() {
        return limitExceeded;
    }

    /**
     * Returns true if the Segments property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasSegments() {
        return segments != null && !(segments instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Segment documents for the segments and subsegments that comprise the trace.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSegments()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Segment documents for the segments and subsegments that comprise the trace.
     */
    public final List<Segment> segments() {
        return segments;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(duration());
        hashCode = 31 * hashCode + Objects.hashCode(limitExceeded());
        hashCode = 31 * hashCode + Objects.hashCode(hasSegments() ? segments() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Trace)) {
            return false;
        }
        Trace other = (Trace) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(duration(), other.duration())
                && Objects.equals(limitExceeded(), other.limitExceeded()) && hasSegments() == other.hasSegments()
                && Objects.equals(segments(), other.segments());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Trace").add("Id", id()).add("Duration", duration()).add("LimitExceeded", limitExceeded())
                .add("Segments", hasSegments() ? segments() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Duration":
            return Optional.ofNullable(clazz.cast(duration()));
        case "LimitExceeded":
            return Optional.ofNullable(clazz.cast(limitExceeded()));
        case "Segments":
            return Optional.ofNullable(clazz.cast(segments()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Trace, T> g) {
        return obj -> g.apply((Trace) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Trace> {
        /**
         * <p>
         * The unique identifier for the request that generated the trace's segments and subsegments.
         * </p>
         * 
         * @param id
         *        The unique identifier for the request that generated the trace's segments and subsegments.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The length of time in seconds between the start time of the root segment and the end time of the last segment
         * that completed.
         * </p>
         * 
         * @param duration
         *        The length of time in seconds between the start time of the root segment and the end time of the last
         *        segment that completed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder duration(Double duration);

        /**
         * <p>
         * LimitExceeded is set to true when the trace has exceeded one of the defined quotas. For more information
         * about quotas, see <a href="https://docs.aws.amazon.com/general/latest/gr/xray.html">AWS X-Ray endpoints and
         * quotas</a>.
         * </p>
         * 
         * @param limitExceeded
         *        LimitExceeded is set to true when the trace has exceeded one of the defined quotas. For more
         *        information about quotas, see <a href="https://docs.aws.amazon.com/general/latest/gr/xray.html">AWS
         *        X-Ray endpoints and quotas</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder limitExceeded(Boolean limitExceeded);

        /**
         * <p>
         * Segment documents for the segments and subsegments that comprise the trace.
         * </p>
         * 
         * @param segments
         *        Segment documents for the segments and subsegments that comprise the trace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder segments(Collection<Segment> segments);

        /**
         * <p>
         * Segment documents for the segments and subsegments that comprise the trace.
         * </p>
         * 
         * @param segments
         *        Segment documents for the segments and subsegments that comprise the trace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder segments(Segment... segments);

        /**
         * <p>
         * Segment documents for the segments and subsegments that comprise the trace.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Segment>.Builder} avoiding the need to
         * create one manually via {@link List<Segment>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Segment>.Builder#build()} is called immediately and its
         * result is passed to {@link #segments(List<Segment>)}.
         * 
         * @param segments
         *        a consumer that will call methods on {@link List<Segment>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #segments(List<Segment>)
         */
        Builder segments(Consumer<Segment.Builder>... segments);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private Double duration;

        private Boolean limitExceeded;

        private List<Segment> segments = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Trace model) {
            id(model.id);
            duration(model.duration);
            limitExceeded(model.limitExceeded);
            segments(model.segments);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final Double getDuration() {
            return duration;
        }

        @Override
        public final Builder duration(Double duration) {
            this.duration = duration;
            return this;
        }

        public final void setDuration(Double duration) {
            this.duration = duration;
        }

        public final Boolean getLimitExceeded() {
            return limitExceeded;
        }

        @Override
        public final Builder limitExceeded(Boolean limitExceeded) {
            this.limitExceeded = limitExceeded;
            return this;
        }

        public final void setLimitExceeded(Boolean limitExceeded) {
            this.limitExceeded = limitExceeded;
        }

        public final List<Segment.Builder> getSegments() {
            List<Segment.Builder> result = SegmentListCopier.copyToBuilder(this.segments);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder segments(Collection<Segment> segments) {
            this.segments = SegmentListCopier.copy(segments);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder segments(Segment... segments) {
            segments(Arrays.asList(segments));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder segments(Consumer<Segment.Builder>... segments) {
            segments(Stream.of(segments).map(c -> Segment.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setSegments(Collection<Segment.BuilderImpl> segments) {
            this.segments = SegmentListCopier.copyFromBuilder(segments);
        }

        @Override
        public Trace build() {
            return new Trace(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
