/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.xray.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information that describes an insight.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InsightSummary implements SdkPojo, Serializable, ToCopyableBuilder<InsightSummary.Builder, InsightSummary> {
    private static final SdkField<String> INSIGHT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InsightId").getter(getter(InsightSummary::insightId)).setter(setter(Builder::insightId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InsightId").build()).build();

    private static final SdkField<String> GROUP_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GroupARN").getter(getter(InsightSummary::groupARN)).setter(setter(Builder::groupARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GroupARN").build()).build();

    private static final SdkField<String> GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GroupName").getter(getter(InsightSummary::groupName)).setter(setter(Builder::groupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GroupName").build()).build();

    private static final SdkField<ServiceId> ROOT_CAUSE_SERVICE_ID_FIELD = SdkField.<ServiceId> builder(MarshallingType.SDK_POJO)
            .memberName("RootCauseServiceId").getter(getter(InsightSummary::rootCauseServiceId))
            .setter(setter(Builder::rootCauseServiceId)).constructor(ServiceId::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RootCauseServiceId").build())
            .build();

    private static final SdkField<List<String>> CATEGORIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Categories")
            .getter(getter(InsightSummary::categoriesAsStrings))
            .setter(setter(Builder::categoriesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Categories").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(InsightSummary::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTime").getter(getter(InsightSummary::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndTime").getter(getter(InsightSummary::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndTime").build()).build();

    private static final SdkField<String> SUMMARY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Summary")
            .getter(getter(InsightSummary::summary)).setter(setter(Builder::summary))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Summary").build()).build();

    private static final SdkField<RequestImpactStatistics> CLIENT_REQUEST_IMPACT_STATISTICS_FIELD = SdkField
            .<RequestImpactStatistics> builder(MarshallingType.SDK_POJO)
            .memberName("ClientRequestImpactStatistics")
            .getter(getter(InsightSummary::clientRequestImpactStatistics))
            .setter(setter(Builder::clientRequestImpactStatistics))
            .constructor(RequestImpactStatistics::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientRequestImpactStatistics")
                    .build()).build();

    private static final SdkField<RequestImpactStatistics> ROOT_CAUSE_SERVICE_REQUEST_IMPACT_STATISTICS_FIELD = SdkField
            .<RequestImpactStatistics> builder(MarshallingType.SDK_POJO)
            .memberName("RootCauseServiceRequestImpactStatistics")
            .getter(getter(InsightSummary::rootCauseServiceRequestImpactStatistics))
            .setter(setter(Builder::rootCauseServiceRequestImpactStatistics))
            .constructor(RequestImpactStatistics::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("RootCauseServiceRequestImpactStatistics").build()).build();

    private static final SdkField<List<AnomalousService>> TOP_ANOMALOUS_SERVICES_FIELD = SdkField
            .<List<AnomalousService>> builder(MarshallingType.LIST)
            .memberName("TopAnomalousServices")
            .getter(getter(InsightSummary::topAnomalousServices))
            .setter(setter(Builder::topAnomalousServices))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TopAnomalousServices").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AnomalousService> builder(MarshallingType.SDK_POJO)
                                            .constructor(AnomalousService::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> LAST_UPDATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdateTime").getter(getter(InsightSummary::lastUpdateTime)).setter(setter(Builder::lastUpdateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdateTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSIGHT_ID_FIELD,
            GROUP_ARN_FIELD, GROUP_NAME_FIELD, ROOT_CAUSE_SERVICE_ID_FIELD, CATEGORIES_FIELD, STATE_FIELD, START_TIME_FIELD,
            END_TIME_FIELD, SUMMARY_FIELD, CLIENT_REQUEST_IMPACT_STATISTICS_FIELD,
            ROOT_CAUSE_SERVICE_REQUEST_IMPACT_STATISTICS_FIELD, TOP_ANOMALOUS_SERVICES_FIELD, LAST_UPDATE_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String insightId;

    private final String groupARN;

    private final String groupName;

    private final ServiceId rootCauseServiceId;

    private final List<String> categories;

    private final String state;

    private final Instant startTime;

    private final Instant endTime;

    private final String summary;

    private final RequestImpactStatistics clientRequestImpactStatistics;

    private final RequestImpactStatistics rootCauseServiceRequestImpactStatistics;

    private final List<AnomalousService> topAnomalousServices;

    private final Instant lastUpdateTime;

    private InsightSummary(BuilderImpl builder) {
        this.insightId = builder.insightId;
        this.groupARN = builder.groupARN;
        this.groupName = builder.groupName;
        this.rootCauseServiceId = builder.rootCauseServiceId;
        this.categories = builder.categories;
        this.state = builder.state;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.summary = builder.summary;
        this.clientRequestImpactStatistics = builder.clientRequestImpactStatistics;
        this.rootCauseServiceRequestImpactStatistics = builder.rootCauseServiceRequestImpactStatistics;
        this.topAnomalousServices = builder.topAnomalousServices;
        this.lastUpdateTime = builder.lastUpdateTime;
    }

    /**
     * <p>
     * The insights unique identifier.
     * </p>
     * 
     * @return The insights unique identifier.
     */
    public final String insightId() {
        return insightId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the group that the insight belongs to.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the group that the insight belongs to.
     */
    public final String groupARN() {
        return groupARN;
    }

    /**
     * <p>
     * The name of the group that the insight belongs to.
     * </p>
     * 
     * @return The name of the group that the insight belongs to.
     */
    public final String groupName() {
        return groupName;
    }

    /**
     * Returns the value of the RootCauseServiceId property for this object.
     * 
     * @return The value of the RootCauseServiceId property for this object.
     */
    public final ServiceId rootCauseServiceId() {
        return rootCauseServiceId;
    }

    /**
     * <p>
     * Categories The categories that label and describe the type of insight.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCategories} method.
     * </p>
     * 
     * @return Categories The categories that label and describe the type of insight.
     */
    public final List<InsightCategory> categories() {
        return InsightCategoryListCopier.copyStringToEnum(categories);
    }

    /**
     * For responses, this returns true if the service returned a value for the Categories property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasCategories() {
        return categories != null && !(categories instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Categories The categories that label and describe the type of insight.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCategories} method.
     * </p>
     * 
     * @return Categories The categories that label and describe the type of insight.
     */
    public final List<String> categoriesAsStrings() {
        return categories;
    }

    /**
     * <p>
     * The current state of the insight.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link InsightState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the insight.
     * @see InsightState
     */
    public final InsightState state() {
        return InsightState.fromValue(state);
    }

    /**
     * <p>
     * The current state of the insight.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link InsightState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the insight.
     * @see InsightState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The time, in Unix seconds, at which the insight began.
     * </p>
     * 
     * @return The time, in Unix seconds, at which the insight began.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The time, in Unix seconds, at which the insight ended.
     * </p>
     * 
     * @return The time, in Unix seconds, at which the insight ended.
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * A brief description of the insight.
     * </p>
     * 
     * @return A brief description of the insight.
     */
    public final String summary() {
        return summary;
    }

    /**
     * <p>
     * The impact statistics of the client side service. This includes the number of requests to the client service and
     * whether the requests were faults or okay.
     * </p>
     * 
     * @return The impact statistics of the client side service. This includes the number of requests to the client
     *         service and whether the requests were faults or okay.
     */
    public final RequestImpactStatistics clientRequestImpactStatistics() {
        return clientRequestImpactStatistics;
    }

    /**
     * <p>
     * The impact statistics of the root cause service. This includes the number of requests to the client service and
     * whether the requests were faults or okay.
     * </p>
     * 
     * @return The impact statistics of the root cause service. This includes the number of requests to the client
     *         service and whether the requests were faults or okay.
     */
    public final RequestImpactStatistics rootCauseServiceRequestImpactStatistics() {
        return rootCauseServiceRequestImpactStatistics;
    }

    /**
     * For responses, this returns true if the service returned a value for the TopAnomalousServices property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTopAnomalousServices() {
        return topAnomalousServices != null && !(topAnomalousServices instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The service within the insight that is most impacted by the incident.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTopAnomalousServices} method.
     * </p>
     * 
     * @return The service within the insight that is most impacted by the incident.
     */
    public final List<AnomalousService> topAnomalousServices() {
        return topAnomalousServices;
    }

    /**
     * <p>
     * The time, in Unix seconds, that the insight was last updated.
     * </p>
     * 
     * @return The time, in Unix seconds, that the insight was last updated.
     */
    public final Instant lastUpdateTime() {
        return lastUpdateTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(insightId());
        hashCode = 31 * hashCode + Objects.hashCode(groupARN());
        hashCode = 31 * hashCode + Objects.hashCode(groupName());
        hashCode = 31 * hashCode + Objects.hashCode(rootCauseServiceId());
        hashCode = 31 * hashCode + Objects.hashCode(hasCategories() ? categoriesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(summary());
        hashCode = 31 * hashCode + Objects.hashCode(clientRequestImpactStatistics());
        hashCode = 31 * hashCode + Objects.hashCode(rootCauseServiceRequestImpactStatistics());
        hashCode = 31 * hashCode + Objects.hashCode(hasTopAnomalousServices() ? topAnomalousServices() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdateTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InsightSummary)) {
            return false;
        }
        InsightSummary other = (InsightSummary) obj;
        return Objects.equals(insightId(), other.insightId()) && Objects.equals(groupARN(), other.groupARN())
                && Objects.equals(groupName(), other.groupName())
                && Objects.equals(rootCauseServiceId(), other.rootCauseServiceId()) && hasCategories() == other.hasCategories()
                && Objects.equals(categoriesAsStrings(), other.categoriesAsStrings())
                && Objects.equals(stateAsString(), other.stateAsString()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(endTime(), other.endTime()) && Objects.equals(summary(), other.summary())
                && Objects.equals(clientRequestImpactStatistics(), other.clientRequestImpactStatistics())
                && Objects.equals(rootCauseServiceRequestImpactStatistics(), other.rootCauseServiceRequestImpactStatistics())
                && hasTopAnomalousServices() == other.hasTopAnomalousServices()
                && Objects.equals(topAnomalousServices(), other.topAnomalousServices())
                && Objects.equals(lastUpdateTime(), other.lastUpdateTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InsightSummary").add("InsightId", insightId()).add("GroupARN", groupARN())
                .add("GroupName", groupName()).add("RootCauseServiceId", rootCauseServiceId())
                .add("Categories", hasCategories() ? categoriesAsStrings() : null).add("State", stateAsString())
                .add("StartTime", startTime()).add("EndTime", endTime()).add("Summary", summary())
                .add("ClientRequestImpactStatistics", clientRequestImpactStatistics())
                .add("RootCauseServiceRequestImpactStatistics", rootCauseServiceRequestImpactStatistics())
                .add("TopAnomalousServices", hasTopAnomalousServices() ? topAnomalousServices() : null)
                .add("LastUpdateTime", lastUpdateTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InsightId":
            return Optional.ofNullable(clazz.cast(insightId()));
        case "GroupARN":
            return Optional.ofNullable(clazz.cast(groupARN()));
        case "GroupName":
            return Optional.ofNullable(clazz.cast(groupName()));
        case "RootCauseServiceId":
            return Optional.ofNullable(clazz.cast(rootCauseServiceId()));
        case "Categories":
            return Optional.ofNullable(clazz.cast(categoriesAsStrings()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "EndTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "Summary":
            return Optional.ofNullable(clazz.cast(summary()));
        case "ClientRequestImpactStatistics":
            return Optional.ofNullable(clazz.cast(clientRequestImpactStatistics()));
        case "RootCauseServiceRequestImpactStatistics":
            return Optional.ofNullable(clazz.cast(rootCauseServiceRequestImpactStatistics()));
        case "TopAnomalousServices":
            return Optional.ofNullable(clazz.cast(topAnomalousServices()));
        case "LastUpdateTime":
            return Optional.ofNullable(clazz.cast(lastUpdateTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InsightSummary, T> g) {
        return obj -> g.apply((InsightSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InsightSummary> {
        /**
         * <p>
         * The insights unique identifier.
         * </p>
         * 
         * @param insightId
         *        The insights unique identifier.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder insightId(String insightId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the group that the insight belongs to.
         * </p>
         * 
         * @param groupARN
         *        The Amazon Resource Name (ARN) of the group that the insight belongs to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupARN(String groupARN);

        /**
         * <p>
         * The name of the group that the insight belongs to.
         * </p>
         * 
         * @param groupName
         *        The name of the group that the insight belongs to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupName(String groupName);

        /**
         * Sets the value of the RootCauseServiceId property for this object.
         *
         * @param rootCauseServiceId
         *        The new value for the RootCauseServiceId property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rootCauseServiceId(ServiceId rootCauseServiceId);

        /**
         * Sets the value of the RootCauseServiceId property for this object.
         *
         * This is a convenience method that creates an instance of the {@link ServiceId.Builder} avoiding the need to
         * create one manually via {@link ServiceId#builder()}.
         *
         * When the {@link Consumer} completes, {@link ServiceId.Builder#build()} is called immediately and its result
         * is passed to {@link #rootCauseServiceId(ServiceId)}.
         * 
         * @param rootCauseServiceId
         *        a consumer that will call methods on {@link ServiceId.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rootCauseServiceId(ServiceId)
         */
        default Builder rootCauseServiceId(Consumer<ServiceId.Builder> rootCauseServiceId) {
            return rootCauseServiceId(ServiceId.builder().applyMutation(rootCauseServiceId).build());
        }

        /**
         * <p>
         * Categories The categories that label and describe the type of insight.
         * </p>
         * 
         * @param categories
         *        Categories The categories that label and describe the type of insight.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder categoriesWithStrings(Collection<String> categories);

        /**
         * <p>
         * Categories The categories that label and describe the type of insight.
         * </p>
         * 
         * @param categories
         *        Categories The categories that label and describe the type of insight.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder categoriesWithStrings(String... categories);

        /**
         * <p>
         * Categories The categories that label and describe the type of insight.
         * </p>
         * 
         * @param categories
         *        Categories The categories that label and describe the type of insight.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder categories(Collection<InsightCategory> categories);

        /**
         * <p>
         * Categories The categories that label and describe the type of insight.
         * </p>
         * 
         * @param categories
         *        Categories The categories that label and describe the type of insight.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder categories(InsightCategory... categories);

        /**
         * <p>
         * The current state of the insight.
         * </p>
         * 
         * @param state
         *        The current state of the insight.
         * @see InsightState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InsightState
         */
        Builder state(String state);

        /**
         * <p>
         * The current state of the insight.
         * </p>
         * 
         * @param state
         *        The current state of the insight.
         * @see InsightState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InsightState
         */
        Builder state(InsightState state);

        /**
         * <p>
         * The time, in Unix seconds, at which the insight began.
         * </p>
         * 
         * @param startTime
         *        The time, in Unix seconds, at which the insight began.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The time, in Unix seconds, at which the insight ended.
         * </p>
         * 
         * @param endTime
         *        The time, in Unix seconds, at which the insight ended.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * A brief description of the insight.
         * </p>
         * 
         * @param summary
         *        A brief description of the insight.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder summary(String summary);

        /**
         * <p>
         * The impact statistics of the client side service. This includes the number of requests to the client service
         * and whether the requests were faults or okay.
         * </p>
         * 
         * @param clientRequestImpactStatistics
         *        The impact statistics of the client side service. This includes the number of requests to the client
         *        service and whether the requests were faults or okay.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientRequestImpactStatistics(RequestImpactStatistics clientRequestImpactStatistics);

        /**
         * <p>
         * The impact statistics of the client side service. This includes the number of requests to the client service
         * and whether the requests were faults or okay.
         * </p>
         * This is a convenience method that creates an instance of the {@link RequestImpactStatistics.Builder} avoiding
         * the need to create one manually via {@link RequestImpactStatistics#builder()}.
         *
         * When the {@link Consumer} completes, {@link RequestImpactStatistics.Builder#build()} is called immediately
         * and its result is passed to {@link #clientRequestImpactStatistics(RequestImpactStatistics)}.
         * 
         * @param clientRequestImpactStatistics
         *        a consumer that will call methods on {@link RequestImpactStatistics.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #clientRequestImpactStatistics(RequestImpactStatistics)
         */
        default Builder clientRequestImpactStatistics(Consumer<RequestImpactStatistics.Builder> clientRequestImpactStatistics) {
            return clientRequestImpactStatistics(RequestImpactStatistics.builder().applyMutation(clientRequestImpactStatistics)
                    .build());
        }

        /**
         * <p>
         * The impact statistics of the root cause service. This includes the number of requests to the client service
         * and whether the requests were faults or okay.
         * </p>
         * 
         * @param rootCauseServiceRequestImpactStatistics
         *        The impact statistics of the root cause service. This includes the number of requests to the client
         *        service and whether the requests were faults or okay.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rootCauseServiceRequestImpactStatistics(RequestImpactStatistics rootCauseServiceRequestImpactStatistics);

        /**
         * <p>
         * The impact statistics of the root cause service. This includes the number of requests to the client service
         * and whether the requests were faults or okay.
         * </p>
         * This is a convenience method that creates an instance of the {@link RequestImpactStatistics.Builder} avoiding
         * the need to create one manually via {@link RequestImpactStatistics#builder()}.
         *
         * When the {@link Consumer} completes, {@link RequestImpactStatistics.Builder#build()} is called immediately
         * and its result is passed to {@link #rootCauseServiceRequestImpactStatistics(RequestImpactStatistics)}.
         * 
         * @param rootCauseServiceRequestImpactStatistics
         *        a consumer that will call methods on {@link RequestImpactStatistics.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rootCauseServiceRequestImpactStatistics(RequestImpactStatistics)
         */
        default Builder rootCauseServiceRequestImpactStatistics(
                Consumer<RequestImpactStatistics.Builder> rootCauseServiceRequestImpactStatistics) {
            return rootCauseServiceRequestImpactStatistics(RequestImpactStatistics.builder()
                    .applyMutation(rootCauseServiceRequestImpactStatistics).build());
        }

        /**
         * <p>
         * The service within the insight that is most impacted by the incident.
         * </p>
         * 
         * @param topAnomalousServices
         *        The service within the insight that is most impacted by the incident.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder topAnomalousServices(Collection<AnomalousService> topAnomalousServices);

        /**
         * <p>
         * The service within the insight that is most impacted by the incident.
         * </p>
         * 
         * @param topAnomalousServices
         *        The service within the insight that is most impacted by the incident.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder topAnomalousServices(AnomalousService... topAnomalousServices);

        /**
         * <p>
         * The service within the insight that is most impacted by the incident.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<AnomalousService>.Builder} avoiding
         * the need to create one manually via {@link List<AnomalousService>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AnomalousService>.Builder#build()} is called immediately and
         * its result is passed to {@link #topAnomalousServices(List<AnomalousService>)}.
         * 
         * @param topAnomalousServices
         *        a consumer that will call methods on {@link List<AnomalousService>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #topAnomalousServices(List<AnomalousService>)
         */
        Builder topAnomalousServices(Consumer<AnomalousService.Builder>... topAnomalousServices);

        /**
         * <p>
         * The time, in Unix seconds, that the insight was last updated.
         * </p>
         * 
         * @param lastUpdateTime
         *        The time, in Unix seconds, that the insight was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdateTime(Instant lastUpdateTime);
    }

    static final class BuilderImpl implements Builder {
        private String insightId;

        private String groupARN;

        private String groupName;

        private ServiceId rootCauseServiceId;

        private List<String> categories = DefaultSdkAutoConstructList.getInstance();

        private String state;

        private Instant startTime;

        private Instant endTime;

        private String summary;

        private RequestImpactStatistics clientRequestImpactStatistics;

        private RequestImpactStatistics rootCauseServiceRequestImpactStatistics;

        private List<AnomalousService> topAnomalousServices = DefaultSdkAutoConstructList.getInstance();

        private Instant lastUpdateTime;

        private BuilderImpl() {
        }

        private BuilderImpl(InsightSummary model) {
            insightId(model.insightId);
            groupARN(model.groupARN);
            groupName(model.groupName);
            rootCauseServiceId(model.rootCauseServiceId);
            categoriesWithStrings(model.categories);
            state(model.state);
            startTime(model.startTime);
            endTime(model.endTime);
            summary(model.summary);
            clientRequestImpactStatistics(model.clientRequestImpactStatistics);
            rootCauseServiceRequestImpactStatistics(model.rootCauseServiceRequestImpactStatistics);
            topAnomalousServices(model.topAnomalousServices);
            lastUpdateTime(model.lastUpdateTime);
        }

        public final String getInsightId() {
            return insightId;
        }

        public final void setInsightId(String insightId) {
            this.insightId = insightId;
        }

        @Override
        public final Builder insightId(String insightId) {
            this.insightId = insightId;
            return this;
        }

        public final String getGroupARN() {
            return groupARN;
        }

        public final void setGroupARN(String groupARN) {
            this.groupARN = groupARN;
        }

        @Override
        public final Builder groupARN(String groupARN) {
            this.groupARN = groupARN;
            return this;
        }

        public final String getGroupName() {
            return groupName;
        }

        public final void setGroupName(String groupName) {
            this.groupName = groupName;
        }

        @Override
        public final Builder groupName(String groupName) {
            this.groupName = groupName;
            return this;
        }

        public final ServiceId.Builder getRootCauseServiceId() {
            return rootCauseServiceId != null ? rootCauseServiceId.toBuilder() : null;
        }

        public final void setRootCauseServiceId(ServiceId.BuilderImpl rootCauseServiceId) {
            this.rootCauseServiceId = rootCauseServiceId != null ? rootCauseServiceId.build() : null;
        }

        @Override
        public final Builder rootCauseServiceId(ServiceId rootCauseServiceId) {
            this.rootCauseServiceId = rootCauseServiceId;
            return this;
        }

        public final Collection<String> getCategories() {
            if (categories instanceof SdkAutoConstructList) {
                return null;
            }
            return categories;
        }

        public final void setCategories(Collection<String> categories) {
            this.categories = InsightCategoryListCopier.copy(categories);
        }

        @Override
        public final Builder categoriesWithStrings(Collection<String> categories) {
            this.categories = InsightCategoryListCopier.copy(categories);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder categoriesWithStrings(String... categories) {
            categoriesWithStrings(Arrays.asList(categories));
            return this;
        }

        @Override
        public final Builder categories(Collection<InsightCategory> categories) {
            this.categories = InsightCategoryListCopier.copyEnumToString(categories);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder categories(InsightCategory... categories) {
            categories(Arrays.asList(categories));
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(InsightState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final String getSummary() {
            return summary;
        }

        public final void setSummary(String summary) {
            this.summary = summary;
        }

        @Override
        public final Builder summary(String summary) {
            this.summary = summary;
            return this;
        }

        public final RequestImpactStatistics.Builder getClientRequestImpactStatistics() {
            return clientRequestImpactStatistics != null ? clientRequestImpactStatistics.toBuilder() : null;
        }

        public final void setClientRequestImpactStatistics(RequestImpactStatistics.BuilderImpl clientRequestImpactStatistics) {
            this.clientRequestImpactStatistics = clientRequestImpactStatistics != null ? clientRequestImpactStatistics.build()
                    : null;
        }

        @Override
        public final Builder clientRequestImpactStatistics(RequestImpactStatistics clientRequestImpactStatistics) {
            this.clientRequestImpactStatistics = clientRequestImpactStatistics;
            return this;
        }

        public final RequestImpactStatistics.Builder getRootCauseServiceRequestImpactStatistics() {
            return rootCauseServiceRequestImpactStatistics != null ? rootCauseServiceRequestImpactStatistics.toBuilder() : null;
        }

        public final void setRootCauseServiceRequestImpactStatistics(
                RequestImpactStatistics.BuilderImpl rootCauseServiceRequestImpactStatistics) {
            this.rootCauseServiceRequestImpactStatistics = rootCauseServiceRequestImpactStatistics != null ? rootCauseServiceRequestImpactStatistics
                    .build() : null;
        }

        @Override
        public final Builder rootCauseServiceRequestImpactStatistics(
                RequestImpactStatistics rootCauseServiceRequestImpactStatistics) {
            this.rootCauseServiceRequestImpactStatistics = rootCauseServiceRequestImpactStatistics;
            return this;
        }

        public final List<AnomalousService.Builder> getTopAnomalousServices() {
            List<AnomalousService.Builder> result = AnomalousServiceListCopier.copyToBuilder(this.topAnomalousServices);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTopAnomalousServices(Collection<AnomalousService.BuilderImpl> topAnomalousServices) {
            this.topAnomalousServices = AnomalousServiceListCopier.copyFromBuilder(topAnomalousServices);
        }

        @Override
        public final Builder topAnomalousServices(Collection<AnomalousService> topAnomalousServices) {
            this.topAnomalousServices = AnomalousServiceListCopier.copy(topAnomalousServices);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder topAnomalousServices(AnomalousService... topAnomalousServices) {
            topAnomalousServices(Arrays.asList(topAnomalousServices));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder topAnomalousServices(Consumer<AnomalousService.Builder>... topAnomalousServices) {
            topAnomalousServices(Stream.of(topAnomalousServices).map(c -> AnomalousService.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Instant getLastUpdateTime() {
            return lastUpdateTime;
        }

        public final void setLastUpdateTime(Instant lastUpdateTime) {
            this.lastUpdateTime = lastUpdateTime;
        }

        @Override
        public final Builder lastUpdateTime(Instant lastUpdateTime) {
            this.lastUpdateTime = lastUpdateTime;
            return this;
        }

        @Override
        public InsightSummary build() {
            return new InsightSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
