/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.xray;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.xray.internal.XRayServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.xray.model.BatchGetTracesRequest;
import software.amazon.awssdk.services.xray.model.BatchGetTracesResponse;
import software.amazon.awssdk.services.xray.model.CreateGroupRequest;
import software.amazon.awssdk.services.xray.model.CreateGroupResponse;
import software.amazon.awssdk.services.xray.model.CreateSamplingRuleRequest;
import software.amazon.awssdk.services.xray.model.CreateSamplingRuleResponse;
import software.amazon.awssdk.services.xray.model.DeleteGroupRequest;
import software.amazon.awssdk.services.xray.model.DeleteGroupResponse;
import software.amazon.awssdk.services.xray.model.DeleteResourcePolicyRequest;
import software.amazon.awssdk.services.xray.model.DeleteResourcePolicyResponse;
import software.amazon.awssdk.services.xray.model.DeleteSamplingRuleRequest;
import software.amazon.awssdk.services.xray.model.DeleteSamplingRuleResponse;
import software.amazon.awssdk.services.xray.model.GetEncryptionConfigRequest;
import software.amazon.awssdk.services.xray.model.GetEncryptionConfigResponse;
import software.amazon.awssdk.services.xray.model.GetGroupRequest;
import software.amazon.awssdk.services.xray.model.GetGroupResponse;
import software.amazon.awssdk.services.xray.model.GetGroupsRequest;
import software.amazon.awssdk.services.xray.model.GetGroupsResponse;
import software.amazon.awssdk.services.xray.model.GetInsightEventsRequest;
import software.amazon.awssdk.services.xray.model.GetInsightEventsResponse;
import software.amazon.awssdk.services.xray.model.GetInsightImpactGraphRequest;
import software.amazon.awssdk.services.xray.model.GetInsightImpactGraphResponse;
import software.amazon.awssdk.services.xray.model.GetInsightRequest;
import software.amazon.awssdk.services.xray.model.GetInsightResponse;
import software.amazon.awssdk.services.xray.model.GetInsightSummariesRequest;
import software.amazon.awssdk.services.xray.model.GetInsightSummariesResponse;
import software.amazon.awssdk.services.xray.model.GetSamplingRulesRequest;
import software.amazon.awssdk.services.xray.model.GetSamplingRulesResponse;
import software.amazon.awssdk.services.xray.model.GetSamplingStatisticSummariesRequest;
import software.amazon.awssdk.services.xray.model.GetSamplingStatisticSummariesResponse;
import software.amazon.awssdk.services.xray.model.GetSamplingTargetsRequest;
import software.amazon.awssdk.services.xray.model.GetSamplingTargetsResponse;
import software.amazon.awssdk.services.xray.model.GetServiceGraphRequest;
import software.amazon.awssdk.services.xray.model.GetServiceGraphResponse;
import software.amazon.awssdk.services.xray.model.GetTimeSeriesServiceStatisticsRequest;
import software.amazon.awssdk.services.xray.model.GetTimeSeriesServiceStatisticsResponse;
import software.amazon.awssdk.services.xray.model.GetTraceGraphRequest;
import software.amazon.awssdk.services.xray.model.GetTraceGraphResponse;
import software.amazon.awssdk.services.xray.model.GetTraceSummariesRequest;
import software.amazon.awssdk.services.xray.model.GetTraceSummariesResponse;
import software.amazon.awssdk.services.xray.model.InvalidPolicyRevisionIdException;
import software.amazon.awssdk.services.xray.model.InvalidRequestException;
import software.amazon.awssdk.services.xray.model.ListResourcePoliciesRequest;
import software.amazon.awssdk.services.xray.model.ListResourcePoliciesResponse;
import software.amazon.awssdk.services.xray.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.xray.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.xray.model.LockoutPreventionException;
import software.amazon.awssdk.services.xray.model.MalformedPolicyDocumentException;
import software.amazon.awssdk.services.xray.model.PolicyCountLimitExceededException;
import software.amazon.awssdk.services.xray.model.PolicySizeLimitExceededException;
import software.amazon.awssdk.services.xray.model.PutEncryptionConfigRequest;
import software.amazon.awssdk.services.xray.model.PutEncryptionConfigResponse;
import software.amazon.awssdk.services.xray.model.PutResourcePolicyRequest;
import software.amazon.awssdk.services.xray.model.PutResourcePolicyResponse;
import software.amazon.awssdk.services.xray.model.PutTelemetryRecordsRequest;
import software.amazon.awssdk.services.xray.model.PutTelemetryRecordsResponse;
import software.amazon.awssdk.services.xray.model.PutTraceSegmentsRequest;
import software.amazon.awssdk.services.xray.model.PutTraceSegmentsResponse;
import software.amazon.awssdk.services.xray.model.ResourceNotFoundException;
import software.amazon.awssdk.services.xray.model.RuleLimitExceededException;
import software.amazon.awssdk.services.xray.model.TagResourceRequest;
import software.amazon.awssdk.services.xray.model.TagResourceResponse;
import software.amazon.awssdk.services.xray.model.ThrottledException;
import software.amazon.awssdk.services.xray.model.TooManyTagsException;
import software.amazon.awssdk.services.xray.model.UntagResourceRequest;
import software.amazon.awssdk.services.xray.model.UntagResourceResponse;
import software.amazon.awssdk.services.xray.model.UpdateGroupRequest;
import software.amazon.awssdk.services.xray.model.UpdateGroupResponse;
import software.amazon.awssdk.services.xray.model.UpdateSamplingRuleRequest;
import software.amazon.awssdk.services.xray.model.UpdateSamplingRuleResponse;
import software.amazon.awssdk.services.xray.model.XRayException;
import software.amazon.awssdk.services.xray.transform.BatchGetTracesRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.CreateGroupRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.CreateSamplingRuleRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.DeleteGroupRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.DeleteResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.DeleteSamplingRuleRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetEncryptionConfigRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetGroupRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetGroupsRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetInsightEventsRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetInsightImpactGraphRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetInsightRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetInsightSummariesRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetSamplingRulesRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetSamplingStatisticSummariesRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetSamplingTargetsRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetServiceGraphRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetTimeSeriesServiceStatisticsRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetTraceGraphRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.GetTraceSummariesRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.ListResourcePoliciesRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.PutEncryptionConfigRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.PutResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.PutTelemetryRecordsRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.PutTraceSegmentsRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.UpdateGroupRequestMarshaller;
import software.amazon.awssdk.services.xray.transform.UpdateSamplingRuleRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link XRayAsyncClient}.
 *
 * @see XRayAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultXRayAsyncClient implements XRayAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultXRayAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultXRayAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Retrieves a list of traces specified by ID. Each trace is a collection of segment documents that originates from
     * a single request. Use <code>GetTraceSummaries</code> to get a list of trace IDs.
     * </p>
     *
     * @param batchGetTracesRequest
     * @return A Java Future containing the result of the BatchGetTraces operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.BatchGetTraces
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/BatchGetTraces" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetTracesResponse> batchGetTraces(BatchGetTracesRequest batchGetTracesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchGetTracesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetTracesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetTraces");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetTracesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGetTracesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGetTracesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetTracesRequest, BatchGetTracesResponse>()
                            .withOperationName("BatchGetTraces").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchGetTracesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchGetTracesRequest));
            CompletableFuture<BatchGetTracesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a group resource with a name and a filter expression.
     * </p>
     *
     * @param createGroupRequest
     * @return A Java Future containing the result of the CreateGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.CreateGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/CreateGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGroupResponse> createGroup(CreateGroupRequest createGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createGroupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGroupRequest, CreateGroupResponse>()
                            .withOperationName("CreateGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createGroupRequest));
            CompletableFuture<CreateGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a rule to control sampling behavior for instrumented applications. Services retrieve rules with <a
     * href="https://docs.aws.amazon.com/xray/latest/api/API_GetSamplingRules.html">GetSamplingRules</a>, and evaluate
     * each rule in ascending order of <i>priority</i> for each request. If a rule matches, the service records a trace,
     * borrowing it from the reservoir size. After 10 seconds, the service reports back to X-Ray with <a
     * href="https://docs.aws.amazon.com/xray/latest/api/API_GetSamplingTargets.html">GetSamplingTargets</a> to get
     * updated versions of each in-use rule. The updated rule contains a trace quota that the service can use instead of
     * borrowing from the reservoir.
     * </p>
     *
     * @param createSamplingRuleRequest
     * @return A Java Future containing the result of the CreateSamplingRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>RuleLimitExceededException You have reached the maximum number of sampling rules.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.CreateSamplingRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/CreateSamplingRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSamplingRuleResponse> createSamplingRule(CreateSamplingRuleRequest createSamplingRuleRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSamplingRuleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSamplingRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSamplingRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSamplingRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSamplingRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSamplingRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSamplingRuleRequest, CreateSamplingRuleResponse>()
                            .withOperationName("CreateSamplingRule").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateSamplingRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createSamplingRuleRequest));
            CompletableFuture<CreateSamplingRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a group resource.
     * </p>
     *
     * @param deleteGroupRequest
     * @return A Java Future containing the result of the DeleteGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.DeleteGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/DeleteGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteGroupResponse> deleteGroup(DeleteGroupRequest deleteGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteGroupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteGroupRequest, DeleteGroupResponse>()
                            .withOperationName("DeleteGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteGroupRequest));
            CompletableFuture<DeleteGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a resource policy from the target Amazon Web Services account.
     * </p>
     *
     * @param deleteResourcePolicyRequest
     * @return A Java Future containing the result of the DeleteResourcePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>InvalidPolicyRevisionIdException A policy revision id was provided which does not match the latest
     *         policy revision. This exception is also if a policy revision id of 0 is provided via
     *         <code>PutResourcePolicy</code> and a policy with the same name already exists.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.DeleteResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/DeleteResourcePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteResourcePolicyResponse> deleteResourcePolicy(
            DeleteResourcePolicyRequest deleteResourcePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourcePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteResourcePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteResourcePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteResourcePolicyRequest, DeleteResourcePolicyResponse>()
                            .withOperationName("DeleteResourcePolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteResourcePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteResourcePolicyRequest));
            CompletableFuture<DeleteResourcePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a sampling rule.
     * </p>
     *
     * @param deleteSamplingRuleRequest
     * @return A Java Future containing the result of the DeleteSamplingRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.DeleteSamplingRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/DeleteSamplingRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSamplingRuleResponse> deleteSamplingRule(DeleteSamplingRuleRequest deleteSamplingRuleRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSamplingRuleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSamplingRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSamplingRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSamplingRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSamplingRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSamplingRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSamplingRuleRequest, DeleteSamplingRuleResponse>()
                            .withOperationName("DeleteSamplingRule").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteSamplingRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSamplingRuleRequest));
            CompletableFuture<DeleteSamplingRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the current encryption configuration for X-Ray data.
     * </p>
     *
     * @param getEncryptionConfigRequest
     * @return A Java Future containing the result of the GetEncryptionConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetEncryptionConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetEncryptionConfig" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetEncryptionConfigResponse> getEncryptionConfig(
            GetEncryptionConfigRequest getEncryptionConfigRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEncryptionConfigRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEncryptionConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEncryptionConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEncryptionConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEncryptionConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEncryptionConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEncryptionConfigRequest, GetEncryptionConfigResponse>()
                            .withOperationName("GetEncryptionConfig").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetEncryptionConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getEncryptionConfigRequest));
            CompletableFuture<GetEncryptionConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves group resource details.
     * </p>
     *
     * @param getGroupRequest
     * @return A Java Future containing the result of the GetGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetGroupResponse> getGroup(GetGroupRequest getGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getGroupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetGroupRequest, GetGroupResponse>().withOperationName("GetGroup")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetGroupRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getGroupRequest));
            CompletableFuture<GetGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves all active group details.
     * </p>
     *
     * @param getGroupsRequest
     * @return A Java Future containing the result of the GetGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetGroupsResponse> getGroups(GetGroupsRequest getGroupsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getGroupsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetGroupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetGroupsRequest, GetGroupsResponse>().withOperationName("GetGroups")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetGroupsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getGroupsRequest));
            CompletableFuture<GetGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the summary information of an insight. This includes impact to clients and root cause services, the top
     * anomalous services, the category, the state of the insight, and the start and end time of the insight.
     * </p>
     *
     * @param getInsightRequest
     * @return A Java Future containing the result of the GetInsight operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetInsight
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetInsight" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetInsightResponse> getInsight(GetInsightRequest getInsightRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getInsightRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInsightRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInsight");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetInsightResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetInsightResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetInsightResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetInsightRequest, GetInsightResponse>().withOperationName("GetInsight")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetInsightRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getInsightRequest));
            CompletableFuture<GetInsightResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * X-Ray reevaluates insights periodically until they're resolved, and records each intermediate state as an event.
     * You can review an insight's events in the Impact Timeline on the Inspect page in the X-Ray console.
     * </p>
     *
     * @param getInsightEventsRequest
     * @return A Java Future containing the result of the GetInsightEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetInsightEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetInsightEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetInsightEventsResponse> getInsightEvents(GetInsightEventsRequest getInsightEventsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getInsightEventsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInsightEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInsightEvents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetInsightEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetInsightEventsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetInsightEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetInsightEventsRequest, GetInsightEventsResponse>()
                            .withOperationName("GetInsightEvents").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetInsightEventsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getInsightEventsRequest));
            CompletableFuture<GetInsightEventsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a service graph structure filtered by the specified insight. The service graph is limited to only
     * structural information. For a complete service graph, use this API with the GetServiceGraph API.
     * </p>
     *
     * @param getInsightImpactGraphRequest
     * @return A Java Future containing the result of the GetInsightImpactGraph operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetInsightImpactGraph
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetInsightImpactGraph" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetInsightImpactGraphResponse> getInsightImpactGraph(
            GetInsightImpactGraphRequest getInsightImpactGraphRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getInsightImpactGraphRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInsightImpactGraphRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInsightImpactGraph");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetInsightImpactGraphResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetInsightImpactGraphResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetInsightImpactGraphResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetInsightImpactGraphRequest, GetInsightImpactGraphResponse>()
                            .withOperationName("GetInsightImpactGraph").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetInsightImpactGraphRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getInsightImpactGraphRequest));
            CompletableFuture<GetInsightImpactGraphResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the summaries of all insights in the specified group matching the provided filter values.
     * </p>
     *
     * @param getInsightSummariesRequest
     * @return A Java Future containing the result of the GetInsightSummaries operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetInsightSummaries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetInsightSummaries" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetInsightSummariesResponse> getInsightSummaries(
            GetInsightSummariesRequest getInsightSummariesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getInsightSummariesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInsightSummariesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInsightSummaries");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetInsightSummariesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetInsightSummariesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetInsightSummariesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetInsightSummariesRequest, GetInsightSummariesResponse>()
                            .withOperationName("GetInsightSummaries").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetInsightSummariesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getInsightSummariesRequest));
            CompletableFuture<GetInsightSummariesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves all sampling rules.
     * </p>
     *
     * @param getSamplingRulesRequest
     * @return A Java Future containing the result of the GetSamplingRules operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetSamplingRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetSamplingRules" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSamplingRulesResponse> getSamplingRules(GetSamplingRulesRequest getSamplingRulesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSamplingRulesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSamplingRulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSamplingRules");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSamplingRulesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSamplingRulesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSamplingRulesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSamplingRulesRequest, GetSamplingRulesResponse>()
                            .withOperationName("GetSamplingRules").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSamplingRulesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSamplingRulesRequest));
            CompletableFuture<GetSamplingRulesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about recent sampling results for all sampling rules.
     * </p>
     *
     * @param getSamplingStatisticSummariesRequest
     * @return A Java Future containing the result of the GetSamplingStatisticSummaries operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetSamplingStatisticSummaries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetSamplingStatisticSummaries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSamplingStatisticSummariesResponse> getSamplingStatisticSummaries(
            GetSamplingStatisticSummariesRequest getSamplingStatisticSummariesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSamplingStatisticSummariesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getSamplingStatisticSummariesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSamplingStatisticSummaries");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSamplingStatisticSummariesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSamplingStatisticSummariesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSamplingStatisticSummariesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSamplingStatisticSummariesRequest, GetSamplingStatisticSummariesResponse>()
                            .withOperationName("GetSamplingStatisticSummaries").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSamplingStatisticSummariesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSamplingStatisticSummariesRequest));
            CompletableFuture<GetSamplingStatisticSummariesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Requests a sampling quota for rules that the service is using to sample requests.
     * </p>
     *
     * @param getSamplingTargetsRequest
     * @return A Java Future containing the result of the GetSamplingTargets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetSamplingTargets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetSamplingTargets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSamplingTargetsResponse> getSamplingTargets(GetSamplingTargetsRequest getSamplingTargetsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSamplingTargetsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSamplingTargetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSamplingTargets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSamplingTargetsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSamplingTargetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSamplingTargetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSamplingTargetsRequest, GetSamplingTargetsResponse>()
                            .withOperationName("GetSamplingTargets").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSamplingTargetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSamplingTargetsRequest));
            CompletableFuture<GetSamplingTargetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a document that describes services that process incoming requests, and downstream services that they
     * call as a result. Root services process incoming requests and make calls to downstream services. Root services
     * are applications that use the <a href="https://docs.aws.amazon.com/xray/index.html">Amazon Web Services X-Ray
     * SDK</a>. Downstream services can be other applications, Amazon Web Services resources, HTTP web APIs, or SQL
     * databases.
     * </p>
     *
     * @param getServiceGraphRequest
     * @return A Java Future containing the result of the GetServiceGraph operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetServiceGraph
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetServiceGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetServiceGraphResponse> getServiceGraph(GetServiceGraphRequest getServiceGraphRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getServiceGraphRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServiceGraphRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetServiceGraph");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetServiceGraphResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetServiceGraphResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetServiceGraphResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetServiceGraphRequest, GetServiceGraphResponse>()
                            .withOperationName("GetServiceGraph").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetServiceGraphRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getServiceGraphRequest));
            CompletableFuture<GetServiceGraphResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get an aggregation of service statistics defined by a specific time range.
     * </p>
     *
     * @param getTimeSeriesServiceStatisticsRequest
     * @return A Java Future containing the result of the GetTimeSeriesServiceStatistics operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetTimeSeriesServiceStatistics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetTimeSeriesServiceStatistics"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTimeSeriesServiceStatisticsResponse> getTimeSeriesServiceStatistics(
            GetTimeSeriesServiceStatisticsRequest getTimeSeriesServiceStatisticsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTimeSeriesServiceStatisticsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getTimeSeriesServiceStatisticsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTimeSeriesServiceStatistics");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTimeSeriesServiceStatisticsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTimeSeriesServiceStatisticsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTimeSeriesServiceStatisticsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTimeSeriesServiceStatisticsRequest, GetTimeSeriesServiceStatisticsResponse>()
                            .withOperationName("GetTimeSeriesServiceStatistics").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTimeSeriesServiceStatisticsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTimeSeriesServiceStatisticsRequest));
            CompletableFuture<GetTimeSeriesServiceStatisticsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a service graph for one or more specific trace IDs.
     * </p>
     *
     * @param getTraceGraphRequest
     * @return A Java Future containing the result of the GetTraceGraph operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetTraceGraph
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetTraceGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTraceGraphResponse> getTraceGraph(GetTraceGraphRequest getTraceGraphRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTraceGraphRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTraceGraphRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTraceGraph");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTraceGraphResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetTraceGraphResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTraceGraphResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTraceGraphRequest, GetTraceGraphResponse>()
                            .withOperationName("GetTraceGraph").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTraceGraphRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTraceGraphRequest));
            CompletableFuture<GetTraceGraphResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves IDs and annotations for traces available for a specified time frame using an optional filter. To get
     * the full traces, pass the trace IDs to <code>BatchGetTraces</code>.
     * </p>
     * <p>
     * A filter expression can target traced requests that hit specific service nodes or edges, have errors, or come
     * from a known user. For example, the following filter expression targets traces that pass through
     * <code>api.example.com</code>:
     * </p>
     * <p>
     * <code>service("api.example.com")</code>
     * </p>
     * <p>
     * This filter expression finds traces that have an annotation named <code>account</code> with the value
     * <code>12345</code>:
     * </p>
     * <p>
     * <code>annotation.account = "12345"</code>
     * </p>
     * <p>
     * For a full list of indexed fields and keywords that you can use in filter expressions, see <a
     * href="https://docs.aws.amazon.com/xray/latest/devguide/xray-console-filters.html">Using Filter Expressions</a> in
     * the <i>Amazon Web Services X-Ray Developer Guide</i>.
     * </p>
     *
     * @param getTraceSummariesRequest
     * @return A Java Future containing the result of the GetTraceSummaries operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.GetTraceSummaries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/GetTraceSummaries" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTraceSummariesResponse> getTraceSummaries(GetTraceSummariesRequest getTraceSummariesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTraceSummariesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTraceSummariesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTraceSummaries");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTraceSummariesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTraceSummariesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTraceSummariesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTraceSummariesRequest, GetTraceSummariesResponse>()
                            .withOperationName("GetTraceSummaries").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTraceSummariesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTraceSummariesRequest));
            CompletableFuture<GetTraceSummariesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the list of resource policies in the target Amazon Web Services account.
     * </p>
     *
     * @param listResourcePoliciesRequest
     * @return A Java Future containing the result of the ListResourcePolicies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.ListResourcePolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/ListResourcePolicies" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListResourcePoliciesResponse> listResourcePolicies(
            ListResourcePoliciesRequest listResourcePoliciesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listResourcePoliciesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourcePoliciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResourcePolicies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResourcePoliciesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListResourcePoliciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResourcePoliciesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResourcePoliciesRequest, ListResourcePoliciesResponse>()
                            .withOperationName("ListResourcePolicies").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListResourcePoliciesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listResourcePoliciesRequest));
            CompletableFuture<ListResourcePoliciesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of tags that are applied to the specified Amazon Web Services X-Ray group or sampling rule.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>ResourceNotFoundException The resource was not found. Verify that the name or Amazon Resource Name
     *         (ARN) of the resource is correct.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the encryption configuration for X-Ray data.
     * </p>
     *
     * @param putEncryptionConfigRequest
     * @return A Java Future containing the result of the PutEncryptionConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.PutEncryptionConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/PutEncryptionConfig" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutEncryptionConfigResponse> putEncryptionConfig(
            PutEncryptionConfigRequest putEncryptionConfigRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putEncryptionConfigRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putEncryptionConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutEncryptionConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutEncryptionConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutEncryptionConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutEncryptionConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutEncryptionConfigRequest, PutEncryptionConfigResponse>()
                            .withOperationName("PutEncryptionConfig").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutEncryptionConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putEncryptionConfigRequest));
            CompletableFuture<PutEncryptionConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets the resource policy to grant one or more Amazon Web Services services and accounts permissions to access
     * X-Ray. Each resource policy will be associated with a specific Amazon Web Services account. Each Amazon Web
     * Services account can have a maximum of 5 resource policies, and each policy name must be unique within that
     * account. The maximum size of each resource policy is 5KB.
     * </p>
     *
     * @param putResourcePolicyRequest
     * @return A Java Future containing the result of the PutResourcePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>MalformedPolicyDocumentException Invalid policy document provided in request.</li>
     *         <li>LockoutPreventionException The provided resource policy would prevent the caller of this request from
     *         calling PutResourcePolicy in the future.</li>
     *         <li>InvalidPolicyRevisionIdException A policy revision id was provided which does not match the latest
     *         policy revision. This exception is also if a policy revision id of 0 is provided via
     *         <code>PutResourcePolicy</code> and a policy with the same name already exists.</li>
     *         <li>PolicySizeLimitExceededException Exceeded the maximum size for a resource policy.</li>
     *         <li>PolicyCountLimitExceededException Exceeded the maximum number of resource policies for a target
     *         Amazon Web Services account.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.PutResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/PutResourcePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutResourcePolicyResponse> putResourcePolicy(PutResourcePolicyRequest putResourcePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutResourcePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutResourcePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutResourcePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutResourcePolicyRequest, PutResourcePolicyResponse>()
                            .withOperationName("PutResourcePolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutResourcePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putResourcePolicyRequest));
            CompletableFuture<PutResourcePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used by the Amazon Web Services X-Ray daemon to upload telemetry.
     * </p>
     *
     * @param putTelemetryRecordsRequest
     * @return A Java Future containing the result of the PutTelemetryRecords operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.PutTelemetryRecords
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/PutTelemetryRecords" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutTelemetryRecordsResponse> putTelemetryRecords(
            PutTelemetryRecordsRequest putTelemetryRecordsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putTelemetryRecordsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putTelemetryRecordsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutTelemetryRecords");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutTelemetryRecordsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutTelemetryRecordsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutTelemetryRecordsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutTelemetryRecordsRequest, PutTelemetryRecordsResponse>()
                            .withOperationName("PutTelemetryRecords").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutTelemetryRecordsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putTelemetryRecordsRequest));
            CompletableFuture<PutTelemetryRecordsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Uploads segment documents to Amazon Web Services X-Ray. The <a
     * href="https://docs.aws.amazon.com/xray/index.html">X-Ray SDK</a> generates segment documents and sends them to
     * the X-Ray daemon, which uploads them in batches. A segment document can be a completed segment, an in-progress
     * segment, or an array of subsegments.
     * </p>
     * <p>
     * Segments must include the following fields. For the full segment document schema, see <a
     * href="https://docs.aws.amazon.com/xray/latest/devguide/xray-api-segmentdocuments.html">Amazon Web Services X-Ray
     * Segment Documents</a> in the <i>Amazon Web Services X-Ray Developer Guide</i>.
     * </p>
     * <p class="title">
     * <b>Required segment document fields</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>name</code> - The name of the service that handled the request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>id</code> - A 64-bit identifier for the segment, unique among segments in the same trace, in 16 hexadecimal
     * digits.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>trace_id</code> - A unique identifier that connects all segments and subsegments originating from a single
     * client request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>start_time</code> - Time the segment or subsegment was created, in floating point seconds in epoch time,
     * accurate to milliseconds. For example, <code>1480615200.010</code> or <code>1.480615200010E9</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>end_time</code> - Time the segment or subsegment was closed. For example, <code>1480615200.090</code> or
     * <code>1.480615200090E9</code>. Specify either an <code>end_time</code> or <code>in_progress</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>in_progress</code> - Set to <code>true</code> instead of specifying an <code>end_time</code> to record that
     * a segment has been started, but is not complete. Send an in-progress segment when your application receives a
     * request that will take a long time to serve, to trace that the request was received. When the response is sent,
     * send the complete segment to overwrite the in-progress segment.
     * </p>
     * </li>
     * </ul>
     * <p>
     * A <code>trace_id</code> consists of three numbers separated by hyphens. For example,
     * 1-58406520-a006649127e371903a2de979. This includes:
     * </p>
     * <p class="title">
     * <b>Trace ID Format</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * The version number, for instance, <code>1</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * The time of the original request, in Unix epoch time, in 8 hexadecimal digits. For example, 10:00AM December 2nd,
     * 2016 PST in epoch time is <code>1480615200</code> seconds, or <code>58406520</code> in hexadecimal.
     * </p>
     * </li>
     * <li>
     * <p>
     * A 96-bit identifier for the trace, globally unique, in 24 hexadecimal digits.
     * </p>
     * </li>
     * </ul>
     *
     * @param putTraceSegmentsRequest
     * @return A Java Future containing the result of the PutTraceSegments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.PutTraceSegments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/PutTraceSegments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutTraceSegmentsResponse> putTraceSegments(PutTraceSegmentsRequest putTraceSegmentsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putTraceSegmentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putTraceSegmentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutTraceSegments");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutTraceSegmentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutTraceSegmentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutTraceSegmentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutTraceSegmentsRequest, PutTraceSegmentsResponse>()
                            .withOperationName("PutTraceSegments").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutTraceSegmentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putTraceSegmentsRequest));
            CompletableFuture<PutTraceSegmentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Applies tags to an existing Amazon Web Services X-Ray group or sampling rule.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>ResourceNotFoundException The resource was not found. Verify that the name or Amazon Resource Name
     *         (ARN) of the resource is correct.</li>
     *         <li>TooManyTagsException You have exceeded the maximum number of tags you can apply to this resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from an Amazon Web Services X-Ray group or sampling rule. You cannot edit or delete system tags
     * (those with an <code>aws:</code> prefix).
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>ResourceNotFoundException The resource was not found. Verify that the name or Amazon Resource Name
     *         (ARN) of the resource is correct.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a group resource.
     * </p>
     *
     * @param updateGroupRequest
     * @return A Java Future containing the result of the UpdateGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.UpdateGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/UpdateGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateGroupResponse> updateGroup(UpdateGroupRequest updateGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateGroupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateGroupRequest, UpdateGroupResponse>()
                            .withOperationName("UpdateGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateGroupRequest));
            CompletableFuture<UpdateGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies a sampling rule's configuration.
     * </p>
     *
     * @param updateSamplingRuleRequest
     * @return A Java Future containing the result of the UpdateSamplingRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is missing required parameters or has invalid parameters.</li>
     *         <li>ThrottledException The request exceeds the maximum number of requests per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>XRayException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample XRayAsyncClient.UpdateSamplingRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/UpdateSamplingRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSamplingRuleResponse> updateSamplingRule(UpdateSamplingRuleRequest updateSamplingRuleRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateSamplingRuleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSamplingRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "XRay");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSamplingRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSamplingRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSamplingRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSamplingRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSamplingRuleRequest, UpdateSamplingRuleResponse>()
                            .withOperationName("UpdateSamplingRule").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateSamplingRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateSamplingRuleRequest));
            CompletableFuture<UpdateSamplingRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final XRayServiceClientConfiguration serviceClientConfiguration() {
        return new XRayServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(XRayException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MalformedPolicyDocumentException")
                                .exceptionBuilderSupplier(MalformedPolicyDocumentException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPolicyRevisionIdException")
                                .exceptionBuilderSupplier(InvalidPolicyRevisionIdException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottledException")
                                .exceptionBuilderSupplier(ThrottledException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RuleLimitExceededException")
                                .exceptionBuilderSupplier(RuleLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PolicyCountLimitExceededException")
                                .exceptionBuilderSupplier(PolicyCountLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LockoutPreventionException")
                                .exceptionBuilderSupplier(LockoutPreventionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PolicySizeLimitExceededException")
                                .exceptionBuilderSupplier(PolicySizeLimitExceededException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        XRayServiceClientConfigurationBuilder serviceConfigBuilder = new XRayServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
