/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.xray.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * X-Ray reevaluates insights periodically until they are resolved, and records each intermediate state in an event. You
 * can review incident events in the Impact Timeline on the Inspect page in the X-Ray console.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InsightEvent implements SdkPojo, Serializable, ToCopyableBuilder<InsightEvent.Builder, InsightEvent> {
    private static final SdkField<String> SUMMARY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Summary")
            .getter(getter(InsightEvent::summary)).setter(setter(Builder::summary))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Summary").build()).build();

    private static final SdkField<Instant> EVENT_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EventTime").getter(getter(InsightEvent::eventTime)).setter(setter(Builder::eventTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventTime").build()).build();

    private static final SdkField<RequestImpactStatistics> CLIENT_REQUEST_IMPACT_STATISTICS_FIELD = SdkField
            .<RequestImpactStatistics> builder(MarshallingType.SDK_POJO)
            .memberName("ClientRequestImpactStatistics")
            .getter(getter(InsightEvent::clientRequestImpactStatistics))
            .setter(setter(Builder::clientRequestImpactStatistics))
            .constructor(RequestImpactStatistics::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientRequestImpactStatistics")
                    .build()).build();

    private static final SdkField<RequestImpactStatistics> ROOT_CAUSE_SERVICE_REQUEST_IMPACT_STATISTICS_FIELD = SdkField
            .<RequestImpactStatistics> builder(MarshallingType.SDK_POJO)
            .memberName("RootCauseServiceRequestImpactStatistics")
            .getter(getter(InsightEvent::rootCauseServiceRequestImpactStatistics))
            .setter(setter(Builder::rootCauseServiceRequestImpactStatistics))
            .constructor(RequestImpactStatistics::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("RootCauseServiceRequestImpactStatistics").build()).build();

    private static final SdkField<List<AnomalousService>> TOP_ANOMALOUS_SERVICES_FIELD = SdkField
            .<List<AnomalousService>> builder(MarshallingType.LIST)
            .memberName("TopAnomalousServices")
            .getter(getter(InsightEvent::topAnomalousServices))
            .setter(setter(Builder::topAnomalousServices))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TopAnomalousServices").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AnomalousService> builder(MarshallingType.SDK_POJO)
                                            .constructor(AnomalousService::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SUMMARY_FIELD,
            EVENT_TIME_FIELD, CLIENT_REQUEST_IMPACT_STATISTICS_FIELD, ROOT_CAUSE_SERVICE_REQUEST_IMPACT_STATISTICS_FIELD,
            TOP_ANOMALOUS_SERVICES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String summary;

    private final Instant eventTime;

    private final RequestImpactStatistics clientRequestImpactStatistics;

    private final RequestImpactStatistics rootCauseServiceRequestImpactStatistics;

    private final List<AnomalousService> topAnomalousServices;

    private InsightEvent(BuilderImpl builder) {
        this.summary = builder.summary;
        this.eventTime = builder.eventTime;
        this.clientRequestImpactStatistics = builder.clientRequestImpactStatistics;
        this.rootCauseServiceRequestImpactStatistics = builder.rootCauseServiceRequestImpactStatistics;
        this.topAnomalousServices = builder.topAnomalousServices;
    }

    /**
     * <p>
     * A brief description of the event.
     * </p>
     * 
     * @return A brief description of the event.
     */
    public final String summary() {
        return summary;
    }

    /**
     * <p>
     * The time, in Unix seconds, at which the event was recorded.
     * </p>
     * 
     * @return The time, in Unix seconds, at which the event was recorded.
     */
    public final Instant eventTime() {
        return eventTime;
    }

    /**
     * <p>
     * The impact statistics of the client side service. This includes the number of requests to the client service and
     * whether the requests were faults or okay.
     * </p>
     * 
     * @return The impact statistics of the client side service. This includes the number of requests to the client
     *         service and whether the requests were faults or okay.
     */
    public final RequestImpactStatistics clientRequestImpactStatistics() {
        return clientRequestImpactStatistics;
    }

    /**
     * <p>
     * The impact statistics of the root cause service. This includes the number of requests to the client service and
     * whether the requests were faults or okay.
     * </p>
     * 
     * @return The impact statistics of the root cause service. This includes the number of requests to the client
     *         service and whether the requests were faults or okay.
     */
    public final RequestImpactStatistics rootCauseServiceRequestImpactStatistics() {
        return rootCauseServiceRequestImpactStatistics;
    }

    /**
     * For responses, this returns true if the service returned a value for the TopAnomalousServices property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTopAnomalousServices() {
        return topAnomalousServices != null && !(topAnomalousServices instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The service during the event that is most impacted by the incident.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTopAnomalousServices} method.
     * </p>
     * 
     * @return The service during the event that is most impacted by the incident.
     */
    public final List<AnomalousService> topAnomalousServices() {
        return topAnomalousServices;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(summary());
        hashCode = 31 * hashCode + Objects.hashCode(eventTime());
        hashCode = 31 * hashCode + Objects.hashCode(clientRequestImpactStatistics());
        hashCode = 31 * hashCode + Objects.hashCode(rootCauseServiceRequestImpactStatistics());
        hashCode = 31 * hashCode + Objects.hashCode(hasTopAnomalousServices() ? topAnomalousServices() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InsightEvent)) {
            return false;
        }
        InsightEvent other = (InsightEvent) obj;
        return Objects.equals(summary(), other.summary()) && Objects.equals(eventTime(), other.eventTime())
                && Objects.equals(clientRequestImpactStatistics(), other.clientRequestImpactStatistics())
                && Objects.equals(rootCauseServiceRequestImpactStatistics(), other.rootCauseServiceRequestImpactStatistics())
                && hasTopAnomalousServices() == other.hasTopAnomalousServices()
                && Objects.equals(topAnomalousServices(), other.topAnomalousServices());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InsightEvent").add("Summary", summary()).add("EventTime", eventTime())
                .add("ClientRequestImpactStatistics", clientRequestImpactStatistics())
                .add("RootCauseServiceRequestImpactStatistics", rootCauseServiceRequestImpactStatistics())
                .add("TopAnomalousServices", hasTopAnomalousServices() ? topAnomalousServices() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Summary":
            return Optional.ofNullable(clazz.cast(summary()));
        case "EventTime":
            return Optional.ofNullable(clazz.cast(eventTime()));
        case "ClientRequestImpactStatistics":
            return Optional.ofNullable(clazz.cast(clientRequestImpactStatistics()));
        case "RootCauseServiceRequestImpactStatistics":
            return Optional.ofNullable(clazz.cast(rootCauseServiceRequestImpactStatistics()));
        case "TopAnomalousServices":
            return Optional.ofNullable(clazz.cast(topAnomalousServices()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Summary", SUMMARY_FIELD);
        map.put("EventTime", EVENT_TIME_FIELD);
        map.put("ClientRequestImpactStatistics", CLIENT_REQUEST_IMPACT_STATISTICS_FIELD);
        map.put("RootCauseServiceRequestImpactStatistics", ROOT_CAUSE_SERVICE_REQUEST_IMPACT_STATISTICS_FIELD);
        map.put("TopAnomalousServices", TOP_ANOMALOUS_SERVICES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<InsightEvent, T> g) {
        return obj -> g.apply((InsightEvent) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InsightEvent> {
        /**
         * <p>
         * A brief description of the event.
         * </p>
         * 
         * @param summary
         *        A brief description of the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder summary(String summary);

        /**
         * <p>
         * The time, in Unix seconds, at which the event was recorded.
         * </p>
         * 
         * @param eventTime
         *        The time, in Unix seconds, at which the event was recorded.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventTime(Instant eventTime);

        /**
         * <p>
         * The impact statistics of the client side service. This includes the number of requests to the client service
         * and whether the requests were faults or okay.
         * </p>
         * 
         * @param clientRequestImpactStatistics
         *        The impact statistics of the client side service. This includes the number of requests to the client
         *        service and whether the requests were faults or okay.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientRequestImpactStatistics(RequestImpactStatistics clientRequestImpactStatistics);

        /**
         * <p>
         * The impact statistics of the client side service. This includes the number of requests to the client service
         * and whether the requests were faults or okay.
         * </p>
         * This is a convenience method that creates an instance of the {@link RequestImpactStatistics.Builder} avoiding
         * the need to create one manually via {@link RequestImpactStatistics#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RequestImpactStatistics.Builder#build()} is called immediately
         * and its result is passed to {@link #clientRequestImpactStatistics(RequestImpactStatistics)}.
         * 
         * @param clientRequestImpactStatistics
         *        a consumer that will call methods on {@link RequestImpactStatistics.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #clientRequestImpactStatistics(RequestImpactStatistics)
         */
        default Builder clientRequestImpactStatistics(Consumer<RequestImpactStatistics.Builder> clientRequestImpactStatistics) {
            return clientRequestImpactStatistics(RequestImpactStatistics.builder().applyMutation(clientRequestImpactStatistics)
                    .build());
        }

        /**
         * <p>
         * The impact statistics of the root cause service. This includes the number of requests to the client service
         * and whether the requests were faults or okay.
         * </p>
         * 
         * @param rootCauseServiceRequestImpactStatistics
         *        The impact statistics of the root cause service. This includes the number of requests to the client
         *        service and whether the requests were faults or okay.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rootCauseServiceRequestImpactStatistics(RequestImpactStatistics rootCauseServiceRequestImpactStatistics);

        /**
         * <p>
         * The impact statistics of the root cause service. This includes the number of requests to the client service
         * and whether the requests were faults or okay.
         * </p>
         * This is a convenience method that creates an instance of the {@link RequestImpactStatistics.Builder} avoiding
         * the need to create one manually via {@link RequestImpactStatistics#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RequestImpactStatistics.Builder#build()} is called immediately
         * and its result is passed to {@link #rootCauseServiceRequestImpactStatistics(RequestImpactStatistics)}.
         * 
         * @param rootCauseServiceRequestImpactStatistics
         *        a consumer that will call methods on {@link RequestImpactStatistics.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rootCauseServiceRequestImpactStatistics(RequestImpactStatistics)
         */
        default Builder rootCauseServiceRequestImpactStatistics(
                Consumer<RequestImpactStatistics.Builder> rootCauseServiceRequestImpactStatistics) {
            return rootCauseServiceRequestImpactStatistics(RequestImpactStatistics.builder()
                    .applyMutation(rootCauseServiceRequestImpactStatistics).build());
        }

        /**
         * <p>
         * The service during the event that is most impacted by the incident.
         * </p>
         * 
         * @param topAnomalousServices
         *        The service during the event that is most impacted by the incident.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder topAnomalousServices(Collection<AnomalousService> topAnomalousServices);

        /**
         * <p>
         * The service during the event that is most impacted by the incident.
         * </p>
         * 
         * @param topAnomalousServices
         *        The service during the event that is most impacted by the incident.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder topAnomalousServices(AnomalousService... topAnomalousServices);

        /**
         * <p>
         * The service during the event that is most impacted by the incident.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.xray.model.AnomalousService.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.xray.model.AnomalousService#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.xray.model.AnomalousService.Builder#build()} is called immediately and
         * its result is passed to {@link #topAnomalousServices(List<AnomalousService>)}.
         * 
         * @param topAnomalousServices
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.xray.model.AnomalousService.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #topAnomalousServices(java.util.Collection<AnomalousService>)
         */
        Builder topAnomalousServices(Consumer<AnomalousService.Builder>... topAnomalousServices);
    }

    static final class BuilderImpl implements Builder {
        private String summary;

        private Instant eventTime;

        private RequestImpactStatistics clientRequestImpactStatistics;

        private RequestImpactStatistics rootCauseServiceRequestImpactStatistics;

        private List<AnomalousService> topAnomalousServices = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(InsightEvent model) {
            summary(model.summary);
            eventTime(model.eventTime);
            clientRequestImpactStatistics(model.clientRequestImpactStatistics);
            rootCauseServiceRequestImpactStatistics(model.rootCauseServiceRequestImpactStatistics);
            topAnomalousServices(model.topAnomalousServices);
        }

        public final String getSummary() {
            return summary;
        }

        public final void setSummary(String summary) {
            this.summary = summary;
        }

        @Override
        public final Builder summary(String summary) {
            this.summary = summary;
            return this;
        }

        public final Instant getEventTime() {
            return eventTime;
        }

        public final void setEventTime(Instant eventTime) {
            this.eventTime = eventTime;
        }

        @Override
        public final Builder eventTime(Instant eventTime) {
            this.eventTime = eventTime;
            return this;
        }

        public final RequestImpactStatistics.Builder getClientRequestImpactStatistics() {
            return clientRequestImpactStatistics != null ? clientRequestImpactStatistics.toBuilder() : null;
        }

        public final void setClientRequestImpactStatistics(RequestImpactStatistics.BuilderImpl clientRequestImpactStatistics) {
            this.clientRequestImpactStatistics = clientRequestImpactStatistics != null ? clientRequestImpactStatistics.build()
                    : null;
        }

        @Override
        public final Builder clientRequestImpactStatistics(RequestImpactStatistics clientRequestImpactStatistics) {
            this.clientRequestImpactStatistics = clientRequestImpactStatistics;
            return this;
        }

        public final RequestImpactStatistics.Builder getRootCauseServiceRequestImpactStatistics() {
            return rootCauseServiceRequestImpactStatistics != null ? rootCauseServiceRequestImpactStatistics.toBuilder() : null;
        }

        public final void setRootCauseServiceRequestImpactStatistics(
                RequestImpactStatistics.BuilderImpl rootCauseServiceRequestImpactStatistics) {
            this.rootCauseServiceRequestImpactStatistics = rootCauseServiceRequestImpactStatistics != null ? rootCauseServiceRequestImpactStatistics
                    .build() : null;
        }

        @Override
        public final Builder rootCauseServiceRequestImpactStatistics(
                RequestImpactStatistics rootCauseServiceRequestImpactStatistics) {
            this.rootCauseServiceRequestImpactStatistics = rootCauseServiceRequestImpactStatistics;
            return this;
        }

        public final List<AnomalousService.Builder> getTopAnomalousServices() {
            List<AnomalousService.Builder> result = AnomalousServiceListCopier.copyToBuilder(this.topAnomalousServices);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTopAnomalousServices(Collection<AnomalousService.BuilderImpl> topAnomalousServices) {
            this.topAnomalousServices = AnomalousServiceListCopier.copyFromBuilder(topAnomalousServices);
        }

        @Override
        public final Builder topAnomalousServices(Collection<AnomalousService> topAnomalousServices) {
            this.topAnomalousServices = AnomalousServiceListCopier.copy(topAnomalousServices);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder topAnomalousServices(AnomalousService... topAnomalousServices) {
            topAnomalousServices(Arrays.asList(topAnomalousServices));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder topAnomalousServices(Consumer<AnomalousService.Builder>... topAnomalousServices) {
            topAnomalousServices(Stream.of(topAnomalousServices).map(c -> AnomalousService.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public InsightEvent build() {
            return new InsightEvent(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
