/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.xray.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Retrieved collection of spans with matching trace IDs.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RetrievedTrace implements SdkPojo, Serializable, ToCopyableBuilder<RetrievedTrace.Builder, RetrievedTrace> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(RetrievedTrace::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<Double> DURATION_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("Duration").getter(getter(RetrievedTrace::duration)).setter(setter(Builder::duration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Duration").build()).build();

    private static final SdkField<List<Span>> SPANS_FIELD = SdkField
            .<List<Span>> builder(MarshallingType.LIST)
            .memberName("Spans")
            .getter(getter(RetrievedTrace::spans))
            .setter(setter(Builder::spans))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Spans").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Span> builder(MarshallingType.SDK_POJO)
                                            .constructor(Span::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, DURATION_FIELD,
            SPANS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String id;

    private final Double duration;

    private final List<Span> spans;

    private RetrievedTrace(BuilderImpl builder) {
        this.id = builder.id;
        this.duration = builder.duration;
        this.spans = builder.spans;
    }

    /**
     * <p>
     * The unique identifier for the span.
     * </p>
     * 
     * @return The unique identifier for the span.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The length of time in seconds between the start time of the root span and the end time of the last span that
     * completed.
     * </p>
     * 
     * @return The length of time in seconds between the start time of the root span and the end time of the last span
     *         that completed.
     */
    public final Double duration() {
        return duration;
    }

    /**
     * For responses, this returns true if the service returned a value for the Spans property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasSpans() {
        return spans != null && !(spans instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Spans that comprise the trace.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSpans} method.
     * </p>
     * 
     * @return Spans that comprise the trace.
     */
    public final List<Span> spans() {
        return spans;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(duration());
        hashCode = 31 * hashCode + Objects.hashCode(hasSpans() ? spans() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RetrievedTrace)) {
            return false;
        }
        RetrievedTrace other = (RetrievedTrace) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(duration(), other.duration()) && hasSpans() == other.hasSpans()
                && Objects.equals(spans(), other.spans());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RetrievedTrace").add("Id", id()).add("Duration", duration())
                .add("Spans", hasSpans() ? spans() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Duration":
            return Optional.ofNullable(clazz.cast(duration()));
        case "Spans":
            return Optional.ofNullable(clazz.cast(spans()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Id", ID_FIELD);
        map.put("Duration", DURATION_FIELD);
        map.put("Spans", SPANS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RetrievedTrace, T> g) {
        return obj -> g.apply((RetrievedTrace) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RetrievedTrace> {
        /**
         * <p>
         * The unique identifier for the span.
         * </p>
         * 
         * @param id
         *        The unique identifier for the span.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The length of time in seconds between the start time of the root span and the end time of the last span that
         * completed.
         * </p>
         * 
         * @param duration
         *        The length of time in seconds between the start time of the root span and the end time of the last
         *        span that completed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder duration(Double duration);

        /**
         * <p>
         * Spans that comprise the trace.
         * </p>
         * 
         * @param spans
         *        Spans that comprise the trace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder spans(Collection<Span> spans);

        /**
         * <p>
         * Spans that comprise the trace.
         * </p>
         * 
         * @param spans
         *        Spans that comprise the trace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder spans(Span... spans);

        /**
         * <p>
         * Spans that comprise the trace.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.xray.model.Span.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.xray.model.Span#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.xray.model.Span.Builder#build()}
         * is called immediately and its result is passed to {@link #spans(List<Span>)}.
         * 
         * @param spans
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.xray.model.Span.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #spans(java.util.Collection<Span>)
         */
        Builder spans(Consumer<Span.Builder>... spans);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private Double duration;

        private List<Span> spans = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(RetrievedTrace model) {
            id(model.id);
            duration(model.duration);
            spans(model.spans);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final Double getDuration() {
            return duration;
        }

        public final void setDuration(Double duration) {
            this.duration = duration;
        }

        @Override
        public final Builder duration(Double duration) {
            this.duration = duration;
            return this;
        }

        public final List<Span.Builder> getSpans() {
            List<Span.Builder> result = SpanListCopier.copyToBuilder(this.spans);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSpans(Collection<Span.BuilderImpl> spans) {
            this.spans = SpanListCopier.copyFromBuilder(spans);
        }

        @Override
        public final Builder spans(Collection<Span> spans) {
            this.spans = SpanListCopier.copy(spans);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder spans(Span... spans) {
            spans(Arrays.asList(spans));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder spans(Consumer<Span.Builder>... spans) {
            spans(Stream.of(spans).map(c -> Span.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public RetrievedTrace build() {
            return new RetrievedTrace(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
