/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.lambda.powertools.idempotency.persistence;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectWriter;
import io.burt.jmespath.Expression;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.OptionalLong;
import java.util.Spliterators;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.utils.StringUtils;
import software.amazon.lambda.powertools.idempotency.IdempotencyConfig;
import software.amazon.lambda.powertools.idempotency.exceptions.IdempotencyItemAlreadyExistsException;
import software.amazon.lambda.powertools.idempotency.exceptions.IdempotencyItemNotFoundException;
import software.amazon.lambda.powertools.idempotency.exceptions.IdempotencyKeyException;
import software.amazon.lambda.powertools.idempotency.exceptions.IdempotencyValidationException;
import software.amazon.lambda.powertools.idempotency.internal.cache.LRUCache;
import software.amazon.lambda.powertools.idempotency.persistence.DataRecord;
import software.amazon.lambda.powertools.idempotency.persistence.PersistenceStore;
import software.amazon.lambda.powertools.utilities.JsonConfig;

public abstract class BasePersistenceStore
implements PersistenceStore {
    private static final Logger LOG = LoggerFactory.getLogger(BasePersistenceStore.class);
    protected boolean payloadValidationEnabled = false;
    private String functionName = "";
    private boolean configured = false;
    private long expirationInSeconds = 3600L;
    private boolean useLocalCache = false;
    private LRUCache<String, DataRecord> cache;
    private String eventKeyJMESPath;
    private Expression<JsonNode> eventKeyCompiledJMESPath;
    private Expression<JsonNode> validationKeyJMESPath;
    private boolean throwOnNoIdempotencyKey = false;
    private String hashFunctionName;

    public void configure(IdempotencyConfig config, String functionName) {
        String funcEnv = System.getenv("AWS_LAMBDA_FUNCTION_NAME");
        String string = this.functionName = funcEnv != null ? funcEnv : "testFunction";
        if (!StringUtils.isEmpty((CharSequence)functionName)) {
            this.functionName = String.valueOf(this.functionName) + "." + functionName;
        }
        if (this.configured) {
            return;
        }
        this.eventKeyJMESPath = config.getEventKeyJMESPath();
        if (this.eventKeyJMESPath != null) {
            this.eventKeyCompiledJMESPath = JsonConfig.get().getJmesPath().compile(this.eventKeyJMESPath);
        }
        if (config.getPayloadValidationJMESPath() != null) {
            this.validationKeyJMESPath = JsonConfig.get().getJmesPath().compile(config.getPayloadValidationJMESPath());
            this.payloadValidationEnabled = true;
        }
        this.throwOnNoIdempotencyKey = config.throwOnNoIdempotencyKey();
        this.useLocalCache = config.useLocalCache();
        if (this.useLocalCache) {
            this.cache = new LRUCache(config.getLocalCacheMaxItems());
        }
        this.expirationInSeconds = config.getExpirationInSeconds();
        this.hashFunctionName = config.getHashFunction();
        this.configured = true;
    }

    public void saveSuccess(JsonNode data, Object result, Instant now) {
        ObjectWriter writer = JsonConfig.get().getObjectMapper().writer();
        try {
            String responseJson = result instanceof String ? (String)result : writer.writeValueAsString(result);
            Optional<String> hashedIdempotencyKey = this.getHashedIdempotencyKey(data);
            if (!hashedIdempotencyKey.isPresent()) {
                return;
            }
            DataRecord record = new DataRecord(hashedIdempotencyKey.get(), DataRecord.Status.COMPLETED, this.getExpiryEpochSecond(now), responseJson, this.getHashedPayload(data));
            LOG.debug("Function successfully executed. Saving record to persistence store with idempotency key: {}", (Object)record.getIdempotencyKey());
            this.updateRecord(record);
            this.saveToCache(record);
        }
        catch (JsonProcessingException e) {
            throw new RuntimeException("Error while serializing the response", e);
        }
    }

    public void saveInProgress(JsonNode data, Instant now, OptionalInt remainingTimeInMs) throws IdempotencyItemAlreadyExistsException {
        Optional<String> hashedIdempotencyKey = this.getHashedIdempotencyKey(data);
        if (!hashedIdempotencyKey.isPresent()) {
            return;
        }
        String idempotencyKey = hashedIdempotencyKey.get();
        if (this.retrieveFromCache(idempotencyKey, now) != null) {
            throw new IdempotencyItemAlreadyExistsException();
        }
        OptionalLong inProgressExpirationMsTimestamp = OptionalLong.empty();
        if (remainingTimeInMs.isPresent()) {
            inProgressExpirationMsTimestamp = OptionalLong.of(now.plus((long)remainingTimeInMs.getAsInt(), ChronoUnit.MILLIS).toEpochMilli());
        }
        DataRecord record = new DataRecord(idempotencyKey, DataRecord.Status.INPROGRESS, this.getExpiryEpochSecond(now), null, this.getHashedPayload(data), inProgressExpirationMsTimestamp);
        LOG.debug("saving in progress record for idempotency key: {}", (Object)record.getIdempotencyKey());
        this.putRecord(record, now);
    }

    public void deleteRecord(JsonNode data, Throwable throwable) {
        Optional<String> hashedIdempotencyKey = this.getHashedIdempotencyKey(data);
        if (!hashedIdempotencyKey.isPresent()) {
            return;
        }
        String idemPotencyKey = hashedIdempotencyKey.get();
        LOG.debug("Function raised an exception {}. Clearing in progress record in persistence store for idempotency key: {}", throwable.getClass(), (Object)idemPotencyKey);
        this.deleteRecord(idemPotencyKey);
        this.deleteFromCache(idemPotencyKey);
    }

    public DataRecord getRecord(JsonNode data, Instant now) throws IdempotencyValidationException, IdempotencyItemNotFoundException {
        Optional<String> hashedIdempotencyKey = this.getHashedIdempotencyKey(data);
        if (!hashedIdempotencyKey.isPresent()) {
            return null;
        }
        String idemPotencyKey = hashedIdempotencyKey.get();
        DataRecord cachedRecord = this.retrieveFromCache(idemPotencyKey, now);
        if (cachedRecord != null) {
            LOG.debug("Idempotency record found in cache with idempotency key: {}", (Object)idemPotencyKey);
            this.validatePayload(data, cachedRecord);
            return cachedRecord;
        }
        DataRecord record = this.getRecord(idemPotencyKey);
        this.saveToCache(record);
        this.validatePayload(data, record);
        return record;
    }

    private Optional<String> getHashedIdempotencyKey(JsonNode data) {
        JsonNode node = data;
        if (this.eventKeyJMESPath != null) {
            node = (JsonNode)this.eventKeyCompiledJMESPath.search((Object)data);
        }
        if (this.isMissingIdemPotencyKey(node)) {
            if (this.throwOnNoIdempotencyKey) {
                throw new IdempotencyKeyException("No data found to create a hashed idempotency key");
            }
            LOG.warn("No data found to create a hashed idempotency key. JMESPath: {}", (Object)this.eventKeyJMESPath);
            return Optional.empty();
        }
        String hash = this.generateHash(node);
        hash = String.valueOf(this.functionName) + "#" + hash;
        return Optional.of(hash);
    }

    private boolean isMissingIdemPotencyKey(JsonNode data) {
        if (data.isContainerNode()) {
            Stream<Map.Entry> stream = StreamSupport.stream(Spliterators.spliteratorUnknownSize(data.fields(), 16), false);
            return stream.allMatch(e -> ((JsonNode)e.getValue()).isNull());
        }
        return data.isNull();
    }

    private String getHashedPayload(JsonNode data) {
        if (!this.payloadValidationEnabled) {
            return "";
        }
        JsonNode object = (JsonNode)this.validationKeyJMESPath.search((Object)data);
        return this.generateHash(object);
    }

    String generateHash(JsonNode data) {
        Object node = data.isContainerNode() ? data.toString() : (data.isTextual() ? data.asText() : (data.isInt() ? Integer.valueOf(data.asInt()) : (data.isLong() ? Long.valueOf(data.asLong()) : (data.isDouble() ? Double.valueOf(data.asDouble()) : (data.isFloat() ? Float.valueOf(data.floatValue()) : (data.isBigInteger() ? data.bigIntegerValue() : (data.isBigDecimal() ? data.decimalValue() : (data.isBoolean() ? Boolean.valueOf(data.asBoolean()) : data))))))));
        MessageDigest hashAlgorithm = this.getHashAlgorithm();
        byte[] digest = hashAlgorithm.digest(node.toString().getBytes(StandardCharsets.UTF_8));
        return String.format("%032x", new BigInteger(1, digest));
    }

    private MessageDigest getHashAlgorithm() {
        MessageDigest hashAlgorithm;
        try {
            hashAlgorithm = MessageDigest.getInstance(this.hashFunctionName);
        }
        catch (NoSuchAlgorithmException noSuchAlgorithmException) {
            LOG.warn("Error instantiating {} hash function, trying with MD5", (Object)this.hashFunctionName);
            try {
                hashAlgorithm = MessageDigest.getInstance("MD5");
            }
            catch (NoSuchAlgorithmException ex) {
                throw new RuntimeException("Unable to instantiate MD5 digest", ex);
            }
        }
        return hashAlgorithm;
    }

    private void validatePayload(JsonNode data, DataRecord dataRecord) throws IdempotencyValidationException {
        String dataHash;
        if (this.payloadValidationEnabled && !StringUtils.equals((String)(dataHash = this.getHashedPayload(data)), (String)dataRecord.getPayloadHash())) {
            throw new IdempotencyValidationException("Payload does not match stored record for this event key");
        }
    }

    private long getExpiryEpochSecond(Instant now) {
        return now.plus(this.expirationInSeconds, ChronoUnit.SECONDS).getEpochSecond();
    }

    private void saveToCache(DataRecord dataRecord) {
        if (!this.useLocalCache) {
            return;
        }
        if (dataRecord.getStatus().equals((Object)DataRecord.Status.INPROGRESS)) {
            return;
        }
        this.cache.put(dataRecord.getIdempotencyKey(), dataRecord);
    }

    private DataRecord retrieveFromCache(String idempotencyKey, Instant now) {
        if (!this.useLocalCache) {
            return null;
        }
        DataRecord record = (DataRecord)this.cache.get(idempotencyKey);
        if (record != null) {
            if (!record.isExpired(now)) {
                return record;
            }
            LOG.debug("Removing expired local cache record for idempotency key: {}", (Object)idempotencyKey);
            this.deleteFromCache(idempotencyKey);
        }
        return null;
    }

    private void deleteFromCache(String idempotencyKey) {
        if (!this.useLocalCache) {
            return;
        }
        this.cache.remove(idempotencyKey);
    }

    void configure(IdempotencyConfig config, String functionName, LRUCache<String, DataRecord> cache) {
        this.configure(config, functionName);
        this.cache = cache;
    }
}

