/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.lambda.powertools.sqs.internal;

import com.amazonaws.services.lambda.runtime.events.SQSEvent;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.services.sqs.SqsClient;
import software.amazon.awssdk.services.sqs.model.DeleteMessageBatchRequest;
import software.amazon.awssdk.services.sqs.model.DeleteMessageBatchRequestEntry;
import software.amazon.awssdk.services.sqs.model.DeleteMessageBatchResponse;
import software.amazon.awssdk.services.sqs.model.GetQueueAttributesRequest;
import software.amazon.awssdk.services.sqs.model.GetQueueAttributesResponse;
import software.amazon.awssdk.services.sqs.model.MessageAttributeValue;
import software.amazon.awssdk.services.sqs.model.QueueAttributeName;
import software.amazon.awssdk.services.sqs.model.SendMessageBatchRequest;
import software.amazon.awssdk.services.sqs.model.SendMessageBatchRequestEntry;
import software.amazon.awssdk.services.sqs.model.SendMessageBatchResponse;
import software.amazon.lambda.powertools.sqs.SQSBatchProcessingException;
import software.amazon.lambda.powertools.sqs.SqsUtils;

public final class BatchContext {
    private static final Logger LOG = LoggerFactory.getLogger(BatchContext.class);
    private static final Map<String, String> QUEUE_ARN_TO_DLQ_URL_MAPPING = new HashMap<String, String>();
    private final Map<SQSEvent.SQSMessage, Exception> messageToException = new HashMap<SQSEvent.SQSMessage, Exception>();
    private final List<SQSEvent.SQSMessage> success = new ArrayList<SQSEvent.SQSMessage>();
    private final SqsClient client;

    public BatchContext(SqsClient client) {
        this.client = client;
    }

    public void addSuccess(SQSEvent.SQSMessage event) {
        this.success.add(event);
    }

    public void addFailure(SQSEvent.SQSMessage event, Exception e) {
        this.messageToException.put(event, e);
    }

    @SafeVarargs
    public final <T> void processSuccessAndHandleFailed(List<T> successReturns, boolean suppressException, boolean deleteNonRetryableMessageFromQueue, Class<? extends Exception> ... nonRetryableExceptions) {
        if (this.hasFailures()) {
            boolean isMovedToDlq;
            ArrayList<Exception> exceptions = new ArrayList<Exception>();
            ArrayList<SQSEvent.SQSMessage> failedMessages = new ArrayList<SQSEvent.SQSMessage>();
            HashMap<SQSEvent.SQSMessage, Exception> nonRetryableMessageToException = new HashMap<SQSEvent.SQSMessage, Exception>();
            if (nonRetryableExceptions.length == 0) {
                exceptions.addAll(this.messageToException.values());
                failedMessages.addAll(this.messageToException.keySet());
            } else {
                this.messageToException.forEach((sqsMessage, exception) -> {
                    boolean nonRetryableException = this.isNonRetryableException((Exception)exception, nonRetryableExceptions);
                    if (nonRetryableException) {
                        nonRetryableMessageToException.put((SQSEvent.SQSMessage)sqsMessage, (Exception)exception);
                    } else {
                        exceptions.add((Exception)exception);
                        failedMessages.add((SQSEvent.SQSMessage)sqsMessage);
                    }
                });
            }
            ArrayList<SQSEvent.SQSMessage> messagesToBeDeleted = new ArrayList<SQSEvent.SQSMessage>(this.success);
            if (!nonRetryableMessageToException.isEmpty() && deleteNonRetryableMessageFromQueue) {
                messagesToBeDeleted.addAll(nonRetryableMessageToException.keySet());
            } else if (!nonRetryableMessageToException.isEmpty() && !(isMovedToDlq = this.moveNonRetryableMessagesToDlqIfConfigured(nonRetryableMessageToException))) {
                exceptions.addAll(nonRetryableMessageToException.values());
                failedMessages.addAll(nonRetryableMessageToException.keySet());
            }
            this.deleteMessagesFromQueue(messagesToBeDeleted);
            this.processFailedMessages(successReturns, suppressException, exceptions, failedMessages);
        }
    }

    private <T> void processFailedMessages(List<T> successReturns, boolean suppressException, List<Exception> exceptions, List<SQSEvent.SQSMessage> failedMessages) {
        if (failedMessages.isEmpty()) {
            return;
        }
        if (!suppressException) {
            throw new SQSBatchProcessingException(exceptions, failedMessages, successReturns);
        }
        List messageIds = failedMessages.stream().map(SQSEvent.SQSMessage::getMessageId).collect(Collectors.toList());
        LOG.debug(String.format("[%s] records failed processing, but exceptions are suppressed. Failed messages %s", failedMessages.size(), messageIds));
    }

    private boolean isNonRetryableException(Exception exception, Class<? extends Exception>[] nonRetryableExceptions) {
        return Arrays.stream(nonRetryableExceptions).anyMatch(aClass -> aClass.isInstance(exception));
    }

    private boolean moveNonRetryableMessagesToDlqIfConfigured(Map<SQSEvent.SQSMessage, Exception> nonRetryableMessageToException) {
        Optional<String> dlqUrl = this.fetchDlqUrl(nonRetryableMessageToException);
        if (!dlqUrl.isPresent()) {
            return false;
        }
        List dlqMessages = nonRetryableMessageToException.keySet().stream().map(sqsMessage -> {
            HashMap messageAttributesMap = new HashMap();
            sqsMessage.getMessageAttributes().forEach((s, messageAttribute) -> {
                MessageAttributeValue.Builder builder = MessageAttributeValue.builder();
                builder.dataType(messageAttribute.getDataType()).stringValue(messageAttribute.getStringValue());
                if (messageAttribute.getBinaryValue() != null) {
                    builder.binaryValue(SdkBytes.fromByteBuffer((ByteBuffer)messageAttribute.getBinaryValue()));
                }
                messageAttributesMap.put(s, (MessageAttributeValue)builder.build());
            });
            return (SendMessageBatchRequestEntry)SendMessageBatchRequestEntry.builder().messageBody(sqsMessage.getBody()).id(sqsMessage.getMessageId()).messageAttributes(messageAttributesMap).build();
        }).collect(Collectors.toList());
        List<SendMessageBatchResponse> sendMessageBatchResponses = this.batchRequest(dlqMessages, 10, entriesToSend -> {
            SendMessageBatchResponse sendMessageBatchResponse = this.client.sendMessageBatch((SendMessageBatchRequest)SendMessageBatchRequest.builder().entries((Collection)entriesToSend).queueUrl((String)dlqUrl.get()).build());
            LOG.debug("Response from send batch message to DLQ request {}", (Object)sendMessageBatchResponse);
            return sendMessageBatchResponse;
        });
        return sendMessageBatchResponses.stream().filter(response -> response != null && response.hasFailed()).peek(sendMessageBatchResponse -> LOG.error("Failed sending message to the DLQ. Entire batch will be re processed. Check if needed permissions are configured for the function. Response: {}", sendMessageBatchResponse)).count() == 0L;
    }

    private Optional<String> fetchDlqUrl(Map<SQSEvent.SQSMessage, Exception> nonRetryableMessageToException) {
        return nonRetryableMessageToException.keySet().stream().findFirst().map(sqsMessage -> QUEUE_ARN_TO_DLQ_URL_MAPPING.computeIfAbsent(sqsMessage.getEventSourceArn(), sourceArn -> {
            String queueUrl = this.url((String)sourceArn);
            GetQueueAttributesResponse queueAttributes = this.client.getQueueAttributes((GetQueueAttributesRequest)GetQueueAttributesRequest.builder().attributeNames(new QueueAttributeName[]{QueueAttributeName.REDRIVE_POLICY}).queueUrl(queueUrl).build());
            return Optional.ofNullable((String)queueAttributes.attributes().get(QueueAttributeName.REDRIVE_POLICY)).map(policy -> {
                try {
                    return SqsUtils.objectMapper().readTree(policy);
                }
                catch (JsonProcessingException e) {
                    LOG.debug("Unable to parse Re drive policy for queue {}. Even if DLQ exists, failed messages will be send back to main queue.", (Object)queueUrl, (Object)e);
                    return null;
                }
            }).map(node -> node.get("deadLetterTargetArn")).map(JsonNode::asText).map(this::url).orElse(null);
        }));
    }

    private boolean hasFailures() {
        return !this.messageToException.isEmpty();
    }

    private void deleteMessagesFromQueue(List<SQSEvent.SQSMessage> messages) {
        if (!messages.isEmpty()) {
            List entries = messages.stream().map(m -> (DeleteMessageBatchRequestEntry)DeleteMessageBatchRequestEntry.builder().id(m.getMessageId()).receiptHandle(m.getReceiptHandle()).build()).collect(Collectors.toList());
            this.batchRequest(entries, 10, entriesToDelete -> {
                DeleteMessageBatchRequest request = (DeleteMessageBatchRequest)DeleteMessageBatchRequest.builder().queueUrl(this.url(((SQSEvent.SQSMessage)messages.get(0)).getEventSourceArn())).entries((Collection)entriesToDelete).build();
                DeleteMessageBatchResponse deleteMessageBatchResponse = this.client.deleteMessageBatch(request);
                LOG.debug("Response from delete request {}", (Object)deleteMessageBatchResponse);
                return deleteMessageBatchResponse;
            });
        }
    }

    private <T, R> List<R> batchRequest(List<T> listOFEntries, int size, Function<List<T>, R> batchLogic) {
        return IntStream.range(0, listOFEntries.size()).filter(index -> index % size == 0).mapToObj(index -> listOFEntries.subList(index, Math.min(index + size, listOFEntries.size()))).map(batchLogic).collect(Collectors.toList());
    }

    private String url(String queueArn) {
        String[] arnArray = queueArn.split(":");
        return String.format("https://sqs.%s.amazonaws.com/%s/%s", arnArray[3], arnArray[4], arnArray[5]);
    }
}

