/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.nio.spi.s3;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.stats.CacheStats;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ReadableByteChannel;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.core.BytesWrapper;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.services.s3.S3AsyncClient;
import software.amazon.nio.spi.s3.S3Path;
import software.amazon.nio.spi.s3.S3SeekableByteChannel;
import software.amazon.nio.spi.s3.util.TimeOutUtils;

public class S3ReadAheadByteChannel
implements ReadableByteChannel {
    private final S3AsyncClient client;
    private final S3Path path;
    private final S3SeekableByteChannel delegator;
    private final int maxFragmentSize;
    private final int maxNumberFragments;
    private final int numFragmentsInObject;
    private final long size;
    private final Long timeout;
    private final TimeUnit timeUnit;
    private boolean open;
    private final Cache<Integer, CompletableFuture<ByteBuffer>> readAheadBuffersCache;
    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    public S3ReadAheadByteChannel(S3Path path, int maxFragmentSize, int maxNumberFragments, S3AsyncClient client, S3SeekableByteChannel delegator, Long timeout, TimeUnit timeUnit) throws IOException {
        Objects.requireNonNull(path);
        Objects.requireNonNull(client);
        Objects.requireNonNull(delegator);
        if (maxFragmentSize < 1) {
            throw new IllegalArgumentException("maxFragmentSize must be >= 1");
        }
        if (maxNumberFragments < 2) {
            throw new IllegalArgumentException("maxNumberFragments must be >= 2");
        }
        this.logger.debug("max read ahead fragments '{}' with size '{}' bytes", (Object)maxNumberFragments, (Object)maxFragmentSize);
        this.client = client;
        this.path = path;
        this.delegator = delegator;
        this.size = delegator.size();
        this.maxFragmentSize = maxFragmentSize;
        this.numFragmentsInObject = (int)Math.ceil((float)this.size / (float)maxFragmentSize);
        this.readAheadBuffersCache = Caffeine.newBuilder().maximumSize((long)maxNumberFragments).recordStats().build();
        this.maxNumberFragments = maxNumberFragments;
        this.open = true;
        this.timeout = timeout != null ? timeout : 5L;
        this.timeUnit = timeUnit != null ? timeUnit : TimeUnit.MINUTES;
    }

    @Override
    public int read(ByteBuffer dst) throws IOException {
        Objects.requireNonNull(dst);
        long channelPosition = this.delegator.position();
        this.logger.debug("delegator position: {}", (Object)channelPosition);
        if (channelPosition >= this.size) {
            return -1;
        }
        Integer fragmentIndex = this.fragmentIndexForByteNumber(channelPosition);
        this.logger.debug("fragment index: {}", (Object)fragmentIndex);
        int fragmentOffset = (int)(channelPosition - fragmentIndex.longValue() * (long)this.maxFragmentSize);
        this.logger.debug("fragment {} offset: {}", (Object)fragmentIndex, (Object)fragmentOffset);
        try {
            ByteBuffer fragment = ((ByteBuffer)Objects.requireNonNull((CompletableFuture)this.readAheadBuffersCache.get((Object)fragmentIndex, this::computeFragmentFuture)).get(this.timeout, this.timeUnit)).asReadOnlyBuffer();
            fragment.position(fragmentOffset);
            this.logger.debug("fragment remaining: {}", (Object)fragment.remaining());
            this.logger.debug("dst remaining: {}", (Object)dst.remaining());
            int limit = Math.min(fragment.remaining(), dst.remaining());
            this.logger.debug("byte limit: {}", (Object)limit);
            byte[] copiedBytes = new byte[limit];
            fragment.get(copiedBytes, 0, limit);
            dst.put(copiedBytes);
            if (fragment.position() >= fragment.limit() / 2) {
                this.clearPriorFragments(fragmentIndex);
                int maxFragmentsToLoad = Math.min(this.maxNumberFragments - 1, this.numFragmentsInObject - fragmentIndex - 1);
                for (int i = 0; i < maxFragmentsToLoad; ++i) {
                    int idxToLoad = i + fragmentIndex + 1;
                    if (this.readAheadBuffersCache.asMap().containsKey(idxToLoad)) continue;
                    this.logger.debug("initiate pre-loading fragment with index '{}' from '{}'", (Object)idxToLoad, (Object)this.path.toUri());
                    this.readAheadBuffersCache.put((Object)idxToLoad, this.computeFragmentFuture(idxToLoad));
                }
            }
            this.delegator.position(channelPosition + (long)copiedBytes.length);
            return copiedBytes.length;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException(e);
        }
        catch (ExecutionException e) {
            this.logger.error("an exception occurred while reading bytes from {} that was not recovered by the S3 Client RetryCondition(s)", (Object)this.path.toUri());
            throw new IOException(e);
        }
        catch (TimeoutException e) {
            throw TimeOutUtils.logAndGenerateExceptionOnTimeOut(this.logger, "read", 5L, TimeUnit.MINUTES);
        }
    }

    private void clearPriorFragments(int currentFragIndx) {
        Set<@NonNull T> priorIndexes = this.readAheadBuffersCache.asMap().keySet().stream().filter(idx -> idx < currentFragIndx).collect(Collectors.toSet());
        if (priorIndexes.size() > 0) {
            this.logger.debug("invalidating fragment(s) '{}' from '{}'", (Object)priorIndexes.stream().map(Objects::toString).collect(Collectors.joining(", ")), (Object)this.path.toUri());
            this.readAheadBuffersCache.invalidateAll(priorIndexes);
        }
    }

    @Override
    public boolean isOpen() {
        return this.open;
    }

    @Override
    public void close() {
        this.open = false;
        this.readAheadBuffersCache.invalidateAll();
        this.readAheadBuffersCache.cleanUp();
    }

    protected int numberOfCachedFragments() {
        this.readAheadBuffersCache.cleanUp();
        return (int)this.readAheadBuffersCache.estimatedSize();
    }

    protected CacheStats cacheStatistics() {
        return this.readAheadBuffersCache.stats();
    }

    private CompletableFuture<ByteBuffer> computeFragmentFuture(int fragmentIndex) {
        long readFrom = (long)fragmentIndex * (long)this.maxFragmentSize;
        long readTo = Math.min(readFrom + (long)this.maxFragmentSize, this.size) - 1L;
        String range = "bytes=" + readFrom + "-" + readTo;
        this.logger.debug("byte range for {} is '{}'", (Object)this.path.getKey(), (Object)range);
        return this.client.getObject(builder -> builder.bucket(this.path.bucketName()).key(this.path.getKey()).range(range), AsyncResponseTransformer.toBytes()).thenApply(BytesWrapper::asByteBuffer);
    }

    protected Integer fragmentIndexForByteNumber(long byteNumber) {
        return Math.toIntExact(Math.floorDiv(byteNumber, (long)this.maxFragmentSize));
    }
}

