/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.serverless.ssmcachingclient;

import java.beans.ConstructorProperties;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.services.ssm.SsmClient;
import software.amazon.awssdk.services.ssm.model.GetParameterRequest;
import software.amazon.awssdk.services.ssm.model.GetParameterResponse;
import software.amazon.awssdk.services.ssm.model.GetParametersByPathRequest;
import software.amazon.awssdk.services.ssm.model.Parameter;
import software.amazon.awssdk.services.ssm.model.ParameterNotFoundException;
import software.amazon.awssdk.services.ssm.model.ParameterType;
import software.amazon.awssdk.services.ssm.paginators.GetParametersByPathIterable;

public class SsmParameterCachingClient {
    private static final Logger log = LoggerFactory.getLogger(SsmParameterCachingClient.class);
    private final SsmClient ssm;
    private final Duration ttl;
    private final String pathPrefix;
    private final boolean allowStaleValues;
    private final boolean bulkLoad;
    private final Clock clock;
    private volatile ConcurrentMap<String, CachedParameter> cache;

    SsmParameterCachingClient(@NonNull SsmClient ssm, Duration ttl, String pathPrefix, boolean allowStaleValues, boolean bulkLoad, Clock clock) {
        if (ssm == null) {
            throw new IllegalArgumentException("ssm is null");
        }
        this.ssm = ssm;
        this.ttl = ttl;
        this.pathPrefix = pathPrefix;
        this.allowStaleValues = allowStaleValues;
        this.bulkLoad = bulkLoad;
        this.clock = clock;
        this.cache = new ConcurrentHashMap<String, CachedParameter>();
    }

    public SsmParameterCachingClient(SsmClient ssm, Duration ttl, String pathPrefix, boolean allowStaleValues, boolean bulkLoad) {
        this(ssm, ttl, pathPrefix, allowStaleValues, bulkLoad, Clock.systemUTC());
    }

    public SsmParameterCachingClient(SsmClient ssm, Duration ttl, String pathPrefix) {
        this(ssm, ttl, pathPrefix, true, true);
    }

    public SsmParameterCachingClient(SsmClient ssm, Duration ttl) {
        this(ssm, ttl, null);
    }

    public Parameter get(String name) {
        CachedParameter cached = this.getCachedParameter(name);
        if (this.isStale(cached)) {
            log.debug("SSM Parameter cache miss for pathPrefix={}, name={}. Attempting to load value from SSM.", (Object)this.pathPrefix, (Object)name);
            try {
                cached = this.load(name);
            }
            catch (Exception e) {
                if (e instanceof ParameterNotFoundException) {
                    throw e;
                }
                if (this.allowStaleValues && cached != null) {
                    log.warn(String.format("Failed to update cache from SSM for pathPrefix=%s, name=%s. Returning stale value instead.", this.pathPrefix, name), (Throwable)e);
                    return cached.getParameter();
                }
                throw e;
            }
        } else {
            log.debug("SSM Parameter cache hit for pathPrefix={}, name={}. Returning cached value.", (Object)this.pathPrefix, (Object)name);
        }
        return cached.parameter;
    }

    private CachedParameter getCachedParameter(String name) {
        String resolvedName = this.resolvedName(name);
        return (CachedParameter)this.cache.get(resolvedName);
    }

    private String resolvedName(String name) {
        if (this.pathPrefix == null) {
            return name;
        }
        return this.pathPrefix + name;
    }

    private boolean isStale(CachedParameter cached) {
        return cached == null || Instant.now(this.clock).isAfter(cached.getLoadedAt().plus(this.ttl));
    }

    private CachedParameter load(String name) {
        String resolvedName = this.resolvedName(name);
        if (this.bulkLoad && this.pathPrefix != null) {
            return this.bulkLoad(resolvedName);
        }
        return this.loadSingleValue(resolvedName);
    }

    private CachedParameter bulkLoad(String resolvedName) {
        log.debug("Bulk loading and caching all parameters for pathPrefix={} from SSM", (Object)this.pathPrefix);
        GetParametersByPathRequest request = (GetParametersByPathRequest)GetParametersByPathRequest.builder().path(this.pathPrefix).recursive(Boolean.valueOf(true)).withDecryption(Boolean.valueOf(true)).build();
        GetParametersByPathIterable iterable = this.ssm.getParametersByPathPaginator(request);
        ConcurrentHashMap<String, CachedParameter> newCache = new ConcurrentHashMap<String, CachedParameter>();
        iterable.stream().flatMap(r -> r.parameters().stream()).forEach(p -> newCache.put(p.name(), new CachedParameter((Parameter)p, Instant.now(this.clock))));
        if (!newCache.containsKey(resolvedName)) {
            log.debug("Bulk load results did not contain parameter {}", (Object)resolvedName);
            this.loadSingleValue(resolvedName);
            throw new IllegalStateException(String.format("Could not find SSM parameter via GetParametersByPath(path=%s), but found it with GetParameter(name=%s). This is probably a bug in the caching client", this.pathPrefix, resolvedName));
        }
        this.cache = newCache;
        return (CachedParameter)this.cache.get(resolvedName);
    }

    private CachedParameter loadSingleValue(String resolvedName) {
        log.debug("Loading and caching single value {} from SSM", (Object)resolvedName);
        GetParameterRequest request = (GetParameterRequest)GetParameterRequest.builder().name(resolvedName).withDecryption(Boolean.valueOf(true)).build();
        GetParameterResponse response = this.ssm.getParameter(request);
        CachedParameter cached = new CachedParameter(response.parameter(), Instant.now(this.clock));
        this.cache.put(response.parameter().name(), cached);
        return cached;
    }

    public String getAsString(String name) {
        return this.get(name).value();
    }

    public List<String> getAsStringList(String name) {
        Parameter parameter = this.get(name);
        if (parameter.type() != ParameterType.STRING_LIST) {
            throw new IllegalArgumentException(String.format("Parameter %s is of type %s, expected StringList", parameter.name(), parameter.type()));
        }
        return Arrays.asList(parameter.value().split(","));
    }

    private static final class CachedParameter {
        private final Parameter parameter;
        private final Instant loadedAt;

        @ConstructorProperties(value={"parameter", "loadedAt"})
        public CachedParameter(Parameter parameter, Instant loadedAt) {
            this.parameter = parameter;
            this.loadedAt = loadedAt;
        }

        public Parameter getParameter() {
            return this.parameter;
        }

        public Instant getLoadedAt() {
            return this.loadedAt;
        }

        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (!(o instanceof CachedParameter)) {
                return false;
            }
            CachedParameter other = (CachedParameter)o;
            Parameter this$parameter = this.getParameter();
            Parameter other$parameter = other.getParameter();
            if (this$parameter == null ? other$parameter != null : !this$parameter.equals(other$parameter)) {
                return false;
            }
            Instant this$loadedAt = this.getLoadedAt();
            Instant other$loadedAt = other.getLoadedAt();
            return !(this$loadedAt == null ? other$loadedAt != null : !((Object)this$loadedAt).equals(other$loadedAt));
        }

        public int hashCode() {
            int PRIME = 59;
            int result = 1;
            Parameter $parameter = this.getParameter();
            result = result * 59 + ($parameter == null ? 43 : $parameter.hashCode());
            Instant $loadedAt = this.getLoadedAt();
            result = result * 59 + ($loadedAt == null ? 43 : ((Object)$loadedAt).hashCode());
            return result;
        }

        public String toString() {
            return "SsmParameterCachingClient.CachedParameter(parameter=" + this.getParameter() + ", loadedAt=" + this.getLoadedAt() + ")";
        }
    }
}

