/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.timestream.jdbc;

import com.amazonaws.services.timestreamquery.model.Datum;
import com.amazonaws.services.timestreamquery.model.TimeSeriesDataPoint;
import com.google.common.collect.ImmutableList;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.sql.Date;
import java.sql.JDBCType;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Time;
import java.sql.Timestamp;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeParseException;
import java.util.EnumMap;
import java.util.Map;
import java.util.StringJoiner;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.timestream.jdbc.Constants;
import software.amazon.timestream.jdbc.Error;
import software.amazon.timestream.jdbc.JdbcType;
import software.amazon.timestream.jdbc.TimestreamConvertFunction;
import software.amazon.timestream.jdbc.TimestreamDataType;
import software.amazon.timestream.jdbc.Warning;

class Conversions {
    private static final Logger LOGGER = LoggerFactory.getLogger(Conversions.class);
    static final Map<TimestreamDataType, Map<JdbcType, TimestreamConvertFunction<?>>> CONVERSIONS = Conversions.populateConversionMap();
    private static final BigDecimal BYTE_MIN = new BigDecimal(-128);
    private static final BigDecimal BYTE_MAX = new BigDecimal(127);
    private static final BigDecimal SHORT_MIN = new BigDecimal(Short.MIN_VALUE);
    private static final BigDecimal SHORT_MAX = new BigDecimal(Short.MAX_VALUE);
    private static final BigDecimal INT_MIN = new BigDecimal(Integer.MIN_VALUE);
    private static final BigDecimal INT_MAX = new BigDecimal(Integer.MAX_VALUE);
    private static final BigDecimal LONG_MIN = new BigDecimal(Long.MIN_VALUE);
    private static final BigDecimal LONG_MAX = new BigDecimal(Long.MAX_VALUE);

    Conversions() {
    }

    static Map<TimestreamDataType, Map<JdbcType, TimestreamConvertFunction<?>>> populateConversionMap() {
        EnumMap conversionMap = new EnumMap(TimestreamDataType.class);
        Conversions.populateIntConversions(conversionMap);
        Conversions.populateDoubleConversions(conversionMap);
        Conversions.populateBooleanConversions(conversionMap);
        Conversions.populateLongConversions(conversionMap);
        Conversions.populateDateConversions(conversionMap);
        Conversions.populateTimeConversions(conversionMap);
        Conversions.populateTimestampConversions(conversionMap);
        Conversions.populateStringConversions(conversionMap);
        Conversions.populateArrayConversions(conversionMap);
        Conversions.populateRowConversions(conversionMap);
        Conversions.populateTimeSeriesConversions(conversionMap);
        Conversions.populateIntervalConversions(conversionMap);
        return conversionMap;
    }

    private static void addCommonNumericConversions(Map<JdbcType, TimestreamConvertFunction<?>> conversions) {
        conversions.put(JdbcType.TINYINT, (data, callback) -> Byte.valueOf(data.getScalarValue()));
        conversions.put(JdbcType.SMALLINT, (data, callback) -> Short.valueOf(data.getScalarValue()));
        conversions.put(JdbcType.INTEGER, (data, callback) -> Integer.valueOf(data.getScalarValue()));
        conversions.put(JdbcType.BIGINT, (data, callback) -> Long.valueOf(data.getScalarValue()));
        conversions.put(JdbcType.DOUBLE, (data, callback) -> Double.valueOf(data.getScalarValue()));
        conversions.put(JdbcType.FLOAT, (data, callback) -> Float.valueOf(data.getScalarValue()));
        conversions.put(JdbcType.DECIMAL, (data, callback) -> new BigDecimal(data.getScalarValue()));
        conversions.put(JdbcType.VARCHAR, (data, callback) -> data.getScalarValue());
    }

    private static void populateIntConversions(Map<TimestreamDataType, Map<JdbcType, TimestreamConvertFunction<?>>> map) {
        EnumMap intConversions = new EnumMap(JdbcType.class);
        Conversions.addCommonNumericConversions(intConversions);
        intConversions.put(JdbcType.BOOLEAN, (data, callback) -> Integer.parseInt(data.getScalarValue()) != 0);
        map.put(TimestreamDataType.INTEGER, intConversions);
    }

    private static void populateDoubleConversions(Map<TimestreamDataType, Map<JdbcType, TimestreamConvertFunction<?>>> map) {
        EnumMap doubleConversions = new EnumMap(JdbcType.class);
        Conversions.addCommonNumericConversions(doubleConversions);
        doubleConversions.put(JdbcType.BOOLEAN, (data, callback) -> Double.parseDouble(data.getScalarValue()) != 0.0);
        doubleConversions.put(JdbcType.TINYINT, (data, callback) -> {
            BigDecimal value = Conversions.validateValueRange(data.getScalarValue(), BYTE_MIN, BYTE_MAX, JDBCType.TINYINT);
            if (value.stripTrailingZeros().scale() > 0) {
                callback.accept(new SQLWarning(Warning.lookup(Warning.VALUE_TRUNCATED, new Object[]{TimestreamDataType.DOUBLE, JdbcType.TINYINT})));
            }
            return value.byteValue();
        });
        doubleConversions.put(JdbcType.SMALLINT, (data, callback) -> {
            BigDecimal value = Conversions.validateValueRange(data.getScalarValue(), SHORT_MIN, SHORT_MAX, JDBCType.SMALLINT);
            if (value.stripTrailingZeros().scale() > 0) {
                callback.accept(new SQLWarning(Warning.lookup(Warning.VALUE_TRUNCATED, new Object[]{TimestreamDataType.DOUBLE, JdbcType.SMALLINT})));
            }
            return value.shortValue();
        });
        doubleConversions.put(JdbcType.INTEGER, (data, callback) -> {
            BigDecimal value = Conversions.validateValueRange(data.getScalarValue(), INT_MIN, INT_MAX, JDBCType.INTEGER);
            if (value.stripTrailingZeros().scale() > 0) {
                callback.accept(new SQLWarning(Warning.lookup(Warning.VALUE_TRUNCATED, new Object[]{TimestreamDataType.DOUBLE, JdbcType.INTEGER})));
            }
            return value.intValue();
        });
        doubleConversions.put(JdbcType.BIGINT, (data, callback) -> {
            BigDecimal value = Conversions.validateValueRange(data.getScalarValue(), LONG_MIN, LONG_MAX, JDBCType.BIGINT);
            if (value.stripTrailingZeros().scale() > 0) {
                callback.accept(new SQLWarning(Warning.lookup(Warning.VALUE_TRUNCATED, new Object[]{TimestreamDataType.DOUBLE, JdbcType.BIGINT})));
            }
            return value.longValue();
        });
        map.put(TimestreamDataType.DOUBLE, doubleConversions);
    }

    private static void populateBooleanConversions(Map<TimestreamDataType, Map<JdbcType, TimestreamConvertFunction<?>>> map) {
        EnumMap<JdbcType, TimestreamConvertFunction<Object>> conversions = new EnumMap<JdbcType, TimestreamConvertFunction<Object>>(JdbcType.class);
        conversions.put(JdbcType.BOOLEAN, (data, callback) -> Boolean.valueOf(data.getScalarValue()));
        conversions.put(JdbcType.TINYINT, (data, callback) -> (byte)(Boolean.parseBoolean(data.getScalarValue()) ? 1 : 0));
        conversions.put(JdbcType.SMALLINT, (data, callback) -> (short)(Boolean.parseBoolean(data.getScalarValue()) ? 1 : 0));
        conversions.put(JdbcType.INTEGER, (data, callback) -> Boolean.parseBoolean(data.getScalarValue()) ? 1 : 0);
        conversions.put(JdbcType.BIGINT, (data, callback) -> Boolean.parseBoolean(data.getScalarValue()) ? 1L : 0L);
        conversions.put(JdbcType.FLOAT, (data, callback) -> Float.valueOf(Boolean.parseBoolean(data.getScalarValue()) ? 1.0f : 0.0f));
        conversions.put(JdbcType.DOUBLE, (data, callback) -> Boolean.parseBoolean(data.getScalarValue()) ? 1.0 : 0.0);
        conversions.put(JdbcType.DECIMAL, (data, callback) -> new BigDecimal(Boolean.parseBoolean(data.getScalarValue()) ? 1 : 0));
        conversions.put(JdbcType.VARCHAR, (data, callback) -> data.getScalarValue());
        map.put(TimestreamDataType.BOOLEAN, conversions);
    }

    private static void populateLongConversions(Map<TimestreamDataType, Map<JdbcType, TimestreamConvertFunction<?>>> map) {
        EnumMap longConversions = new EnumMap(JdbcType.class);
        Conversions.addCommonNumericConversions(longConversions);
        longConversions.put(JdbcType.TINYINT, (data, callback) -> {
            BigDecimal value = Conversions.validateValueRange(data.getScalarValue(), BYTE_MIN, BYTE_MAX, JDBCType.TINYINT);
            return value.byteValue();
        });
        longConversions.put(JdbcType.SMALLINT, (data, callback) -> {
            BigDecimal value = Conversions.validateValueRange(data.getScalarValue(), SHORT_MIN, SHORT_MAX, JDBCType.SMALLINT);
            return value.shortValue();
        });
        longConversions.put(JdbcType.BOOLEAN, (data, callback) -> Long.parseLong(data.getScalarValue()) != 0L);
        map.put(TimestreamDataType.BIGINT, longConversions);
    }

    private static void populateDateConversions(Map<TimestreamDataType, Map<JdbcType, TimestreamConvertFunction<?>>> map) {
        EnumMap<JdbcType, TimestreamConvertFunction<Object>> dateConversions = new EnumMap<JdbcType, TimestreamConvertFunction<Object>>(JdbcType.class);
        dateConversions.put(JdbcType.DATE, (data, callback) -> Date.valueOf(data.getScalarValue()));
        dateConversions.put(JdbcType.TIMESTAMP, (data, callback) -> Timestamp.valueOf(LocalDate.parse(data.getScalarValue()).atTime(0, 0, 0)));
        dateConversions.put(JdbcType.VARCHAR, (data, callback) -> data.getScalarValue());
        map.put(TimestreamDataType.DATE, dateConversions);
    }

    private static void populateTimeConversions(Map<TimestreamDataType, Map<JdbcType, TimestreamConvertFunction<?>>> map) {
        EnumMap<JdbcType, TimestreamConvertFunction<Object>> timeConversions = new EnumMap<JdbcType, TimestreamConvertFunction<Object>>(JdbcType.class);
        timeConversions.put(JdbcType.TIME, (data, callback) -> Time.valueOf(LocalTime.parse(data.getScalarValue())));
        timeConversions.put(JdbcType.TIMESTAMP, (data, callback) -> Timestamp.valueOf(LocalDateTime.of(LocalDate.of(1970, 1, 1), LocalTime.parse(data.getScalarValue()))));
        timeConversions.put(JdbcType.VARCHAR, (data, callback) -> data.getScalarValue());
        map.put(TimestreamDataType.TIME, timeConversions);
    }

    private static void populateTimestampConversions(Map<TimestreamDataType, Map<JdbcType, TimestreamConvertFunction<?>>> map) {
        EnumMap<JdbcType, TimestreamConvertFunction<Object>> timestampConversions = new EnumMap<JdbcType, TimestreamConvertFunction<Object>>(JdbcType.class);
        timestampConversions.put(JdbcType.DATE, (data, callback) -> Date.valueOf(LocalDateTime.parse(data.getScalarValue(), Constants.DATE_TIME_FORMATTER).toLocalDate()));
        timestampConversions.put(JdbcType.TIME, (data, callback) -> Time.valueOf(LocalDateTime.parse(data.getScalarValue(), Constants.DATE_TIME_FORMATTER).toLocalTime()));
        timestampConversions.put(JdbcType.TIMESTAMP, (data, callback) -> Timestamp.valueOf(LocalDateTime.parse(data.getScalarValue(), Constants.DATE_TIME_FORMATTER)));
        timestampConversions.put(JdbcType.VARCHAR, (data, callback) -> data.getScalarValue());
        map.put(TimestreamDataType.TIMESTAMP, timestampConversions);
    }

    private static void populateStringConversions(Map<TimestreamDataType, Map<JdbcType, TimestreamConvertFunction<?>>> map) {
        EnumMap stringConversions = new EnumMap(JdbcType.class);
        Conversions.addCommonNumericConversions(stringConversions);
        stringConversions.put(JdbcType.BOOLEAN, (data, callback) -> Boolean.parseBoolean(data.getScalarValue()));
        stringConversions.put(JdbcType.DATE, (data, callback) -> Date.valueOf(LocalDate.parse(data.getScalarValue(), Constants.DATE_FORMATTER)));
        stringConversions.put(JdbcType.TIME, (data, callback) -> Time.valueOf(LocalTime.parse(data.getScalarValue(), Constants.TIME_FORMATTER)));
        stringConversions.put(JdbcType.TIMESTAMP, (data, callback) -> Timestamp.valueOf(LocalDateTime.parse(data.getScalarValue(), Constants.DATE_TIME_FORMATTER)));
        map.put(TimestreamDataType.VARCHAR, stringConversions);
    }

    private static void populateArrayConversions(Map<TimestreamDataType, Map<JdbcType, TimestreamConvertFunction<?>>> map) {
        EnumMap<JdbcType, TimestreamConvertFunction<Object>> arrayConversions = new EnumMap<JdbcType, TimestreamConvertFunction<Object>>(JdbcType.class);
        arrayConversions.put(JdbcType.ARRAY, (data, callback) -> data.getArrayValue());
        arrayConversions.put(JdbcType.VARCHAR, (data, callback) -> {
            StringJoiner result = new StringJoiner(", ", "[", "]");
            for (Datum datum : data.getArrayValue()) {
                result.add(String.valueOf(Conversions.convert(Conversions.getDatumSourceType(datum), JdbcType.VARCHAR, datum, callback)));
            }
            return result.toString();
        });
        arrayConversions.put(JdbcType.JAVA_OBJECT, (data, callback) -> data.getArrayValue());
        map.put(TimestreamDataType.ARRAY, arrayConversions);
    }

    private static void populateRowConversions(Map<TimestreamDataType, Map<JdbcType, TimestreamConvertFunction<?>>> map) {
        EnumMap<JdbcType, TimestreamConvertFunction<Object>> rowConversions = new EnumMap<JdbcType, TimestreamConvertFunction<Object>>(JdbcType.class);
        rowConversions.put(JdbcType.VARCHAR, (data, callback) -> {
            StringJoiner result = new StringJoiner(", ", "(", ")");
            for (Datum datum : data.getRowValue().getData()) {
                result.add(String.valueOf(Conversions.convert(Conversions.getDatumSourceType(datum), JdbcType.VARCHAR, datum, callback)));
            }
            return result.toString();
        });
        rowConversions.put(JdbcType.STRUCT, (data, callback) -> data.getRowValue());
        rowConversions.put(JdbcType.JAVA_OBJECT, (data, callback) -> ImmutableList.of((Object)data.getRowValue()));
        map.put(TimestreamDataType.ROW, rowConversions);
    }

    private static void populateTimeSeriesConversions(Map<TimestreamDataType, Map<JdbcType, TimestreamConvertFunction<?>>> map) {
        EnumMap<JdbcType, TimestreamConvertFunction<Object>> timeSeriesConversions = new EnumMap<JdbcType, TimestreamConvertFunction<Object>>(JdbcType.class);
        timeSeriesConversions.put(JdbcType.VARCHAR, (data, callback) -> {
            StringJoiner result = new StringJoiner(", ", "[", "]");
            for (TimeSeriesDataPoint timeSeriesDataPoint : data.getTimeSeriesValue()) {
                StringJoiner stringJoiner = new StringJoiner(", ", "{", "}");
                stringJoiner.add("time: " + timeSeriesDataPoint.getTime());
                Datum value = timeSeriesDataPoint.getValue();
                stringJoiner.add("value: " + (value == null ? null : Conversions.convert(Conversions.getDatumSourceType(value), JdbcType.VARCHAR, value, callback)));
                result.add(stringJoiner.toString());
            }
            return result.toString();
        });
        timeSeriesConversions.put(JdbcType.JAVA_OBJECT, (data, callback) -> data.getTimeSeriesValue());
        timeSeriesConversions.put(JdbcType.ARRAY, (data, callback) -> data.getTimeSeriesValue());
        map.put(TimestreamDataType.TIMESERIES, timeSeriesConversions);
    }

    private static void populateIntervalConversions(Map<TimestreamDataType, Map<JdbcType, TimestreamConvertFunction<?>>> map) {
        EnumMap<JdbcType, TimestreamConvertFunction<Object>> intervalConversion = new EnumMap<JdbcType, TimestreamConvertFunction<Object>>(JdbcType.class);
        intervalConversion.put(JdbcType.VARCHAR, (data, callback) -> String.valueOf(data.getScalarValue()));
        map.put(TimestreamDataType.INTERVAL_DAY_TO_SECOND, intervalConversion);
        map.put(TimestreamDataType.INTERVAL_YEAR_TO_MONTH, intervalConversion);
    }

    private static BigDecimal validateValueRange(String data, BigDecimal minValue, BigDecimal maxValue, JDBCType targetType) throws SQLException {
        BigDecimal decimalValue = new BigDecimal(data);
        BigInteger value = decimalValue.toBigInteger();
        if (value.compareTo(maxValue.toBigInteger()) > 0 || value.compareTo(minValue.toBigInteger()) < 0) {
            throw Error.createSQLException(LOGGER, Error.VALUE_OUT_OF_RANGE, data, minValue, maxValue, targetType);
        }
        return decimalValue;
    }

    private static TimestreamDataType getDatumSourceType(Datum datum) {
        if (datum.getArrayValue() != null) {
            return TimestreamDataType.ARRAY;
        }
        if (datum.getTimeSeriesValue() != null) {
            return TimestreamDataType.TIMESERIES;
        }
        if (datum.getRowValue() != null) {
            return TimestreamDataType.ROW;
        }
        return TimestreamDataType.VARCHAR;
    }

    static TimestreamConvertFunction<?> retrieveAndValidateConversion(TimestreamDataType sourceType, JdbcType targetType) throws SQLException {
        Map<JdbcType, TimestreamConvertFunction<?>> conversion = CONVERSIONS.get((Object)sourceType);
        if (conversion == null) {
            throw Error.createSQLException(LOGGER, Error.UNSUPPORTED_CONVERSION, sourceType.toString(), String.valueOf((Object)targetType));
        }
        TimestreamConvertFunction<?> fn = conversion.get((Object)targetType);
        if (fn == null) {
            throw Error.createSQLException(LOGGER, Error.UNSUPPORTED_CONVERSION, sourceType.toString(), String.valueOf((Object)targetType));
        }
        return fn;
    }

    static Object convert(TimestreamDataType sourceType, JdbcType targetType, Datum data, Consumer<SQLWarning> callback) throws SQLException {
        TimestreamConvertFunction<?> fn = Conversions.retrieveAndValidateConversion(sourceType, targetType);
        try {
            LOGGER.debug("Converting data {} from source type {} to target type {}", new Object[]{data, sourceType, targetType});
            return fn.convert(data, callback);
        }
        catch (IllegalArgumentException | DateTimeParseException e) {
            LOGGER.warn("Cannot parse data as {}: {}", (Object)sourceType, (Object)e.getMessage());
            throw new SQLException(Error.lookup(Error.INCORRECT_SOURCE_TYPE_AT_CELL, new Object[]{sourceType}), e);
        }
    }
}

