/**
 * <h2>CDK Graph (<code>&#64;aws-prototyping-sdk/cdk-graph</code>)</h2>
 * <p>
 * <img alt="experimental" src="https://img.shields.io/badge/stability-experimental-orange.svg">
 * <img alt="alpha" src="https://img.shields.io/badge/version-alpha-red.svg">
 * <a href="https://aws.github.io/aws-prototyping-sdk/typescript/cdk-graph/index.html"><img alt="API Documetnation" src="https://img.shields.io/badge/view-API_Documentation-blue.svg"></a>
 * <a href="https://github.com/aws/aws-prototyping-sdk/tree/mainline/packages/cdk-graph"><img alt="Source Code" src="https://img.shields.io/badge/view-Source_Code-blue.svg"></a>
 * <p>
 * <blockquote>
 * <p>
 * More comprehensive documentation to come as this package stabilizes
 * <p>
 * </blockquote>
 * <p>
 * This package is the core framework for supporting additional cdk based automation and tooling, such as diagraming, cost modeling, security and compliance, in a holistic and comprehensive way.
 * <p>
 * This package provides the following functionality:
 * <p>
 * <ol>
 * <li>Synthesizes a serialized graph (nodes and edges) from CDK source code.</li>
 * <li>Provides runtime interface for interacting with the graph (in-memory database-like graph store).</li>
 * <li>Provides plugin framework for additional tooling to utilize and extend the graph.</li>
 * </ol>
 * <p>
 * The goal of this framework is to enable bespoke tooling to be built without having to first traverse the CDK Tree and Metadata to build a graph. Projects like <a href="https://github.com/pistazie/cdk-dia">cdk-dia</a> generate a bespoke in-memory graph that is then utilized to generate diagrams; while the diagram generation is the core value it must first have a graph to act upon and currently is required to generate this undifferentiated graph to provide its diagrams. By standardizing on the graph interface necessary to build complex tooling, we can more rapidly build new tooling that focuses on its core value.
 * <p>
 * <hr>
 * <p>
 * <h3>Available Plugins</h3>
 * <p>
 * | Name | Description | Screenshot | Links |
 * |--- | --- | --- | --- |
 * | <strong>Diagram</strong> | Generate cloud infrastructure diagrams from cdk graph | <img src="https://github.com/aws/aws-prototyping-sdk/blob/mainline/packages/cdk-graph-plugin-diagram/docs/examples/default.png?raw=true" style="max-width:200px;max-height:200px" /> | <a href="https://aws.github.io/aws-prototyping-sdk/typescript/cdk-graph/index.html"><img alt="API Documetnation" src="https://img.shields.io/badge/view-API_Documentation-blue.svg"></a> <a href="https://github.com/aws/aws-prototyping-sdk/tree/mainline/packages/cdk-graph"><img alt="Source Code" src="https://img.shields.io/badge/view-Source_Code-blue.svg"></a> |
 * <p>
 * <hr>
 * <p>
 * <h3>Quick Start</h3>
 * <p>
 * <h4>Instrument CDK App with CdkGraph</h4>
 * <p>
 * <blockquote><pre>
 * #!/usr/bin/env node
 * import * as cdk from "aws-cdk-lib";
 * import { MyStack } from "../lib/my-stack";
 * 
 * import { CdkGraph } from "&#64;aws-prototyping-sdk/cdk-graph";
 * 
 * const app = new cdk.App();
 * new MyStack(app, "MyStack");
 * 
 * // Add CdkGraph after other construct added to app
 * new CdkGraph(app);
 * </pre></blockquote>
 * <p>
 * <h4>Using Plugins</h4>
 * <p>
 * <blockquote><pre>
 * #!/usr/bin/env node
 * import * as cdk from "aws-cdk-lib";
 * import { MyStack } from "../lib/my-stack";
 * 
 * import { CdkGraph } from "&#64;aws-prototyping-sdk/cdk-graph";
 * import { ExamplePlugin } from "&#64;aws-prototyping-sdk/cdk-graph-plugin-example"; // does not exist, just example
 * 
 * const app = new cdk.App();
 * new MyStack(app, "MyStack");
 * 
 * // Add CdkGraph after other construct added to app
 * new CdkGraph(app, {
 *   plugins: [new ExamplePlugin()],
 * });
 * </pre></blockquote>
 * <p>
 * <hr>
 * <p>
 * <h3>Config</h3>
 * <p>
 * Configuration is supported through the <code>.cdkgraphrc.js</code> and depending on the plugin, through passing config to the plugin instance.
 * <p>
 * Config precedence follows 1) defaults, 2) cdkgraphrc, 3) instance.
 * <p>
 * <blockquote><pre>
 * // .cdkgraphrc.js
 * module.exports = {
 *   // Defaults to "&lt;cdk.out&gt;/cdkgraph"
 *   outdir: "reports/graph",
 * 
 *   // plugin configuration
 *   example: {
 *     verbose: true,
 *     reportType: "csv",
 *   },
 * };
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * #!/usr/bin/env node
 * import * as cdk from "aws-cdk-lib";
 * import { MyStack } from "../lib/my-stack";
 * 
 * import { CdkGraph } from "&#64;aws-prototyping-sdk/cdk-graph";
 * import { ExamplePlugin } from "&#64;aws-prototyping-sdk/cdk-graph-plugin-example"; // does not exist, just example
 * 
 * const app = new cdk.App();
 * new MyStack(app, "MyStack");
 * 
 * // Add CdkGraph after other construct added to app
 * new CdkGraph(app, {
 *   plugins: [
 *     new ExamplePlugin({
 *       // Will override .cdkgraphrc.js value
 *       verbose: false,
 *     }),
 *   ],
 * });
 * </pre></blockquote>
 * <p>
 * <hr>
 * <p>
 * <h3>Plugin Interface</h3>
 * <p>
 * <blockquote><pre>
 * /** CdkGraph **Plugin** interface *&#47;
 * export interface ICdkGraphPlugin {
 *   /** Unique identifier for this plugin *&#47;
 *   readonly id: string;
 *   /** Plugin version *&#47;
 *   readonly version: Version;
 *   /** List of plugins this plugin depends on, including optional semver version (eg: ["foo", "bar&#64;1.2"]) *&#47;
 *   readonly dependencies?: string[];
 * 
 *   /**
 *    * Binds the plugin to the CdkGraph instance. Enables plugins to receive base configs.
 *    *&#47;
 *   bind: IGraphPluginBindCallback;
 * 
 *   /**
 *    * Node visitor callback for construct tree traversal. This follows IAspect.visit pattern, but the order
 *    * of visitor traversal in managed by the CdkGraph.
 *    *&#47;
 *   inspect?: IGraphVisitorCallback;
 *   /**
 *    * Called during CDK synthesize to generate synchronous artifacts based on the in-memory graph passed
 *    * to the plugin. This is called in fifo order of plugins.
 *    *&#47;
 *   synthesize?: IGraphSynthesizeCallback;
 *   /**
 *    * Generate asynchronous reports based on the graph. This is not automatically called when synthesizing CDK.
 *    * Developer must explicitly add `await graphInstance.report()` to the CDK bin or invoke this outside
 *    * of the CDK synth. In either case, the plugin receives the in-memory graph interface when invoked, as the
 *    * CdkGraph will deserialize the graph prior to invoking the plugin report.
 *    *&#47;
 *   report?: IGraphReportCallback;
 * }
 * </pre></blockquote>
 * <p>
 * Plugin operations are automatically invoked by CdkGraph in the order they are defined in the <code>plugins</code> property. The invocation flow of plugins follows: 1) <code>bind</code>, 2) <code>inspect</code>, 3) <code>synthesize</code>, 4) <code>async report</code>.
 * <p>
 * <h3>Asynchronous Plugins</h3>
 * <p>
 * Some plugins may requiring performing asynchronous requests, or may make expensive operations that are best left outside of the synthesis process.
 * <p>
 * CdkGraph support asynchronous operations through the <code>async report()</code> method of plugins. However, since CDK does not support asynchronous operations during synthesis, this must be wired up a bit differently.
 * <p>
 * <blockquote><pre>
 * #!/usr/bin/env node
 * import * as cdk from "aws-cdk-lib";
 * import { MyStack } from "../lib/my-stack";
 * 
 * import { CdkGraph } from "&#64;aws-prototyping-sdk/cdk-graph";
 * import { ExampleAsyncPlugin } from "&#64;aws-prototyping-sdk/cdk-graph-plugin-async-example"; // does not exist, just example
 * 
 * (async () =&gt; {
 *   const app = new cdk.App();
 *   new MyStack(app, "MyStack");
 * 
 *   // Add CdkGraph after other construct added to app
 *   const graph = new CdkGraph(app, {
 *     plugins: [new ExampleAsyncPlugin()],
 *   });
 * 
 *   // invokes all plugin `report()` operations asynchronously (in order they are defined in `plugins` property)
 *   await graph.report();
 * })();
 * </pre></blockquote>
 * <p>
 * <h3>Example Plugin Implementation</h3>
 * <p>
 * Very basic example of implementing a plugin. Once the first actual plugins have been published this will be updated to reference those as examples.
 * <p>
 * <blockquote><pre>
 * import {
 *   CdkGraph,
 *   CdkGraphContext,
 *   ICdkGraphPlugin,
 * } from "&#64;aws-prototyping-sdk/cdk-graph";
 * 
 * export class CdkGraphExamplePlugin implements ICdkGraphPlugin {
 *   static readonly ARTIFACT_NS = "EXAMPLE";
 *   static readonly ID = "example";
 *   static readonly VERSION = "0.0.0";
 * 
 *   get id(): string {
 *     return CdkGraphDiagramPlugin.ID;
 *   }
 *   get version(): string {
 *     return CdkGraphDiagramPlugin.VERSION;
 *   }
 * 
 *   readonly dependencies?: string[] = [];
 * 
 *   /** &#64;internal *&#47;
 *   private _graph?: CdkGraph;
 * 
 *   bind(graph: CdkGraph): void {
 *     this._graph = graph;
 *   }
 * 
 *   synthesize(context: CdkGraphContext): void {
 *     const pluginConfig = this.config as Required&lt;IPluginConfig&gt;;
 * 
 *     // Get counts of all resources
 *     const cfnResourceCounts = context.store.counts.cfnResources;
 * 
 *     // Write plugin artifact
 *     context.writeArtifact(
 *       this,
 *       "EXAMPLE",
 *       "example.json",
 *       JSON.stringify(cfnResourceCounts, null, 2)
 *     );
 *   }
 * 
 *   async report(context: CdkGraphContext): void {
 *     // perform async operation here utilizing graph store
 *     const cfnResourceCounts = context.store.counts.cfnResources;
 *     const fetchedData = await fetch("https://example.com/data", {
 *       method: "POST",
 *       body: JSON.stringify(cfnResourceCounts),
 *     });
 * 
 *     // Write plugin artifact for fetched data
 *     context.writeArtifact(
 *       this,
 *       "EXAMPLE:FETCHED",
 *       "example-fetched.json",
 *       JSON.stringify(fetchedData, null, 2)
 *     );
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * <h3>Path to Stability</h3>
 * <p>
 * The below is a rough checklist of task necessary to elevate this from experimental to stable.
 * <p>
 * <ul>
 * <li>[ ] Dynamic versioning and Semver enforcement (store, plugins, etc)</li>
 * <li>[ ] Support running <code>async report()</code> method outside of CDK synthesis process</li>
 * <li>[ ] Find alternative synthesis solution that doesn't utilize CDK internals</li>
 * <li>[ ] Support custom Nodes and Edges</li>
 * <li>[ ] Improve logging, bookkeeping, and debugging</li>
 * <li>[ ] Implement store upgrade solution</li>
 * <li>[ ] Battle test the implementation against several plugins</li>
 * <li>[ ] Battle test the implementation in all target languages (currently tested in Typescript, but vended in all PDK supported languages)</li>
 * <li>[ ] Receive community feedback to validate approach</li>
 * </ul>
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Experimental)
package software.aws.awsprototypingsdk.cdkgraph;
