/**
 * <h2>OpenAPI Gateway</h2>
 * <p>
 * Define your APIs using <a href="https://awslabs.github.io/smithy/2.0/">Smithy</a> or <a href="https://swagger.io/specification/">OpenAPI v3</a>, and leverage the power of generated clients and documentation, automatic input validation, and type safe client and server code!
 * <p>
 * This package vends a projen project type which allows you to define an API using either <a href="https://awslabs.github.io/smithy/2.0/">Smithy</a> or <a href="https://swagger.io/specification/">OpenAPI v3</a>, and a construct which manages deploying this API in API Gateway, given an integration (eg a lambda) for every operation.
 * <p>
 * The project will generate models and clients from your api definition in your desired languages, and can be utilised both client side or server side in lambda handlers. The project type also generates a wrapper construct which adds type safety to ensure an integration is provided for every API operation.
 * <p>
 * When you change your API specification, just run <code>npx projen</code> again to regenerate all of this!
 * <p>
 * <h3>Project</h3>
 * <p>
 * Use the provided projen project types to get started with your API quickly! There are options for TypeScript, Python and Java:
 * <p>
 * <ul>
 * <li>Smithy
 * <p>
 * <ul>
 * <li><code>SmithyApiGatewayTsProject</code></li>
 * <li><code>SmithyApiGatewayPythonProject</code></li>
 * <li><code>SmithyApiGatewayJavaProject</code></li>
 * </ul></li>
 * <li>OpenAPI
 * <p>
 * <ul>
 * <li><code>OpenApiGatewayTsProject</code></li>
 * <li><code>OpenApiGatewayPythonProject</code></li>
 * <li><code>OpenApiGatewayJavaProject</code></li>
 * </ul></li>
 * </ul>
 * <p>
 * We recommend using the Smithy-based projects, given that Smithy has a less verbose and more powerful IDL (Interface Definition Language).
 * <p>
 * Choose the project type based on the language you'd like to <em>write your CDK infrastructure in</em>. Whichever option above you choose, you can still write your server-side code in any language.
 * <p>
 * It's recommended that these projects are used as part of an <code>nx-monorepo</code> project, as it makes setting up dependencies much easier, and you will not need to manage the build order of generated clients.
 * <p>
 * <h4>Typescript</h4>
 * <p>
 * <h5>OpenAPI</h5>
 * <p>
 * While it is recommended to use the project within an <code>nx-monorepo</code>, you can still use this as a standalone project if you like (eg <code>npx projen new --from &#64;aws-prototyping-sdk/open-api-gateway open-api-gateway-ts</code>), however you will need to manage build order (ie building the generated client first, followed by the project).
 * <p>
 * For usage in a monorepo:
 * <p>
 * Create the project in your .projenrc:
 * <p>
 * <blockquote><pre>
 * import { ClientLanguage, DocumentationFormat, OpenApiGatewayTsProject } from "&#64;aws-prototyping-sdk/open-api-gateway";
 * 
 * new OpenApiGatewayTsProject({
 *   parent: myNxMonorepo,
 *   defaultReleaseBranch: "mainline",
 *   name: "my-api",
 *   outdir: "packages/api",
 *   clientLanguages: [ClientLanguage.TYPESCRIPT, ClientLanguage.PYTHON, ClientLanguage.JAVA],
 *   documentationFormats: [DocumentationFormat.HTML2, DocumentationFormat.PLANTUML, DocumentationFormat.MARKDOWN],
 * });
 * </pre></blockquote>
 * <p>
 * In the output directory (<code>outdir</code>), you'll find a few files to get you started.
 * <p>
 * <blockquote><pre>
 * |_ src/
 *     |_ spec/
 *         |_ spec.yaml - The OpenAPI specification - edit this to define your API
 *         |_ .parsed-spec.json - A json spec generated from your spec.yaml.
 *     |_ api/
 *         |_ api.ts - A CDK construct which defines the API Gateway resources to deploy your API.
 *         |           This wraps the OpenApiGatewayRestApi construct and provides typed interfaces for integrations specific
 *         |           to your API. You shouldn't need to modify this, instead just extend it as in sample-api.ts.
 *         |_ sample-api.ts - Example usage of the construct defined in api.ts.
 *         |_ sample-api.say-hello.ts - An example lambda handler for the operation defined in spec.yaml, making use of the
 *                                      generated lambda handler wrappers for marshalling and type safety.
 * |_ generated/
 *     |_ typescript/ - A generated typescript API client.
 *     |_ python/ - A generated python API client.
 *     |_ java/ - A generated java API client.
 *     |_ documentation/
 *         |_ html2/ - Generated html documentation
 *         |_ markdown/ - Generated markdown documentation
 *         |_ plantuml/ - Generated plant uml documentation
 * </pre></blockquote>
 * <p>
 * If you would prefer to not generate the sample code, you can pass <code>sampleCode: false</code> to <code>OpenApiGatewayTsProject</code>.
 * <p>
 * To make changes to your api, simply update <code>spec.yaml</code> and run <code>npx projen</code> to synthesize all the typesafe client/server code!
 * <p>
 * The <code>SampleApi</code> construct uses <code>NodejsFunction</code> to declare the example lambda, but you are free to change this!
 * <p>
 * <h5>Smithy</h5>
 * <p>
 * To create a project with Smithy as your interface definition language (IDL), you can use the <code>SmithyApiGatewayTsProject</code>:
 * <p>
 * <blockquote><pre>
 * import { ClientLanguage, DocumentationFormat, SmithyApiGatewayTsProject } from "&#64;aws-prototyping-sdk/open-api-gateway";
 * 
 * new SmithyApiGatewayTsProject({
 *   parent: myNxMonorepo,
 *   defaultReleaseBranch: "mainline",
 *   name: "my-api",
 *   outdir: "packages/api",
 *   clientLanguages: [ClientLanguage.TYPESCRIPT, ClientLanguage.PYTHON, ClientLanguage.JAVA],
 *   documentationFormats: [DocumentationFormat.HTML2, DocumentationFormat.PLANTUML, DocumentationFormat.MARKDOWN],
 * });
 * </pre></blockquote>
 * <p>
 * This will result in a directory structure similar to the following:
 * <p>
 * <blockquote><pre>
 * |_ smithy/
 *     |_ src/
 *         |_ main/
 *             |_ smithy - The Smithy model used to define your API. You can define as many .smithy files in here as you like.
 *     |_ build.gradle - Smithy build gradle file - use this to add dependencies/plugins used in your smithy build
 *     |_ settings.gradle - Setup for the Smithy gradle project
 *     |_ smithy-build.json - Smithy build configuration - managed via options on the projen project
 * |_ src/
 *     |_ spec/
 *         |_ .parsed-spec.json - A json spec generated from your Smithy model.
 *     |_ api/
 *         |_ api.ts - A CDK construct which defines the API Gateway resources to deploy your API.
 *         |           This wraps the OpenApiGatewayRestApi construct and provides typed interfaces for integrations specific
 *         |           to your API. You shouldn't need to modify this, instead just extend it as in sample-api.ts.
 *         |_ sample-api.ts - Example usage of the construct defined in api.ts.
 *         |_ sample-api.say-hello.ts - An example lambda handler for the operation defined in spec.yaml, making use of the
 *                                      generated lambda handler wrappers for marshalling and type safety.
 * |_ generated/
 *     |_ typescript/ - A generated typescript API client.
 *     |_ python/ - A generated python API client.
 *     |_ java/ - A generated java API client.
 *     |_ documentation/
 *         |_ html2/ - Generated html documentation
 *         |_ markdown/ - Generated markdown documentation
 *         |_ plantuml/ - Generated plant uml documentation
 * </pre></blockquote>
 * <p>
 * <h6>Customisation</h6>
 * <p>
 * You can customise the Smithy project with several properties:
 * <p>
 * <blockquote><pre>
 * new SmithyApiGatewayTsProject({
 *   parent: myNxMonorepo,
 *   defaultReleaseBranch: "mainline",
 *   name: "my-api",
 *   outdir: "packages/api",
 *   clientLanguages: [],
 *   // The fully qualified service name for your API, including both the namespace and service name:
 *   serviceName: 'com.mycompany#MyService',
 *   // Custom directory for your smithy model, relative to the project outdir
 *   modelDir: 'api-model',
 *   // By default, the contents of `smithy-build/output` will be ignored by source control.
 *   // Set this to false to include it, for example if you are generating clients directly from the smithy model.
 *   ignoreSmithyBuildOutput: false,
 *   // The gradle wrapper used for the smithy build is copied from the PDK itself if it does not already exist in
 *   // the 'smithy' folder. By default, this gradle wrapper will be ignored by source control.
 *   // Set this to false if you would like to check the gradle wrapper in to source control, for example if you want
 *   // to use a different version of the gradle wrapper in your project.
 *   ignoreGradleWrapper: false,
 *   // Use smithyBuildOptions to control what is added to smithy-build.json. See more details below.
 *   smithyBuildOptions: {
 *     projections: {
 *       "ts-client": {
 *         "plugins": {
 *           "typescript-codegen": {
 *             "package" : "&#64;my-test/smithy-generated-typescript-client",
 *             "packageVersion": "0.0.1"
 *           }
 *         }
 *       }
 *     }
 *   }
 * });
 * </pre></blockquote>
 * <p>
 * <code>smithyBuildOptions</code> allows you to customise the Smithy build, used for generating clients from the Smithy model (eg above), or customising the OpenAPI generation.
 * <p>
 * OpenAPI generation can be customised by referencing the <code>openapi</code> projection as follows:
 * <p>
 * <blockquote><pre>
 * smithyBuildOptions: {
 *   projections: {
 *     openapi: {
 *       plugins: {
 *         openapi: {
 *           // Customise the openapi projection here...
 *           forbidGreedyLabels: true,
 *           ...
 *         }
 *       }
 *     }
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * The OpenAPI specification generated by this projection is used to create the CDK infrastructure, lambda handler wrappers, etc. Options for configuring OpenAPI generation can be found <a href="https://awslabs.github.io/smithy/2.0/guides/converting-to-openapi.html">in the Smithy OpenAPI documentation</a>.
 * <p>
 * Note that any additional dependencies required for projections/plugins can be added by modifying <code>smithyBuildOptions</code>, which in turn will add them to the <code>smithy/build.gradle</code> file:
 * <p>
 * <blockquote><pre>
 * smithyBuildOptions: {
 *   maven: {
 *     dependencies: [
 *       "software.amazon.smithy:smithy-validation-model:1.27.2",
 *     ]
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * Notice <code>smithy-cli</code>, <code>smithy-model</code>, <code>smithy-openapi</code> and <code>smithy-aws-traits</code> are always included in the <code>build.gradle</code> since they are required for converting your Smithy model to OpenAPI. You can customise the version of these dependencies by specifying them in <code>smithyBuildOptions</code>:
 * <p>
 * <blockquote><pre>
 * smithyBuildOptions: {
 *   maven: {
 *     dependencies: [
 *       "software.amazon.smithy:smithy-cli:1.27.0",
 *       "software.amazon.smithy:smithy-model:1.27.0",
 *       "software.amazon.smithy:smithy-openapi:1.27.0",
 *       "software.amazon.smithy:smithy-aws-traits:1.27.0",
 *     ]
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * <h4>Python</h4>
 * <p>
 * <h5>OpenAPI</h5>
 * <p>
 * As well as typescript, you can choose to generate the cdk construct and sample handler in python.
 * <p>
 * <blockquote><pre>
 * new OpenApiGatewayPythonProject({
 *   parent: myNxMonorepo,
 *   outdir: 'packages/myapi',
 *   name: 'myapi',
 *   moduleName: 'myapi',
 *   version: '1.0.0',
 *   authorName: 'jack',
 *   authorEmail: 'me&#64;example.com',
 *   clientLanguages: [ClientLanguage.TYPESCRIPT, ClientLanguage.PYTHON, ClientLanguage.JAVA],
 * });
 * </pre></blockquote>
 * <p>
 * You will need to set up a shared virtual environment and configure dependencies via the monorepo (see README.md for the nx-monorepo package). An example of a full <code>.projenrc.ts</code> might be:
 * <p>
 * <blockquote><pre>
 * import { nx_monorepo } from "aws-prototyping-sdk";
 * import { ClientLanguage, OpenApiGatewayPythonProject } from "&#64;aws-prototyping-sdk/open-api-gateway";
 * import { AwsCdkPythonApp } from "projen/lib/awscdk";
 * 
 * const monorepo = new nx_monorepo.NxMonorepoProject({
 *   defaultReleaseBranch: "main",
 *   devDeps: ["aws-prototyping-sdk", "&#64;aws-prototyping-sdk/open-api-gateway"],
 *   name: "open-api-test",
 * });
 * 
 * const api = new OpenApiGatewayPythonProject({
 *   parent: monorepo,
 *   outdir: 'packages/myapi',
 *   name: 'myapi',
 *   moduleName: 'myapi',
 *   version: '1.0.0',
 *   authorName: 'jack',
 *   authorEmail: 'me&#64;example.com',
 *   clientLanguages: [ClientLanguage.TYPESCRIPT],
 *   venvOptions: {
 *     // Use a shared virtual env dir.
 *     // The generated python client will also use this virtual env dir
 *     envdir: '../../.env',
 *   },
 * });
 * 
 * // Install into virtual env so it's available for the cdk app
 * api.tasks.tryFind('install')!.exec('pip install --editable .');
 * 
 * const app = new AwsCdkPythonApp({
 *   authorName: "jack",
 *   authorEmail: "me&#64;example.com",
 *   cdkVersion: "2.1.0",
 *   moduleName: "myapp",
 *   name: "myapp",
 *   version: "1.0.0",
 *   parent: monorepo,
 *   outdir: "packages/myapp",
 *   deps: [api.moduleName],
 *   venvOptions: {
 *     envdir: '../../.env',
 *   },
 * });
 * 
 * monorepo.addImplicitDependency(app, api);
 * 
 * monorepo.synth();
 * </pre></blockquote>
 * <p>
 * You'll find the following directory structure in <code>packages/myapi</code>:
 * <p>
 * <blockquote><pre>
 * |_ myapi/
 *     |_ spec/
 *         |_ spec.yaml - The OpenAPI specification - edit this to define your API
 *         |_ .parsed-spec.json - A json spec generated from your spec.yaml.
 *     |_ api/
 *         |_ api.py - A CDK construct which defines the API Gateway resources to deploy your API.
 *         |           This wraps the OpenApiGatewayRestApi construct and provides typed interfaces for integrations specific
 *         |           to your API. You shouldn't need to modify this, instead just extend it as in sample_api.py.
 *         |_ sample_api.py - Example usage of the construct defined in api.py.
 *         |_ handlers/
 *              |_ say_hello_handler_sample.py - An example lambda handler for the operation defined in spec.yaml, making use of the
 *                                               generated lambda handler wrappers for marshalling and type safety.
 * |_ generated/
 *     |_ typescript/ - A generated typescript API client.
 *     |_ python/ - A generated python API client.
 *     |_ java/ - A generated java API client.
 * </pre></blockquote>
 * <p>
 * For simplicity, the generated code deploys a lambda layer for the generated code and its dependencies. You may choose to define an entirely separate projen <code>PythonProject</code> for your lambda handlers should you wish to add more dependencies than just the generated code.
 * <p>
 * <h5>Smithy</h5>
 * <p>
 * Similar to typescript, you can use the <code>SmithyApiGatewayPythonProject</code>.
 * <p>
 * <h4>Java</h4>
 * <p>
 * <h5>OpenAPI</h5>
 * <p>
 * As well as TypeScript and Python, you can choose to generate the cdk construct and sample handler in Java.
 * <p>
 * <blockquote><pre>
 * new OpenApiGatewayJavaProject({
 *   parent: monorepo,
 *   outdir: 'packages/myapi',
 *   name: "myapi",
 *   groupId: "com.mycompany",
 *   artifactId: "my-api",
 *   version: "1.0.0",
 *   clientLanguages: [ClientLanguage.PYTHON, ClientLanguage.TYPESCRIPT],
 *   documentationFormats: [DocumentationFormat.HTML2, DocumentationFormat.PLANTUML, DocumentationFormat.MARKDOWN],
 * });
 * </pre></blockquote>
 * <p>
 * The output directory will look a little like this:
 * <p>
 * <blockquote><pre>
 * |_ src/
 *     |_ spec/
 *         |_ spec.yaml - The OpenAPI specification - edit this to define your API
 *     |_ main/
 *         |_ java/
 *             |_ api/
 *                 |_ Api.java - A CDK construct which defines the API Gateway resources to deploy your API.
 *                 |             This wraps the OpenApiGatewayRestApi construct and provides typed interfaces for integrations specific
 *                 |             to your API. You shouldn't need to modify this, instead just extend it as in SampleApi.java.
 *                 |_ ApiProps.java - Defines properties for the CDK construct in Api.java
 *                 |_ SampleApi.java - Example usage of the construct defined in Api.java
 *                 |_ SayHelloHandler.java - An example lambda handler for the operation defined in spec.yaml, making use of the
 *                                         generated lambda handler wrappers for marshalling and type safety.
 *         |_ resources/
 *             |_ .parsed-spec.json - A json spec generated from your spec.yaml. This will be bundled in the project jar.
 * |_ generated/
 *     |_ typescript/ - A generated typescript API client
 *     |_ python/ - A generated python API client.
 *     |_ java/ - A generated java API client.
 *     |_ documentation/
 *         |_ html2/ - Generated html documentation
 *         |_ markdown/ - Generated markdown documentation
 *         |_ plantuml/ - Generated plant uml documentation
 * </pre></blockquote>
 * <p>
 * The <code>SampleApi</code> construct uses a lambda function which deploys the entire project jar as a simple way to get started with an api that deploys out of the box. This jar includes a lot of extra code and dependencies that you don't need in your lambda, so it's recommended that after experimenting with the construct, you separate your lambdas into another <code>JavaProject</code>. Please refer to the <code>Java API Lambda Handlers</code> section of this README for details on how to set this up.
 * <p>
 * <h5>Smithy</h5>
 * <p>
 * Similar to typescript and python, you can use the <code>SmithyApiGatewayJavaProject</code>.
 * <p>
 * <h3>Smithy IDL</h3>
 * <p>
 * Please refer to the <a href="https://awslabs.github.io/smithy/2.0/quickstart.html">Smithy documentation</a> for how to write models in Smithy. A basic example is provided below:
 * <p>
 * <blockquote><pre>
 * $version: "2"
 * namespace example.hello
 * 
 * use aws.protocols#restJson1
 * 
 * &#64;title("A Sample Hello World API")
 * 
 * /// A sample smithy api
 * &#64;restJson1
 * service Hello {
 *     version: "1.0"
 *     operations: [SayHello]
 * }
 * 
 * &#64;readonly
 * &#64;http(method: "GET", uri: "/hello")
 * operation SayHello {
 *     input: SayHelloInput
 *     output: SayHelloOutput
 *     errors: [ApiError]
 * }
 * 
 * string Name
 * string Message
 * 
 * &#64;input
 * structure SayHelloInput {
 *     &#64;httpQuery("name")
 *     &#64;required
 *     name: Name
 * }
 * 
 * &#64;output
 * structure SayHelloOutput {
 *     &#64;required
 *     message: Message
 * }
 * 
 * &#64;error("client")
 * structure ApiError {
 *     &#64;required
 *     errorMessage: Message
 * }
 * </pre></blockquote>
 * <p>
 * <h4>Supported Protocols</h4>
 * <p>
 * Currently only <a href="https://awslabs.github.io/smithy/2.0/aws/protocols/aws-restjson1-protocol.html">AWS restJson1</a> is supported. Please ensure your service is annotated with the <code>&#64;restJson1</code> trait.
 * <p>
 * <h4>Multiple Files</h4>
 * <p>
 * You can split your definition into multiple files and folders, so long as they are all under the <code>model</code> directory in your API project.
 * <p>
 * <h4>Authorizers</h4>
 * <p>
 * Smithy supports <a href="https://awslabs.github.io/smithy/2.0/aws/aws-auth.html">adding API Gateway authorizers in the model itself</a>. Given that at model definition time one usually does not know the ARN of the user pool or lambda function for an authorizer, it is recommended to add the authorizer(s) in your Api CDK construct.
 * <p>
 * If using Smithy generated clients, some authorizer traits (eg sigv4) will include configuring the client for that particular method of authorization, so it can be beneficial to still define authorizers in the model. We therefore support specifying authorizers in both the model and the construct, but note that the construct will take precedence where the authorizer ID is the same.
 * <p>
 * <h3>OpenAPI Specification</h3>
 * <p>
 * Your <code>spec.yaml</code> file defines your api using <a href="https://swagger.io/specification/">OpenAPI Version 3.0.3</a>. An example spec might look like:
 * <p>
 * <blockquote><pre>
 * openapi: 3.0.3
 * info:
 *   version: 1.0.0
 *   title: Example API
 * paths:
 *   /hello:
 *     get:
 *       operationId: sayHello
 *       parameters:
 *         - in: query
 *           name: name
 *           schema:
 *             type: string
 *           required: true
 *       responses:
 *         '200':
 *           description: Successful response
 *           content:
 *             'application/json':
 *               schema:
 *                 $ref: '#/components/schemas/HelloResponse'
 * components:
 *   schemas:
 *     HelloResponse:
 *       type: object
 *       properties:
 *         message:
 *           type: string
 *       required:
 *         - message
 * </pre></blockquote>
 * <p>
 * You can divide your specification into multiple files using <code>$ref</code>.
 * <p>
 * For example, you might choose to structure your spec as follows:
 * <p>
 * <blockquote><pre>
 * |_ spec/
 *     |_ spec.yaml
 *     |_ paths/
 *         |_ index.yaml
 *         |_ sayHello.yaml
 *     |_ schemas/
 *         |_ index.yaml
 *         |_ helloResponse.yaml
 * </pre></blockquote>
 * <p>
 * Where <code>spec.yaml</code> looks as follows:
 * <p>
 * <blockquote><pre>
 * openapi: 3.0.3
 * info:
 *   version: 1.0.0
 *   title: Example API
 * paths:
 *   $ref: './paths/index.yaml'
 * components:
 *   schemas:
 *     $ref: './schemas/index.yaml'
 * </pre></blockquote>
 * <p>
 * <code>paths/index.yaml</code>:
 * <p>
 * <blockquote><pre>
 * /hello:
 *   get:
 *     $ref: './sayHello.yaml'
 * </pre></blockquote>
 * <p>
 * <code>paths/sayHello.yaml</code>:
 * <p>
 * <blockquote><pre>
 * operationId: sayHello
 * parameters:
 *  - in: query
 *    name: name
 *    schema:
 *      type: string
 *    required: true
 * responses:
 *   '200':
 *     description: Successful response
 *     content:
 *       'application/json':
 *         schema:
 *           $ref: '../schemas/helloResponse.yaml'
 * </pre></blockquote>
 * <p>
 * <code>schemas/index.yaml</code>:
 * <p>
 * <blockquote><pre>
 * HelloResponse:
 *   $ref: './helloResponse.yaml'
 * </pre></blockquote>
 * <p>
 * <code>schemas/helloResponse.yaml</code>:
 * <p>
 * <blockquote><pre>
 * type: object
 * properties:
 *   message:
 *     type: string
 * required:
 *   - message
 * </pre></blockquote>
 * <p>
 * <h3>Construct</h3>
 * <p>
 * A sample construct is generated which provides a type-safe interface for creating an API Gateway API based on your OpenAPI specification. You'll get a type error if you forget to define an integration for an operation defined in your api.
 * <p>
 * <blockquote><pre>
 * import { Authorizers, Integrations } from '&#64;aws-prototyping-sdk/open-api-gateway';
 * import { NodejsFunction } from 'aws-cdk-lib/aws-lambda-nodejs';
 * import { Construct } from 'constructs';
 * import { Api } from './api';
 * 
 * /**
 *  * An example of how to wire lambda handler functions to the API
 *  *&#47;
 * export class SampleApi extends Api {
 *   constructor(scope: Construct, id: string) {
 *     super(scope, id, {
 *       defaultAuthorizer: Authorizers.iam(),
 *       integrations: {
 *         // Every operation defined in your API must have an integration defined!
 *         sayHello: {
 *           integration: Integrations.lambda(new NodejsFunction(scope, 'say-hello')),
 *         },
 *       },
 *     });
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * <h4>Sharing Integrations</h4>
 * <p>
 * If you would like to use the same integration for every operation, you can use the <code>Operations.all</code> method from your generated client to save repeating yourself, for example:
 * <p>
 * <blockquote><pre>
 * import { Operations } from 'my-api-typescript-client';
 * import { Authorizers, Integrations } from '&#64;aws-prototyping-sdk/open-api-gateway';
 * import { NodejsFunction } from 'aws-cdk-lib/aws-lambda-nodejs';
 * import { Construct } from 'constructs';
 * import { Api } from './api';
 * 
 * export class SampleApi extends Api {
 *   constructor(scope: Construct, id: string) {
 *     super(scope, id, {
 *       defaultAuthorizer: Authorizers.iam(),
 *       // Use the same integration for every operation.
 *       integrations: Operations.all({
 *         integration: Integrations.lambda(new NodejsFunction(scope, 'say-hello')),
 *       }),
 *     });
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * <h4>Authorizers</h4>
 * <p>
 * The <code>Api</code> construct allows you to define one or more authorizers for securing your API. An integration will use the <code>defaultAuthorizer</code> unless an <code>authorizer</code> is specified at the integration level. The following authorizers are supported:
 * <p>
 * <ul>
 * <li><code>Authorizers.none</code> - No auth</li>
 * <li><code>Authorizers.iam</code> - AWS IAM (Signature Version 4)</li>
 * <li><code>Authorizers.cognito</code> - Cognito user pool</li>
 * <li><code>Authorizers.custom</code> - A custom authorizer</li>
 * </ul>
 * <p>
 * <h5>Cognito Authorizer</h5>
 * <p>
 * To use the Cognito authorizer, one or more user pools must be provided. You can optionally specify the scopes to check if using an access token. You can use the <code>withScopes</code> method to use the same authorizer but verify different scopes for individual integrations, for example:
 * <p>
 * <blockquote><pre>
 * export class SampleApi extends Api {
 *   constructor(scope: Construct, id: string) {
 *     const cognitoAuthorizer = Authorizers.cognito({
 *       authorizerId: 'myCognitoAuthorizer',
 *       userPools: [new UserPool(scope, 'UserPool')],
 *     });
 * 
 *     super(scope, id, {
 *       defaultAuthorizer: cognitoAuthorizer,
 *       integrations: {
 *         // Everyone in the user pool can call this operation:
 *         sayHello: {
 *           integration: Integrations.lambda(new NodejsFunction(scope, 'say-hello')),
 *         },
 *         // Only users with the given scopes can call this operation
 *         myRestrictedOperation: {
 *           integration: Integrations.lambda(new NodejsFunction(scope, 'my-restricted-operation')),
 *           authorizer: cognitoAuthorizer.withScopes('my-resource-server/my-scope'),
 *         },
 *       },
 *     });
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * For more information about scopes or identity and access tokens, please see the <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-integrate-with-cognito.html">API Gateway documentation</a>.
 * <p>
 * <h5>Custom Authorizer</h5>
 * <p>
 * Custom authorizers use lambda functions to handle authorizing requests. These can either be simple token-based authorizers, or more complex request-based authorizers. See the <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html">API Gateway documentation</a> for more details.
 * <p>
 * An example token-based authorizer (default):
 * <p>
 * <blockquote><pre>
 * Authorizers.custom({
 *   authorizerId: 'myTokenAuthorizer',
 *   function: new NodejsFunction(scope, 'authorizer'),
 * });
 * </pre></blockquote>
 * <p>
 * An example request-based handler. By default the identitySource will be <code>method.request.header.Authorization</code>, however you can customise this as per <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-identitysource">the API Gateway documentation</a>.
 * <p>
 * <blockquote><pre>
 * Authorizers.custom({
 *   authorizerId: 'myRequestAuthorizer',
 *   type: CustomAuthorizerType.REQUEST,
 *   identitySource: 'method.request.header.MyCustomHeader, method.request.querystring.myQueryString',
 *   function: new NodejsFunction(scope, 'authorizer'),
 * });
 * </pre></blockquote>
 * <p>
 * <h4>Integrations</h4>
 * <p>
 * Integrations are used by API Gateway to service requests.
 * <p>
 * <h5>Lambda Integration</h5>
 * <p>
 * Currently, the only built-in integration is a lambda integration. You can construct one using <code>Integrations.lambda(yourLambdaFunction)</code>.
 * <p>
 * <h5>Custom Integrations</h5>
 * <p>
 * You can implement your own integrations by inheriting the <code>Integration</code> class and implementing its <code>render</code> method. This method is responsible for returning a snippet of OpenAPI which will be added as the <code>x-amazon-apigateway-integration</code> for an operation. Please refer to the <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-integration.html">API Gateway Swagger Extensions documentation</a> for more details.
 * <p>
 * You can also optionally override the <code>grant</code> method if you need to use CDK to grant API Gateway access to invoke your integration.
 * <p>
 * <h3>Generated Client</h3>
 * <p>
 * <h4>Typescript</h4>
 * <p>
 * The <a href="https://openapi-generator.tech/docs/generators/typescript-fetch/">typescript-fetch</a> OpenAPI generator is used to generate OpenAPI clients for typescript. This requires an implementation of <code>fetch</code> to be passed to the client. In the browser one can pass the built in fetch, or in NodeJS you can use an implementation such as <a href="https://www.npmjs.com/package/node-fetch">node-fetch</a>.
 * <p>
 * Example usage of the client in a website:
 * <p>
 * <blockquote><pre>
 * import { Configuration, DefaultApi } from "my-api-typescript-client";
 * 
 * const client = new DefaultApi(new Configuration({
 *   basePath: "https://xxxxxxxxxx.execute-api.ap-southeast-2.amazonaws.com",
 *   fetchApi: window.fetch.bind(window),
 * }));
 * 
 * await client.sayHello({ name: "Jack" });
 * </pre></blockquote>
 * <p>
 * <h4>Python</h4>
 * <p>
 * The <a href="https://openapi-generator.tech/docs/generators/python-experimental">python-experimental</a> OpenAPI generator is used to generate OpenAPI clients for python.
 * <p>
 * Example usage:
 * <p>
 * <blockquote><pre>
 * from my_api_python import ApiClient, Configuration
 * from my_api_python.api.default_api import DefaultApi
 * 
 * configuration = Configuration(
 *     host = "https://xxxxxxxxxx.execute-api.ap-southeast-2.amazonaws.com"
 * )
 * 
 * with ApiClient(configuration) as api_client:
 *     client = DefaultApi(api_client)
 * 
 *     client.say_hello(
 *         query_params={
 *             'name': "name_example",
 *         },
 *     )
 * </pre></blockquote>
 * <p>
 * You'll find details about how to use the python client in the README.md alongside your generated client.
 * <p>
 * <h4>Java</h4>
 * <p>
 * The <a href="https://openapi-generator.tech/docs/generators/java/">java</a> OpenAPI generator is used to generate OpenAPI clients for Java.
 * <p>
 * Example usage:
 * <p>
 * <blockquote><pre>
 * import com.generated.api.myapijava.client.api.DefaultApi;
 * import com.generated.api.myapijava.client.ApiClient;
 * import com.generated.api.myapijava.client.Configuration;
 * import com.generated.api.myapijava.client.models.HelloResponse;
 * 
 * ApiClient client = Configuration.getDefaultApiClient();
 * client.setBasePath("https://xxxxxxxxxx.execute-api.ap-southeast-2.amazonaws.com");
 * 
 * DefaultApi api = new DefaultApi(client);
 * HelloResponse response = api.sayHello("Adrian").execute()
 * </pre></blockquote>
 * <p>
 * You'll find more details about how to use the Java client in the README.md alongside your generated client.
 * <p>
 * <h3>Lambda Handler Wrappers</h3>
 * <p>
 * Lambda handler wrappers are also importable from the generated client. These provide input/output type safety, ensuring that your API handlers return outputs that correspond to your specification.
 * <p>
 * <h4>Typescript</h4>
 * <p>
 * <blockquote><pre>
 * import { sayHelloHandler } from "my-api-typescript-client";
 * 
 * export const handler = sayHelloHandler(async ({ input }) =&gt; {
 *   return {
 *     statusCode: 200,
 *     body: {
 *       message: `Hello ${input.requestParameters.name}!`,
 *     },
 *   };
 * });
 * </pre></blockquote>
 * <p>
 * <h5>Handler Router</h5>
 * <p>
 * The lambda handler wrappers can be used in isolation as handler methods for separate lambdas. If you would like to use a single lambda function to serve all requests, you can do so with the <code>handlerRouter</code>.
 * <p>
 * <blockquote><pre>
 * import { handlerRouter, sayHelloHandler, sayGoodbyeHandler } from "my-api-typescript-client";
 * import { corsInterceptor } from "./interceptors";
 * import { sayGoodbye } from "./handlers/say-goodbye";
 * 
 * const sayHello = sayHelloHandler(async ({ input }) =&gt; {
 *   return {
 *     statusCode: 200,
 *     body: {
 *       message: `Hello ${input.requestParameters.name}!`,
 *     },
 *   };
 * });
 * 
 * export const handler = handlerRouter({
 *   // Interceptors declared in this list will apply to all operations
 *   interceptors: [corsInterceptor],
 *   // Assign handlers to each operation here
 *   handlers: {
 *     sayHello,
 *     sayGoodbye,
 *   },
 * });
 * </pre></blockquote>
 * <p>
 * <h4>Python</h4>
 * <p>
 * <blockquote><pre>
 * from myapi_python.apis.tags.default_api_operation_config import say_hello_handler, SayHelloRequest, ApiResponse, SayHelloOperationResponses
 * from myapi_python.model.api_error import ApiError
 * from myapi_python.model.hello_response import HelloResponse
 * 
 * &#64;say_hello_handler
 * def handler(input: SayHelloRequest, **kwargs) -&gt; SayHelloOperationResponses:
 *     return ApiResponse(
 *         status_code=200,
 *         body=HelloResponse(message="Hello {}!".format(input.request_parameters["name"])),
 *         headers={}
 *     )
 * </pre></blockquote>
 * <p>
 * <h5>Handler Router</h5>
 * <p>
 * The lambda handler wrappers can be used in isolation as handler methods for separate lambdas. If you would like to use a single lambda function to serve all requests, you can do so with the <code>handler_router</code>.
 * <p>
 * <blockquote><pre>
 * from myapi_python.apis.tags.default_api_operation_config import say_hello_handler, SayHelloRequest, ApiResponse, SayHelloOperationResponses, handler_router, HandlerRouterHandlers
 * from myapi_python.model.api_error import ApiError
 * from myapi_python.model.hello_response import HelloResponse
 * from other_handlers import say_goodbye
 * from my_interceptors import cors_interceptor
 * 
 * &#64;say_hello_handler
 * def say_hello(input: SayHelloRequest, **kwargs) -&gt; SayHelloOperationResponses:
 *     return ApiResponse(
 *         status_code=200,
 *         body=HelloResponse(message="Hello {}!".format(input.request_parameters["name"])),
 *         headers={}
 *     )
 * 
 * handler = handler_router(
 *   # Interceptors defined here will apply to all operations
 *   interceptors=[cors_interceptor],
 *   handlers=HandlerRouterHandlers(
 *     say_hello=say_hello,
 *     say_goodbye=say_goodbye
 *   )
 * )
 * </pre></blockquote>
 * <p>
 * <h4>Java</h4>
 * <p>
 * <blockquote><pre>
 * import com.generated.api.myapijava.client.api.Handlers.SayHello;
 * import com.generated.api.myapijava.client.api.Handlers.SayHello200Response;
 * import com.generated.api.myapijava.client.api.Handlers.SayHelloRequestInput;
 * import com.generated.api.myapijava.client.api.Handlers.SayHelloResponse;
 * import com.generated.api.myapijava.client.model.HelloResponse;
 * 
 * 
 * public class SayHelloHandler extends SayHello {
 *     &#64;Override
 *     public SayHelloResponse handle(SayHelloRequestInput sayHelloRequestInput) {
 *         return SayHello200Response.of(HelloResponse.builder()
 *                 .message(String.format("Hello %s", sayHelloRequestInput.getInput().getRequestParameters().getName()))
 *                 .build());
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * <h5>Handler Router</h5>
 * <p>
 * The lambda handler wrappers can be used in isolation as handler methods for separate lambdas. If you would like to use a single lambda function to serve all requests, you can do so by extending the <code>HandlerRouter</code> class.
 * <p>
 * <blockquote><pre>
 * import com.generated.api.myapijava.client.api.Handlers.SayGoodbye;
 * import com.generated.api.myapijava.client.api.Handlers.HandlerRouter;
 * import com.generated.api.myapijava.client.api.Handlers.Interceptors;
 * import com.generated.api.myapijava.client.api.Handlers.SayHello;
 * 
 * import java.util.Arrays;
 * import java.util.List;
 * 
 * // Interceptors defined here apply to all operations
 * &#64;Interceptors({ TimingInterceptor.class })
 * public class ApiHandlerRouter extends HandlerRouter {
 *     // You must implement a method to return a handler for every operation
 *     &#64;Override
 *     public SayHello sayHello() {
 *         return new SayHelloHandler();
 *     }
 * 
 *     &#64;Override
 *     public SayGoodbye sayGoodbye() {
 *         return new SayGoodbyeHandler();
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * <h3>Interceptors</h3>
 * <p>
 * The lambda handler wrappers allow you to pass in a <em>chain</em> of handler functions to handle the request. This allows you to implement middleware / interceptors for handling requests. Each handler function may choose whether or not to continue the handler chain by invoking <code>chain.next</code>.
 * <p>
 * <h4>Typescript</h4>
 * <p>
 * In typescript, interceptors are passed as separate arguments to the generated handler wrapper, in the order in which they should be executed. Call <code>request.chain.next(request)</code> from an interceptor to delegate to the rest of the chain to handle a request. Note that the last handler in the chain (ie the actual request handler which transforms the input to the output) should not call <code>chain.next</code>.
 * <p>
 * <blockquote><pre>
 * import {
 *   sayHelloHandler,
 *   ChainedRequestInput,
 *   OperationResponse,
 * } from "my-api-typescript-client";
 * 
 * // Interceptor to wrap invocations in a try/catch, returning a 500 error for any unhandled exceptions.
 * const tryCatchInterceptor = async &lt;RequestParameters, RequestArrayParameters, RequestBody, Response&gt;(
 *   request: ChainedRequestInput&lt;RequestParameters, RequestArrayParameters, RequestBody, Response&gt;
 * ): Promise&lt;Response | OperationResponse&lt;500, { errorMessage: string }&gt;&gt; =&gt; {
 *   try {
 *     return await request.chain.next(request);
 *   } catch (e: any) {
 *     return { statusCode: 500, body: { errorMessage: e.message }};
 *   }
 * };
 * 
 * // tryCatchInterceptor is passed first, so it runs first and calls the second argument function (the request handler) via chain.next
 * export const handler = sayHelloHandler(tryCatchInterceptor, async ({ input }) =&gt; {
 *   return {
 *     statusCode: 200,
 *     body: {
 *       message: `Hello ${input.requestParameters.name}!`,
 *     },
 *   };
 * });
 * </pre></blockquote>
 * <p>
 * Another example interceptor might be to record request time metrics. The example below includes the full generic type signature for an interceptor:
 * <p>
 * <blockquote><pre>
 * import {
 *   ChainedRequestInput,
 * } from 'my-api-typescript-client';
 * 
 * const timingInterceptor = async &lt;RequestParameters, RequestArrayParameters, RequestBody, Response&gt;(
 *   request: ChainedRequestInput&lt;RequestParameters, RequestArrayParameters, RequestBody, Response&gt;
 * ): Promise&lt;Response&gt; =&gt; {
 *   const start = Date.now();
 *   const response = await request.chain.next(request);
 *   const end = Date.now();
 *   console.log(`Took ${end - start} ms`);
 *   return response;
 * };
 * </pre></blockquote>
 * <p>
 * Interceptors may mutate the <code>interceptorContext</code> to pass state to further interceptors or the final lambda handler, for example an <code>identityInterceptor</code> might want to extract the authenticated user from the request so that it is available in handlers.
 * <p>
 * <blockquote><pre>
 * import {
 *   LambdaRequestParameters,
 *   LambdaHandlerChain,
 * } from 'my-api-typescript-client';
 * 
 * const identityInterceptor = async &lt;RequestParameters, RequestArrayParameters, RequestBody, Response&gt;(
 *   request: ChainedRequestInput&lt;RequestParameters, RequestArrayParameters, RequestBody, Response&gt;
 * ): Promise&lt;Response&gt; =&gt; {
 *   const authenticatedUser = await getAuthenticatedUser(request.event);
 *   return await request.chain.next({
 *     ...request,
 *     interceptorContext: {
 *       ...request.interceptorContext,
 *       authenticatedUser,
 *     },
 *   });
 * };
 * </pre></blockquote>
 * <p>
 * <h4>Python</h4>
 * <p>
 * In Python, a list of interceptors can be passed as a keyword argument to the generated lambda handler decorator, for example:
 * <p>
 * <blockquote><pre>
 * from myapi_python.apis.tags.default_api_operation_config import say_hello_handler, SayHelloRequest, ApiResponse, SayHelloOperationResponses
 * from myapi_python.model.api_error import ApiError
 * from myapi_python.model.hello_response import HelloResponse
 * 
 * &#64;say_hello_handler(interceptors=[timing_interceptor, try_catch_interceptor])
 * def handler(input: SayHelloRequest, **kwargs) -&gt; SayHelloOperationResponses:
 *     return ApiResponse(
 *         status_code=200,
 *         body=HelloResponse(message="Hello {}!".format(input.request_parameters["name"])),
 *         headers={}
 *     )
 * </pre></blockquote>
 * <p>
 * Writing an interceptor is just like writing a lambda handler. Call <code>chain.next(input)</code> from an interceptor to delegate to the rest of the chain to handle a request.
 * <p>
 * <blockquote><pre>
 * import time
 * from myapi_python.apis.tags.default_api_operation_config import ChainedApiRequest, ApiResponse
 * 
 * def timing_interceptor(input: ChainedApiRequest) -&gt; ApiResponse:
 *     start = int(round(time.time() * 1000))
 *     response = input.chain.next(input)
 *     end = int(round(time.time() * 1000))
 *     print("Took {} ms".format(end - start))
 *     return response
 * </pre></blockquote>
 * <p>
 * Interceptors may choose to return different responses, for example to return a 500 response for any unhandled exceptions:
 * <p>
 * <blockquote><pre>
 * import time
 * from myapi_python.model.api_error import ApiError
 * from myapi_python.apis.tags.default_api_operation_config import ChainedApiRequest, ApiResponse
 * 
 * def try_catch_interceptor(input: ChainedApiRequest) -&gt; ApiResponse:
 *     try:
 *         return input.chain.next(input)
 *     except Exception as e:
 *         return ApiResponse(
 *             status_code=500,
 *             body=ApiError(errorMessage=str(e)),
 *             headers={}
 *         )
 * </pre></blockquote>
 * <p>
 * Interceptors are permitted to mutate the "interceptor context", which is a <code>Dict[str, Any]</code>. Each interceptor in the chain, and the final handler, can access this context:
 * <p>
 * <blockquote><pre>
 * def identity_interceptor(input: ChainedApiRequest) -&gt; ApiResponse:
 *     input.interceptor_context["AuthenticatedUser"] = get_authenticated_user(input.event)
 *     return input.chain.next(input)
 * </pre></blockquote>
 * <p>
 * Interceptors can also mutate the response returned by the handler chain. An example use case might be adding cross-origin resource sharing headers:
 * <p>
 * <blockquote><pre>
 * def add_cors_headers_interceptor(input: ChainedApiRequest) -&gt; ApiResponse:
 *     response = input.chain.next(input)
 *     return ApiResponse(
 *         status_code=response.status_code,
 *         body=response.body,
 *         headers={
 *             **response.headers,
 *             "Access-Control-Allow-Origin": "*",
 *             "Access-Control-Allow-Headers": "*"
 *         }
 *     )
 * </pre></blockquote>
 * <p>
 * <h4>Java</h4>
 * <p>
 * In Java, interceptors can be added to a handler via the <code>&#64;Interceptors</code> class annotation:
 * <p>
 * <blockquote><pre>
 * import com.generated.api.myjavaapi.client.api.Handlers.Interceptors;
 * 
 * &#64;Interceptors({ TimingInterceptor.class, TryCatchInterceptor.class })
 * public class SayHelloHandler extends SayHello {
 *     &#64;Override
 *     public SayHelloResponse handle(SayHelloRequestInput sayHelloRequestInput) {
 *         return SayHello200Response.of(HelloResponse.builder()
 *                 .message(String.format("Hello %s", sayHelloRequestInput.getInput().getRequestParameters().getName()))
 *                 .build());
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * To write an interceptor, you can implement the <code>Interceptor</code> interface. For example, a timing interceptor:
 * <p>
 * <blockquote><pre>
 * import com.generated.api.myjavaapi.client.api.Handlers.Interceptor;
 * import com.generated.api.myjavaapi.client.api.Handlers.ChainedRequestInput;
 * import com.generated.api.myjavaapi.client.api.Handlers.Response;
 * 
 * public class TimingInterceptor&lt;Input&gt; implements Interceptor&lt;Input&gt; {
 *     &#64;Override
 *     public Response handle(ChainedRequestInput&lt;Input&gt; input) {
 *         long start = System.currentTimeMillis();
 *         Response res = input.getChain().next(input);
 *         long end = System.currentTimeMillis();
 *         System.out.printf("Took %d ms%n", end - start);
 *         return res;
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * Interceptors may choose to return different responses, for example to return a 500 response for any unhandled exceptions:
 * <p>
 * <blockquote><pre>
 * import com.generated.api.myjavaapi.client.api.Handlers.Interceptor;
 * import com.generated.api.myjavaapi.client.api.Handlers.ChainedRequestInput;
 * import com.generated.api.myjavaapi.client.api.Handlers.Response;
 * import com.generated.api.myjavaapi.client.api.Handlers.ApiResponse;
 * import com.generated.api.myjavaapi.client.model.ApiError;
 * 
 * public class TryCatchInterceptor&lt;Input&gt; implements Interceptor&lt;Input&gt; {
 *     &#64;Override
 *     public Response handle(ChainedRequestInput&lt;Input&gt; input) {
 *         try {
 *             return input.getChain().next(input);
 *         } catch (Exception e) {
 *             return ApiResponse.builder()
 *                     .statusCode(500)
 *                     .body(ApiError.builder()
 *                             .errorMessage(e.getMessage())
 *                             .build().toJson())
 *                     .build();
 *         }
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * Interceptors are permitted to mutate the "interceptor context", which is a <code>Map&lt;String, Object&gt;</code>. Each interceptor in the chain, and the final handler, can access this context:
 * <p>
 * <blockquote><pre>
 * public class IdentityInterceptor&lt;Input&gt; implements Interceptor&lt;Input&gt; {
 *     &#64;Override
 *     public Response handle(ChainedRequestInput&lt;Input&gt; input) {
 *         input.getInterceptorContext().put("AuthenticatedUser", this.getAuthenticatedUser(input.getEvent()));
 *         return input.getChain().next(input);
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * Interceptors can also mutate the response returned by the handler chain. An example use case might be adding cross-origin resource sharing headers:
 * <p>
 * <blockquote><pre>
 * public static class AddCorsHeadersInterceptor&lt;Input&gt; implements Interceptor&lt;Input&gt; {
 *     &#64;Override
 *     public Response handle(ChainedRequestInput&lt;Input&gt; input) {
 *         Response res = input.getChain().next(input);
 *         res.getHeaders().put("Access-Control-Allow-Origin", "*");
 *         res.getHeaders().put("Access-Control-Allow-Headers", "*");
 *         return res;
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * <h5>Interceptors with Dependency Injection</h5>
 * <p>
 * Interceptors referenced by the <code>&#64;Interceptors</code> annotation must be constructable with no arguments. If more complex instantiation of your interceptor is required (for example if you are using dependency injection or wish to pass configuration to your interceptor), you may instead override the <code>getInterceptors</code> method in your handler:
 * <p>
 * <blockquote><pre>
 * public class SayHelloHandler extends SayHello {
 *     &#64;Override
 *     public List&lt;Interceptor&lt;SayHelloInput&gt;&gt; getInterceptors() {
 *         return Arrays.asList(
 *                 new MyConfiguredInterceptor&lt;&gt;(42),
 *                 new MyOtherConfiguredInterceptor&lt;&gt;("configuration"));
 *     }
 * 
 *     &#64;Override
 *     public SayHelloResponse handle(SayHelloRequestInput sayHelloRequestInput) {
 *         return SayHello200Response.of(HelloResponse.builder()
 *                 .message(String.format("Hello %s!", sayHelloRequestInput.getInput().getRequestParameters().getName()))
 *                 .build());
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * <h3>Other Details</h3>
 * <p>
 * <h4>Workspaces and <code>OpenApiGatewayTsProject</code></h4>
 * <p>
 * <code>OpenApiGatewayTsProject</code> can be used as part of a monorepo using YARN/NPM/PNPM workspaces. When used in a monorepo, a dependency is established between <code>OpenApiGatewayTsProject</code> and the generated typescript client, which is expected to be managed by the parent monorepo (ie both <code>OpenApiGatewayTsProject</code> and the generated typescript client are parented by the monorepo).
 * <p>
 * During initial project synthesis, the dependency between <code>OpenApiGatewayTsProject</code> and the generated client is established via workspace configuration local to <code>OpenApiGatewayTsProject</code>, since the parent monorepo will not have updated to include the new packages in time for the initial "install".
 * <p>
 * When the package manager is PNPM, this initial workspace is configured by creating a local <code>pnpm-workspace.yaml</code> file, and thus if you specify your own for an instance of <code>OpenApiGatewayTsProject</code>, synthesis will fail. It is most likely that you will not need to define this file locally in <code>OpenApiGatewayTsProject</code> since the monorepo copy should be used to manage all packages within the repo, however if managing this file at the <code>OpenApiGatewayTsProject</code> level is required, please use the <code>pnpmWorkspace</code> property of <code>OpenApiGatewayTsProject</code>.
 * <p>
 * <h4>Customising Generated Client Projects</h4>
 * <p>
 * By default, the generated clients are configured automatically, including project names. You can customise the generated client code using the <code>&lt;language&gt;ProjectOptions</code> properties when constructing your projen project.
 * <p>
 * <h5>Python Shared Virtual Environment</h5>
 * <p>
 * For adding dependencies between python projects within a monorepo you can use a single shared virtual environment, and install your python projects into that environment with <code>pip install --editable .</code> in the dependee. The generated python client will automatically do this if it detects it is within a monorepo.
 * <p>
 * The following example shows how to configure the generated client to use a shared virtual environment:
 * <p>
 * <blockquote><pre>
 * const api = new OpenApiGatewayTsProject({
 *   parent: monorepo,
 *   name: 'api',
 *   outdir: 'packages/api',
 *   defaultReleaseBranch: 'main',
 *   clientLanguages: [ClientLanguage.PYTHON],
 *   pythonClientOptions: {
 *     moduleName: 'my_api_python',
 *     name: 'my_api_python',
 *     authorName: 'jack',
 *     authorEmail: 'me&#64;example.com',
 *     version: '1.0.0',
 *     venvOptions: {
 *       // Directory relative to the generated python client (in this case packages/api/generated/python)
 *       envdir: '../../../../.env',
 *     },
 *   },
 * });
 * 
 * new PythonProject({
 *   parent: monorepo,
 *   outdir: 'packages/my-python-lib',
 *   moduleName: 'my_python_lib',
 *   name: 'my_python_lib',
 *   authorName: 'jack',
 *   authorEmail: 'me&#64;example.com',
 *   version: '1.0.0',
 *   venvOptions: {
 *     // Directory relative to the python lib (in this case packages/my-python-lib)
 *     envdir: '../../.env',
 *   },
 *   // Generated client can be safely cast to a PythonProject
 *   deps: [(api.generatedClients[ClientLanguage.PYTHON] as PythonProject).moduleName],
 * });
 * </pre></blockquote>
 * <p>
 * <h4>Java API Lambda Handlers</h4>
 * <p>
 * To build your lambda handlers in Java, it's recommended to create a separate <code>JavaProject</code> in your <code>.projenrc</code>. This needs to build a "super jar" with all of your dependencies packed into a single jar. You can use the <code>maven-shade-plugin</code> to achieve this (see <a href="https://docs.aws.amazon.com/lambda/latest/dg/java-package.html">the java lambda docs for details</a>). You'll need to add a dependency on the generated java client for the handler wrappers. For example, your <code>.projenrc.ts</code> might look like:
 * <p>
 * <blockquote><pre>
 * const api = new OpenApiGatewayTsProject({
 *   parent: monorepo,
 *   name: '&#64;my-test/api',
 *   outdir: 'packages/api',
 *   defaultReleaseBranch: 'main',
 *   clientLanguages: [ClientLanguage.JAVA],
 * });
 * 
 * const apiJavaClient = (api.generatedClients[ClientLanguage.JAVA] as JavaProject);
 * 
 * const javaLambdaProject = new JavaProject({
 *   parent: monorepo,
 *   outdir: 'packages/java-lambdas',
 *   artifactId: "my-java-lambdas",
 *   groupId: "com.mycompany",
 *   name: "javalambdas",
 *   version: "1.0.0",
 *   // Add a dependency on the java client
 *   deps: [`${apiJavaClient.pom.groupId}/${apiJavaClient.pom.artifactId}&#64;${apiJavaClient.pom.version}`],
 * });
 * 
 * // Set up the dependency on the generated lambda client
 * monorepo.addImplicitDependency(javaLambdaProject, apiJavaClient);
 * javaLambdaProject.pom.addRepository({
 *   url: `file://../api/generated/java/dist/java`,
 *   id: 'java-api-client',
 * });
 * 
 * // Use the maven-shade-plugin as part of the maven package task
 * javaLambdaProject.pom.addPlugin('org.apache.maven.plugins/maven-shade-plugin&#64;3.2.2', {
 *   configuration: {
 *     createDependencyReducedPom: false,
 *     finalName: 'my-java-lambdas',
 *   },
 *   executions: [{
 *     id: 'shade-task',
 *     phase: 'package', // &lt;- NB "phase" is supported in projen ^0.61.37
 *     goals: ['shade'],
 *   }],
 * });
 * 
 * // Build the "super jar" as part of the project's package task
 * javaLambdaProject.packageTask.exec('mvn clean install');
 * </pre></blockquote>
 * <p>
 * You can then implement your lambda handlers in your <code>java-lambdas</code> project using the generated lambda handler wrappers (see above).
 * <p>
 * Finally, you can create a lambda function in your CDK infrastructure which points to the resultant "super jar":
 * <p>
 * <blockquote><pre>
 * new Api(this, 'JavaApi', {
 *   integrations: {
 *     sayHello: {
 *       function: new Function(this, 'SayHelloJava', {
 *         code: Code.fromAsset('../java-lambdas/target/my-java-lambdas.jar'),
 *         handler: 'com.mycompany.SayHelloHandler',
 *         runtime: Runtime.JAVA_11,
 *         timeout: Duration.seconds(30),
 *       }),
 *     },
 *   },
 * });
 * </pre></blockquote>
 * <p>
 * Note that to ensure the jar is built before the CDK infrastructure which consumes it, you must add a dependency, eg:
 * <p>
 * <blockquote><pre>
 * monorepo.addImplicitDependency(infra, javaLambdaProject);
 * </pre></blockquote>
 * <p>
 * <h4>AWS WAFv2 Web ACL</h4>
 * <p>
 * By default, a <a href="https://docs.aws.amazon.com/waf/latest/developerguide/web-acl.html">Web ACL</a> is deployed and attached to your API Gateway Rest API with the "<a href="https://docs.aws.amazon.com/waf/latest/developerguide/aws-managed-rule-groups-baseline.html">AWSManagedRulesCommonRuleSet</a>", which provides protection against exploitation of a wide range of vulnerabilities, including some of the high risk and commonly occurring vulnerabilities described in OWASP publications such as <a href="https://owasp.org/www-project-top-ten/">OWASP Top 10</a>.
 * <p>
 * You can customise the Web ACL configuration via the <code>webAclOptions</code> of your <code>Api</code> CDK construct, eg:
 * <p>
 * <blockquote><pre>
 * export class SampleApi extends Api {
 *   constructor(scope: Construct, id: string) {
 *     super(scope, id, {
 *       integrations: { ... },
 *       webAclOptions: {
 *         // Allow access only to specific CIDR ranges
 *         cidrAllowList: {
 *           cidrType: 'IPV4',
 *           cidrRanges: ['1.2.3.4/5'],
 *         },
 *         // Pick from the set here: https://docs.aws.amazon.com/waf/latest/developerguide/aws-managed-rule-groups-list.html
 *         managedRules: [
 *           { vendor: 'AWS', name: 'AWSManagedRulesSQLiRuleSet' },
 *         ],
 *       },
 *     });
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * You can remove the Web ACL entirely with <code>webAclOptions: { disable: true }</code> - you may wish to use this if you'd like to set up a Web ACL yourself with more control over the rules.
 * <p>
 * <h4>Smithy IntelliJ Plugin</h4>
 * <p>
 * The Smithy-based projects are compatible with the <a href="https://github.com/iancaffey/smithy-intellij-plugin">Smithy IntelliJ Plugin</a>, which provides syntax highlighting and auto-complete for your Smithy model. To make use of it, perform the following steps:
 * <p>
 * <ul>
 * <li>Install the "Smithy" plugin (under <code>Preferences -&gt; Plugins</code>)</li>
 * <li>Right-click on the <code>smithy/build.gradle</code> file in your Smithy API project</li>
 * <li>Select "Link Gradle Project"</li>
 * </ul>
 * <p>
 * <h3>Breaking Changes</h3>
 * <p>
 * <ul>
 * <li><code>v0.14.0</code> - see https://github.com/aws/aws-prototyping-sdk/pull/280
 * <p>
 * <ul>
 * <li>Moved smithy model files from <code>model</code> directory to <code>smithy/src/main/smithy</code> - please move these manually as part of upgrading to <code>0.14.0</code>, and delete your <code>model</code> directory when done.</li>
 * <li>Moved smithy gradle files from <code>smithy-build</code> directory to <code>smithy</code> - if you have added any dependencies to your <code>smithy-build/build.gradle</code> file you will need to copy them across into <code>smithy/build.gradle</code> (note dependencies in the new gradle file start with <code>implementation</code> rather than <code>smithy</code>).</li>
 * <li>Deprecated <code>gradleWrapperPath</code> option on SmithApiGateway projects in favour of <code>ignoreGradleWrapper: false</code> - the gradle wrapper in <code>smithy</code> directory is always used (and generated automatically if not found). If you used a custom gradle wrapper, copy it into the <code>smithy</code> directory, set <code>ignoreGradleWrapper: false</code> and check it in to your repository.</li>
 * </ul></li>
 * </ul>
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Experimental)
package software.aws.awsprototypingsdk.openapigateway;
