/**
 * <h2>OpenAPI Gateway</h2>
 * <p>
 * Define your APIs using <a href="https://swagger.io/specification/">OpenAPI v3</a>, and leverage the power of generated clients and documentation, automatic input validation, and type safe client and server code!
 * <p>
 * This package vends a projen project type which allows you to define an API using <a href="https://swagger.io/specification/">OpenAPI v3</a>, and a construct which manages deploying this API in API Gateway, given a lambda integration for every operation.
 * <p>
 * The project will generate models and clients from your OpenAPI spec in your desired languages, and can be utilised both client side or server side in lambda handlers. The project type also generates a wrapper construct which adds type safety to ensure a lambda integration is provided for every API integration.
 * <p>
 * When you change your API specification, just run <code>npx projen</code> again to regenerate all of this!
 * <p>
 * <h3>Project</h3>
 * <p>
 * <h4>Typescript</h4>
 * <p>
 * It's recommended that this project is used as part of an <code>nx_monorepo</code> project. You can still use this as a standalone project if you like (eg <code>npx projen new --from &#64;aws-prototyping-sdk/open-api-gateway open-api-gateway-ts</code>), however you will need to manage build order (ie building the generated client first, followed by the project).
 * <p>
 * For usage in a monorepo:
 * <p>
 * Create the project in your .projenrc:
 * <p>
 * <blockquote><pre>
 * import { ClientLanguage, DocumentationFormat, OpenApiGatewayTsProject } from "&#64;aws-prototyping-sdk/open-api-gateway";
 * 
 * new OpenApiGatewayTsProject({
 *   parent: myNxMonorepo,
 *   defaultReleaseBranch: "mainline",
 *   name: "my-api",
 *   outdir: "packages/api",
 *   clientLanguages: [ClientLanguage.TYPESCRIPT, ClientLanguage.PYTHON, ClientLanguage.JAVA],
 *   documentationFormats: [DocumentationFormat.HTML2, DocumentationFormat.PLANTUML, DocumentationFormat.MARKDOWN],
 * });
 * </pre></blockquote>
 * <p>
 * In the output directory (<code>outdir</code>), you'll find a few files to get you started.
 * <p>
 * <blockquote><pre>
 * |_ src/
 *     |_ spec/
 *         |_ spec.yaml - The OpenAPI specification - edit this to define your API
 *         |_ .parsed-spec.json - A json spec generated from your spec.yaml.
 *     |_ api/
 *         |_ api.ts - A CDK construct which defines the API Gateway resources to deploy your API.
 *         |           This wraps the OpenApiGatewayLambdaApi construct and provides typed interfaces for integrations specific
 *         |           to your API. You shouldn't need to modify this, instead just extend it as in sample-api.ts.
 *         |_ sample-api.ts - Example usage of the construct defined in api.ts.
 *         |_ sample-api.say-hello.ts - An example lambda handler for the operation defined in spec.yaml, making use of the
 *                                      generated lambda handler wrappers for marshalling and type safety.
 * |_ generated/
 *     |_ typescript/ - A generated typescript API client, including generated lambda handler wrappers
 *     |_ python/ - A generated python API client.
 *     |_ java/ - A generated java API client.
 *     |_ documentation/
 *         |_ html2/ - Generated html documentation
 *         |_ markdown/ - Generated markdown documentation
 *         |_ plantuml/ - Generated plant uml documentation
 * </pre></blockquote>
 * <p>
 * If you would prefer to not generate the sample code, you can pass <code>sampleCode: false</code> to <code>OpenApiGatewayTsProject</code>.
 * <p>
 * To make changes to your api, simply update <code>spec.yaml</code> and run <code>npx projen</code> to synthesize all the typesafe client/server code!
 * <p>
 * The <code>SampleApi</code> construct uses <code>NodejsFunction</code> to declare the example lambda, but you are free to change this!
 * <p>
 * <h4>Python</h4>
 * <p>
 * As well as typescript, you can choose to generate the cdk construct and sample handler in python.
 * <p>
 * <blockquote><pre>
 * new OpenApiGatewayPythonProject({
 *   parent: myNxMonorepo,
 *   outdir: 'packages/myapi',
 *   name: 'myapi',
 *   moduleName: 'myapi',
 *   version: '1.0.0',
 *   authorName: 'jack',
 *   authorEmail: 'me&#64;example.com',
 *   clientLanguages: [ClientLanguage.TYPESCRIPT, ClientLanguage.PYTHON, ClientLanguage.JAVA],
 * });
 * </pre></blockquote>
 * <p>
 * You will need to set up a shared virtual environment and configure dependencies via the monorepo (see README.md for the nx-monorepo package). An example of a full <code>.projenrc.ts</code> might be:
 * <p>
 * <blockquote><pre>
 * import { nx_monorepo } from "aws-prototyping-sdk";
 * import { ClientLanguage, OpenApiGatewayPythonProject } from "&#64;aws-prototyping-sdk/open-api-gateway";
 * import { AwsCdkPythonApp } from "projen/lib/awscdk";
 * 
 * const monorepo = new nx_monorepo.NxMonorepoProject({
 *   defaultReleaseBranch: "main",
 *   devDeps: ["aws-prototyping-sdk", "&#64;aws-prototyping-sdk/open-api-gateway"],
 *   name: "open-api-test",
 * });
 * 
 * const api = new OpenApiGatewayPythonProject({
 *   parent: monorepo,
 *   outdir: 'packages/myapi',
 *   name: 'myapi',
 *   moduleName: 'myapi',
 *   version: '1.0.0',
 *   authorName: 'jack',
 *   authorEmail: 'me&#64;example.com',
 *   clientLanguages: [ClientLanguage.TYPESCRIPT],
 *   venvOptions: {
 *     // Use a shared virtual env dir.
 *     // The generated python client will also use this virtual env dir
 *     envdir: '../../.env',
 *   },
 * });
 * 
 * // Install into virtual env so it's available for the cdk app
 * api.tasks.tryFind('install')!.exec('pip install --editable .');
 * 
 * const app = new AwsCdkPythonApp({
 *   authorName: "jack",
 *   authorEmail: "me&#64;example.com",
 *   cdkVersion: "2.1.0",
 *   moduleName: "myapp",
 *   name: "myapp",
 *   version: "1.0.0",
 *   parent: monorepo,
 *   outdir: "packages/myapp",
 *   deps: [api.moduleName],
 *   venvOptions: {
 *     envdir: '../../.env',
 *   },
 * });
 * 
 * monorepo.addImplicitDependency(app, api);
 * 
 * monorepo.synth();
 * </pre></blockquote>
 * <p>
 * You'll find the following directory structure in <code>packages/myapi</code>:
 * <p>
 * <blockquote><pre>
 * |_ myapi/
 *     |_ spec/
 *         |_ spec.yaml - The OpenAPI specification - edit this to define your API
 *         |_ .parsed-spec.json - A json spec generated from your spec.yaml.
 *     |_ api/
 *         |_ api.py - A CDK construct which defines the API Gateway resources to deploy your API.
 *         |           This wraps the OpenApiGatewayLambdaApi construct and provides typed interfaces for integrations specific
 *         |           to your API. You shouldn't need to modify this, instead just extend it as in sample_api.py.
 *         |_ sample_api.py - Example usage of the construct defined in api.py.
 *         |_ handlers/
 *              |_ say_hello_handler_sample.py - An example lambda handler for the operation defined in spec.yaml, making use of the
 *                                               generated lambda handler wrappers for marshalling and type safety.
 * |_ generated/
 *     |_ typescript/ - A generated typescript API client.
 *     |_ python/ - A generated python API client, including generated lambda handler wrappers.
 *     |_ java/ - A generated java API client.
 * </pre></blockquote>
 * <p>
 * For simplicity, the generated code deploys a lambda layer for the generated code and its dependencies. You may choose to define an entirely separate projen <code>PythonProject</code> for your lambda handlers should you wish to add more dependencies than just the generated code.
 * <p>
 * <h3>OpenAPI Specification</h3>
 * <p>
 * Your <code>spec.yaml</code> file defines your api using <a href="https://swagger.io/specification/">OpenAPI Version 3.0.3</a>. An example spec might look like:
 * <p>
 * <blockquote><pre>
 * openapi: 3.0.3
 * info:
 *   version: 1.0.0
 *   title: Example API
 * paths:
 *   /hello:
 *     get:
 *       operationId: sayHello
 *       parameters:
 *         - in: query
 *           name: name
 *           schema:
 *             type: string
 *           required: true
 *       responses:
 *         '200':
 *           description: Successful response
 *           content:
 *             'application/json':
 *               schema:
 *                 $ref: '#/components/schemas/HelloResponse'
 * components:
 *   schemas:
 *     HelloResponse:
 *       type: object
 *       properties:
 *         message:
 *           type: string
 *       required:
 *         - message
 * </pre></blockquote>
 * <p>
 * You can divide your specification into multiple files using <code>$ref</code>.
 * <p>
 * For example, you might choose to structure your spec as follows:
 * <p>
 * <blockquote><pre>
 * |_ spec/
 *     |_ spec.yaml
 *     |_ paths/
 *         |_ index.yaml
 *         |_ sayHello.yaml
 *     |_ schemas/
 *         |_ index.yaml
 *         |_ helloResponse.yaml
 * </pre></blockquote>
 * <p>
 * Where <code>spec.yaml</code> looks as follows:
 * <p>
 * <blockquote><pre>
 * openapi: 3.0.3
 * info:
 *   version: 1.0.0
 *   title: Example API
 * paths:
 *   $ref: './paths/index.yaml'
 * components:
 *   schemas:
 *     $ref: './schemas/index.yaml'
 * </pre></blockquote>
 * <p>
 * <code>paths/index.yaml</code>:
 * <p>
 * <blockquote><pre>
 * /hello:
 *   get:
 *     $ref: './sayHello.yaml'
 * </pre></blockquote>
 * <p>
 * <code>paths/sayHello.yaml</code>:
 * <p>
 * <blockquote><pre>
 * operationId: sayHello
 * parameters:
 *  - in: query
 *    name: name
 *    schema:
 *      type: string
 *    required: true
 * responses:
 *   '200':
 *     description: Successful response
 *     content:
 *       'application/json':
 *         schema:
 *           $ref: '../schemas/helloResponse.yaml'
 * </pre></blockquote>
 * <p>
 * <code>schemas/index.yaml</code>:
 * <p>
 * <blockquote><pre>
 * HelloResponse:
 *   $ref: './helloResponse.yaml'
 * </pre></blockquote>
 * <p>
 * <code>schemas/helloResponse.yaml</code>:
 * <p>
 * <blockquote><pre>
 * type: object
 * properties:
 *   message:
 *     type: string
 * required:
 *   - message
 * </pre></blockquote>
 * <p>
 * <h3>Construct</h3>
 * <p>
 * A sample construct is generated which provides a type-safe interface for creating an API Gateway API based on your OpenAPI specification. You'll get a type error if you forget to define an integration for an operation defined in your api.
 * <p>
 * <blockquote><pre>
 * import * as path from 'path';
 * import { Authorizers } from '&#64;aws-prototyping-sdk/open-api-gateway';
 * import { NodejsFunction } from 'aws-cdk-lib/aws-lambda-nodejs';
 * import { Construct } from 'constructs';
 * import { Api } from './api';
 * 
 * /**
 *  * An example of how to wire lambda handler functions to the API
 *  *&#47;
 * export class SampleApi extends Api {
 *   constructor(scope: Construct, id: string) {
 *     super(scope, id, {
 *       defaultAuthorizer: Authorizers.iam(),
 *       integrations: {
 *         // Every operation defined in your API must have an integration defined!
 *         sayHello: {
 *           function: new NodejsFunction(scope, 'say-hello'),
 *         },
 *       },
 *     });
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * <h4>Authorizers</h4>
 * <p>
 * The <code>Api</code> construct allows you to define one or more authorizers for securing your API. An integration will use the <code>defaultAuthorizer</code> unless an <code>authorizer</code> is specified at the integration level. The following authorizers are supported:
 * <p>
 * <ul>
 * <li><code>Authorizers.none</code> - No auth</li>
 * <li><code>Authorizers.iam</code> - AWS IAM (Signature Version 4)</li>
 * <li><code>Authorizers.cognito</code> - Cognito user pool</li>
 * <li><code>Authorizers.custom</code> - A custom authorizer</li>
 * </ul>
 * <p>
 * <h5>Cognito Authorizer</h5>
 * <p>
 * To use the Cognito authorizer, one or more user pools must be provided. You can optionally specify the scopes to check if using an access token. You can use the <code>withScopes</code> method to use the same authorizer but verify different scopes for individual integrations, for example:
 * <p>
 * <blockquote><pre>
 * export class SampleApi extends Api {
 *   constructor(scope: Construct, id: string) {
 *     const cognitoAuthorizer = Authorizers.cognito({
 *       authorizerId: 'myCognitoAuthorizer',
 *       userPools: [new UserPool(scope, 'UserPool')],
 *     });
 * 
 *     super(scope, id, {
 *       defaultAuthorizer: cognitoAuthorizer,
 *       integrations: {
 *         // Everyone in the user pool can call this operation:
 *         sayHello: {
 *           function: new NodejsFunction(scope, 'say-hello'),
 *         },
 *         // Only users with the given scopes can call this operation
 *         myRestrictedOperation: {
 *           function: new NodejsFunction(scope, 'my-restricted-operation'),
 *           authorizer: cognitoAuthorizer.withScopes('my-resource-server/my-scope'),
 *         },
 *       },
 *     });
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * For more information about scopes or identity and access tokens, please see the <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-integrate-with-cognito.html">API Gateway documentation</a>.
 * <p>
 * <h5>Custom Authorizer</h5>
 * <p>
 * Custom authorizers use lambda functions to handle authorizing requests. These can either be simple token-based authorizers, or more complex request-based authorizers. See the <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html">API Gateway documentation</a> for more details.
 * <p>
 * An example token-based authorizer (default):
 * <p>
 * <blockquote><pre>
 * Authorizers.custom({
 *   authorizerId: 'myTokenAuthorizer',
 *   function: new NodejsFunction(scope, 'authorizer'),
 * });
 * </pre></blockquote>
 * <p>
 * An example request-based handler. By default the identitySource will be <code>method.request.header.Authorization</code>, however you can customise this as per <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-identitysource">the API Gateway documentation</a>.
 * <p>
 * <blockquote><pre>
 * Authorizers.custom({
 *   authorizerId: 'myRequestAuthorizer',
 *   type: CustomAuthorizerType.REQUEST,
 *   identitySource: 'method.request.header.MyCustomHeader, method.request.querystring.myQueryString',
 *   function: new NodejsFunction(scope, 'authorizer'),
 * });
 * </pre></blockquote>
 * <p>
 * <h3>Generated Client</h3>
 * <p>
 * <h4>Typescript</h4>
 * <p>
 * The <a href="https://openapi-generator.tech/docs/generators/typescript-fetch/">typescript-fetch</a> OpenAPI generator is used to generate OpenAPI clients for typescript. This requires an implementation of <code>fetch</code> to be passed to the client. In the browser one can pass the built in fetch, or in NodeJS you can use an implementation such as <a href="https://www.npmjs.com/package/node-fetch">node-fetch</a>.
 * <p>
 * Example usage of the client in a website:
 * <p>
 * <blockquote><pre>
 * import { Configuration, DefaultApi } from "my-api-typescript-client";
 * 
 * const client = new DefaultApi(new Configuration({
 *   basePath: "https://xxxxxxxxxx.execute-api.ap-southeast-2.amazonaws.com",
 *   fetchApi: window.fetch.bind(window),
 * }));
 * 
 * await client.sayHello({ name: "Jack" });
 * </pre></blockquote>
 * <p>
 * <h4>Python</h4>
 * <p>
 * The <a href="https://openapi-generator.tech/docs/generators/python-experimental">python-experimental</a> OpenAPI generator is used to generate OpenAPI clients for python.
 * <p>
 * Example usage:
 * <p>
 * <blockquote><pre>
 * from my_api_python import ApiClient, Configuration
 * from my_api_python.api.default_api import DefaultApi
 * 
 * configuration = Configuration(
 *     host = "https://xxxxxxxxxx.execute-api.ap-southeast-2.amazonaws.com"
 * )
 * 
 * with ApiClient(configuration) as api_client:
 *     client = DefaultApi(api_client)
 * 
 *     client.say_hello(
 *         query_params={
 *             'name': "name_example",
 *         },
 *     )
 * </pre></blockquote>
 * <p>
 * You'll find details about how to use the python client in the README.md alongside your generated client.
 * <p>
 * <h4>Java</h4>
 * <p>
 * The <a href="https://openapi-generator.tech/docs/generators/java/">java</a> OpenAPI generator is used to generate OpenAPI clients for Java.
 * <p>
 * Example usage:
 * <p>
 * <blockquote><pre>
 * import com.generated.api.myapijava.client.api.DefaultApi;
 * import com.generated.api.myapijava.client.ApiClient;
 * import com.generated.api.myapijava.client.Configuration;
 * import com.generated.api.myapijava.client.models.HelloResponse;
 * 
 * ApiClient client = Configuration.getDefaultApiClient();
 * client.setBasePath("https://xxxxxxxxxx.execute-api.ap-southeast-2.amazonaws.com");
 * 
 * DefaultApi api = new DefaultApi(client);
 * HelloResponse response = api.sayHello("Adrian").execute()
 * </pre></blockquote>
 * <p>
 * You'll find more details about how to use the Java client in the README.md alongside your generated client.
 * <p>
 * <h3>Lambda Handler Wrappers</h3>
 * <p>
 * Lambda handler wrappers are also importable from the generated client. These provide input/output type safety, ensuring that your API handlers return outputs that correspond to your specification.
 * <p>
 * <h4>Typescript</h4>
 * <p>
 * <blockquote><pre>
 * import { sayHelloHandler } from "my-api-typescript-client";
 * 
 * export const handler = sayHelloHandler(async (input) =&gt; {
 *   return {
 *     statusCode: 200,
 *     body: {
 *       message: `Hello ${input.requestParameters.name}!`,
 *     },
 *   };
 * });
 * </pre></blockquote>
 * <p>
 * <h4>Python</h4>
 * <p>
 * <blockquote><pre>
 * from myapi_python.api.default_api_operation_config import say_hello_handler, SayHelloRequest, ApiResponse
 * from myapi_python.model.api_error import ApiError
 * from myapi_python.model.hello_response import HelloResponse
 * 
 * &#64;say_hello_handler
 * def handler(input: SayHelloRequest, **kwargs) -&gt; ApiResponse[HelloResponse, ApiError]:
 *     return ApiResponse(
 *         status_code=200,
 *         body=HelloResponse(message="Hello {}!".format(input.request_parameters["name"])),
 *         headers={}
 *     )
 * </pre></blockquote>
 * <p>
 * <h4>Java</h4>
 * <p>
 * <blockquote><pre>
 * import com.generated.api.myapijava.client.api.Handlers.SayHello;
 * import com.generated.api.myapijava.client.api.Handlers.SayHello200Response;
 * import com.generated.api.myapijava.client.api.Handlers.SayHelloRequestInput;
 * import com.generated.api.myapijava.client.api.Handlers.SayHelloResponse;
 * import com.generated.api.myapijava.client.model.HelloResponse;
 * 
 * 
 * public class SayHelloHandler extends SayHello {
 *     &#64;Override
 *     public SayHelloResponse handle(SayHelloRequestInput sayHelloRequestInput) {
 *         return SayHello200Response.of(HelloResponse.builder()
 *                 .message(String.format("Hello %s", sayHelloRequestInput.getInput().getRequestParameters().getName()))
 *                 .build());
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * <h3>Interceptors</h3>
 * <p>
 * The lambda handler wrappers allow you to pass in a <em>chain</em> of handler functions to handle the request. This allows you to implement middleware / interceptors for handling requests. Each handler function may choose whether or not to continue the handler chain by invoking <code>chain.next</code>.
 * <p>
 * <h4>Typescript</h4>
 * <p>
 * In typescript, interceptors are passed as separate arguments to the generated handler wrapper, in the order in which they should be executed. Call <code>chain.next(input, event, context)</code> from an interceptor to delegate to the rest of the chain to handle a request. Note that the last handler in the chain (ie the actual request handler which transforms the input to the output) should not call <code>chain.next</code>.
 * <p>
 * <blockquote><pre>
 * import {
 *   sayHelloHandler,
 *   LambdaRequestParameters,
 *   LambdaHandlerChain,
 * } from "my-api-typescript-client";
 * 
 * // Interceptor to wrap invocations in a try/catch, returning a 500 error for any unhandled exceptions.
 * const tryCatchInterceptor = async &lt;
 *   RequestParameters,
 *   RequestArrayParameters,
 *   RequestBody,
 *   Response
 * &gt;(
 *   input: LambdaRequestParameters&lt;RequestParameters, RequestArrayParameters, RequestBody&gt;,
 *   event: any,
 *   context: any,
 *   chain: LambdaHandlerChain&lt;RequestParameters, RequestArrayParameters, RequestBody, Response&gt;,
 * ): Promise&lt;Response | OperationResponse&lt;500, { errorMessage: string }&gt;&gt; =&gt; {
 *   try {
 *     return await chain.next(input, event, context);
 *   } catch (e: any) {
 *     return { statusCode: 500, body: { errorMessage: e.message }};
 *   }
 * };
 * 
 * // tryCatchInterceptor is passed first, so it runs first and calls the second argument function (the request handler) via chain.next
 * export const handler = sayHelloHandler(tryCatchInterceptor, async (input) =&gt; {
 *   return {
 *     statusCode: 200,
 *     body: {
 *       message: `Hello ${input.requestParameters.name}!`,
 *     },
 *   };
 * });
 * </pre></blockquote>
 * <p>
 * Another example interceptor might be to record request time metrics. The example below includes the full generic type signature for an interceptor:
 * <p>
 * <blockquote><pre>
 * import {
 *   LambdaRequestParameters,
 *   LambdaHandlerChain,
 * } from 'my-api-typescript-client';
 * 
 * const timingInterceptor = async &lt;
 *   RequestParameters,
 *   RequestArrayParameters,
 *   RequestBody,
 *   Response
 * &gt;(
 *   input: LambdaRequestParameters&lt;RequestParameters, RequestArrayParameters, RequestBody&gt;,
 *   event: any,
 *   context: any,
 *   chain: LambdaHandlerChain&lt;RequestParameters, RequestArrayParameters, RequestBody, Response&gt;,
 * ): Promise&lt;Response&gt; =&gt; {
 *   const start = Date.now();
 *   const response = await chain.next(input, event, context);
 *   const end = Date.now();
 *   console.log(`Took ${end - start}ms`);
 *   return response;
 * };
 * </pre></blockquote>
 * <p>
 * Interceptors may add extra properties to the <code>context</code> to pass state to further interceptors or the final lambda handler, for example an <code>identityInterceptor</code> might want to extract the authenticated user from the request so that it is available in handlers.
 * <p>
 * <blockquote><pre>
 * import {
 *   LambdaRequestParameters,
 *   LambdaHandlerChain,
 * } from 'my-api-typescript-client';
 * 
 * const identityInterceptor = async &lt;
 *   RequestParameters,
 *   RequestArrayParameters,
 *   RequestBody,
 *   Response
 * &gt;(
 *   input: LambdaRequestParameters&lt;RequestParameters, RequestArrayParameters, RequestBody&gt;,
 *   event: any,
 *   context: any,
 *   chain: LambdaHandlerChain&lt;RequestParameters, RequestArrayParameters, RequestBody, Response&gt;,
 * ): Promise&lt;Response&gt; =&gt; {
 *   const authenticatedUser = await getAuthenticatedUser(event);
 *   return await chain.next(input, event, {
 *     ...context,
 *     authenticatedUser,
 *   });
 * };
 * </pre></blockquote>
 * <p>
 * <h4>Java</h4>
 * <p>
 * In Java, interceptors can be added to a handler via the <code>&#64;Interceptors</code> class annotation:
 * <p>
 * <blockquote><pre>
 * import com.generated.api.myjavaapi.client.api.Handlers.Interceptors;
 * 
 * &#64;Interceptors({ TimingInterceptor.class, TryCatchInterceptor.class })
 * public class SayHelloHandler extends SayHello {
 *     &#64;Override
 *     public SayHelloResponse handle(SayHelloRequestInput sayHelloRequestInput) {
 *         return SayHello200Response.of(HelloResponse.builder()
 *                 .message(String.format("Hello %s", sayHelloRequestInput.getInput().getRequestParameters().getName()))
 *                 .build());
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * To write an interceptor, you can implement the <code>Interceptor</code> interface. For example, a timing interceptor:
 * <p>
 * <blockquote><pre>
 * import com.generated.api.myjavaapi.client.api.Handlers.Interceptor;
 * import com.generated.api.myjavaapi.client.api.Handlers.ChainedRequestInput;
 * import com.generated.api.myjavaapi.client.api.Handlers.Response;
 * 
 * public class TimingInterceptor&lt;Input&gt; implements Interceptor&lt;Input&gt; {
 *     &#64;Override
 *     public Response handle(ChainedRequestInput&lt;Input&gt; input) {
 *         long start = System.currentTimeMillis();
 *         Response res = input.getChain().next(input);
 *         long end = System.currentTimeMillis();
 *         System.out.printf("Took %d ms%n", end - start);
 *         return res;
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * Interceptors may choose to return different responses, for example to catch errors:
 * <p>
 * <blockquote><pre>
 * import com.generated.api.myjavaapi.client.api.Handlers.Interceptor;
 * import com.generated.api.myjavaapi.client.api.Handlers.ChainedRequestInput;
 * import com.generated.api.myjavaapi.client.api.Handlers.Response;
 * import com.generated.api.myjavaapi.client.api.Handlers.ApiResponse;
 * import com.generated.api.myjavaapi.client.model.ApiError;
 * 
 * public class TryCatchInterceptor&lt;Input&gt; implements Interceptor&lt;Input&gt; {
 *     &#64;Override
 *     public Response handle(ChainedRequestInput&lt;Input&gt; input) {
 *         try {
 *             return input.getChain().next(input);
 *         } catch (Exception e) {
 *             return ApiResponse.builder()
 *                     .statusCode(500)
 *                     .body(ApiError.builder()
 *                             .errorMessage(e.getMessage())
 *                             .build().toJson())
 *                     .build();
 *         }
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * Interceptors are permitted to mutate the "interceptor context", which is a <code>Map&lt;String, Object&gt;</code>. Each interceptor in the chain, and the final handler, can access this context:
 * <p>
 * <blockquote><pre>
 * public class IdentityInterceptor&lt;Input&gt; implements Interceptor&lt;Input&gt; {
 *     &#64;Override
 *     public Response handle(ChainedRequestInput&lt;Input&gt; input) {
 *         input.getInterceptorContext().put("AuthenticatedUser", this.getAuthenticatedUser(input.getEvent()));
 *         return input.getChain().next(input);
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * Interceptors can also mutate the response returned by the handler chain. An example use case might be adding cross-origin resource sharing headers:
 * <p>
 * <blockquote><pre>
 * public static class AddCorsHeadersInterceptor&lt;Input&gt; implements Interceptor&lt;Input&gt; {
 *     &#64;Override
 *     public Response handle(ChainedRequestInput&lt;Input&gt; input) {
 *         Response res = input.getChain().next(input);
 *         res.getHeaders().put("Access-Control-Allow-Origin", "*");
 *         res.getHeaders().put("Access-Control-Allow-Headers", "*");
 *         return res;
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * <h5>Interceptors with Dependency Injection</h5>
 * <p>
 * Interceptors referenced by the <code>&#64;Interceptors</code> annotation must be constructable with no arguments. If more complex instantiation of your interceptor is required (for example if you are using dependency injection or wish to pass configuration to your interceptor), you may instead override the <code>getInterceptors</code> method in your handler:
 * <p>
 * <blockquote><pre>
 * public class SayHelloHandler extends SayHello {
 *     &#64;Override
 *     public List&lt;Interceptor&lt;SayHelloInput&gt;&gt; getInterceptors() {
 *         return Arrays.asList(
 *                 new MyConfiguredInterceptor&lt;&gt;(42),
 *                 new MyOtherConfiguredInterceptor&lt;&gt;("configuration"));
 *     }
 * 
 *     &#64;Override
 *     public SayHelloResponse handle(SayHelloRequestInput sayHelloRequestInput) {
 *         return SayHello200Response.of(HelloResponse.builder()
 *                 .message(String.format("Hello %s!", sayHelloRequestInput.getInput().getRequestParameters().getName()))
 *                 .build());
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * <h3>Other Details</h3>
 * <p>
 * <h4>Workspaces and <code>OpenApiGatewayTsProject</code></h4>
 * <p>
 * <code>OpenApiGatewayTsProject</code> can be used as part of a monorepo using YARN/NPM/PNPM workspaces. When used in a monorepo, a dependency is established between <code>OpenApiGatewayTsProject</code> and the generated typescript client, which is expected to be managed by the parent monorepo (ie both <code>OpenApiGatewayTsProject</code> and the generated typescript client are parented by the monorepo).
 * <p>
 * During initial project synthesis, the dependency between <code>OpenApiGatewayTsProject</code> and the generated client is established via workspace configuration local to <code>OpenApiGatewayTsProject</code>, since the parent monorepo will not have updated to include the new packages in time for the initial "install".
 * <p>
 * When the package manager is PNPM, this initial workspace is configured by creating a local <code>pnpm-workspace.yaml</code> file, and thus if you specify your own for an instance of <code>OpenApiGatewayTsProject</code>, synthesis will fail. It is most likely that you will not need to define this file locally in <code>OpenApiGatewayTsProject</code> since the monorepo copy should be used to manage all packages within the repo, however if managing this file at the <code>OpenApiGatewayTsProject</code> level is required, please use the <code>pnpmWorkspace</code> property of <code>OpenApiGatewayTsProject</code>.
 * <p>
 * <h4>Customising Generated Client Projects</h4>
 * <p>
 * By default, the generated clients are configured automatically, including project names. You can customise the generated client code using the <code>&lt;language&gt;ProjectOptions</code> properties when constructing your projen project.
 * <p>
 * <h5>Python Shared Virtual Environment</h5>
 * <p>
 * For adding dependencies between python projects within a monorepo you can use a single shared virtual environment, and install your python projects into that environment with <code>pip install --editable .</code> in the dependee. The generated python client will automatically do this if it detects it is within a monorepo.
 * <p>
 * The following example shows how to configure the generated client to use a shared virtual environment:
 * <p>
 * <blockquote><pre>
 * const api = new OpenApiGatewayTsProject({
 *   parent: monorepo,
 *   name: 'api',
 *   outdir: 'packages/api',
 *   defaultReleaseBranch: 'main',
 *   clientLanguages: [ClientLanguage.PYTHON],
 *   pythonClientOptions: {
 *     moduleName: 'my_api_python',
 *     name: 'my_api_python',
 *     authorName: 'jack',
 *     authorEmail: 'me&#64;example.com',
 *     version: '1.0.0',
 *     venvOptions: {
 *       // Directory relative to the generated python client (in this case packages/api/generated/python)
 *       envdir: '../../../../.env',
 *     },
 *   },
 * });
 * 
 * new PythonProject({
 *   parent: monorepo,
 *   outdir: 'packages/my-python-lib',
 *   moduleName: 'my_python_lib',
 *   name: 'my_python_lib',
 *   authorName: 'jack',
 *   authorEmail: 'me&#64;example.com',
 *   version: '1.0.0',
 *   venvOptions: {
 *     // Directory relative to the python lib (in this case packages/my-python-lib)
 *     envdir: '../../.env',
 *   },
 *   // Generated client can be safely cast to a PythonProject
 *   deps: [(api.generatedClients[ClientLanguage.PYTHON] as PythonProject).moduleName],
 * });
 * </pre></blockquote>
 * <p>
 * <h4>Java API Lambda Handlers</h4>
 * <p>
 * To build your lambda handlers in Java, it's recommended to create a separate <code>JavaProject</code> in your <code>.projenrc</code>. This needs to build a "super jar" with all of your dependencies packed into a single jar. You can use the <code>maven-shade-plugin</code> to achieve this (see <a href="https://docs.aws.amazon.com/lambda/latest/dg/java-package.html">the java lambda docs for details</a>). You'll need to add a dependency on the generated java client for the handler wrappers. For example, your <code>.projenrc.ts</code> might look like:
 * <p>
 * <blockquote><pre>
 * const api = new OpenApiGatewayTsProject({
 *   parent: monorepo,
 *   name: '&#64;my-test/api',
 *   outdir: 'packages/api',
 *   defaultReleaseBranch: 'main',
 *   clientLanguages: [ClientLanguage.JAVA],
 * });
 * 
 * const apiJavaClient = (api.generatedClients[ClientLanguage.JAVA] as JavaProject);
 * 
 * const javaLambdaProject = new JavaProject({
 *   parent: monorepo,
 *   outdir: 'packages/java-lambdas',
 *   artifactId: "my-java-lambdas",
 *   groupId: "com.mycompany",
 *   name: "javalambdas",
 *   version: "1.0.0",
 *   // Add a dependency on the java client
 *   deps: [`${apiJavaClient.pom.groupId}/${apiJavaClient.pom.artifactId}&#64;${apiJavaClient.pom.version}`],
 * });
 * 
 * // Set up the dependency on the generated lambda client
 * monorepo.addImplicitDependency(javaLambdaProject, apiJavaClient);
 * javaLambdaProject.pom.addRepository({
 *   url: `file://../api/generated/java/dist/java`,
 *   id: 'java-api-client',
 * });
 * 
 * // Use the maven-shade-plugin as part of the maven package task
 * javaLambdaProject.pom.addPlugin('org.apache.maven.plugins/maven-shade-plugin&#64;3.2.2', {
 *   configuration: {
 *     createDependencyReducedPom: false,
 *     finalName: 'my-java-lambdas',
 *   },
 *   executions: [{
 *     id: 'shade-task',
 *     phase: 'package', // &lt;- NB "phase" is supported in projen ^0.61.37
 *     goals: ['shade'],
 *   }],
 * });
 * 
 * // Build the "super jar" as part of the project's package task
 * javaLambdaProject.packageTask.exec('mvn clean install');
 * </pre></blockquote>
 * <p>
 * You can then implement your lambda handlers in your <code>java-lambdas</code> project using the generated lambda handler wrappers (see above).
 * <p>
 * Finally, you can create a lambda function in your CDK infrastructure which points to the resultant "super jar":
 * <p>
 * <blockquote><pre>
 * new Api(this, 'JavaApi', {
 *   integrations: {
 *     sayHello: {
 *       function: new Function(this, 'SayHelloJava', {
 *         code: Code.fromAsset('../java-lambdas/target/my-java-lambdas.jar'),
 *         handler: 'com.mycompany.SayHelloHandler',
 *         runtime: Runtime.JAVA_11,
 *         timeout: Duration.seconds(30),
 *       }),
 *     },
 *   },
 * });
 * </pre></blockquote>
 * <p>
 * Note that to ensure the jar is built before the CDK infrastructure which consumes it, you must add a dependency, eg:
 * <p>
 * <blockquote><pre>
 * monorepo.addImplicitDependency(infra, javaLambdaProject);
 * </pre></blockquote>
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Experimental)
package software.aws.awsprototypingsdk.openapigateway;
