/*
 * Copyright (c) 2007, 2020, Oracle and/or its affiliates.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License, version 2.0, as published by the
 * Free Software Foundation.
 *
 * This program is also distributed with certain software (including but not
 * limited to OpenSSL) that is licensed under separate terms, as designated in a
 * particular file or component or in included license documentation. The
 * authors of MySQL hereby grant you an additional permission to link the
 * program and your derivative works with the separately licensed software that
 * they have included with MySQL.
 *
 * Without limiting anything contained in the foregoing, this file, which is
 * part of MySQL Connector/J, is also subject to the Universal FOSS Exception,
 * version 1.0, a copy of which can be found at
 * http://oss.oracle.com/licenses/universal-foss-exception.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License, version 2.0,
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA
 */

package software.aws.rds.jdbc.mysql.shading.com.mysql.cj.jdbc.interceptors;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.function.Supplier;

import software.aws.rds.jdbc.mysql.shading.com.mysql.cj.MysqlConnection;
import software.aws.rds.jdbc.mysql.shading.com.mysql.cj.Query;
import software.aws.rds.jdbc.mysql.shading.com.mysql.cj.exceptions.ExceptionFactory;
import software.aws.rds.jdbc.mysql.shading.com.mysql.cj.interceptors.QueryInterceptor;
import software.aws.rds.jdbc.mysql.shading.com.mysql.cj.jdbc.JdbcConnection;
import software.aws.rds.jdbc.mysql.shading.com.mysql.cj.log.Log;
import software.aws.rds.jdbc.mysql.shading.com.mysql.cj.protocol.Resultset;
import software.aws.rds.jdbc.mysql.shading.com.mysql.cj.protocol.ServerSession;
import software.aws.rds.jdbc.mysql.shading.com.mysql.cj.util.Util;

public class ServerStatusDiffInterceptor implements QueryInterceptor {

    private Map<String, String> preExecuteValues = new HashMap<>();

    private Map<String, String> postExecuteValues = new HashMap<>();

    private JdbcConnection connection;

    private Log log;

    @Override
    public QueryInterceptor init(MysqlConnection conn, Properties props, Log l) {
        this.connection = (JdbcConnection) conn;
        this.log = l;
        return this;
    }

    @Override
    public <T extends Resultset> T postProcess(Supplier<String> sql, Query interceptedQuery, T originalResultSet, ServerSession serverSession) {

        populateMapWithSessionStatusValues(this.postExecuteValues);

        this.log.logInfo("Server status change for query:\n" + Util.calculateDifferences(this.preExecuteValues, this.postExecuteValues));

        return null; // we don't actually modify a result set

    }

    private void populateMapWithSessionStatusValues(Map<String, String> toPopulate) {
        try {
            try (Statement stmt = this.connection.createStatement()) {
                toPopulate.clear();

                try (ResultSet rs = stmt.executeQuery("SHOW SESSION STATUS")) {
                    while (rs.next()) {
                        toPopulate.put(rs.getString(1), rs.getString(2));
                    }
                }
            }
        } catch (SQLException ex) {
            throw ExceptionFactory.createException(ex.getMessage(), ex);
        }
    }

    @Override
    public <T extends Resultset> T preProcess(Supplier<String> sql, Query interceptedQuery) {

        populateMapWithSessionStatusValues(this.preExecuteValues);

        return null; // we don't actually modify a result set
    }

    @Override
    public boolean executeTopLevelOnly() {
        return true;
    }

    @Override
    public void destroy() {
        this.connection = null;
        this.log = null;
    }
}
