/*
 * Decompiled with CFR 0.152.
 */
package software.coolstuff.springframework.owncloud.service.impl.local;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.FileNameMap;
import java.net.URI;
import java.net.URLConnection;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.PostConstruct;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.web.util.UriComponentsBuilder;
import software.coolstuff.springframework.owncloud.exception.resource.OwncloudLocalResourceException;
import software.coolstuff.springframework.owncloud.exception.resource.OwncloudNoDirectoryResourceException;
import software.coolstuff.springframework.owncloud.exception.resource.OwncloudQuotaExceededException;
import software.coolstuff.springframework.owncloud.exception.resource.OwncloudResourceNotFoundException;
import software.coolstuff.springframework.owncloud.model.OwncloudFileResource;
import software.coolstuff.springframework.owncloud.model.OwncloudQuota;
import software.coolstuff.springframework.owncloud.model.OwncloudResource;
import software.coolstuff.springframework.owncloud.model.OwncloudUserDetails;
import software.coolstuff.springframework.owncloud.service.impl.OwncloudUtils;
import software.coolstuff.springframework.owncloud.service.impl.local.OwncloudLocalFileResourceImpl;
import software.coolstuff.springframework.owncloud.service.impl.local.OwncloudLocalProperties;
import software.coolstuff.springframework.owncloud.service.impl.local.OwncloudLocalQuotaImpl;
import software.coolstuff.springframework.owncloud.service.impl.local.OwncloudLocalResourceChecksumService;
import software.coolstuff.springframework.owncloud.service.impl.local.OwncloudLocalResourceExtension;
import software.coolstuff.springframework.owncloud.service.impl.local.OwncloudLocalResourceImpl;
import software.coolstuff.springframework.owncloud.service.impl.local.OwncloudLocalResourceService;
import software.coolstuff.springframework.owncloud.service.impl.local.OwncloudLocalUserDataService;
import software.coolstuff.springframework.owncloud.service.impl.local.OwncloudLocalUserServiceExtension;
import software.coolstuff.springframework.owncloud.service.impl.local.OwncloudLocalUtils;
import software.coolstuff.springframework.owncloud.service.impl.local.PipedOutputStreamAfterCopyEnvironment;
import software.coolstuff.springframework.owncloud.service.impl.local.PipedOutputStreamLocalSynchronizer;

class OwncloudLocalResourceServiceImpl
implements OwncloudLocalResourceService {
    private static final Logger log = LoggerFactory.getLogger(OwncloudLocalResourceServiceImpl.class);
    @Autowired
    private OwncloudLocalProperties properties;
    @Autowired
    private OwncloudLocalResourceChecksumService checksumService;
    @Autowired
    private OwncloudLocalUserDataService userDataService;
    @Autowired
    private OwncloudLocalUserServiceExtension userService;
    private Map<String, OwncloudLocalQuotaImpl> quotas = new HashMap<String, OwncloudLocalQuotaImpl>();

    OwncloudLocalResourceServiceImpl() {
    }

    @PostConstruct
    protected void afterPropertiesSet() throws Exception {
        OwncloudLocalProperties.ResourceServiceProperties resourceProperties = this.properties.getResourceService();
        Validate.notNull((Object)resourceProperties);
        Validate.notNull((Object)resourceProperties.getLocation());
        Path baseLocation = resourceProperties.getLocation();
        OwncloudLocalUtils.checkPrivilegesOnDirectory(baseLocation);
        this.calculateQuotas(baseLocation);
        log.debug("Register Usermodification Callbacks");
        this.userService.registerSaveUserCallback(this::notifyUserModification);
        this.userService.registerDeleteUserCallback(this::notifyRemovedUser);
    }

    private void calculateQuotas(Path baseLocation) throws IOException {
        this.quotas.clear();
        this.userDataService.getUsers().forEach(user -> {
            String username = user.getUsername();
            OwncloudLocalQuotaImpl quota = this.calculateUsedSpace(username, baseLocation);
            quota.setTotal(user.getQuota());
            this.quotas.put(username, quota);
        });
    }

    private OwncloudLocalQuotaImpl calculateUsedSpace(String username, Path baseLocation) {
        Path userBaseLocation = baseLocation.resolve(username);
        if (Files.notExists(userBaseLocation, new LinkOption[0])) {
            return OwncloudLocalQuotaImpl.builder().username(username).location(baseLocation).build();
        }
        try {
            OwncloudLocalQuotaImpl quota = OwncloudLocalQuotaImpl.builder().username(username).location(userBaseLocation).build();
            log.debug("Calculate the Space used by User {}", (Object)username);
            Files.walkFileTree(userBaseLocation, new UsedSpaceFileVisitor(quota::increaseUsed));
            return quota;
        }
        catch (IOException e) {
            String logMessage = "IOException while calculating the used Space of Location " + userBaseLocation.toAbsolutePath().normalize().toString();
            log.error(logMessage);
            throw new OwncloudLocalResourceException(logMessage, e);
        }
    }

    private void notifyUserModification(OwncloudUserDetails userDetails) {
        log.debug("User {} has been changed or created -> change the Quota to {}", (Object)userDetails.getUsername(), (Object)userDetails.getQuota());
        OwncloudLocalQuotaImpl quota = this.quotas.computeIfAbsent(userDetails.getUsername(), this::getOrCreateQuota);
        quota.setTotal(userDetails.getQuota());
    }

    private OwncloudLocalQuotaImpl getOrCreateQuota(String username) {
        OwncloudLocalProperties.ResourceServiceProperties resourceProperties = this.properties.getResourceService();
        return this.calculateUsedSpace(username, resourceProperties.getLocation());
    }

    private void notifyRemovedUser(String username) {
        log.debug("User {} has been removed -> remove the Quota Information", (Object)username);
        this.quotas.remove(username);
    }

    @Override
    public List<OwncloudResource> list(URI relativeTo) {
        Path location = this.resolveLocation(relativeTo);
        return this.resourcesOf(location);
    }

    private Path resolveLocation(URI relativeTo) {
        Path location = this.getRootLocationOfAuthenticatedUser();
        if (relativeTo == null || StringUtils.isBlank((CharSequence)relativeTo.getPath())) {
            return location;
        }
        this.createDirectoryIfNotExists(location);
        String relativeToPath = relativeTo.getPath();
        if (StringUtils.startsWith((CharSequence)relativeToPath, (CharSequence)"/")) {
            relativeToPath = relativeToPath.substring(1);
        }
        return location.resolve(relativeToPath);
    }

    private Path getRootLocationOfAuthenticatedUser() {
        OwncloudLocalProperties.ResourceServiceProperties resourceProperties = this.properties.getResourceService();
        Path location = resourceProperties.getLocation();
        String username = this.getUsername();
        location = location.resolve(username);
        this.checkIfExistingDirectory(location);
        log.debug("Resolved Base Location of User {}: {}", (Object)username, (Object)location);
        return location;
    }

    private String getUsername() {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        return authentication.getName();
    }

    private void checkIfExistingDirectory(Path location) {
        if (this.isNotExistingDirectory(location)) {
            String logMessage = String.format("Existing Path %s is not a Directory", location);
            log.error(logMessage);
            throw new OwncloudLocalResourceException(logMessage);
        }
    }

    private boolean isNotExistingDirectory(Path location) {
        return Files.exists(location, new LinkOption[0]) && !Files.isDirectory(location, new LinkOption[0]);
    }

    private void createDirectoryIfNotExists(Path location) {
        if (Files.notExists(location, new LinkOption[0])) {
            try {
                log.debug("Create Directory {}", (Object)location);
                Files.createDirectories(location, new FileAttribute[0]);
            }
            catch (IOException e) {
                String logMessage = String.format("Could not create Directory %s", location);
                log.error(logMessage, (Throwable)e);
                throw new OwncloudLocalResourceException(logMessage, e);
            }
        }
    }

    private List<OwncloudResource> resourcesOf(Path location) {
        if (Files.isDirectory(location, new LinkOption[0])) {
            return this.getDirectoryResources(location);
        }
        return Stream.of(this.createOwncloudResourceOf(location)).peek(resource -> log.debug("Add Resource {} to the Result", (Object)resource.getHref())).collect(Collectors.toList());
    }

    private List<OwncloudResource> getDirectoryResources(Path location) {
        try {
            ArrayList<OwncloudResource> owncloudResources = new ArrayList<OwncloudResource>();
            owncloudResources.add(this.getActualDirectoryOf(location));
            try (Stream<Path> stream = Files.list(location);){
                stream.map(this::createOwncloudResourceOf).peek(resource -> log.debug("Add Resource {} to the Result", (Object)resource.getHref())).forEach(owncloudResources::add);
            }
            this.getParentDirectoryOf(location).ifPresent(owncloudResources::add);
            return owncloudResources;
        }
        catch (IOException e) {
            throw new OwncloudLocalResourceException(e);
        }
    }

    private OwncloudLocalResourceExtension getActualDirectoryOf(Path location) {
        OwncloudLocalResourceExtension actualDirectory = this.createOwncloudResourceOf(location);
        log.debug("Add actual Directory {} to the Result", (Object)actualDirectory.getHref());
        actualDirectory.setName(".");
        return actualDirectory;
    }

    private OwncloudLocalResourceExtension createOwncloudResourceOf(Path path) {
        Path rootPath = this.getRootLocationOfAuthenticatedUser();
        Path relativePath = rootPath.toAbsolutePath().relativize(path.toAbsolutePath());
        URI href = URI.create(UriComponentsBuilder.fromPath((String)"/").path(relativePath.toString()).toUriString());
        String name = path.getFileName().toString();
        MediaType mediaType = MediaType.APPLICATION_OCTET_STREAM;
        if (Files.isDirectory(path, new LinkOption[0])) {
            href = URI.create(UriComponentsBuilder.fromUri((URI)href).path("/").toUriString());
            mediaType = OwncloudUtils.getDirectoryMediaType();
        } else {
            FileNameMap fileNameMap = URLConnection.getFileNameMap();
            String contentType = fileNameMap.getContentTypeFor(path.getFileName().toString());
            if (StringUtils.isNotBlank((CharSequence)contentType)) {
                mediaType = MediaType.valueOf((String)contentType);
            }
        }
        try {
            LocalDateTime lastModifiedAt = LocalDateTime.ofInstant(Files.getLastModifiedTime(path, new LinkOption[0]).toInstant(), ZoneId.systemDefault());
            Optional<String> checksum = this.checksumService.getChecksum(path);
            if (Files.isSameFile(rootPath, path)) {
                name = "/";
                checksum = Optional.empty();
            }
            OwncloudLocalResourceImpl resource = OwncloudLocalResourceImpl.builder().href(href).name(name).eTag(checksum.orElse(null)).mediaType(mediaType).lastModifiedAt(lastModifiedAt).build();
            if (Files.isDirectory(path, new LinkOption[0])) {
                return resource;
            }
            return OwncloudLocalFileResourceImpl.fileBuilder().owncloudResource(resource).contentLength(Files.size(path)).build();
        }
        catch (NoSuchFileException e) {
            throw new OwncloudResourceNotFoundException(href, this.getUsername());
        }
        catch (IOException e) {
            String logMessage = String.format("Cannot create OwncloudResource from Path %s", path);
            log.error(logMessage, (Throwable)e);
            throw new OwncloudLocalResourceException(logMessage, e);
        }
    }

    private Optional<OwncloudLocalResourceExtension> getParentDirectoryOf(Path location) {
        if (this.isParentDirectoryNotAppendable(location)) {
            return Optional.empty();
        }
        OwncloudLocalResourceExtension superDirectory = this.createOwncloudResourceOf(location.resolve("..").normalize());
        log.debug("Add parent Directory of Location {} ({}) to the Result", (Object)location, (Object)superDirectory.getHref());
        superDirectory.setName("..");
        return Optional.of(superDirectory);
    }

    private boolean isParentDirectoryNotAppendable(Path location) {
        return !this.properties.getResourceService().isAddRelativeDownPath() || this.isRootDirectory(location);
    }

    private boolean isRootDirectory(Path location) {
        if (!Files.isDirectory(location, new LinkOption[0])) {
            return false;
        }
        Path rootLocation = this.getRootLocationOfAuthenticatedUser();
        try {
            return Files.isSameFile(location, rootLocation);
        }
        catch (IOException e) {
            String logMessage = String.format("Cannot determine the equality of path %s to the base Location %s", location, rootLocation);
            log.error(logMessage, (Throwable)e);
            throw new OwncloudLocalResourceException(logMessage, e);
        }
    }

    @Override
    public Optional<OwncloudResource> find(URI path) {
        Path location = this.resolveLocation(path);
        if (Files.notExists(location, new LinkOption[0])) {
            return Optional.empty();
        }
        log.debug("Get Information about Resource %s", (Object)path);
        return Optional.ofNullable(this.createOwncloudResourceOf(location));
    }

    @Override
    public OwncloudResource createDirectory(URI directory) {
        Path location = this.resolveLocation(directory);
        if (Files.exists(location, new LinkOption[0])) {
            if (Files.isDirectory(location, new LinkOption[0])) {
                return this.createOwncloudResourceOf(location);
            }
            throw new OwncloudNoDirectoryResourceException(directory);
        }
        try {
            log.debug("Create Directory {}", (Object)location.toAbsolutePath().normalize());
            Files.createDirectory(location, new FileAttribute[0]);
            this.checksumService.recalculateChecksum(location);
            return this.createOwncloudResourceOf(location);
        }
        catch (IOException e) {
            String logMessage = String.format("Cannot create Directory %s", location.toAbsolutePath().normalize());
            log.error(logMessage, (Throwable)e);
            throw new OwncloudLocalResourceException(logMessage, e);
        }
    }

    @Override
    public void delete(OwncloudResource resource) {
        Path path = this.resolveLocation(resource.getHref());
        this.checkExistence(path, resource);
        this.removeExistingPath(path);
    }

    private void checkExistence(Path path, OwncloudResource resource) {
        if (Files.notExists(path, new LinkOption[0])) {
            Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
            throw new OwncloudResourceNotFoundException(resource.getHref(), authentication.getName());
        }
    }

    private void removeExistingPath(Path path) {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        OwncloudLocalQuotaImpl quota = this.quotas.get(authentication.getName());
        this.removeExistingPathAndRecalculateSpaceAndChecksum(path, quota);
    }

    private void removeExistingPathAndRecalculateSpaceAndChecksum(Path path, OwncloudLocalQuotaImpl quota) {
        try {
            if (Files.isDirectory(path, new LinkOption[0])) {
                log.debug("Remove Directory {} with all its Content and reduce the used Space of User {}", (Object)path.toAbsolutePath().normalize(), (Object)quota.getUsername());
                Files.walkFileTree(path, new DeleteFileVisitor(quota::reduceUsed));
            } else {
                log.debug("Remove File {} and reduce the used Space of User {}", (Object)path.toAbsolutePath().normalize(), (Object)quota.getUsername());
                quota.reduceUsed(Files.size(path));
                Files.delete(path);
            }
        }
        catch (IOException e) {
            String logMessage = String.format("Cannot remove Path %s", path.toAbsolutePath().normalize());
            log.error(logMessage, (Throwable)e);
            throw new OwncloudLocalResourceException(logMessage, e);
        }
        finally {
            this.checksumService.recalculateChecksum(path);
        }
    }

    @Override
    public InputStream getInputStream(OwncloudFileResource resource) {
        Path location = this.resolveLocation(resource.getHref());
        try {
            log.debug("Return InputStream of File {}", (Object)location.toAbsolutePath().normalize());
            return Files.newInputStream(location, new OpenOption[0]);
        }
        catch (NoSuchFileException e) {
            log.warn("File {} not found", (Object)location.toAbsolutePath().normalize());
            throw new OwncloudResourceNotFoundException(resource.getHref(), this.getUsername());
        }
        catch (IOException e) {
            String logMessage = String.format("Cannot get InputStream of File %s", location.toAbsolutePath().normalize());
            log.error(logMessage, (Throwable)e);
            throw new OwncloudLocalResourceException(logMessage, e);
        }
    }

    @Override
    public OutputStream getOutputStream(OwncloudFileResource resource) {
        return this.getOutputStream(resource.getHref(), resource.getMediaType());
    }

    @Override
    public OutputStream getOutputStream(URI path, MediaType mediaType) {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        log.debug("Create a piped OutputStream to control the written Data (because of the Quota of User {}", (Object)authentication.getName());
        PipedOutputStreamLocalSynchronizer pipedStreamSynchronizer = PipedOutputStreamLocalSynchronizer.builder().authentication(authentication).afterCopyCallback(this::afterCopy).owncloudLocalProperties(this.properties).uri(path).uriResolver(this::resolveLocation).build();
        return pipedStreamSynchronizer.getOutputStream();
    }

    private void afterCopy(PipedOutputStreamAfterCopyEnvironment environment) {
        Optional.ofNullable(this.quotas.get(environment.getUsername())).ifPresent(quota -> {
            this.checkSpace((OwncloudQuota)quota, environment);
            quota.increaseUsed(environment.getContentLength());
        });
        if (Files.exists(environment.getPath(), new LinkOption[0])) {
            this.checksumService.recalculateChecksum(environment.getPath());
        }
    }

    private void checkSpace(OwncloudQuota quota, PipedOutputStreamAfterCopyEnvironment environment) {
        if (this.isNoMoreSpaceLeft(quota, environment)) {
            log.error("User {} exceeded its Quota of {} Bytes", (Object)quota.getUsername(), (Object)quota.getTotal());
            this.removeFile(environment);
            throw new OwncloudQuotaExceededException(environment.getUri(), environment.getUsername());
        }
    }

    private boolean isNoMoreSpaceLeft(OwncloudQuota quota, PipedOutputStreamAfterCopyEnvironment environment) {
        return quota.getFree() < environment.getContentLength();
    }

    private void removeFile(PipedOutputStreamAfterCopyEnvironment environment) {
        try {
            log.debug("Remove File {}", (Object)environment.getPath().toAbsolutePath().normalize());
            Files.delete(environment.getPath());
        }
        catch (IOException e) {
            String logMessage = String.format("Error while removing File %s", environment.getPath().toAbsolutePath().normalize());
            log.error(logMessage, (Throwable)e);
            throw new OwncloudLocalResourceException(logMessage, e);
        }
    }

    @Override
    public OwncloudQuota getQuota() {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        log.debug("Return the actual Quota of User {}", (Object)authentication.getName());
        return this.quotas.get(authentication.getName());
    }

    @Override
    public void resetAllUsedSpace() {
        this.quotas.forEach(this::resetUsedSpace);
    }

    private void resetUsedSpace(String username, OwncloudLocalQuotaImpl quota) {
        log.debug("Reset the used Space of User {}", (Object)username);
        quota.setUsed(0L);
    }

    @Override
    public void recalculateAllUsedSpace() {
        OwncloudLocalProperties.ResourceServiceProperties resourceProperties = this.properties.getResourceService();
        Path baseLocation = resourceProperties.getLocation();
        this.quotas.forEach((username, unusedQuota) -> this.quotas.computeIfPresent((String)username, (unusedUsername, existingQuota) -> {
            OwncloudLocalQuotaImpl quota = this.calculateUsedSpace((String)username, baseLocation);
            quota.setTotal(existingQuota.getTotal());
            return quota;
        }));
    }

    private static class DeleteFileVisitor
    extends SimpleFileVisitor<Path> {
        private final Consumer<Long> usedSpaceReductionConsumer;

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            this.usedSpaceReductionConsumer.accept(Files.size(file));
            Files.delete(file);
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
            Files.delete(dir);
            return FileVisitResult.CONTINUE;
        }

        public DeleteFileVisitor(Consumer<Long> usedSpaceReductionConsumer) {
            this.usedSpaceReductionConsumer = usedSpaceReductionConsumer;
        }
    }

    private static class UsedSpaceFileVisitor
    extends SimpleFileVisitor<Path> {
        private final Consumer<Long> usedSpaceIncreaseConsumer;

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            this.usedSpaceIncreaseConsumer.accept(Files.size(file));
            return FileVisitResult.CONTINUE;
        }

        public UsedSpaceFileVisitor(Consumer<Long> usedSpaceIncreaseConsumer) {
            this.usedSpaceIncreaseConsumer = usedSpaceIncreaseConsumer;
        }
    }
}

