package com.etsy.sbt.checkstyle

import com.etsy.sbt.checkstyle.CheckstyleSeverityLevel.*
import com.puppycrawl.tools.checkstyle.ConfigurationLoader.IgnoredModulesOptions
import com.puppycrawl.tools.checkstyle.api.AutomaticBean.OutputStreamOptions
import com.puppycrawl.tools.checkstyle.api.Configuration as CheckstyleConfiguration
import com.puppycrawl.tools.checkstyle.{Checker, ConfigurationLoader, PropertiesExpander, XMLLogger}
import net.sf.saxon.s9api.Processor
import org.xml.sax.InputSource
import sbt.*

import java.io.FileOutputStream
import java.util.Properties
import javax.xml.transform.stream.StreamSource
import scala.collection.JavaConverters.*

/**
 * A Scala wrapper around the Checkstyle Java API
 *
 * @author
 *   Andrew Johnson <ajohnson@etsy.com>
 * @author
 *   Joseph Earl <joe@josephearl.co.uk>
 */
object Checkstyle {

  /**
   * Runs Checkstyle.
   *
   * Note: any non-Java sources passed to this function will be ignored.
   *
   * @param sourceFiles
   *   list of source files to scan.
   * @param configLocation
   *   location of the Checkstyle config to use.
   * @param resources
   *   list of resource files (used if `configLocation` points to the classpath).
   * @param outputFile
   *   file to store the Checkstyle report in.
   * @param xsltTransformations
   *   XML transformations to apply to the Checkstyle report.
   * @param severityLevel
   *   threshold of violations for failing the build if reached.
   * @param log
   *   used to log status of Checkstyle.
   */
  def checkstyle(
      sourceFiles: Seq[File],
      configLocation: CheckstyleConfigLocation,
      resources: Seq[File],
      outputFile: File,
      xsltTransformations: Set[CheckstyleXSLTSettings],
      severityLevel: CheckstyleSeverityLevel,
      log: Logger
  ): Unit = {

    // ensure output dir exists
    IO.createDirectory(outputFile.getParentFile)

    val checker = new Checker()

    try {
      val checkstyleClassLoader = classOf[Checker].getClassLoader
      checker.setModuleClassLoader(checkstyleClassLoader)

      val checkerConfig: CheckstyleConfiguration = oiConfig(configLocation, resources)

      checker.configure(checkerConfig)

      val checkstyleListener = new CheckstyleListener()
      checker.addListener(checkstyleListener)

      val fileListener = new XMLLogger(new FileOutputStream(outputFile), OutputStreamOptions.CLOSE)
      checker.addListener(fileListener)

      val javaSources = sourceFiles.filter(_.ext == "java")
      checker.process(javaSources.asJava)

      log.info(s"Checkstyle complete. ${checkstyleListener.currentStatus}")

      if (xsltTransformations.nonEmpty) {
        applyXSLT(outputFile, xsltTransformations)
      }

      if (severityLevel != CheckstyleSeverityLevel.Ignore && severityLevel <= checkstyleListener.highestErrorLevel) {
        sys.error("Severity of checkstyle errors exceeds project limit")
      }
    } finally {
      checker.destroy()
    }
  }

  private def oiConfig(configLocation: CheckstyleConfigLocation, resources: Seq[File]): CheckstyleConfiguration = {
    val properties = new Properties()
    val configSource = new InputSource(configLocation.load(resources))

    ConfigurationLoader.loadConfiguration(
      configSource,
      new PropertiesExpander(properties),
      IgnoredModulesOptions.EXECUTE
    )
  }

  /**
   * Applies a set of XSLT transformation to the XML file produced by checkstyle
   *
   * @param input
   *   The XML file produced by checkstyle
   * @param transformations
   *   The XSLT transformations to be applied
   */
  private def applyXSLT(
      input: File,
      transformations: Set[CheckstyleXSLTSettings]
  ): Unit = {

    val processor = new Processor(false)
    val source = processor.newDocumentBuilder().build(input)

    transformations foreach { transform: CheckstyleXSLTSettings =>
      val output = processor.newSerializer(transform.output)
      val compiler = processor.newXsltCompiler()
      val executor = compiler.compile(new StreamSource(transform.xslt))
      val transformer = executor.load()
      transformer.setInitialContextNode(source)
      transformer.setDestination(output)
      transformer.transform()
      transformer.close()
      output.close()
    }
  }
}
