package com.etsy.sbt.checkstyle

import com.etsy.sbt.checkstyle.CheckstyleSeverityLevel.CheckstyleSeverityLevel
import sbt.*
import sbt.Def.Initialize
import sbt.Keys.*

/**
 * An SBT plugin to run checkstyle over Java code
 *
 * @author
 *   Andrew Johnson <ajohnson@etsy.com>
 * @author
 *   Alejandro Rivera <alejandro.rivera.lopez@gmail.com>
 * @author
 *   Joseph Earl <joe@josephearl.co.uk>
 */
object CheckstylePlugin extends AutoPlugin {
  override def trigger: PluginTrigger = allRequirements

  object autoImport {
    val checkstyle: TaskKey[Unit] = taskKey[Unit]("Runs checkstyle")
    val checkstyleOutputFile: SettingKey[File] = settingKey[File](
      "The location of the generated checkstyle report"
    )
    val checkstyleConfigLocation: SettingKey[CheckstyleConfigLocation] = settingKey[CheckstyleConfigLocation](
      "The location of the checkstyle configuration file"
    )
    val checkstyleXsltTransformations: SettingKey[Set[CheckstyleXSLTSettings]] =
      settingKey[Set[CheckstyleXSLTSettings]](
        "An optional set of XSLT transformations to be applied to the checkstyle output"
      )
    val checkstyleSeverityLevel: SettingKey[CheckstyleSeverityLevel] =
      settingKey[CheckstyleSeverityLevel]("Sets the severity levels which should fail the build")

    val CheckstyleConfigLocation: com.etsy.sbt.checkstyle.CheckstyleConfigLocation.type =
      com.etsy.sbt.checkstyle.CheckstyleConfigLocation
    val CheckstyleSeverityLevel: com.etsy.sbt.checkstyle.CheckstyleSeverityLevel.type =
      com.etsy.sbt.checkstyle.CheckstyleSeverityLevel
    val CheckstyleXSLTSettings: com.etsy.sbt.checkstyle.CheckstyleXSLTSettings.type =
      com.etsy.sbt.checkstyle.CheckstyleXSLTSettings

    /**
     * Runs checkstyle
     *
     * @param conf
     *   The configuration (Compile or Test) in which context to execute the checkstyle command
     */
    def checkstyleTask(conf: Configuration): Initialize[Task[Unit]] = Def.task {
      Checkstyle.checkstyle(
        (conf / sources).value,
        (conf / checkstyleConfigLocation).value,
        (Compile / resourceDirectories).value,
        (conf / checkstyleOutputFile).value,
        (conf / checkstyleXsltTransformations).value,
        (conf / checkstyleSeverityLevel).value,
        streams.value.log
      )
    }
  }

  // scalastyle:off import.grouping
  import autoImport.*
  // scalastyle:on import.grouping

  private lazy val commonSettings: Seq[Def.Setting[?]] = Seq(
    checkstyleXsltTransformations := Set.empty,
    checkstyleSeverityLevel := CheckstyleSeverityLevel.Ignore
  )

  override lazy val projectSettings: Seq[Def.Setting[?]] = Seq(
    checkstyleOutputFile := target.value / "checkstyle-report.xml",
    Test / checkstyleOutputFile := target.value / "checkstyle-test-report.xml",
    checkstyleConfigLocation := CheckstyleConfigLocation.File("checkstyle-config.xml"),
    Test / checkstyleConfigLocation := checkstyleConfigLocation.value,
    checkstyle := checkstyleTask(Compile).value,
    Test / checkstyle := checkstyleTask(Test).value
  ) ++ commonSettings
}
