

package space.yizhu.kits;

import java.security.MessageDigest;

/**
 * <p>HashKit class.</p>
 *
 * @author yi
 * @version $Id: $Id
 */
public class HashKit {

    /** Constant <code>FNV_OFFSET_BASIS_64=0xcbf29ce484222325L</code> */
    public static final long FNV_OFFSET_BASIS_64 = 0xcbf29ce484222325L;
    /** Constant <code>FNV_PRIME_64=0x100000001b3L</code> */
    public static final long FNV_PRIME_64 = 0x100000001b3L;

    private static final java.security.SecureRandom random = new java.security.SecureRandom();
    private static final char[] HEX_DIGITS = "0123456789abcdef".toCharArray();
    private static final char[] CHAR_ARRAY = "_-0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ".toCharArray();

    /**
     * <p>fnv1a64.</p>
     *
     * @param key a {@link java.lang.String} object.
     * @return a long.
     */
    public static long fnv1a64(String key) {
        long hash = FNV_OFFSET_BASIS_64;
        for (int i = 0, size = key.length(); i < size; i++) {
            hash ^= key.charAt(i);
            hash *= FNV_PRIME_64;
        }
        return hash;
    }

    /**
     * <p>md5.</p>
     *
     * @param srcStr a {@link java.lang.String} object.
     * @return a {@link java.lang.String} object.
     */
    public static String md5(String srcStr) {
        return hash("MD5", srcStr);
    }

    /**
     * <p>sha1.</p>
     *
     * @param srcStr a {@link java.lang.String} object.
     * @return a {@link java.lang.String} object.
     */
    public static String sha1(String srcStr) {
        return hash("SHA-1", srcStr);
    }

    /**
     * <p>sha256.</p>
     *
     * @param srcStr a {@link java.lang.String} object.
     * @return a {@link java.lang.String} object.
     */
    public static String sha256(String srcStr) {
        return hash("SHA-256", srcStr);
    }

    /**
     * <p>sha384.</p>
     *
     * @param srcStr a {@link java.lang.String} object.
     * @return a {@link java.lang.String} object.
     */
    public static String sha384(String srcStr) {
        return hash("SHA-384", srcStr);
    }

    /**
     * <p>sha512.</p>
     *
     * @param srcStr a {@link java.lang.String} object.
     * @return a {@link java.lang.String} object.
     */
    public static String sha512(String srcStr) {
        return hash("SHA-512", srcStr);
    }

    /**
     * <p>hash.</p>
     *
     * @param algorithm a {@link java.lang.String} object.
     * @param srcStr a {@link java.lang.String} object.
     * @return a {@link java.lang.String} object.
     */
    public static String hash(String algorithm, String srcStr) {
        try {
            MessageDigest md = MessageDigest.getInstance(algorithm);
            byte[] bytes = md.digest(srcStr.getBytes("utf-8"));
            return toHex(bytes);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * <p>toHex.</p>
     *
     * @param bytes an array of {@link byte} objects.
     * @return a {@link java.lang.String} object.
     */
    public static String toHex(byte[] bytes) {
        StringBuilder ret = new StringBuilder(bytes.length * 2);
        for (int i = 0; i < bytes.length; i++) {
            ret.append(HEX_DIGITS[(bytes[i] >> 4) & 0x0f]);
            ret.append(HEX_DIGITS[bytes[i] & 0x0f]);
        }
        return ret.toString();
    }

    /**
     * md5 128bit 16bytes
     * sha1 160bit 20bytes
     * sha256 256bit 32bytes
     * sha384 384bit 48bytes
     * sha512 512bit 64bytes
     *
     * @param saltLength 盐长度
     * @return 生成盐
     */
    public static String generateSalt(int saltLength) {
        StringBuilder salt = new StringBuilder(saltLength);
        for (int i = 0; i < saltLength; i++) {
            salt.append(CHAR_ARRAY[random.nextInt(CHAR_ARRAY.length)]);
        }
        return salt.toString();
    }

    /**
     * <p>generateSaltForSha256.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public static String generateSaltForSha256() {
        return generateSalt(32);
    }

    /**
     * <p>generateSaltForSha512.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public static String generateSaltForSha512() {
        return generateSalt(64);
    }

    /**
     * <p>slowEquals.</p>
     *
     * @param a an array of {@link byte} objects.
     * @param b an array of {@link byte} objects.
     * @return a boolean.
     */
    public static boolean slowEquals(byte[] a, byte[] b) {
        if (a == null || b == null) {
            return false;
        }

        int diff = a.length ^ b.length;
        for (int i = 0; i < a.length && i < b.length; i++) {
            diff |= a[i] ^ b[i];
        }
        return diff == 0;
    }
}




