package space.yizhu.record.aop;

import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.concurrent.ConcurrentHashMap;


/**
 * <p>AopFactory class.</p>
 *
 * @author yi
 * @version $Id: $Id
 */
public class AopFactory {

    
    protected ConcurrentHashMap<Class<?>, Object> singletonCache = new ConcurrentHashMap<Class<?>, Object>();

    
    protected HashMap<Class<?>, Class<?>> mapping = null;

    /** Constant <code>MAX_INJECT_DEPTH=7</code> */
    protected static int MAX_INJECT_DEPTH = 7;            

    protected boolean singleton = true;                    
    protected boolean enhance = true;                    
    protected int injectDepth = 3;                        

    /**
     * <p>get.</p>
     *
     * @param targetClass a {@link java.lang.Class} object.
     * @param <T> a T object.
     * @return a T object.
     */
    public <T> T get(Class<T> targetClass) {
        try {
            return doGet(targetClass, injectDepth);
        } catch (ReflectiveOperationException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * <p>get.</p>
     *
     * @param targetClass a {@link java.lang.Class} object.
     * @param injectDepth a int.
     * @param <T> a T object.
     * @return a T object.
     */
    public <T> T get(Class<T> targetClass, int injectDepth) {
        try {
            return doGet(targetClass, injectDepth);
        } catch (ReflectiveOperationException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * <p>doGet.</p>
     *
     * @param targetClass a {@link java.lang.Class} object.
     * @param injectDepth a int.
     * @param <T> a T object.
     * @return a T object.
     * @throws java.lang.ReflectiveOperationException if any.
     */
    @SuppressWarnings("unchecked")
    protected <T> T doGet(Class<T> targetClass, int injectDepth) throws ReflectiveOperationException {
        
        

        targetClass = (Class<T>) getMappingClass(targetClass);

        Singleton si = targetClass.getAnnotation(Singleton.class);
        boolean singleton = (si != null ? si.value() : this.singleton);

        Object ret;
        if (!singleton) {
            ret = createObject(targetClass);
            doInject(targetClass, ret, injectDepth);
            return (T) ret;
        }

        ret = singletonCache.get(targetClass);
        if (ret == null) {
            synchronized (this) {
                ret = singletonCache.get(targetClass);
                if (ret == null) {
                    ret = createObject(targetClass);
                    doInject(targetClass, ret, injectDepth);
                    singletonCache.put(targetClass, ret);
                }
            }
        }

        return (T) ret;
    }

    /**
     * <p>inject.</p>
     *
     * @param targetObject a T object.
     * @param <T> a T object.
     * @return a T object.
     */
    public <T> T inject(T targetObject) {
        try {
            doInject(targetObject.getClass(), targetObject, injectDepth);
            return targetObject;
        } catch (ReflectiveOperationException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * <p>inject.</p>
     *
     * @param targetObject a T object.
     * @param injectDepth a int.
     * @param <T> a T object.
     * @return a T object.
     */
    public <T> T inject(T targetObject, int injectDepth) {
        try {
            doInject(targetObject.getClass(), targetObject, injectDepth);
            return targetObject;
        } catch (ReflectiveOperationException e) {
            throw new RuntimeException(e);
        }
    }

    
    /**
     * <p>inject.</p>
     *
     * @param targetClass a {@link java.lang.Class} object.
     * @param targetObject a T object.
     * @param <T> a T object.
     * @return a T object.
     */
    public <T> T inject(Class<T> targetClass, T targetObject) {
        try {
            doInject(targetClass, targetObject, injectDepth);
            return targetObject;
        } catch (ReflectiveOperationException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * <p>inject.</p>
     *
     * @param targetClass a {@link java.lang.Class} object.
     * @param targetObject a T object.
     * @param injectDepth a int.
     * @param <T> a T object.
     * @return a T object.
     */
    public <T> T inject(Class<T> targetClass, T targetObject, int injectDepth) {
        try {
            doInject(targetClass, targetObject, injectDepth);
            return targetObject;
        } catch (ReflectiveOperationException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * <p>doInject.</p>
     *
     * @param targetClass a {@link java.lang.Class} object.
     * @param targetObject a {@link java.lang.Object} object.
     * @param injectDepth a int.
     * @throws java.lang.ReflectiveOperationException if any.
     */
    protected void doInject(Class<?> targetClass, Object targetObject, int injectDepth) throws ReflectiveOperationException {
        if ((injectDepth--) <= 0) {
            return;
        }

        targetClass = getUsefulClass(targetClass);
        Field[] fields = targetClass.getDeclaredFields();
        if (fields.length == 0) {
            return;
        }

        for (Field field : fields) {
            Inject inject = field.getAnnotation(Inject.class);
            if (inject == null) {
                continue;
            }

            Class<?> fieldInjectedClass = inject.value();
            if (fieldInjectedClass == Void.class) {
                fieldInjectedClass = field.getType();
            }

            Object fieldInjectedObject = doGet(fieldInjectedClass, injectDepth);
            field.setAccessible(true);
            field.set(targetObject, fieldInjectedObject);
        }
    }

    
    /**
     * <p>createObject.</p>
     *
     * @param targetClass a {@link java.lang.Class} object.
     * @return a {@link java.lang.Object} object.
     * @throws java.lang.ReflectiveOperationException if any.
     */
    @SuppressWarnings("deprecation")
    protected Object createObject(Class<?> targetClass) throws ReflectiveOperationException {
        Enhance en = targetClass.getAnnotation(Enhance.class);
        boolean enhance = (en != null ? en.value() : this.enhance);

        return targetClass.newInstance();
    }

    
    /**
     * <p>getUsefulClass.</p>
     *
     * @param clazz a {@link java.lang.Class} object.
     * @return a {@link java.lang.Class} object.
     */
    protected Class<?> getUsefulClass(Class<?> clazz) {
        
        
        return (Class<?>) (clazz.getName().indexOf("$$EnhancerBy") == -1 ? clazz : clazz.getSuperclass());
    }

    
    /**
     * <p>Setter for the field <code>enhance</code>.</p>
     *
     * @param enhance a boolean.
     * @return a {@link space.yizhu.record.aop.AopFactory} object.
     */
    @Deprecated
    public AopFactory setEnhance(boolean enhance) {
        this.enhance = enhance;
        return this;
    }

    
    /**
     * <p>Setter for the field <code>singleton</code>.</p>
     *
     * @param singleton a boolean.
     * @return a {@link space.yizhu.record.aop.AopFactory} object.
     */
    public AopFactory setSingleton(boolean singleton) {
        this.singleton = singleton;
        return this;
    }

    /**
     * <p>isSingleton.</p>
     *
     * @return a boolean.
     */
    public boolean isSingleton() {
        return singleton;
    }

    
    /**
     * <p>Setter for the field <code>injectDepth</code>.</p>
     *
     * @param injectDepth a int.
     * @return a {@link space.yizhu.record.aop.AopFactory} object.
     */
    public AopFactory setInjectDepth(int injectDepth) {
        if (injectDepth <= 0) {
            throw new IllegalArgumentException("注入层数必须大于 0");
        }
        if (injectDepth > MAX_INJECT_DEPTH) {
            throw new IllegalArgumentException("为保障性能，注入层数不能超过 " + MAX_INJECT_DEPTH);
        }

        this.injectDepth = injectDepth;
        return this;
    }

    /**
     * <p>Getter for the field <code>injectDepth</code>.</p>
     *
     * @return a int.
     */
    public int getInjectDepth() {
        return injectDepth;
    }

    /**
     * <p>addSingletonObject.</p>
     *
     * @param singletonObject a {@link java.lang.Object} object.
     * @return a {@link space.yizhu.record.aop.AopFactory} object.
     */
    public AopFactory addSingletonObject(Object singletonObject) {
        if (singletonObject == null) {
            throw new IllegalArgumentException("singletonObject can not be null");
        }
        if (singletonObject instanceof Class) {
            throw new IllegalArgumentException("singletonObject can not be Class type");
        }

        Class<?> type = getUsefulClass(singletonObject.getClass());
        if (singletonCache.putIfAbsent(type, singletonObject) != null) {
            throw new RuntimeException("Singleton object already exists for type : " + type.getName());
        }

        return this;
    }

    /**
     * <p>addMapping.</p>
     *
     * @param from a {@link java.lang.Class} object.
     * @param to a {@link java.lang.Class} object.
     * @param <T> a T object.
     * @return a {@link space.yizhu.record.aop.AopFactory} object.
     */
    public synchronized <T> AopFactory addMapping(Class<T> from, Class<? extends T> to) {
        if (from == null || to == null) {
            throw new IllegalArgumentException("The parameter from and to can not be null");
        }

        if (mapping == null) {
            mapping = new HashMap<Class<?>, Class<?>>(128, 0.25F);
        } else if (mapping.containsKey(from)) {
            throw new RuntimeException("Class already mapped : " + from.getName());
        }

        mapping.put(from, to);
        return this;
    }

    /**
     * <p>addMapping.</p>
     *
     * @param from a {@link java.lang.Class} object.
     * @param to a {@link java.lang.String} object.
     * @param <T> a T object.
     * @return a {@link space.yizhu.record.aop.AopFactory} object.
     */
    public <T> AopFactory addMapping(Class<T> from, String to) {
        try {
            @SuppressWarnings("unchecked")
            Class<T> toClass = (Class<T>) Class.forName(to.trim());
            if (from.isAssignableFrom(toClass)) {
                return addMapping(from, toClass);
            } else {
                throw new IllegalArgumentException("The parameter \"to\" must be the subclass or implementation of the parameter \"from\"");
            }
        } catch (ReflectiveOperationException e) {
            throw new RuntimeException(e);
        }
    }

    
    /**
     * <p>getMappingClass.</p>
     *
     * @param from a {@link java.lang.Class} object.
     * @return a {@link java.lang.Class} object.
     */
    public Class<?> getMappingClass(Class<?> from) {
        if (mapping != null) {
            Class<?> ret = mapping.get(from);
            return ret != null ? ret : from;
        } else {
            return from;
        }
    }
}







