

package space.yizhu.record.plugin.activerecord;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.List;
import javax.sql.DataSource;

import space.yizhu.kits.StrKit;
import space.yizhu.record.plugin.IPlugin;
import space.yizhu.record.plugin.activerecord.cache.ICache;
import space.yizhu.record.plugin.activerecord.dialect.Dialect;
import space.yizhu.record.plugin.activerecord.sql.SqlKit;
import space.yizhu.record.plugin.activerecord.cache.EhCache;
import space.yizhu.record.plugin.activerecord.dialect.MysqlDialect;
import space.yizhu.record.template.Engine;
import space.yizhu.record.template.source.ISource;


/**
 * <p>ActiveRecordPlugin class.</p>
 *
 * @author yi
 * @version $Id: $Id
 */
public class ActiveRecordPlugin implements IPlugin {

    protected IDataSourceProvider dataSourceProvider = null;
    protected Boolean devMode = null;

    protected Config config = null;

    protected volatile boolean isStarted = false;
    protected List<Table> tableList = new ArrayList<Table>();

    /**
     * <p>Constructor for ActiveRecordPlugin.</p>
     *
     * @param configName a {@link java.lang.String} object.
     * @param dataSource a {@link javax.sql.DataSource} object.
     * @param transactionLevel a int.
     */
    public ActiveRecordPlugin(String configName, DataSource dataSource, int transactionLevel) {
        if (StrKit.isBlank(configName)) {
            throw new IllegalArgumentException("configName can not be blank");
        }
        if (dataSource == null) {
            throw new IllegalArgumentException("dataSource can not be null");
        }
        this.config = new Config(configName, dataSource, transactionLevel);
    }

    /**
     * <p>Constructor for ActiveRecordPlugin.</p>
     *
     * @param dataSource a {@link javax.sql.DataSource} object.
     */
    public ActiveRecordPlugin(DataSource dataSource) {
        this(DbConfig.MAIN_CONFIG_NAME, dataSource);
    }

    /**
     * <p>Constructor for ActiveRecordPlugin.</p>
     *
     * @param configName a {@link java.lang.String} object.
     * @param dataSource a {@link javax.sql.DataSource} object.
     */
    public ActiveRecordPlugin(String configName, DataSource dataSource) {
        this(configName, dataSource, DbConfig.DEFAULT_TRANSACTION_LEVEL);
    }

    /**
     * <p>Constructor for ActiveRecordPlugin.</p>
     *
     * @param dataSource a {@link javax.sql.DataSource} object.
     * @param transactionLevel a int.
     */
    public ActiveRecordPlugin(DataSource dataSource, int transactionLevel) {
        this(DbConfig.MAIN_CONFIG_NAME, dataSource, transactionLevel);
    }

    /**
     * <p>Constructor for ActiveRecordPlugin.</p>
     *
     * @param configName a {@link java.lang.String} object.
     * @param dataSourceProvider a {@link space.yizhu.record.plugin.activerecord.IDataSourceProvider} object.
     * @param transactionLevel a int.
     */
    public ActiveRecordPlugin(String configName, IDataSourceProvider dataSourceProvider, int transactionLevel) {
        if (StrKit.isBlank(configName)) {
            throw new IllegalArgumentException("configName can not be blank");
        }
        if (dataSourceProvider == null) {
            throw new IllegalArgumentException("dataSourceProvider can not be null");
        }
        this.dataSourceProvider = dataSourceProvider;
        this.config = new Config(configName, null, transactionLevel);
    }

    /**
     * <p>Constructor for ActiveRecordPlugin.</p>
     *
     * @param dataSourceProvider a {@link space.yizhu.record.plugin.activerecord.IDataSourceProvider} object.
     */
    public ActiveRecordPlugin(IDataSourceProvider dataSourceProvider) {
        this(DbConfig.MAIN_CONFIG_NAME, dataSourceProvider);
    }

    /**
     * <p>Constructor for ActiveRecordPlugin.</p>
     *
     * @param configName a {@link java.lang.String} object.
     * @param dataSourceProvider a {@link space.yizhu.record.plugin.activerecord.IDataSourceProvider} object.
     */
    public ActiveRecordPlugin(String configName, IDataSourceProvider dataSourceProvider) {
        this(configName, dataSourceProvider, DbConfig.DEFAULT_TRANSACTION_LEVEL);
    }

    /**
     * <p>Constructor for ActiveRecordPlugin.</p>
     *
     * @param dataSourceProvider a {@link space.yizhu.record.plugin.activerecord.IDataSourceProvider} object.
     * @param transactionLevel a int.
     */
    public ActiveRecordPlugin(IDataSourceProvider dataSourceProvider, int transactionLevel) {
        this(DbConfig.MAIN_CONFIG_NAME, dataSourceProvider, transactionLevel);
    }

    /**
     * <p>Constructor for ActiveRecordPlugin.</p>
     *
     * @param config a {@link space.yizhu.record.plugin.activerecord.Config} object.
     */
    public ActiveRecordPlugin(Config config) {
        if (config == null) {
            throw new IllegalArgumentException("Config can not be null");
        }
        this.config = config;
    }

    /**
     * <p>addMapping.</p>
     *
     * @param tableName a {@link java.lang.String} object.
     * @param primaryKey a {@link java.lang.String} object.
     * @param modelClass a {@link java.lang.Class} object.
     * @return a {@link space.yizhu.record.plugin.activerecord.ActiveRecordPlugin} object.
     */
    public ActiveRecordPlugin addMapping(String tableName, String primaryKey, Class<? extends Model<?>> modelClass) {
        tableList.add(new Table(tableName, primaryKey, modelClass));
        return this;
    }

    /**
     * <p>addMapping.</p>
     *
     * @param tableName a {@link java.lang.String} object.
     * @param modelClass a {@link java.lang.Class} object.
     * @return a {@link space.yizhu.record.plugin.activerecord.ActiveRecordPlugin} object.
     */
    public ActiveRecordPlugin addMapping(String tableName, Class<? extends Model<?>> modelClass) {
        tableList.add(new Table(tableName, modelClass));
        return this;
    }

    /**
     * <p>addSqlTemplate.</p>
     *
     * @param sqlTemplate a {@link java.lang.String} object.
     * @return a {@link space.yizhu.record.plugin.activerecord.ActiveRecordPlugin} object.
     */
    public ActiveRecordPlugin addSqlTemplate(String sqlTemplate) {
        config.sqlKit.addSqlTemplate(sqlTemplate);
        return this;
    }

    /**
     * <p>addSqlTemplate.</p>
     *
     * @param sqlTemplate a {@link space.yizhu.record.template.source.ISource} object.
     * @return a {@link space.yizhu.record.plugin.activerecord.ActiveRecordPlugin} object.
     */
    public ActiveRecordPlugin addSqlTemplate(ISource sqlTemplate) {
        config.sqlKit.addSqlTemplate(sqlTemplate);
        return this;
    }

    /**
     * <p>setBaseSqlTemplatePath.</p>
     *
     * @param baseSqlTemplatePath a {@link java.lang.String} object.
     * @return a {@link space.yizhu.record.plugin.activerecord.ActiveRecordPlugin} object.
     */
    public ActiveRecordPlugin setBaseSqlTemplatePath(String baseSqlTemplatePath) {
        config.sqlKit.setBaseSqlTemplatePath(baseSqlTemplatePath);
        return this;
    }

    /**
     * <p>getSqlKit.</p>
     *
     * @return a {@link space.yizhu.record.plugin.activerecord.sql.SqlKit} object.
     */
    public SqlKit getSqlKit() {
        return config.sqlKit;
    }

    /**
     * <p>getEngine.</p>
     *
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public Engine getEngine() {
        return getSqlKit().getEngine();
    }

    
    /**
     * <p>setTransactionLevel.</p>
     *
     * @param transactionLevel a int.
     * @return a {@link space.yizhu.record.plugin.activerecord.ActiveRecordPlugin} object.
     */
    public ActiveRecordPlugin setTransactionLevel(int transactionLevel) {
        config.setTransactionLevel(transactionLevel);
        return this;
    }

    /**
     * <p>setCache.</p>
     *
     * @param cache a {@link space.yizhu.record.plugin.activerecord.cache.ICache} object.
     * @return a {@link space.yizhu.record.plugin.activerecord.ActiveRecordPlugin} object.
     */
    public ActiveRecordPlugin setCache(ICache cache) {
        if (cache == null) {
            throw new IllegalArgumentException("cache can not be null");
        }
        config.cache = cache;
        return this;
    }

    /**
     * <p>setShowSql.</p>
     *
     * @param showSql a boolean.
     * @return a {@link space.yizhu.record.plugin.activerecord.ActiveRecordPlugin} object.
     */
    public ActiveRecordPlugin setShowSql(boolean showSql) {
        config.showSql = showSql;
        return this;
    }

    /**
     * <p>Setter for the field <code>devMode</code>.</p>
     *
     * @param devMode a boolean.
     * @return a {@link space.yizhu.record.plugin.activerecord.ActiveRecordPlugin} object.
     */
    public ActiveRecordPlugin setDevMode(boolean devMode) {
        this.devMode = devMode;
        config.setDevMode(devMode);
        return this;
    }

    /**
     * <p>Getter for the field <code>devMode</code>.</p>
     *
     * @return a {@link java.lang.Boolean} object.
     */
    public Boolean getDevMode() {
        return devMode;
    }

    /**
     * <p>setDialect.</p>
     *
     * @param dialect a {@link space.yizhu.record.plugin.activerecord.dialect.Dialect} object.
     * @return a {@link space.yizhu.record.plugin.activerecord.ActiveRecordPlugin} object.
     */
    public ActiveRecordPlugin setDialect(Dialect dialect) {
        if (dialect == null) {
            throw new IllegalArgumentException("dialect can not be null");
        }
        config.dialect = dialect;
        if (config.transactionLevel == Connection.TRANSACTION_REPEATABLE_READ && dialect.isOracle()) {
            
            config.transactionLevel = Connection.TRANSACTION_READ_COMMITTED;
        }
        return this;
    }

    /**
     * <p>setContainerFactory.</p>
     *
     * @param containerFactory a {@link space.yizhu.record.plugin.activerecord.IContainerFactory} object.
     * @return a {@link space.yizhu.record.plugin.activerecord.ActiveRecordPlugin} object.
     */
    public ActiveRecordPlugin setContainerFactory(IContainerFactory containerFactory) {
        if (containerFactory == null) {
            throw new IllegalArgumentException("containerFactory can not be null");
        }
        config.containerFactory = containerFactory;
        return this;
    }

    /**
     * <p>setDbProFactory.</p>
     *
     * @param dbProFactory a {@link space.yizhu.record.plugin.activerecord.IDbProFactory} object.
     * @return a {@link space.yizhu.record.plugin.activerecord.ActiveRecordPlugin} object.
     */
    public ActiveRecordPlugin setDbProFactory(IDbProFactory dbProFactory) {
        if (dbProFactory == null) {
            throw new IllegalArgumentException("dbProFactory can not be null");
        }
        config.dbProFactory = dbProFactory;
        return this;
    }

    
    /**
     * <p>setPrimaryKey.</p>
     *
     * @param tableName a {@link java.lang.String} object.
     * @param primaryKey a {@link java.lang.String} object.
     */
    public void setPrimaryKey(String tableName, String primaryKey) {
        for (Table table : tableList) {
            if (table.getName().equalsIgnoreCase(tableName.trim())) {
                table.setPrimaryKey(primaryKey);
            }
        }
    }

    /**
     * <p>start.</p>
     *
     * @return a boolean.
     */
    public boolean start() {
        if (isStarted) {
            return true;
        }
        if (config.dataSource == null && dataSourceProvider != null) {
            config.dataSource = dataSourceProvider.getDataSource();
        }
        if (config.dataSource == null) {
            throw new RuntimeException("ActiveRecord start error: ActiveRecordPlugin need DataSource or DataSourceProvider");
        }

        config.sqlKit.parseSqlTemplate();

        new TableBuilder().build(tableList, config);
        DbConfig.addConfig(config);
        isStarted = true;
        return true;
    }

    /**
     * <p>stop.</p>
     *
     * @return a boolean.
     */
    public boolean stop() {
        DbConfig.removeConfig(config.getName());
        isStarted = false;
        return true;
    }

    
    /**
     * <p>useAsDataTransfer.</p>
     *
     * @param dialect a {@link space.yizhu.record.plugin.activerecord.dialect.Dialect} object.
     * @param containerFactory a {@link space.yizhu.record.plugin.activerecord.IContainerFactory} object.
     * @param cache a {@link space.yizhu.record.plugin.activerecord.cache.ICache} object.
     */
    public static void useAsDataTransfer(Dialect dialect, IContainerFactory containerFactory, ICache cache) {
        if (dialect == null) {
            throw new IllegalArgumentException("dialect can not be null");
        }
        if (containerFactory == null) {
            throw new IllegalArgumentException("containerFactory can not be null");
        }
        if (cache == null) {
            throw new IllegalArgumentException("cache can not be null");
        }
        ActiveRecordPlugin arp = new ActiveRecordPlugin(new NullDataSource());
        arp.setDialect(dialect);
        arp.setContainerFactory(containerFactory);
        arp.setCache(cache);
        arp.start();
        DbConfig.brokenConfig = arp.config;
    }

    
    /**
     * <p>useAsDataTransfer.</p>
     *
     * @param containerFactory a {@link space.yizhu.record.plugin.activerecord.IContainerFactory} object.
     */
    public static void useAsDataTransfer(IContainerFactory containerFactory) {
        useAsDataTransfer(new MysqlDialect(), containerFactory, new EhCache());
    }

    
    /**
     * <p>useAsDataTransfer.</p>
     *
     * @param dialect a {@link space.yizhu.record.plugin.activerecord.dialect.Dialect} object.
     * @param containerFactory a {@link space.yizhu.record.plugin.activerecord.IContainerFactory} object.
     */
    public static void useAsDataTransfer(Dialect dialect, IContainerFactory containerFactory) {
        useAsDataTransfer(dialect, containerFactory, new EhCache());
    }

    
    /**
     * <p>useAsDataTransfer.</p>
     *
     * @param dialect a {@link space.yizhu.record.plugin.activerecord.dialect.Dialect} object.
     */
    public static void useAsDataTransfer(Dialect dialect) {
        useAsDataTransfer(dialect, IContainerFactory.defaultContainerFactory, new EhCache());
    }

    
    /**
     * <p>useAsDataTransfer.</p>
     */
    public static void useAsDataTransfer() {
        useAsDataTransfer(new MysqlDialect(), IContainerFactory.defaultContainerFactory, new EhCache());
    }

    /**
     * <p>Getter for the field <code>config</code>.</p>
     *
     * @return a {@link space.yizhu.record.plugin.activerecord.Config} object.
     */
    public Config getConfig() {
        return config;
    }
}






