

package space.yizhu.record.plugin.activerecord;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import javax.sql.DataSource;

import space.yizhu.kits.LogKit;
import space.yizhu.kits.StrKit;
import space.yizhu.record.plugin.activerecord.cache.EhCache;
import space.yizhu.record.plugin.activerecord.cache.ICache;
import space.yizhu.record.plugin.activerecord.dialect.Dialect;
import space.yizhu.record.plugin.activerecord.dialect.MysqlDialect;
import space.yizhu.record.plugin.activerecord.sql.SqlKit;

/**
 * <p>Config class.</p>
 *
 * @author yi
 * @version $Id: $Id
 */
public class Config {
    private final ThreadLocal<Connection> threadLocal = new ThreadLocal<Connection>();

    String name;
    DataSource dataSource;

    Dialect dialect;
    boolean showSql;
    boolean devMode;
    int transactionLevel;
    IContainerFactory containerFactory;
    IDbProFactory dbProFactory = IDbProFactory.defaultDbProFactory;
    ICache cache;

    SqlKit sqlKit;

    
    Config(String name, DataSource dataSource, int transactionLevel) {
        init(name, dataSource, new MysqlDialect(), false, false, transactionLevel, IContainerFactory.defaultContainerFactory, new EhCache());
    }

    
    /**
     * <p>Constructor for Config.</p>
     *
     * @param name a {@link java.lang.String} object.
     * @param dataSource a {@link javax.sql.DataSource} object.
     * @param dialect a {@link space.yizhu.record.plugin.activerecord.dialect.Dialect} object.
     * @param showSql a boolean.
     * @param devMode a boolean.
     * @param transactionLevel a int.
     * @param containerFactory a {@link space.yizhu.record.plugin.activerecord.IContainerFactory} object.
     * @param cache a {@link space.yizhu.record.plugin.activerecord.cache.ICache} object.
     */
    public Config(String name, DataSource dataSource, Dialect dialect, boolean showSql, boolean devMode, int transactionLevel, IContainerFactory containerFactory, ICache cache) {
        if (dataSource == null) {
            throw new IllegalArgumentException("DataSource can not be null");
        }
        init(name, dataSource, dialect, showSql, devMode, transactionLevel, containerFactory, cache);
    }

    private void init(String name, DataSource dataSource, Dialect dialect, boolean showSql, boolean devMode, int transactionLevel, IContainerFactory containerFactory, ICache cache) {
        if (StrKit.isBlank(name)) {
            throw new IllegalArgumentException("Config name can not be blank");
        }
        if (dialect == null) {
            throw new IllegalArgumentException("Dialect can not be null");
        }
        if (containerFactory == null) {
            throw new IllegalArgumentException("ContainerFactory can not be null");
        }
        if (cache == null) {
            throw new IllegalArgumentException("Cache can not be null");
        }

        this.name = name.trim();
        this.dataSource = dataSource;
        this.dialect = dialect;
        this.showSql = showSql;
        this.devMode = devMode;
        
        this.setTransactionLevel(transactionLevel);
        this.containerFactory = containerFactory;
        this.cache = cache;

        this.sqlKit = new SqlKit(this.name, this.devMode);
    }

    
    /**
     * <p>Constructor for Config.</p>
     *
     * @param name a {@link java.lang.String} object.
     * @param dataSource a {@link javax.sql.DataSource} object.
     */
    public Config(String name, DataSource dataSource) {
        this(name, dataSource, new MysqlDialect());
    }

    
    /**
     * <p>Constructor for Config.</p>
     *
     * @param name a {@link java.lang.String} object.
     * @param dataSource a {@link javax.sql.DataSource} object.
     * @param dialect a {@link space.yizhu.record.plugin.activerecord.dialect.Dialect} object.
     */
    public Config(String name, DataSource dataSource, Dialect dialect) {
        this(name, dataSource, dialect, false, false, DbConfig.DEFAULT_TRANSACTION_LEVEL, IContainerFactory.defaultContainerFactory, new EhCache());
    }

    private Config() {

    }

    void setDevMode(boolean devMode) {
        this.devMode = devMode;
        this.sqlKit.setDevMode(devMode);
    }

    void setTransactionLevel(int transactionLevel) {
        int t = transactionLevel;
        if (t != 0 && t != 1 && t != 2 && t != 4 && t != 8) {
            throw new IllegalArgumentException("The transactionLevel only be 0, 1, 2, 4, 8");
        }
        this.transactionLevel = transactionLevel;
    }

    
    static Config createBrokenConfig() {
        Config ret = new Config();
        ret.dialect = new MysqlDialect();
        ret.showSql = false;
        ret.devMode = false;
        ret.transactionLevel = DbConfig.DEFAULT_TRANSACTION_LEVEL;
        ret.containerFactory = IContainerFactory.defaultContainerFactory;
        ret.cache = new EhCache();
        return ret;
    }

    /**
     * <p>Getter for the field <code>name</code>.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getName() {
        return name;
    }

    /**
     * <p>Getter for the field <code>sqlKit</code>.</p>
     *
     * @return a {@link space.yizhu.record.plugin.activerecord.sql.SqlKit} object.
     */
    public SqlKit getSqlKit() {
        return sqlKit;
    }

    /**
     * <p>Getter for the field <code>dialect</code>.</p>
     *
     * @return a {@link space.yizhu.record.plugin.activerecord.dialect.Dialect} object.
     */
    public Dialect getDialect() {
        return dialect;
    }

    /**
     * <p>Getter for the field <code>cache</code>.</p>
     *
     * @return a {@link space.yizhu.record.plugin.activerecord.cache.ICache} object.
     */
    public ICache getCache() {
        return cache;
    }

    /**
     * <p>Getter for the field <code>transactionLevel</code>.</p>
     *
     * @return a int.
     */
    public int getTransactionLevel() {
        return transactionLevel;
    }

    /**
     * <p>Getter for the field <code>dataSource</code>.</p>
     *
     * @return a {@link javax.sql.DataSource} object.
     */
    public DataSource getDataSource() {
        return dataSource;
    }

    /**
     * <p>Getter for the field <code>containerFactory</code>.</p>
     *
     * @return a {@link space.yizhu.record.plugin.activerecord.IContainerFactory} object.
     */
    public IContainerFactory getContainerFactory() {
        return containerFactory;
    }

    /**
     * <p>Getter for the field <code>dbProFactory</code>.</p>
     *
     * @return a {@link space.yizhu.record.plugin.activerecord.IDbProFactory} object.
     */
    public IDbProFactory getDbProFactory() {
        return dbProFactory;
    }

    /**
     * <p>isShowSql.</p>
     *
     * @return a boolean.
     */
    public boolean isShowSql() {
        return showSql;
    }

    /**
     * <p>isDevMode.</p>
     *
     * @return a boolean.
     */
    public boolean isDevMode() {
        return devMode;
    }

    

    
    /**
     * <p>setThreadLocalConnection.</p>
     *
     * @param connection a {@link java.sql.Connection} object.
     */
    public void setThreadLocalConnection(Connection connection) {
        threadLocal.set(connection);
    }

    /**
     * <p>removeThreadLocalConnection.</p>
     */
    public void removeThreadLocalConnection() {
        threadLocal.remove();
    }

    
    /**
     * <p>getConnection.</p>
     *
     * @return a {@link java.sql.Connection} object.
     * @throws java.sql.SQLException if any.
     */
    public Connection getConnection() throws SQLException {
        Connection conn = threadLocal.get();
        if (conn != null)
            return conn;
        return showSql ? new SqlReporter(dataSource.getConnection()).getConnection() : dataSource.getConnection();
    }

    
    /**
     * <p>getThreadLocalConnection.</p>
     *
     * @return a {@link java.sql.Connection} object.
     */
    public Connection getThreadLocalConnection() {
        return threadLocal.get();
    }

    
    /**
     * <p>isInTransaction.</p>
     *
     * @return a boolean.
     */
    public boolean isInTransaction() {
        return threadLocal.get() != null;
    }

    
    /**
     * <p>close.</p>
     *
     * @param rs a {@link java.sql.ResultSet} object.
     * @param st a {@link java.sql.Statement} object.
     * @param conn a {@link java.sql.Connection} object.
     */
    public void close(ResultSet rs, Statement st, Connection conn) {
        if (rs != null) {
            try {
                rs.close();
            } catch (SQLException e) {
                LogKit.error(e.getMessage(), e);
            }
        }
        if (st != null) {
            try {
                st.close();
            } catch (SQLException e) {
                LogKit.error(e.getMessage(), e);
            }
        }

        if (threadLocal.get() == null) {    
            if (conn != null) {
                try {
                    conn.close();
                } catch (SQLException e) {
                    throw new ActiveRecordException(e);
                }
            }
        }
    }

    /**
     * <p>close.</p>
     *
     * @param st a {@link java.sql.Statement} object.
     * @param conn a {@link java.sql.Connection} object.
     */
    public void close(Statement st, Connection conn) {
        if (st != null) {
            try {
                st.close();
            } catch (SQLException e) {
                LogKit.error(e.getMessage(), e);
            }
        }

        if (threadLocal.get() == null) {    
            if (conn != null) {
                try {
                    conn.close();
                } catch (SQLException e) {
                    throw new ActiveRecordException(e);
                }
            }
        }
    }

    /**
     * <p>close.</p>
     *
     * @param conn a {@link java.sql.Connection} object.
     */
    public void close(Connection conn) {
        if (threadLocal.get() == null)        
            if (conn != null)
                try {
                    conn.close();
                } catch (SQLException e) {
                    throw new ActiveRecordException(e);
                }
    }
}



