

package space.yizhu.record.plugin.activerecord.dialect;

import space.yizhu.record.plugin.activerecord.*;
import space.yizhu.record.plugin.activerecord.*;
import space.yizhu.record.plugin.activerecord.builder.KeepByteAndShortModelBuilder;
import space.yizhu.record.plugin.activerecord.builder.KeepByteAndShortRecordBuilder;
import org.postgresql.util.PGobject;
import space.yizhu.kits.DateKit;

import java.math.BigInteger;
import java.sql.*;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;


/**
 * <p>Abstract Dialect class.</p>
 *
 * @author yi
 * @version $Id: $Id
 */
public abstract class Dialect {

    
    protected boolean keepByteAndShort = false;
    protected ModelBuilder modelBuilder = ModelBuilder.me;
    protected RecordBuilder recordBuilder = RecordBuilder.me;

    
    /**
     * <p>forTableBuilderDoBuild.</p>
     *
     * @param tableName a {@link java.lang.String} object.
     * @return a {@link java.lang.String} object.
     */
    public abstract String forTableBuilderDoBuild(String tableName);

    /**
     * <p>forPaginate.</p>
     *
     * @param pageNumber a int.
     * @param pageSize a int.
     * @param findSql a {@link java.lang.StringBuilder} object.
     * @return a {@link java.lang.String} object.
     */
    public abstract String forPaginate(int pageNumber, int pageSize, StringBuilder findSql);

    
    /**
     * <p>forModelFindById.</p>
     *
     * @param table a {@link space.yizhu.record.plugin.activerecord.Table} object.
     * @param columns a {@link java.lang.String} object.
     * @return a {@link java.lang.String} object.
     */
    public abstract String forModelFindById(Table table, String columns);

    /**
     * <p>forModelDeleteById.</p>
     *
     * @param table a {@link space.yizhu.record.plugin.activerecord.Table} object.
     * @return a {@link java.lang.String} object.
     */
    public abstract String forModelDeleteById(Table table);

    /**
     * <p>forModelSave.</p>
     *
     * @param table a {@link space.yizhu.record.plugin.activerecord.Table} object.
     * @param attrs a {@link java.util.Map} object.
     * @param sql a {@link java.lang.StringBuilder} object.
     * @param paras a {@link java.util.List} object.
     */
    public abstract void forModelSave(Table table, Map<String, Object> attrs, StringBuilder sql, List<Object> paras);

    /**
     * <p>forModelUpdate.</p>
     *
     * @param table a {@link space.yizhu.record.plugin.activerecord.Table} object.
     * @param attrs a {@link java.util.Map} object.
     * @param modifyFlag a {@link java.util.Set} object.
     * @param sql a {@link java.lang.StringBuilder} object.
     * @param paras a {@link java.util.List} object.
     */
    public abstract void forModelUpdate(Table table, Map<String, Object> attrs, Set<String> modifyFlag, StringBuilder sql, List<Object> paras);

    
    /**
     * <p>forDbFindById.</p>
     *
     * @param tableName a {@link java.lang.String} object.
     * @param pKeys an array of {@link java.lang.String} objects.
     * @return a {@link java.lang.String} object.
     */
    public abstract String forDbFindById(String tableName, String[] pKeys);

    /**
     * <p>forDbDeleteById.</p>
     *
     * @param tableName a {@link java.lang.String} object.
     * @param pKeys an array of {@link java.lang.String} objects.
     * @return a {@link java.lang.String} object.
     */
    public abstract String forDbDeleteById(String tableName, String[] pKeys);

    /**
     * <p>forDbSave.</p>
     *
     * @param tableName a {@link java.lang.String} object.
     * @param pKeys an array of {@link java.lang.String} objects.
     * @param record a {@link space.yizhu.record.plugin.activerecord.Record} object.
     * @param sql a {@link java.lang.StringBuilder} object.
     * @param paras a {@link java.util.List} object.
     */
    public abstract void forDbSave(String tableName, String[] pKeys, Record record, StringBuilder sql, List<Object> paras);

    /**
     * <p>forDbUpdate.</p>
     *
     * @param tableName a {@link java.lang.String} object.
     * @param pKeys an array of {@link java.lang.String} objects.
     * @param ids an array of {@link java.lang.Object} objects.
     * @param record a {@link space.yizhu.record.plugin.activerecord.Record} object.
     * @param sql a {@link java.lang.StringBuilder} object.
     * @param paras a {@link java.util.List} object.
     */
    public abstract void forDbUpdate(String tableName, String[] pKeys, Object[] ids, Record record, StringBuilder sql, List<Object> paras);

    /**
     * <p>forFindAll.</p>
     *
     * @param tableName a {@link java.lang.String} object.
     * @return a {@link java.lang.String} object.
     */
    public String forFindAll(String tableName) {
        return "select * from " + tableName;
    }

    
    /**
     * <p>isKeepByteAndShort.</p>
     *
     * @return a boolean.
     */
    public boolean isKeepByteAndShort() {
        return keepByteAndShort;
    }

    
    /**
     * <p>Setter for the field <code>keepByteAndShort</code>.</p>
     *
     * @param keepByteAndShort a boolean.
     * @return a {@link space.yizhu.record.plugin.activerecord.dialect.Dialect} object.
     */
    public Dialect setKeepByteAndShort(boolean keepByteAndShort) {
        this.keepByteAndShort = keepByteAndShort;
        
        if (keepByteAndShort) {
            if (modelBuilder.getClass() == ModelBuilder.class) {
                modelBuilder = KeepByteAndShortModelBuilder.me;
            }
            if (recordBuilder.getClass() == RecordBuilder.class) {
                recordBuilder = KeepByteAndShortRecordBuilder.me;
            }
        } else {
            if (modelBuilder.getClass() == KeepByteAndShortModelBuilder.class) {
                modelBuilder = ModelBuilder.me;
            }
            if (recordBuilder.getClass() == KeepByteAndShortRecordBuilder.class) {
                recordBuilder = RecordBuilder.me;
            }
        }
        return this;
    }

    
    /**
     * <p>Setter for the field <code>modelBuilder</code>.</p>
     *
     * @param modelBuilder a {@link space.yizhu.record.plugin.activerecord.ModelBuilder} object.
     * @return a {@link space.yizhu.record.plugin.activerecord.dialect.Dialect} object.
     */
    public Dialect setModelBuilder(ModelBuilder modelBuilder) {
        this.modelBuilder = modelBuilder;
        return this;
    }

    
    /**
     * <p>Setter for the field <code>recordBuilder</code>.</p>
     *
     * @param recordBuilder a {@link space.yizhu.record.plugin.activerecord.RecordBuilder} object.
     * @return a {@link space.yizhu.record.plugin.activerecord.dialect.Dialect} object.
     */
    public Dialect setRecordBuilder(RecordBuilder recordBuilder) {
        this.recordBuilder = recordBuilder;
        return this;
    }

    /**
     * <p>buildModelList.</p>
     *
     * @param rs a {@link java.sql.ResultSet} object.
     * @param modelClass a {@link java.lang.Class} object.
     * @param <T> a T object.
     * @return a {@link java.util.List} object.
     * @throws java.sql.SQLException if any.
     * @throws java.lang.ReflectiveOperationException if any.
     */
    @SuppressWarnings("rawtypes")
    public <T> List<T> buildModelList(ResultSet rs, Class<? extends Model> modelClass) throws SQLException, ReflectiveOperationException {
        return modelBuilder.build(rs, modelClass);
    }

    /**
     * <p>buildRecordList.</p>
     *
     * @param config a {@link space.yizhu.record.plugin.activerecord.Config} object.
     * @param rs a {@link java.sql.ResultSet} object.
     * @return a {@link java.util.List} object.
     * @throws java.sql.SQLException if any.
     */
    public List<Record> buildRecordList(Config config, ResultSet rs) throws SQLException {
        return recordBuilder.build(config, rs);
    }

    
    /**
     * <p>getModelGeneratedKey.</p>
     *
     * @param model a {@link space.yizhu.record.plugin.activerecord.Model} object.
     * @param pst a {@link java.sql.PreparedStatement} object.
     * @param table a {@link space.yizhu.record.plugin.activerecord.Table} object.
     * @throws java.sql.SQLException if any.
     */
    public void getModelGeneratedKey(Model<?> model, PreparedStatement pst, Table table) throws SQLException {
        String[] pKeys = table.getPrimaryKey();
        ResultSet rs = pst.getGeneratedKeys();
        for (String pKey : pKeys) {
            if (model.get(pKey) == null || isOracle()) {
                if (rs.next()) {
                    Class<?> colType = table.getColumnType(pKey);
                    if (colType != null) {    
                        if (colType == Integer.class || colType == int.class) {
                            model.set(pKey, rs.getInt(1));
                        } else if (colType == Long.class || colType == long.class) {
                            model.set(pKey, rs.getLong(1));
                        } else if (colType == BigInteger.class) {
                            processGeneratedBigIntegerKey(model, pKey, rs.getObject(1));
                        } else {
                            model.set(pKey, rs.getObject(1));    
                        }
                    }
                }
            }
        }
        rs.close();
    }

    
    /**
     * <p>processGeneratedBigIntegerKey.</p>
     *
     * @param model a {@link space.yizhu.record.plugin.activerecord.Model} object.
     * @param pKey a {@link java.lang.String} object.
     * @param v a {@link java.lang.Object} object.
     */
    protected void processGeneratedBigIntegerKey(Model<?> model, String pKey, Object v) {
        if (v instanceof BigInteger) {
            model.set(pKey, (BigInteger) v);
        } else if (v instanceof Number) {
            Number n = (Number) v;
            model.set(pKey, BigInteger.valueOf(n.longValue()));
        } else {
            model.set(pKey, v);
        }
    }

    
    /**
     * <p>getRecordGeneratedKey.</p>
     *
     * @param pst a {@link java.sql.PreparedStatement} object.
     * @param record a {@link space.yizhu.record.plugin.activerecord.Record} object.
     * @param pKeys an array of {@link java.lang.String} objects.
     * @throws java.sql.SQLException if any.
     */
    public void getRecordGeneratedKey(PreparedStatement pst, Record record, String[] pKeys) throws SQLException {
        ResultSet rs = pst.getGeneratedKeys();
        for (String pKey : pKeys) {
            if (record.get(pKey) == null || isOracle()) {
                if (rs.next()) {
                    record.set(pKey, rs.getObject(1));    
                }
            }
        }
        rs.close();
    }

    /**
     * <p>isOracle.</p>
     *
     * @return a boolean.
     */
    public boolean isOracle() {
        return false;
    }

    /**
     * <p>isTakeOverDbPaginate.</p>
     *
     * @return a boolean.
     */
    public boolean isTakeOverDbPaginate() {
        return false;
    }

    /**
     * <p>takeOverDbPaginate.</p>
     *
     * @param conn a {@link java.sql.Connection} object.
     * @param pageNumber a int.
     * @param pageSize a int.
     * @param isGroupBySql a {@link java.lang.Boolean} object.
     * @param totalRowSql a {@link java.lang.String} object.
     * @param findSql a {@link java.lang.StringBuilder} object.
     * @param paras a {@link java.lang.Object} object.
     * @return a {@link space.yizhu.record.plugin.activerecord.Page} object.
     * @throws java.sql.SQLException if any.
     */
    public Page<Record> takeOverDbPaginate(Connection conn, int pageNumber, int pageSize, Boolean isGroupBySql, String totalRowSql, StringBuilder findSql, Object... paras) throws SQLException {
        throw new RuntimeException("You should implements this method in " + getClass().getName());
    }

    /**
     * <p>isTakeOverModelPaginate.</p>
     *
     * @return a boolean.
     */
    public boolean isTakeOverModelPaginate() {
        return false;
    }

    /**
     * <p>takeOverModelPaginate.</p>
     *
     * @param conn a {@link java.sql.Connection} object.
     * @param modelClass a {@link java.lang.Class} object.
     * @param pageNumber a int.
     * @param pageSize a int.
     * @param isGroupBySql a {@link java.lang.Boolean} object.
     * @param totalRowSql a {@link java.lang.String} object.
     * @param findSql a {@link java.lang.StringBuilder} object.
     * @param paras a {@link java.lang.Object} object.
     * @return a {@link space.yizhu.record.plugin.activerecord.Page} object.
     * @throws java.lang.Exception if any.
     */
    @SuppressWarnings("rawtypes")
    public Page takeOverModelPaginate(Connection conn, Class<? extends Model> modelClass, int pageNumber, int pageSize, Boolean isGroupBySql, String totalRowSql, StringBuilder findSql, Object... paras) throws Exception {
        throw new RuntimeException("You should implements this method in " + getClass().getName());
    }

    /**
     * <p>fillStatement.</p>
     *
     * @param pst a {@link java.sql.PreparedStatement} object.
     * @param paras a {@link java.util.List} object.
     * @throws java.sql.SQLException if any.
     */
    public void fillStatement(PreparedStatement pst, List<Object> paras) throws SQLException {
        for (int i = 0, size = paras.size(); i < size; i++) {
            pst.setObject(i + 1, paras.get(i));
        }
    }

    /**
     * <p>fillStatement.</p>
     *
     * @param pst a {@link java.sql.PreparedStatement} object.
     * @param paras a {@link java.lang.Object} object.
     * @throws java.sql.SQLException if any.
     */
    public void fillStatement(PreparedStatement pst, Object... paras) throws SQLException {
        for (int i = 0; i < paras.length; i++) {
            pst.setObject(i + 1, paras[i]);
        }
    }

    /**
     * <p>getDefaultPrimaryKey.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getDefaultPrimaryKey() {
        return "id";
    }

    /**
     * <p>isPrimaryKey.</p>
     *
     * @param colName a {@link java.lang.String} object.
     * @param pKeys an array of {@link java.lang.String} objects.
     * @return a boolean.
     */
    public boolean isPrimaryKey(String colName, String[] pKeys) {
        for (String pKey : pKeys) {
            if (colName.equalsIgnoreCase(pKey)) {
                return true;
            }
        }
        return false;
    }

    
    /**
     * <p>trimPrimaryKeys.</p>
     *
     * @param pKeys an array of {@link java.lang.String} objects.
     */
    public void trimPrimaryKeys(String[] pKeys) {
        for (int i = 0; i < pKeys.length; i++) {
            pKeys[i] = pKeys[i].trim();
        }
    }

    /**
     * <p>replaceOrderBy.</p>
     *
     * @param sql a {@link java.lang.String} object.
     * @return a {@link java.lang.String} object.
     */
    public String replaceOrderBy(String sql) {
        return Holder.ORDER_BY_PATTERN.matcher(sql).replaceAll("");
    }

    
    /**
     * <p>fillStatementHandleDateType.</p>
     *
     * @param pst a {@link java.sql.PreparedStatement} object.
     * @param paras a {@link java.util.List} object.
     * @throws java.sql.SQLException if any.
     */
    protected void fillStatementHandleDateType(PreparedStatement pst, List<Object> paras) throws SQLException {
        for (int i = 0, size = paras.size(); i < size; i++) {
            Object value = paras.get(i);
            doStatementType(pst, i, value);
        }
    }

    private void doStatementType(PreparedStatement pst, int i, Object value) throws SQLException {
        if (value instanceof java.util.Date) {
            if (value instanceof Date) {
                pst.setDate(i + 1, (Date) value);
            } else if (value instanceof Timestamp) {
                pst.setTimestamp(i + 1, (Timestamp) value);
            } else {
                
                java.util.Date d = (java.util.Date) value;
                pst.setTimestamp(i + 1, new Timestamp(d.getTime()));
            }
        } else if (value != null &&
                value.toString().length() - value.toString().replaceAll("-", "").length() == 2
                &&
                value.toString().length() - value.toString().replaceAll(" ", "").length() == 1
                &&
                value.toString().length() - value.toString().replaceAll(":", "").length() == 2) {
            
            pst.setTimestamp(i + 1, DateKit.string2Timestamp(value.toString()));
        } else if (value != null && ((value.toString().startsWith("[") && value.toString().endsWith("]")) || (value.toString().startsWith("{") && value.toString().endsWith("}")))) {
            PGobject jsonObject = new PGobject();
            jsonObject.setType("json");
            jsonObject.setValue(String.valueOf(value));
            pst.setObject(i + 1, jsonObject);
        } else {
            pst.setObject(i + 1, value);
        }
    }

    
    /**
     * <p>fillStatementHandleDateType.</p>
     *
     * @param pst a {@link java.sql.PreparedStatement} object.
     * @param paras a {@link java.lang.Object} object.
     * @throws java.sql.SQLException if any.
     */
    protected void fillStatementHandleDateType(PreparedStatement pst, Object... paras) throws SQLException {
        for (int i = 0; i < paras.length; i++) {
            Object value = paras[i];

            doStatementType(pst, i, value);
        }
    }

    protected static class Holder {
        
        private static final Pattern ORDER_BY_PATTERN = Pattern.compile(
                "order\\s+by\\s+[^,\\s]+(\\s+asc|\\s+desc)?(\\s*,\\s*[^,\\s]+(\\s+asc|\\s+desc)?)*",
                Pattern.CASE_INSENSITIVE | Pattern.MULTILINE);
    }
}






