

package space.yizhu.record.plugin.activerecord.generator;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import javax.sql.DataSource;

import space.yizhu.kits.LogKit;
import space.yizhu.kits.StrKit;


/**
 * <p>DataDictionaryGenerator class.</p>
 *
 * @author yi
 * @version $Id: $Id
 */
public class DataDictionaryGenerator {

    protected DataSource dataSource;
    protected String dataDictionaryOutputDir;
    protected String dataDictionaryFileName = "_DataDictionary.txt";

    /**
     * <p>Constructor for DataDictionaryGenerator.</p>
     *
     * @param dataSource a {@link javax.sql.DataSource} object.
     * @param dataDictionaryOutputDir a {@link java.lang.String} object.
     */
    public DataDictionaryGenerator(DataSource dataSource, String dataDictionaryOutputDir) {
        this.dataSource = dataSource;
        this.dataDictionaryOutputDir = dataDictionaryOutputDir;
    }

    /**
     * <p>Setter for the field <code>dataDictionaryOutputDir</code>.</p>
     *
     * @param dataDictionaryOutputDir a {@link java.lang.String} object.
     */
    public void setDataDictionaryOutputDir(String dataDictionaryOutputDir) {
        if (StrKit.notBlank(dataDictionaryOutputDir)) {
            this.dataDictionaryOutputDir = dataDictionaryOutputDir;
        }
    }

    /**
     * <p>Getter for the field <code>dataDictionaryOutputDir</code>.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getDataDictionaryOutputDir() {
        return dataDictionaryOutputDir;
    }

    /**
     * <p>Setter for the field <code>dataDictionaryFileName</code>.</p>
     *
     * @param dataDictionaryFileName a {@link java.lang.String} object.
     */
    public void setDataDictionaryFileName(String dataDictionaryFileName) {
        if (StrKit.notBlank(dataDictionaryFileName)) {
            this.dataDictionaryFileName = dataDictionaryFileName;
        }
    }

    /**
     * <p>Getter for the field <code>dataDictionaryFileName</code>.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getDataDictionaryFileName() {
        return dataDictionaryFileName;
    }

    /**
     * <p>generate.</p>
     *
     * @param tableMetas a {@link java.util.List} object.
     */
    public void generate(List<TableMeta> tableMetas) {
        System.out.println("Generate DataDictionary file ...");
        System.out.println("Data Dictionary Output Dir: " + dataDictionaryOutputDir);
        rebuildColumnMetas(tableMetas);

        StringBuilder ret = new StringBuilder();
        for (TableMeta tableMeta : tableMetas) {
            generateTable(tableMeta, ret);
        }

        writeToFile(ret.toString());
    }

    /**
     * <p>generateTable.</p>
     *
     * @param tableMeta a {@link space.yizhu.record.plugin.activerecord.generator.TableMeta} object.
     * @param ret a {@link java.lang.StringBuilder} object.
     */
    protected void generateTable(TableMeta tableMeta, StringBuilder ret) {
        ret.append("Table: ").append(tableMeta.name);
        if (StrKit.notBlank(tableMeta.remarks)) {
            ret.append("\tRemarks: ").append(tableMeta.remarks);
        }
        ret.append("\n");

        String sparateLine = genSeparateLine(tableMeta);
        ret.append(sparateLine);
        genTableHead(tableMeta, ret);
        ret.append(sparateLine);
        for (ColumnMeta columnMeta : tableMeta.columnMetas) {
            genColumn(tableMeta, columnMeta, ret);
        }
        ret.append(sparateLine);
        ret.append("\n");
    }

    
    /**
     * <p>genCell.</p>
     *
     * @param columnMaxLen a int.
     * @param preChar a {@link java.lang.String} object.
     * @param value a {@link java.lang.String} object.
     * @param fillChar a {@link java.lang.String} object.
     * @param postChar a {@link java.lang.String} object.
     * @param ret a {@link java.lang.StringBuilder} object.
     */
    protected void genCell(int columnMaxLen, String preChar, String value, String fillChar, String postChar, StringBuilder ret) {
        ret.append(preChar);
        ret.append(value);
        for (int i = 0, n = columnMaxLen - value.length() + 1; i < n; i++) {
            ret.append(fillChar);    
        }
        ret.append(postChar);
    }

    /**
     * <p>genSeparateLine.</p>
     *
     * @param tm a {@link space.yizhu.record.plugin.activerecord.generator.TableMeta} object.
     * @return a {@link java.lang.String} object.
     */
    protected String genSeparateLine(TableMeta tm) {
        StringBuilder ret = new StringBuilder();
        genCell(tm.colNameMaxLen, "-", "---", "-", "+", ret);
        genCell(tm.colTypeMaxLen, "-", "---", "-", "+", ret);
        genCell("Null".length(), "-", "---", "-", "+", ret);
        genCell("Key".length(), "-", "---", "-", "+", ret);
        genCell(tm.colDefaultValueMaxLen, "-", "---", "-", "+", ret);
        genCell("Remarks".length(), "-", "---", "-", "", ret);
        ret.append("\n");
        return ret.toString();
    }

    /**
     * <p>genTableHead.</p>
     *
     * @param tm a {@link space.yizhu.record.plugin.activerecord.generator.TableMeta} object.
     * @param ret a {@link java.lang.StringBuilder} object.
     */
    protected void genTableHead(TableMeta tm, StringBuilder ret) {
        genCell(tm.colNameMaxLen, " ", "Field", " ", "|", ret);
        genCell(tm.colTypeMaxLen, " ", "Type", " ", "|", ret);
        genCell("Null".length(), " ", "Null", " ", "|", ret);
        genCell("Key".length(), " ", "Key", " ", "|", ret);
        genCell(tm.colDefaultValueMaxLen, " ", "Default", " ", "|", ret);
        genCell("Remarks".length(), " ", "Remarks", " ", "", ret);
        ret.append("\n");
    }

    /**
     * <p>genColumn.</p>
     *
     * @param tableMeta a {@link space.yizhu.record.plugin.activerecord.generator.TableMeta} object.
     * @param columnMeta a {@link space.yizhu.record.plugin.activerecord.generator.ColumnMeta} object.
     * @param ret a {@link java.lang.StringBuilder} object.
     */
    protected void genColumn(TableMeta tableMeta, ColumnMeta columnMeta, StringBuilder ret) {
        genCell(tableMeta.colNameMaxLen, " ", columnMeta.name, " ", "|", ret);
        genCell(tableMeta.colTypeMaxLen, " ", columnMeta.type, " ", "|", ret);
        genCell("Null".length(), " ", columnMeta.isNullable, " ", "|", ret);
        genCell("Key".length(), " ", columnMeta.isPrimaryKey, " ", "|", ret);
        genCell(tableMeta.colDefaultValueMaxLen, " ", columnMeta.defaultValue, " ", "|", ret);
        genCell("Remarks".length(), " ", columnMeta.remarks, " ", "", ret);
        ret.append("\n");
    }

    /**
     * <p>rebuildColumnMetas.</p>
     *
     * @param tableMetas a {@link java.util.List} object.
     */
    protected void rebuildColumnMetas(List<TableMeta> tableMetas) {
        Connection conn = null;
        try {
            conn = dataSource.getConnection();
            DatabaseMetaData dbMeta = conn.getMetaData();
            for (TableMeta tableMeta : tableMetas) {
                
                tableMeta.columnMetas = new ArrayList<ColumnMeta>();
                
                ResultSet rs = dbMeta.getColumns(conn.getCatalog(), null, tableMeta.name, null);
                while (rs.next()) {
                    ColumnMeta columnMeta = new ColumnMeta();
                    columnMeta.name = rs.getString("COLUMN_NAME");            

                    columnMeta.type = rs.getString("TYPE_NAME");            
                    if (columnMeta.type == null) {
                        columnMeta.type = "";
                    }

                    int columnSize = rs.getInt("COLUMN_SIZE");                
                    if (columnSize > 0) {
                        columnMeta.type = columnMeta.type + "(" + columnSize;
                        int decimalDigits = rs.getInt("DECIMAL_DIGITS");    
                        if (decimalDigits > 0) {
                            columnMeta.type = columnMeta.type + "," + decimalDigits;
                        }
                        columnMeta.type = columnMeta.type + ")";
                    }

                    columnMeta.isNullable = rs.getString("IS_NULLABLE");    
                    if (columnMeta.isNullable == null) {
                        columnMeta.isNullable = "";
                    }

                    columnMeta.isPrimaryKey = "   ";
                    String[] keys = tableMeta.primaryKey.split(",");
                    for (String key : keys) {
                        if (key.equalsIgnoreCase(columnMeta.name)) {
                            columnMeta.isPrimaryKey = "PRI";
                            break;
                        }
                    }

                    columnMeta.defaultValue = rs.getString("COLUMN_DEF");    
                    if (columnMeta.defaultValue == null) {
                        columnMeta.defaultValue = "";
                    }

                    columnMeta.remarks = rs.getString("REMARKS");            
                    if (columnMeta.remarks == null) {
                        columnMeta.remarks = "";
                    }

                    if (tableMeta.colNameMaxLen < columnMeta.name.length()) {
                        tableMeta.colNameMaxLen = columnMeta.name.length();
                    }
                    if (tableMeta.colTypeMaxLen < columnMeta.type.length()) {
                        tableMeta.colTypeMaxLen = columnMeta.type.length();
                    }
                    if (tableMeta.colDefaultValueMaxLen < columnMeta.defaultValue.length()) {
                        tableMeta.colDefaultValueMaxLen = columnMeta.defaultValue.length();
                    }

                    tableMeta.columnMetas.add(columnMeta);
                }
                rs.close();
            }
        } catch (SQLException e) {
            throw new RuntimeException(e);
        } finally {
            if (conn != null) {
                try {
                    conn.close();
                } catch (SQLException e) {
                    LogKit.error(e.getMessage(), e);
                }
            }
        }
    }

    
    /**
     * <p>writeToFile.</p>
     *
     * @param ret a {@link java.lang.String} object.
     */
    protected void writeToFile(String ret) {
        File dir = new File(dataDictionaryOutputDir);
        if (!dir.exists()) {
            dir.mkdirs();
        }

        String target = dataDictionaryOutputDir + File.separator + dataDictionaryFileName;
        OutputStreamWriter osw = null;
        try {
            osw = new OutputStreamWriter(new FileOutputStream(target), "UTF-8");
            osw.write(ret);
        } catch (IOException e) {
            throw new RuntimeException(e);
        } finally {
            if (osw != null) {
                try {
                    osw.close();
                } catch (IOException e) {
                    LogKit.error(e.getMessage(), e);
                }
            }
        }
    }
}
